/*==========================================================================
  HWCURSOR.C

  Functions to define, enable, disable, and position the Mach64 hardware
  cursor.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

/* --------------------------------------------------------------------------
  SET_HWCURSOR - define hardware cursor bitmap

  This function sets up the hardware cursor data region according to the
  given bitmap data. The data region is located at (0, y).

  The hardware cursor be vary in size from 1x1 to 64x64. The expected bitmap
  format is LSB to MSB. The LSB will be drawn first in a left to right
  direction. Since the cursor position is NOT set in this routine, the
  position should be set (set_hwcursor_pos) prior to enabling the cursor.
-------------------------------------------------------------------------- */
void set_hwcursor (int y, int width, int height, int hot_x, int hot_y,
                   unsigned long color0, unsigned long color1,
                   unsigned int *bitmap)
{
    unsigned long cur_offset, cur_size, cur_pitch;
    unsigned long temp1, temp2, temp3, temp4, temp5, temp6;
    unsigned long bitdata;
    int i, index, dataindex, start, widthwords, temp;

    // Check that cursor size is within limits
    if ((width < 1) || (width > 64)) return;
    if ((height < 1) || (height > 64)) return;
    if ((hot_x < 0) || (hot_x >= width)) return;
    if ((hot_y < 0) || (hot_y >= height)) return;

    // set cursor dimensions
    cursordata.y = y;
    cursordata.width = width;
    cursordata.height = height;
    cursordata.hot_x = hot_x;
    cursordata.hot_y = hot_y;
    cursordata.color0 = color0;
    cursordata.color1 = color1;

    // set hwcursor bitmap to transparent
    for (index = 0; index < (HWCURHEIGHT * HWCURWIDTH); index++)
    {
        cursordata.bitmap[index] = 0xAAAA;
    }

    // load user hwcursor data into bitmap
    dataindex = 0;
    widthwords = width / 8;
    if (width > widthwords * 8)
    {
        widthwords++;
    }
    start = HWCURWIDTH - widthwords;
    for (index = start; index < (HWCURWIDTH * height); index += HWCURWIDTH)
    {
        i = 0;
        do
        {
            cursordata.bitmap[index + i] = *(bitmap + dataindex);
            if (width < 8)
            {
                cursordata.bitmap[index + i] = cursordata.bitmap[index + i] <<
                                               ((8 - width) * 2);
            }
            dataindex++;
            i++;
        } while (i < widthwords);
    }

    // calculate offset of coordinate (0, y) in qwords
    cursordata.cur_offset = cur_offset = get_xy_offset (0, y) / 8;

    // calculate cursor pitch (assuming 16 bpp)
    cur_pitch = HWCURWIDTH / 8;
    cur_pitch = cur_pitch << 22;

    if (querydata.dac_type == DAC_IBM514)
    {
        // select index registers
        iow8 (ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xFC) | 1);

        // enable auto-increment
        iow8 (ioDAC_REGS + DAC_R_INDEX, 1);

        // set cursor RAM write address to start of cursor array - offset 100h
        iow8 (ioDAC_REGS + DAC_W_INDEX, 0);
        iow8 (ioDAC_REGS + DAC_DATA, 1);

        // write given cursor data to cursor array on chip - same as Mach64
        // hardware cursor interface
        for (index = 0; index < 512; index++)
        {
            // 2 bytes per loop
            temp = cursordata.bitmap[index];
            iow8 (ioDAC_REGS + DAC_MASK, (temp & 0xFF));
            iow8 (ioDAC_REGS + DAC_MASK, ((temp >> 8) & 0xFF));
        }

        // set hot-spot registers
        iow8 (ioDAC_REGS + DAC_W_INDEX, 0x35);
        iow8 (ioDAC_REGS + DAC_DATA, 0);
        iow8 (ioDAC_REGS + DAC_MASK,
              64 - (unsigned char) width + (unsigned char) hot_x);
        iow8 (ioDAC_REGS + DAC_MASK, (unsigned char) hot_y);

        // select default palette registers
        iow8 (ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xFC);
    }
    else // write to video memory for other DACs (Mach64 hardware cursor)
    {
        // ---- Use 16 bpp to setup hardware cursor bitmap ----

        // save vital registers
        wait_for_idle ();
        temp1 = regr (DP_PIX_WIDTH);
        temp2 = regr (DP_CHAIN_MASK);
        temp3 = regr (DST_OFF_PITCH);
        temp4 = regr (DP_SRC);
        temp5 = regr (DP_MIX);
        temp6 = regr (DST_CNTL);

        // load bitmap data to hardware cursor bitmap data area
        wait_for_fifo (10);
        regw (DP_PIX_WIDTH, (temp1 & 0xFFC00000) | BYTE_ORDER_LSB_TO_MSB |
                            HOST_16BPP | SRC_16BPP | DST_16BPP);
        regw (DP_CHAIN_MASK, 0x0410);  // chain mask for 16 bpp
        regw (DST_OFF_PITCH, cur_pitch | cur_offset);
        regw (DP_SRC, FRGD_SRC_HOST);
        regw (DP_MIX, FRGD_MIX_S | BKGD_MIX_D);
        regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw (DST_X, 0);
        regw (DST_Y, 0);
        regw (DST_HEIGHT, HWCURHEIGHT);
        regw (DST_WIDTH, HWCURWIDTH);
        for (index = 0; index < (HWCURHEIGHT * HWCURWIDTH * 2); index += 2)
        {
            wait_for_fifo (1);
            bitdata = (unsigned long) (cursordata.bitmap[index+1]);
            bitdata = (bitdata << 16) | (cursordata.bitmap[index]);
            regw (HOST_DATA0, bitdata);
        }
        wait_for_idle ();

        // set cursor size offsets
        cur_size = (unsigned long) (64 - height);
        cur_size = (unsigned long) ((cur_size << 16) | (64 - width));

        wait_for_fifo (8);

        regw (CUR_HORZ_VERT_OFF, cur_size);

        // set cursor offset to cursor data region
        regw (CUR_OFFSET, cur_offset);

        // restore used registers
        regw (DP_PIX_WIDTH, temp1);
        regw (DP_CHAIN_MASK, temp2);
        regw (DST_OFF_PITCH, temp3);
        regw (DP_SRC, temp4);
        regw (DP_MIX, temp5);
        regw (DST_CNTL, temp6);
    } // end else

    // set cursor colors
    set_cursor_colors (color0, color1);
}

/* --------------------------------------------------------------------------
  ENABLE_HWCURSOR - turn on the hardware cursor
-------------------------------------------------------------------------- */
void enable_hwcursor (void)
{
    // enable hardware cursor
    iow16 (ioGEN_TEST_CNTL, ior16 (ioGEN_TEST_CNTL) | HWCURSOR_ENABLE);

    // special enabling required for IBM514 DAC
    if (querydata.dac_type == DAC_IBM514)
    {
        // access cursor control register
        iow8 (ioDAC_CNTL, (ior8 (ioDAC_CNTL) & 0xFC) | 0x01);
        iow8 (ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8 (ioDAC_REGS + DAC_W_INDEX, 0x30);
        iow8 (ioDAC_REGS + DAC_DATA, 0);                    // index 0x0030
        iow8 (ioDAC_REGS + DAC_MASK, 0x0E);                 // XGA cursor type
        iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);
    }
}

/* --------------------------------------------------------------------------
  DISABLE_HWCURSOR - turn off the hardware cursor
-------------------------------------------------------------------------- */
void disable_hwcursor (void)
{
    // disable hardware cursor
    iow16 (ioGEN_TEST_CNTL, ior16 (ioGEN_TEST_CNTL) & (~HWCURSOR_ENABLE));

    // special disabling required for IBM514 DAC
    if (querydata.dac_type == DAC_IBM514)
    {
        // access cursor control register
        iow8 (ioDAC_CNTL, (ior8 (ioDAC_CNTL) & 0xFC) | 0x01);
        iow8 (ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8 (ioDAC_REGS + DAC_W_INDEX, 0x30);
        iow8 (ioDAC_REGS + DAC_DATA, 0);                    // index 0x0030
        iow8 (ioDAC_REGS + DAC_MASK, 0);                    // disable
        iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);
    }
}

/* --------------------------------------------------------------------------
  SET_HWCURSOR_POS - set the hardware cursor position
-------------------------------------------------------------------------- */
void set_hwcursor_pos (int x, int y)
{
    unsigned long cur_pos, cur_size, cur_offset;

    // check for coordinate violations
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    cursordata.current_x = x;
    cursordata.current_y = y;

    if (querydata.dac_type == DAC_IBM514)
    {
        // IBM514 DAC
        cur_pos = (unsigned long) (y);
        cur_pos = (unsigned long) ((cur_pos << 16) | x);
        iow8 (ioDAC_CNTL, (ior8 (ioDAC_CNTL) & 0xFC) | 0x01);
        iow8 (ioDAC_REGS + DAC_R_INDEX, 1);                 // auto-increment
        iow8 (ioDAC_REGS + DAC_W_INDEX, 0x31);
        iow8 (ioDAC_REGS + DAC_DATA, 0);                    // index = 0x31

        // X low
        iow8 (ioDAC_REGS + DAC_MASK, (unsigned char) (cur_pos & 0xFF));

        // X high
        iow8 (ioDAC_REGS + DAC_MASK, (unsigned char) ((cur_pos >> 8) & 0xFF));

        // Y low
        iow8 (ioDAC_REGS + DAC_MASK, (unsigned char) ((cur_pos >> 16) & 0xFF));

        // Y high
        iow8 (ioDAC_REGS + DAC_MASK, (unsigned char) ((cur_pos >> 24) & 0xFF));

        iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);
    }
    else
    {
        // Set the cursor position such that the (x, y) coordinates equal the
        // hot spot.
        if (y < cursordata.hot_y)
        {
            // adjust CUR_OFFSET to point to (hot_y - y) lines into the
            // cursor data region. Also increase CUR_VERT_OFFSET by the
            // same number of lines that CUR_OFFSET was adjusted by. Since
            // these two registers are being used to set the cursor position,
            // set CUR_VERT_POSN to zero.
            cur_size = (unsigned long) (64 - cursordata.height +
                                        cursordata.hot_y - y);
            cur_pos = 0;
            cur_offset = cursordata.cur_offset +
                         (unsigned long) ((cursordata.hot_y - y) * 2);
        }
        else
        {
            // Apply normal cursor defining parameters when y >= hot_y
            cur_size = (unsigned long)(64 - cursordata.height);
            cur_pos = (unsigned long)(y - cursordata.hot_y);
            cur_offset = cursordata.cur_offset;
        }

        // Shift up Y quantities by 16 bits as they are in the upper word for
        // each cursor register.
        cur_size = cur_size << 16;
        cur_pos = cur_pos << 16;

        if (x < cursordata.hot_x)
        {
            // Set the cursor position by adding (hot_x - x) cursor pixels to
            // CUR_HORZ_OFFSET. As in the Y coordinate case, set
            // CUR_HORZ_POSN to zero.
            cur_size = (unsigned long) (cur_size | (64 - cursordata.width +
                       cursordata.hot_x - x));
        }
        else
        {
            // Apply normal cursor defining parameters when x >= hot_x.
            cur_size = (unsigned long) (cur_size | (64 - cursordata.width));
            cur_pos = (unsigned long) (cur_pos | (x - cursordata.hot_x));
        }

        // Write cursor positioning parameters to the cursor registers.
        regw (CUR_OFFSET, cur_offset);
        regw (CUR_HORZ_VERT_OFF, cur_size);
        regw (CUR_HORZ_VERT_POSN, cur_pos);
    }
}

/* --------------------------------------------------------------------------
  get_hwcursor_pos - get the hardware cursor position
-------------------------------------------------------------------------- */
void get_hwcursor_pos (POINT *position)
{
    position->x = cursordata.current_x;
    position->y = cursordata.current_y;
}

/* --------------------------------------------------------------------------
  set_cursor_colors - set hardware cursor colors

  This function is called exclusively by set_hwcursor().
-------------------------------------------------------------------------- */
void set_cursor_colors (unsigned long color0, unsigned long color1)
{
    unsigned long curclr0, curclr1;
    unsigned long red0, green0, blue0, red1, green1, blue1;
    unsigned long rshift, gshift, bshift;
    unsigned long temp;
    int index;
    PALETTE entry;

    // setup cursor colors for 4, 8 bpp modes
    if ((modeinfo.bpp == 4) || (modeinfo.bpp == 8))
    {
        // get color componets from palette using color indices
        if ((querydata.dac_type == DAC_ATI68860_1) ||
            (querydata.dac_type == DAC_ATI68860_2) ||
            (querydata.dac_type == DAC_IBM514) ||
            (ior16 (ioCONFIG_CHIP_ID) == CHIP_CT_ID))
        {
            // ensure palette registers are available
            iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);

            // Cursor colors are 8 bit values, palette colors are 6 bit values
            entry = get_palette ((int) (color0 & 0xFF));
            red0 = (unsigned long) (entry.red) << 2;
            green0 = (unsigned long) (entry.green) << 2;
            blue0 = (unsigned long) (entry.blue) << 2;
            entry = get_palette ((int) (color1 & 0xFF));
            red1 = (unsigned long) (entry.red) << 2;
            green1 = (unsigned long) (entry.green) << 2;
            blue1 = (unsigned long) (entry.blue) << 2;

            // assign cursor color values for Mach64 CT internal DACs
            if ((ior16 (ioCONFIG_CHIP_ID) == CHIP_CT_ID))
            {
                curclr0 = (red0 << 24) | (green0 << 16) | (blue0 << 8) | color0;
                curclr1 = (red1 << 24) | (green1 << 16) | (blue1 << 8) | color1;
            }
        }
        else // use color indices for standard DACs
        {
            curclr0 = (int) (color0 & 0xFF);
            curclr1 = (int) (color1 & 0xFF);
        }
    }
    else // setup cursor color for 15, 16, 24, 32 bpp modes
    {
        // determine color component shifting values
        switch (modeinfo.bpp)
        {
            case 16:
                if (modeinfo.depth == 555)
                {
                    rshift = 3;
                    gshift = 3;
                    bshift = 3;
                }
                else  // 565 weight
                {
                    rshift = 3;
                    gshift = 2;
                    bshift = 3;
                }
                break;

            case 24:
            case 32:
                rshift = 0;
                gshift = 0;
                bshift = 0;
                break;
        }

        // get cursor color 0 components
        red0 = get_primary_color (RED, color0) << rshift;
        green0 = get_primary_color (GREEN, color0) << gshift;
        blue0 = get_primary_color (BLUE, color0) << bshift;

        // get cursor color 1 components
        red1 = get_primary_color (RED, color1) << rshift;
        green1 = get_primary_color (GREEN, color1) << gshift;
        blue1 = get_primary_color (BLUE, color1) << bshift;

        // special case for Brooktree DAC in 24 (BGR) bpp mode
        if ((modeinfo.bpp == 24) &&
            ((querydata.color_depth_support & 0x8) == 0x8) &&
            (querydata.dac_type == DAC_BT481))
        {
            // swap RED and BLUE components
            temp = blue0;
            blue0 = red0;
            red0 = temp;
            temp = blue1;
            blue1 = red1;
            red1 = temp;
        }

        // assign cursor color values for standard DACs
        if ((querydata.dac_type != DAC_ATI68860_1) &&
            (querydata.dac_type != DAC_ATI68860_2) &&
            (querydata.dac_type != DAC_IBM514))
        {
            curclr0 = (red0 << 24) | (green0 << 16) | (blue0 << 8);
            curclr1 = (red1 << 24) | (green1 << 16) | (blue1 << 8);
        }
    }

    // write to hardware to set cursor colors for ATI68860,TVP3026,IBM514 DACs
    if ((querydata.dac_type == DAC_ATI68860_1) ||
        (querydata.dac_type == DAC_ATI68860_1) ||
        (querydata.dac_type == DAC_IBM514))
    {
        // special setup for ATI68860, TVP3026, and IBM514 DACs for cursor
        // colors

        // access cursor color registers
        iow8 (ioDAC_CNTL, (ior8 (ioDAC_CNTL) & 0xFC) | 0x01);

        // select 1st cursor color register set
        switch (querydata.dac_type)
        {
            case DAC_ATI68860_1:
            case DAC_ATI68860_2:
                iow8 (ioDAC_REGS + DAC_W_INDEX, 0);
                index = DAC_DATA;
                break;

            case DAC_IBM514:
                iow8 (ioDAC_REGS + DAC_R_INDEX, 1);     // auto-increment
                iow8 (ioDAC_REGS + DAC_W_INDEX, 0x40);
                iow8 (ioDAC_REGS + DAC_DATA, 0);        // index 0x40
                index = DAC_MASK;
                break;
        }

        // set 1st cursor color - 8 bit values required
        iow8 (ioDAC_REGS + index, (unsigned char) red0);
        iow8 (ioDAC_REGS + index, (unsigned char) green0);
        iow8 (ioDAC_REGS + index, (unsigned char) blue0);

        // set 2nd cursor color - 8 bit values required
        iow8 (ioDAC_REGS + index, (unsigned char) red1);
        iow8 (ioDAC_REGS + index, (unsigned char) green1);
        iow8 (ioDAC_REGS + index, (unsigned char) blue1);

        // return DAC register set to palette registers
        iow8 (ioDAC_CNTL, ior8 (ioDAC_CNTL) & 0xFC);
    }
    else // write to hardware to set cursor colors for standard DACs
    {
        regw (CUR_CLR0, curclr0);
        regw (CUR_CLR1, curclr1);
    }
}
