/*==========================================================================
  INIT.C

  MACH 64 functions to initialize and set a standard engine context.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

/* --------------------------------------------------------------------------
  detect_mach64 - determine if a mach64 based video adapter is installed.

  This routine identifies if a mach64 based video adapter is installed. This
  is done by writing and reading the SCRATCH_REG0 or SCRATCH_REG1 io based
  registers. These registers must be saved and restored since the mach64
  BIOS uses their contents.

  Returns YES_MACH64 if detected, NO_MACH64 if not.
-------------------------------------------------------------------------- */
int detect_mach64 (int instance)
{
    unsigned long save_value;
    int result, index;
    int match = 0;
    unsigned int rom_segment[25] =
    {
        0xC000, 0xC800, 0xC900, 0xCA00, 0xCB00, 0xCC00, 0xCD00, 0xCE00,
        0xCF00, 0xD000, 0xD100, 0xD200, 0xD300, 0xD400, 0xD500, 0xD600,
        0xD700, 0xD800, 0xD900, 0xDA00, 0xDB00, 0xDC00, 0xDD00, 0xDE00,
        0xDF00
    };

    // assume failure
    result = NO_MACH64;

    // find ATI mach64 ROM
    for (index = 0; index < 25; index ++)
    {
        if (is_mach64_rom (rom_segment[index]))
        {
            if (match == instance)
            {
                instanceinfo.rom_segment = rom_segment[index];
                instanceinfo.instance = instance;
                break;
            }
            match++;
        }
    }

    if (instanceinfo.rom_segment == 0)
    {
        return (NO_MACH64);
    }

    // set ROM segment for ROM functions
    set_rom_base (instanceinfo.rom_segment);

    // attempt to get base IO address and type. If unsuccessful,
    // assume the default case
    if (!get_io_base (&iobaseinfo))
    {
        iobaseinfo.io_base = 0x2EC;
        iobaseinfo.io_type = 0;
    }

    // setup and initialize IO functions according to IO base address
    // and type
    init_io_reg (iobaseinfo.io_base, iobaseinfo.io_type);

    // save old value
    save_value = ior32 (ioSCRATCH_REG0);

    // test odd bits for readability
    iow32 (ioSCRATCH_REG0, 0x55555555);
    if (ior32 (ioSCRATCH_REG0) == 0x55555555)
    {
        // test even bits for readability
        iow32 (ioSCRATCH_REG0, 0xAAAAAAAA);
        if (ior32 (ioSCRATCH_REG0) == 0xAAAAAAAA)
        {
            result = YES_MACH64;
        }
    }

    // restore old value
    iow32 (ioSCRATCH_REG0, save_value);

    // place code to check chip revisions here. watch out for rev E

    return (result);
}

/* --------------------------------------------------------------------------
  INIT_4BPP - set the MACH64 engine to a 4bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_4bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_4BPP | SRC_4BPP | DST_4BPP |
                        BYTE_ORDER_MSB_TO_LSB);
    regw (DP_CHAIN_MASK, 0x8888);
}

/* --------------------------------------------------------------------------
  INIT_8BPP - set the MACH64 engine to a 8bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_8bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_8BPP | SRC_8BPP | DST_8BPP |
                        BYTE_ORDER_LSB_TO_MSB);
    regw (DP_CHAIN_MASK, 0x8080);
}

/* --------------------------------------------------------------------------
  INIT_15BPP - set the MACH64 engine to a 15bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_15bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_15BPP | SRC_15BPP | DST_15BPP |
                        BYTE_ORDER_LSB_TO_MSB);
    regw (DP_CHAIN_MASK, 0x4210);
}

/* --------------------------------------------------------------------------
  INIT_16BPP - set the MACH64 engine to a 16bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_16bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_16BPP | SRC_16BPP | DST_16BPP |
                        BYTE_ORDER_LSB_TO_MSB);
    regw (DP_CHAIN_MASK, 0x8410);
}

/* --------------------------------------------------------------------------
  INIT_24BPP - set the MACH64 engine to a 24bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_24bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_8BPP | SRC_8BPP | DST_8BPP |
                        BYTE_ORDER_LSB_TO_MSB);
    regw (DP_CHAIN_MASK, 0x8080);
}

/* --------------------------------------------------------------------------
  INIT_32BPP - set the MACH64 engine to a 32bpp standard context.

  This routine is used in conjuction with INIT_ENGINE ().
-------------------------------------------------------------------------- */
void init_32bpp (void)
{
    wait_for_fifo (2);
    regw (DP_PIX_WIDTH, HOST_32BPP | SRC_32BPP | DST_32BPP |
                        BYTE_ORDER_LSB_TO_MSB);
    regw (DP_CHAIN_MASK, 0x8080);
}

/* --------------------------------------------------------------------------
  RESET_ENGINE - reset engine and clear any FIFO errors

  This function resets the GUI engine and clears any FIFO errors.
-------------------------------------------------------------------------- */
void reset_engine (void)
{
    // reset engine
    iow32 (ioGEN_TEST_CNTL, (ior32 (ioGEN_TEST_CNTL) & 0xFFFFFEFF));

    // enable engine
    iow32 (ioGEN_TEST_CNTL, (ior32 (ioGEN_TEST_CNTL) | GUI_ENGINE_ENABLE));

    // ensure engine is not locked up by clearing any FIFO or HOST errors
    iow32 (ioBUS_CNTL, (ior32 (ioBUS_CNTL) & 0xFF00FFFF) | 0x00AE0000);
}

/* --------------------------------------------------------------------------
  INIT_ENGINE - set the MACH64 engine to a standard context.

  This routine configures the MACH64 engine to a known typical context.
  The context consists of:

     -engine reset and enabling

     -fifo control including clearing fifo overflow errors

     -source and destination pitch

     -source and destination offset into video memory

     -color depth

     -host data control

     -pattern data control

     -line and rectangle control

     -color source, mix, and compare control

     -scissor settings

     -write mask

-------------------------------------------------------------------------- */
void init_engine (void)
{
    unsigned long pitch_value, xres, yres;

    // determine modal information from global mode structure
    xres = (unsigned long) (modeinfo.xres);
    yres = (unsigned long) (modeinfo.yres);
    pitch_value = (unsigned long) (modeinfo.pitch);
    if (modeinfo.bpp == 24)
    {
        // In 24 bpp, the engine is in 8 bpp - this requires that all
        // horizontal coordinates and widths must be adjusted
        pitch_value = pitch_value * 3;
    }

    // Reset engine, enable, and clear any engine errors
    reset_engine ();

    // Ensure that vga page pointers are set to zero - the upper page
    // pointers are set to 1 to handle overflows in the lower page
    iow32 (ioMEM_VGA_WP_SEL, 0x00010000);
    iow32 (ioMEM_VGA_RP_SEL, 0x00010000);

    // ---- Setup standard engine context ----

    // All GUI registers here are FIFOed - therefore, wait for the
    // appropriate number of empty FIFO entries
    wait_for_fifo (14);

    // enable all registers to be loaded for context loads
    regw (CONTEXT_MASK, 0xFFFFFFFF);

    // set destination pitch to modal pitch, set offset to zero
    regw (DST_OFF_PITCH, (pitch_value / 8) << 22);

    // zero these registers (set them to a known state)
    regw (DST_Y_X, 0);
    regw (DST_HEIGHT, 0);
    regw (DST_BRES_ERR, 0);
    regw (DST_BRES_INC, 0);
    regw (DST_BRES_DEC, 0);

    // set destination drawing attributes
    regw (DST_CNTL, DST_LAST_PEL | DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);

    // set source pitch to modal pitch, set offset to zero
    regw (SRC_OFF_PITCH, (pitch_value / 8) << 22);

    // zero these registers (set them to a known state)
    regw (SRC_Y_X, 0);
    regw (SRC_HEIGHT1_WIDTH1, 0);
    regw (SRC_Y_X_START, 0);
    regw (SRC_HEIGHT2_WIDTH2, 0);

    // set source pixel retrieving attributes
    regw (SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT);

    // set host attributes
    wait_for_fifo (13);
    regw (HOST_CNTL, 0);

    // set pattern attributes
    regw (PAT_REG0, 0);
    regw (PAT_REG1, 0);
    regw (PAT_CNTL, 0);

    // set scissors to modal size
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_BOTTOM, yres-1);
    regw (SC_RIGHT, pitch_value-1);

    // set background color to minimum value (usually BLACK)
    regw (DP_BKGD_CLR, 0);

    // set foreground color to maximum value (usually WHITE)
    regw (DP_FRGD_CLR, 0xFFFFFFFF);

    // set write mask to effect all pixel bits
    regw (DP_WRITE_MASK, 0xFFFFFFFF);

    // set foreground mix to overpaint and background mix to no-effect
    regw (DP_MIX, FRGD_MIX_S | BKGD_MIX_D);

    // set primary source pixel channel to foreground color register
    regw (DP_SRC, FRGD_SRC_FRGD_CLR);

    // set compare funtionality to false (no-effect on destination)
    wait_for_fifo (3);
    regw (CLR_CMP_CLR, 0);
    regw (CLR_CMP_MASK, 0xFFFFFFFF);
    regw (CLR_CMP_CNTL, 0);

    // set pixel depth
    switch (modeinfo.bpp)
    {
        case 4 :
            init_4bpp ();
            break;

        case 8 :
            init_8bpp ();
            break;

        case 16:
            if (modeinfo.depth == 555)
            {
                init_15bpp ();   // 555 color weighting
            }
            else
            {
                init_16bpp ();   // 565 color weighting
            }
            break;

        case 24:
            init_24bpp ();
            break;

        case 32:
            init_32bpp ();
            break;
    }

    wait_for_idle ();      // insure engine is idle before leaving
}

/* --------------------------------------------------------------------------
  OPEN_MODE - set an accelerator mode for the Mach 64

  This function sets an accelerator mode, initializes a global mode
  structure, and determines the access method for the GUI memory mapped
  registers of the Mach 64. This function should be called before any other
  engine function since they depend on the global mode information structure
  that is filled by this function. It is assumed that a Mach 64 based video
  adapter has been detected before calling this function (detect_mach64()).

  Inputs :

    Mode code:  MODE_640x480,
                MODE_800x600,
                MODE_1024x768,
                MODE_1280x1024,
                MODE_1600x1200

    Pitch code: PITCH_1024,
                PITCH_XRES

    Color code: COLOR_DEPTH_4,    // palettized 4 bpp
                COLOR_DEPTH_8,    // palettized 8 bpp
                COLOR_DEPTH_15,   // direct color 15 bpp - 555 weighting
                COLOR_DEPTH_16,   // direct color 16 bpp - 565 weighting
                COLOR_DEPTH_24,   // direct color 24 bpp - RGB weighting
                COLOR_DEPTH_32    // direct color 32 bpp - RGBa weighting

  Returns:

    NO_ERROR      - the accelerator mode has been set successfully
    YES_ERROR     - an error has occurred
    NOT_SUPPORTED - the accelerator mode is not supported
-------------------------------------------------------------------------- */
int open_mode (int mode_code, int pitch_code, int color_code)
{
    int retval;

    // Handle cases where mode is not supported due to hardware
    // restrictions before continuing:

    // 4 bpp modes are not supported on ATI68860_1 DACs
    if ((querydata.dac_type == DAC_ATI68860_1) && (color_code == COLOR_DEPTH_4))
        return (YES_ERROR);

    // Setup apertures
    if (init_aperture () == APERTURE_ENABLE_FAILURE)
        return (YES_ERROR);

    update_aperture_status ();

    // Insure that memory is shared between VGA and acelerator for full
    // memory access
    mem_cntl = ior16 (ioMEM_CNTL+2);
    iow16 (ioMEM_CNTL+2, 0);

    // attempt to set an accelerator mode
    retval = load_and_set_mode (mode_code, pitch_code, color_code);
    if (retval == NO_ERROR)
    {
        // Fill modal information structure
        modeinfo.depth = 0;
        switch (mode_code)
        {
            case MODE_640x480:
                modeinfo.xres = 640;
                modeinfo.yres = 480;
                break;

            case MODE_800x600:
                modeinfo.xres = 800;
                modeinfo.yres = 600;
                break;

            case MODE_1024x768:
                modeinfo.xres = 1024;
                modeinfo.yres = 768;
                break;

            case MODE_1280x1024:
                modeinfo.xres = 1280;
                modeinfo.yres = 1024;
                break;

            case MODE_1600x1200:
                modeinfo.xres = 1600;
                modeinfo.yres = 1200;
                break;
        } // end switch mode_code
        switch (pitch_code)
        {
            case PITCH_1024:
                modeinfo.pitch = 1024;
                break;

            case PITCH_XRES:
                modeinfo.pitch = modeinfo.xres;
                break;
        } // end switch pitch_code
        switch (color_code)
        {
            case COLOR_DEPTH_4:
                modeinfo.bpp = 4;
                break;

            case COLOR_DEPTH_8:
                modeinfo.bpp = 8;
                break;

            case COLOR_DEPTH_15:
                modeinfo.bpp = 16;
                modeinfo.depth = DEPTH_SUPPORT_555;
                break;

            case COLOR_DEPTH_16:
                modeinfo.bpp = 16;
                modeinfo.depth = DEPTH_SUPPORT_565;
                break;

            case COLOR_DEPTH_24:
                modeinfo.bpp = 24;
                // Set support for the R,G,and B ordering according to
                // the color depth supported or according to precedence
                // if more than one format is supported.
                if (querydata.color_depth_support & DEPTH_SUPPORT_RGB)
                    modeinfo.depth = DEPTH_SUPPORT_RGB;
                else
                    modeinfo.depth = DEPTH_SUPPORT_BGR;
                break;

            case COLOR_DEPTH_32:
                modeinfo.bpp = 32;
                // Set support for the R,G,B, and A ordering according to
                // the color depth supported or according to precedence
                // if more than one format is supported.
                if (querydata.color_depth_support & DEPTH_SUPPORT_RGBA)
                    modeinfo.depth = DEPTH_SUPPORT_RGBA;
                else if (querydata.color_depth_support & DEPTH_SUPPORT_ARGB)
                    modeinfo.depth = DEPTH_SUPPORT_ARGB;
                else if (querydata.color_depth_support & DEPTH_SUPPORT_BGRA)
                    modeinfo.depth = DEPTH_SUPPORT_BGRA;
                else
                    modeinfo.depth = DEPTH_SUPPORT_ABGR;
                break;
        } // end switch color_code

        // Insure that engine is clear of any idle or fifo errors
        reset_engine ();

        // Setup palette if 4 or 8 bpp mode
        if ((modeinfo.bpp == 4) || (modeinfo.bpp == 8))
        {
            if (ior16 (ioCONFIG_CHIP_ID) == CHIP_CT_ID)
            {
                // bit must be cleared
                iow8 (ioDAC_CNTL + 1, ior8(ioDAC_CNTL + 1) & 0xFE);
            }
            // Save the default palette.
            save_palette (savepalette);
            // Initialize the new palette.
            init_palette ();
        }
        else  // for palettized 15, 16, 24, and 32 bpp modes (chip CT case)
        {
            if (ior16 (ioCONFIG_CHIP_ID) == CHIP_CT_ID)
            {
                // bit must be set to enable 8 bit DAC operation
                iow8 (ioDAC_CNTL + 1, ior8 (ioDAC_CNTL + 1) | 0x01);
                // Save the default palette.
                save_palette (savepalette);
                // Initialize the new palette.
                init_palettized ();
            } // end if
        } // end else
    } // end if NO_ERROR
    else
    {
        // Insure VGA apeture is disabled for proper operation of VGA
        // controller (in text mode) before leaving
        disable_vga_aperture ();
    }

    return (retval);
}

/* --------------------------------------------------------------------------
  CLOSE_MODE - switch back to VGA mode from an accelerator mode

  This function closes a Mach 64 accelerator mode and switches the display
  back to VGA control.
-------------------------------------------------------------------------- */
void close_mode (void)
{
    union REGS regs;

    // Insure DAC is set to 6 bit operation if chip is a CT.
    if (ior16 (ioCONFIG_CHIP_ID) == CHIP_CT_ID)
    {
        iow8 (ioDAC_CNTL + 1, ior8 (ioDAC_CNTL + 1) & 0xFE);
    }

    // Disable accelerator mode and go back to VGA mode
    set_display_mode (VGA_MODE);

    // Restore the default palette if it was modified.
    if ((modeinfo.bpp == 4) || (modeinfo.bpp == 8) ||
        (ior16(ioCONFIG_CHIP_ID) == CHIP_CT_ID))
        restore_palette (savepalette);

    // Disable VGA aperture BEFORE reinitializing the VGA controller
    disable_vga_aperture ();

    // Since the accelerator and VGA controllers share the same video memory,
    // it is necessary to reinitialize the VGA text mode before exiting. Text
    // mode 3 is used here.
    regs.h.ah = 0x00;
    regs.h.al = 0x03;
    int86 (0x10, &regs, &regs);

    // restore MEM_CNTL register
    iow16 (ioMEM_CNTL+2, mem_cntl);
}

/* --------------------------------------------------------------------------
  ENABLE_VGA_APERTURE - enable the vga aperture

  This function enables the VGA aperture and sets the modeinfo
  structure contents accordingly. The sample code functions will use the new
  settings.
-------------------------------------------------------------------------- */
int enable_vga_aperture(void)
{
    iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) | VGA_APERTURE_ENABLE);
    if ((ior16 (ioCONFIG_CNTL) & VGA_APERTURE_ENABLE) == VGA_APERTURE_ENABLE)
    {
        modeinfo.vga_aperture_status = VGA_APERTURE_ENABLED;
        return (APERTURE_VGA_ENABLE_SUCCESS);
    }
    else
    {
        return (APERTURE_ENABLE_FAILURE);
    }
}

/* --------------------------------------------------------------------------
  DISABLE_VGA_APERTURE - disable the vga aperture

  This function disables the VGA aperture and sets the modeinfo
  structure contents accordingly. The sample code functions will use the new
  settings. It is important to note that if both the linear aperture and
  the vga aperture are disabled, the sample code functions will fail to
  operate correctly. It is the application's responsibility to insure that
  this does not occur.
-------------------------------------------------------------------------- */
void disable_vga_aperture(void)
{
    iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) & (~VGA_APERTURE_ENABLE));
    modeinfo.vga_aperture_status = VGA_APERTURE_DISABLED;
}

/* --------------------------------------------------------------------------
  ENABLE_LINEAR_APERTURE - enable the linear aperture

  This function enables the linear aperture for 4M or 8M. If the system bus
  type is PCI, the aperture will only be enabled for 8M. The global query
  structure must be filled prior to calling this function in order to
  determine the bus type. The linear aperture address is set by the ROM at
  boot time. The address is assumed to be correct. The sample code functions
  will use the new settings.
-------------------------------------------------------------------------- */
int enable_linear_aperture(int aperture_size)
{
    int result;

    // assume failure in case of invalid aperture size flag
    result = APERTURE_ENABLE_FAILURE;

    // attempt to set the aperture size
    if (aperture_size == APERTURE_4M_ENABLE)  // 4M size
    {
        // attempt to set a 4MB aperture
        iow16 (ioCONFIG_CNTL,
               (ior16 (ioCONFIG_CNTL) & 0xFFFC) | APERTURE_4M_ENABLE);
        // if successful, update modeinfo structure
        if ((ior16 (ioCONFIG_CNTL) & APERTURE_4M_ENABLE) == APERTURE_4M_ENABLE)
        {
            modeinfo.linear_memreg_offset = 0x3FFC00;
            modeinfo.linear_aperture_size = APERTURE_4M_ENABLE;
            result = APERTURE_4M_ENABLE_SUCCESS;
        }
        else
        {
            result = APERTURE_ENABLE_FAILURE;
        }
    }
    else if (aperture_size == APERTURE_8M_ENABLE) // 8M size
    {
        // attempt to set an 8MB aperture
        iow16 (ioCONFIG_CNTL,
               (ior16 (ioCONFIG_CNTL) & 0xFFFC) | APERTURE_8M_ENABLE);
        // if successful, update modeinfo structure
        if ((ior16 (ioCONFIG_CNTL) & APERTURE_8M_ENABLE) == APERTURE_8M_ENABLE)
        {
            modeinfo.linear_memreg_offset = 0x7FFC00;
            modeinfo.linear_aperture_size = APERTURE_8M_ENABLE;
            result = APERTURE_8M_ENABLE_SUCCESS;
        }
        else
        {
            result = APERTURE_ENABLE_FAILURE;
        }
    }

    return (result);
}

/* --------------------------------------------------------------------------
  DISABLE_LINEAR_APERTURE - disable the linear aperture

  This function disables the linear aperture. It is important to note that
  if both the linear aperture and the vga aperture are disabled, the PGL
  functions will fail to operate correctly. It is the application's
  responsibility to insure that this does not occur.
-------------------------------------------------------------------------- */
void disable_linear_aperture (void)
{
    iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) &
                          (~(APERTURE_4M_ENABLE | APERTURE_8M_ENABLE)));
}

/* --------------------------------------------------------------------------
  INIT_APERTURE - set the apertures according to Mach 64 hardware

  This function selects the aperture according to the installed hardware
  for performance and functionality. This function is called by open_mode()
  each time a new mode is set.
-------------------------------------------------------------------------- */
int init_aperture (void)
{
    // If the VGA controller is enabled, use VGA aperture. If it is disabled,
    // use the linear aperture.
    if (querydata.vga_type == VGA_ENABLE)
    {
        // Enable VGA aperture
        iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) | VGA_APERTURE_ENABLE);
        // if successful, return. otherwise, attempt to set a 4MB aperture
        if ((ior16 (ioCONFIG_CNTL) & VGA_APERTURE_ENABLE) ==
            VGA_APERTURE_ENABLE)
        {
            return APERTURE_VGA_ENABLE_SUCCESS;
        }
    }

    // attepmt to enable a 4M linear aperture
    iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) | APERTURE_4M_ENABLE);
    // if successful, return. otherwise, attempt to set an 8MB aperture
    if ((ior16 (ioCONFIG_CNTL) & APERTURE_4M_ENABLE) == APERTURE_4M_ENABLE)
        return APERTURE_4M_ENABLE_SUCCESS;

    // attepmt to enable an 8M linear aperture
    iow16 (ioCONFIG_CNTL, ior16 (ioCONFIG_CNTL) | APERTURE_8M_ENABLE);
    // if successful, return success code. Otherwise, return failure code
    if ((ior16 (ioCONFIG_CNTL) & APERTURE_8M_ENABLE) == APERTURE_8M_ENABLE)
        return APERTURE_8M_ENABLE_SUCCESS;

    return APERTURE_ENABLE_FAILURE;
}

/* --------------------------------------------------------------------------
  UPDATE_APERTURE_STATUS - update aperture status for PGL register access

  This function is always called when a new mode is set through open_mode().
  The aperture status information in the MODECFG structure is updated
  according the current aperture settings of the Mach 64 hardware. This
  function is also useful if the aperture settings change during an active
  session. The aperture status information is used by the sample functions to
  determine the method of access for the Mach 64 memory mapped registers.
-------------------------------------------------------------------------- */
void update_aperture_status (void)
{
    int config1, config2;

    /* ---- determine status of VGA and linear apertures ---- */
    config1 = ior16 (ioCONFIG_CNTL);
    config2 = ior16 (ioCONFIG_CNTL + 2);

    // determine addresses and offsets for linear aperture
    modeinfo.aperture_address = (unsigned long) (config1 & 0x3FF0);
    modeinfo.aperture_address = modeinfo.aperture_address << 18;
    if ((config1 & APERTURE_4M_ENABLE) == APERTURE_4M_ENABLE)
    {
        // 4M size
        modeinfo.linear_memreg_offset = 0x3FFC00;
        modeinfo.linear_aperture_size = APERTURE_4M_ENABLE;
    }
    else
    {
        // 8M size
        modeinfo.linear_memreg_offset = 0x7FFC00;
        modeinfo.linear_aperture_size = APERTURE_8M_ENABLE;
    }

    // determine linear aperture status and address
    if ((config1 & 3) != 0)
    {
        modeinfo.linear_aperture_status = LINEAR_APERTURE_ENABLED;
    }
    else
    {
        modeinfo.linear_aperture_status = LINEAR_APERTURE_DISABLED;
    }

    // determine addresses and offsets for vga aperture
    modeinfo.vga_memreg_offset = 0xB000FC00;    // segment:offset

    // determine vga aperture status
    if ((config1 & VGA_APERTURE_ENABLE) == VGA_APERTURE_ENABLE)
    {
        modeinfo.vga_aperture_status = VGA_APERTURE_ENABLED;
    }
    else
    {
        modeinfo.vga_aperture_status = VGA_APERTURE_DISABLED;
    }
}

/* --------------------------------------------------------------------------
  QUERY_HARDWARE - query the Mach 64 hardware and fill a global query
                   structure

  This function calls the Mach 64 ROM to fill a global query structure that
  is used by other functions. It should be called before using any other
  function including init_aperture() and open_mode(). The return code should
  be checked before proceeding.

  Returns:

    NO_ERROR      - the query structure has been successfully filled
    YES_ERROR     - an error has occurred
-------------------------------------------------------------------------- */
int query_hardware (void)
{
    int structure_size, error;
    unsigned char far *ptr;
    QUERY_STRUCTURE far *query_ptr;

    // assume error
    error = YES_ERROR;

    // call the BIOS query functions to obtain its size and contents
    structure_size = get_query_size (HEADER_ONLY);
    if (structure_size != 0)
    {
        // allocate memory buffer for query structure
        ptr = (unsigned char far *) malloc (structure_size);
        if (ptr != NULL)
        {
            // call the BIOS to fill memory
            if (fill_query_structure (HEADER_ONLY, ptr) == NO_ERROR)
            {
                // fill global query structure from memory buffer
                query_ptr = (QUERY_STRUCTURE far *) (ptr);
                memcpy (&querydata, query_ptr, sizeof (QUERY_STRUCTURE));

                // set no error condition
                error = NO_ERROR;
            }
            free (ptr);
        }
    }

    return (error);
}

/* --------------------------------------------------------------------------
  SET_PACKED_PIXEL - set VGA controller into packed pixel mode

  The VGA controller is normally set in planar mode. Data transfer
  through the VGA aperture (low and high 32K pages) requires that the
  VGA controller be set in a packed pixel mode where the pixel data
  is arranged contigiously.
-------------------------------------------------------------------------- */
void set_packed_pixel (void)
{
    union REGS regs;

    if (ior16 (ioCONFIG_CHIP_ID) != CHIP_CT_ID)
    {
        // GX, CX:  Both of these chips are still VGAWonder compatible, so
        // we can set the VGA controller into packed pixel mode by setting
        // ATI SuperVGA mode 62h.
        regs.x.ax = 0x62;
        int86 (0x10, &regs, &regs);

        // set VGA controller space to 128k (A0000h-BFFFFh) - this is necessary
        // to allow access to GUI registers in VGA space (BFC00h) since setting
        // mode 62h sets VGA controller space to 64K (A0000h-AFFFFh).
        outpw (0x3CE, 0x0106);      // graf06 = 01  (128K, graphics mode)
    }
    else
    {
        // CT:  This chip is not VGAWonder compatible, so we have to set the
        // VGA controller into packed pixel mode by setting VGA standard
        // mode 13h and manually reprogram the other VGA registers.
        regs.x.ax = 0x13;
        int86 (0x10, &regs, &regs);

        // set VGA controller into packed pixel mode with 128K (A0000h-BFFFFh)
        // - this is necessary to allow access to GUI registers in VGA space
        // (BFC00h) since setting mode 13h sets VGA controller space to
        // 64K (A0000h-AFFFFh).
        outpw (0x3C4, 0xF02);       // seq02  = 0F  (enable all maps)
        outpw (0x3C4, 0xA04);       // seq04  = 0A  (memory mode for 256 color)
        outpw (0x3CE, 0x1000);      // graf00 = 10  (standard set/reset)
        outpw (0x3CE, 0x0001);      // graf01 = 00  (standard enable set/reset)
        outpw (0x3CE, 0x0002);      // graf02 = 00  (standard color compare)
        outpw (0x3CE, 0x0003);      // graf03 = 00  (standard data rotate)
        outpw (0x3CE, 0x0004);      // graf04 = 00  (read mode 0)
        outpw (0x3CE, 0x0005);      // graf05 = 00  (write mode 0)
        outpw (0x3CE, 0x0106);      // graf06 = 01  (128K, graphics mode)
        outpw (0x3CE, 0x0F07);      // graf07 = 0F  (pass all planes)
        outpw (0x3CE, 0xFF08);      // graf08 = FF  (pass all bits through mask)

        // enable linear addressing through VGA aperture
        iow8 (ioCRTC_GEN_CNTL + 3, ior8 (ioCRTC_GEN_CNTL + 3) | 0x08);
    }
}


/* --------------------------------------------------------------------------
  PROCESS_COMMAND_LINE - process application command line arguments.

  This function sets the default resolution and color depth settings
  for each application to 640x480 8bpp and overrides these defaults with
  the first occurance of valid options in the command line argument list.
-------------------------------------------------------------------------- */
void process_command_line (int argc, char *argv[])
{
    int i;

    // Set default resolution and color depth in global variables.
    gmode_res = MODE_640x480;
    gclr_depth = COLOR_DEPTH_8;

    // Override defaults with the first instance of valid command line
    // arguments.
    if (argc > 1)
    {
        // Check for valid resolution options.
        for (i = 1; i <= argc; i++)
        {
            if (strcmp (argv[i], "640") == 0)
            {
                gmode_res = MODE_640x480;
                break;
            }
            else if (strcmp (argv[i], "800") == 0)
            {
                gmode_res = MODE_800x600;
                break;
            }
            else if (strcmp (argv[i], "1024") == 0)
            {
                gmode_res = MODE_1024x768;
                break;
            }
            else if (strcmp (argv[i], "1280") == 0)
            {
                gmode_res = MODE_1280x1024;
                break;
            }
            else if (strcmp (argv[i], "1600") == 0)
            {
                gmode_res = MODE_1600x1200;
                break;
            }
        } // end for

        // Check for valid color depth options.
        for (i = 1; i <= argc; i++)
        {
            if (strcmp (argv[i], "4") == 0)
            {
                gclr_depth = COLOR_DEPTH_4;
                break;
            }
            else if (strcmp (argv[i], "8") == 0)
            {
                gclr_depth = COLOR_DEPTH_8;
                break;
            }
            else if (strcmp (argv[i], "15") == 0)
            {
                gclr_depth = COLOR_DEPTH_15;
                break;
            }
            else if (strcmp (argv[i], "16") == 0)
            {
                gclr_depth = COLOR_DEPTH_16;
                break;
            }
            else if (strcmp (argv[i], "24") == 0)
            {
                gclr_depth = COLOR_DEPTH_24;
                break;
            }
            else if (strcmp (argv[i], "32") == 0)
            {
                gclr_depth = COLOR_DEPTH_32;
                break;
            }
        }  // end for
    } // end if argc
}
