; ===========================================================================
; ROMSEG.ASM
;
; Routines to find mach64 ROM and intialize parameters for ROM functions
;
; Compiling:
;   masm /Ml /D<memory model> romseg.asm;
;       <memory model> = mem_S for SMALL model,
;                        mem_M for MEDIUM model,
;                        mem_L for LARGE model
;
; Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
; ===========================================================================

include atim64.inc

IFDEF mem_S
PARM        equ     4   ; passed parameters start at bp+4 for small model
ELSE
PARM        equ     6   ; passed parameters start at bp+6 for other models
ENDIF

IFDEF mem_S
.MODEL  SMALL, C
ELSEIFDEF mem_M
.MODEL  MEDIUM, C
ELSE
.MODEL  LARGE, C
ENDIF

.DATA

ati_sig     db      '761295520', 0
m64_sig1    db      'MACH64', 0
m64_sig2    db      'GXCX', 0           ; older ROMs
rom_seg     dw      0000h     ; set to 0 to signify invalid address

.CODE
.286

; Macro for 'call' model handling
Mcall       macro   routine
IFDEF mem_S
            call    NEAR PTR routine
ELSE
            call    FAR PTR routine
ENDIF
            endm

; ---------------------------------------------------------------------------
; IS_ATI_ROM
;
; Check if ROM has ATI signature. Normally, the starting location of the 9
; character signature is at ROM_SEGMENT:0031h. In an MCA system, this is not
; the case and can be anywhere in the first 128 bytes of the ROM. For this
; reason, a signature scan method is used.
;
; Inputs : none
;
; Outputs: Returns 1 in ax if signature was found, 0 if not
; ---------------------------------------------------------------------------
            public  is_ati_rom

IFDEF mem_S
is_ati_rom  proc    near
ELSE
is_ati_rom  proc    far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    di
            push    es
            push    si
            push    cx

            ; load ROM segment to be verified into ax
            mov     ax, WORD PTR [bp+PARM]
            call    NEAR PTR find_ati_rom

            ; restore saved registers
            pop     cx
            pop     si
            pop     es
            pop     di

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

is_ati_rom  endp

; ---------------------------------------------------------------------------
; ---------------------------------------------------------------------------
            public  is_mach64_rom

IFDEF mem_S
is_mach64_rom proc  near
ELSE
is_mach64_rom proc  far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; save registers used
            push    di
            push    es
            push    si
            push    cx

            ; load ROM segment to be verified into ax
            mov     ax, WORD PTR [bp+PARM]
            call    NEAR PTR find_ati_rom

            ; check if ATI ROM was found. If so, check if mach64 ROM
            cmp     ax, 1
            jne     m64_rom_not_found

            ; scan ROM for first mach64 signature
            ; ds:si - pointer to source compare string
            ; es:di - pointer to test string
            mov     ax, 0
find_m64_sig1:
            mov     cx, 6
            mov     di, ax
            mov     si, offset m64_sig1
            repe    cmpsb
            jcxz    m64_rom_found
            inc     ax
            cmp     ax, 1024
            jne     find_m64_sig1

            ; scan ROM for second mach64 signature
            ; ds:si - pointer to source compare string
            ; es:di - pointer to test string
            mov     ax, 0
find_m64_sig2:
            mov     cx, 4
            mov     di, ax
            mov     si, offset m64_sig2
            repe    cmpsb
            jcxz    m64_rom_found
            inc     ax
            cmp     ax, 1024
            jne     find_m64_sig2
            jmp     m64_rom_not_found

m64_rom_found:
            mov     ax, 1
            jmp     exit_is_mach64_rom

m64_rom_not_found:
            mov     ax, 0

exit_is_mach64_rom:
            ; restore saved registers
            pop     cx
            pop     si
            pop     es
            pop     di

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

is_mach64_rom endp


; ---------------------------------------------------------------------------
;  find_ati_rom : internal procedure to find the ATI ROM
;  only to be called from functions within the module.
;  registers used must be saved by calling procs as they may be used again
;  by calling procs before being restored.
;
;  input - ax = ROM segment to be verified
;  output - ax - 1, success, 0 failure
; ---------------------------------------------------------------------------
find_ati_rom:

            ; load ROM segment to be verified (in ax) into es
            mov     es, ax

            ; check offset 0 into ROM for ATI ROM ID (AA55h)
            mov     di, 0
            mov     ax, es:[di]
            cmp     ax, 0AA55h
            jne     ati_rom_not_found

            ; scan first 1K of ROM for ATI signature
            ; ds:si - pointer to source compare string
            ; es:di - pointer to test string
            cld
            mov     ax, 0

find_ati_sig:
            mov     cx, 9       ;length of compare string
            mov     di, ax
            mov     si, offset ati_sig
            repe    cmpsb
            jcxz    ati_rom_found
            inc     ax
            cmp     ax, 1024
            jne     find_ati_sig
            jmp     ati_rom_not_found

ati_rom_found:
            ; the ATI signature was found, therefore, assign success code
            mov     ax, 1
            jmp     exit_find_ati_rom

ati_rom_not_found:
            mov     ax, 0

exit_find_ati_rom:

            retn
;__________________________________________________________________________
            public  set_rom_base

IFDEF mem_S
set_rom_base proc   near
ELSE
set_rom_base proc   far
ENDIF

            ; create frame pointer
            push    bp
            mov     bp, sp

            ; set internal ROM segment variable
            mov     ax, WORD PTR [bp+PARM]
            mov     WORD PTR rom_seg, ax

            ; remove frame pointer
            mov     sp, bp
            pop     bp

            ret

set_rom_base endp


;__________________________________________________________________________
            public  get_rom_base

IFDEF mem_S
get_rom_base proc   near
ELSE
get_rom_base proc   far
ENDIF

            ; load ax with internal ROM segment variable
            mov     ax, WORD PTR rom_seg

            ret

get_rom_base endp

            end

