/*==========================================================================
* CONTEXT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved *
*                                                                          *
* Mach64 context load functions.                                           *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_uploadcontext - upload context array to video memory

  This function uploads the given context array to video memory. Each
  context pointer represents 256 bytes of video memory. The context load
  address decreases toward the start of video memory as the context load
  pointer increases. The video address is calculated as follows:

    ((~load_pointer) & memmask) << 8

  'memmask' is determined as being the physical address of top of memory - 1
  shifted down by 8 bits. If the linear aperture is being used, the video
  address is added to the linear aperture base address. Also, do not use load
  pointers 0-3 if the linear aperture size equals the video memory size since
  this will cause an overwrite of the GUI registers. The load pointer should
  not exceed the video memory size.
-------------------------------------------------------------------------- */
void PGL_uploadcontext(unsigned int loadpointer, PGL_context far *context)
{
    unsigned long loadaddr, memmask;
    unsigned long writepage, writeoffset;
    unsigned char far *ptr;
    unsigned int index;

    // determine memory mask to apply to load address calculation
    switch(PGL_querydata.memory_size)
    {
        case 0: memmask = 0x07ff; break;  // 512K
        case 1: memmask = 0x0fff; break;  // 1M
        case 2: memmask = 0x1fff; break;  // 2M
        case 3: memmask = 0x3fff; break;  // 4M
        case 4: memmask = 0x5fff; break;  // 6M
        case 5: memmask = 0x7fff; break;  // 8M
    }

    // determine context load address
    // - the load pointer is complemented, ANDed with a memory mask, and
    //   then shifted up by 8 bits since each pointer represents 256 bytes
    //   of memory
    //
    loadaddr = ((~(unsigned long)loadpointer) & memmask) << 8;

    // load context array according to aperture type
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // set write pointer to address
        writepage = loadaddr / 0x8000;
        writeoffset = loadaddr - (writepage * 0x8000);

        // set 32k VGA aperture pointer with overflow in 2nd aperture
        regw(MEM_VGA_WP_SEL, ((writepage + 1) << 16) | writepage);

        // load context array using VGA aperture
        ptr = (unsigned char far *)(context);
        for (index = 0; index < 256; index++)
        {
            *((unsigned char far *)(0xa0000000 + writeoffset + index)) =
            *((unsigned char far *)(ptr + index));
        }
    }
    else
    {
        // if linear aperture size = memory size, and the load pointer is 0
        // to 3, the context array upload will conflict with GUI register
        // space of the linear aperture - return error in this case

        // load context array using linear aperture
        pgl_movemem((void far *)(context),
                    PGL_modecfg.aperture_address + loadaddr,
                    256 / 2,
                    MEM_WRITE);
    }
}

/* --------------------------------------------------------------------------
  PGL_loadcontext - load a context with the given attributes

  This function loads a context into the Mach64 GUI registers from a
  previously uploaded context array to video memory using
  PGL_uploadcontext(). The context load pointer should be set to the
  desired context array in video memory to load. The attributes determine
  the action after the register load takes place. The attribute value is
  directly placed into the CONTEXT_LOAD_CNTL register.
  The attributes are:

    CONTEXT_NO_LOAD,
    CONTEXT_LOAD,
    CONTEXT_LOAD_AND_DO_FILL,
    CONTEXT_LOAD_AND_DO_LINE,
    CONTEXT_CMD_DISABLE

  These are mutually exclusive except CONTEXT_CMD_DISABLE which can be ORed
  in with the other attribute values. Note that the upper 16 bits of the
  CONTEXT_LOAD_CNTL register is used for the context load attribute.
-------------------------------------------------------------------------- */
void PGL_loadcontext(unsigned int loadpointer, unsigned long attribute)
{
    PGL_waitforfifo(1);
    regw(CONTEXT_LOAD_CNTL, (attribute & 0xffff0000) | (loadpointer & 0x7fff));
}

