/*==========================================================================
* MODELIST.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved *
*                                                                          *
* PGL functions to get available modes from the query structure.           *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include <malloc.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_loadmodelist - return a pointer to an array of available modes

  This function allocates a buffer and calls the ROM query functions to fill
  it with an array of available modes. The number of available modes is
  assigned to the 'modes' variable. This function is intended to provide
  only the information required to call PGL_initmode() with available mode
  information. If more information about a mode is required, the application
  will need to call the PGL query functions directly. If an error occurs, a
  NULL pointer is returned and the 'modes' variable will be set to zero. If
  successful, the buffer should be deallocated by calling PGL_unloadmodelist()
  before exiting the application.
-------------------------------------------------------------------------- */
PGL_modedef *PGL_loadmodelist(int *modes)
{
    PGL_queryheader *queryinfo;
    unsigned char *queryptr;
    unsigned char *qptr;
    unsigned char *qptr2;
    PGL_modetable *modeptr;
    PGL_modetable *modeptr2;
    unsigned char *bufferptr;
    unsigned char *mptr;
    PGL_modedef *modelist;
    int bpplist[6] = {4, 8, 15, 16, 24, 32};
    int bppselect, nmodes, qmodes, totalmodes;
    int querysize;
    int i;
    int skip;
    PGL_modetable *modetable;

    // assume failure
    *modes = 0;

    // blank screen during query calls
    PGL_blankscreen();

    // get query size in bytes of query structure to be filled by ROM
    querysize = PGL_getquerysize(HEADER_AND_MODE);
    if (querysize == 0)
    {
        PGL_unblankscreen();
        return (NULL);
    }

    // allocate buffer for query structure
    queryptr = (unsigned char *) malloc (querysize);
    if (queryptr == NULL)
    {
        PGL_unblankscreen();
        return (NULL);
    }

    // fill buffer
    if (PGL_queryhardware(HEADER_AND_MODE, (void far *)queryptr) != NO_ERROR)
    {
        PGL_unblankscreen();
        free(queryptr);
        return (NULL);
    }

    // unblank screen
    PGL_unblankscreen();

    // assign pointer to query header
    queryinfo = (PGL_queryheader *)(queryptr);

    // check query information about mode tables
    if (queryinfo->mode_tables == 0)
    {
        free(queryptr);
        return (NULL);
    }
    if ((queryinfo->mode_table_offset == 0) || (queryinfo->mode_table_size == 0))
    {
        free(queryptr);
        return (NULL);
    }

    // cycle through query structure to determine number of actual modes
    nmodes = 0;
    qptr = (unsigned char *)(queryptr + queryinfo->mode_table_offset);
    modeptr = (PGL_modetable *)(qptr);
    for (qmodes = 0; qmodes < (int)(queryinfo->mode_tables); qmodes++)
    {
        // expand query modes into actual modes
        switch(modeptr->max_bpp)
        {
            case 1: // 4 bpp
            case 2: // 8 bpp
            case 3: // 15 bpp
            case 4: // 16 bpp
            case 5: // 24 bpp
            case 6: // 32 bpp
                nmodes = nmodes + (int)(modeptr->max_bpp);
                break;
        }

        // next mode table
        qptr = (unsigned char *)(qptr + queryinfo->mode_table_size);
        modeptr = (PGL_modetable *)(qptr);
    }
    if (nmodes == 0)
    {
        free(queryptr);
        return (NULL);
    }

    // allocate memory for PGL available mode structures
    totalmodes = nmodes;
    bufferptr = (unsigned char *) malloc (totalmodes * sizeof(PGL_modedef));
    if (bufferptr == NULL)
    {
        free(queryptr);
        return (NULL);
    }

    // Cycle through query structure again to fill PGL available mode
    // structures. Some modes may have duplicate resolutions. For this
    // case, the highest bpp is choosen.
    nmodes = 0;
    qptr = (unsigned char *)(queryptr + queryinfo->mode_table_offset);
    modeptr = (PGL_modetable *)(qptr);
    for (qmodes = 0; qmodes < (int)(queryinfo->mode_tables); qmodes++)
    {
        // set mode table entries
        modetable = modeptr;

        // do a search of qmodes for duplication before adding modes
        skip = 0;
        qptr2 = (unsigned char *)(queryptr + queryinfo->mode_table_offset);
        modeptr2 = (PGL_modetable *)(qptr2);
        for (i = 0; i < (int)(queryinfo->mode_tables); i++)
        {
            // don't compare if search mode is the same as query mode
            if (i != qmodes)
            {
                // is x resolution the same
                if (modeptr2->xres == modeptr->xres)
                {
                    // is y resolution the same
                    if (modeptr2->yres == modeptr->yres)
                    {
                        //
                        // skip adding modes if
                        //  (1) search bpp > query bpp
                        //  (2) search bpp = query bpp and
                        //      search index < query index
                        //
                        if ((modeptr2->max_bpp > modeptr->max_bpp) ||
                            ((modeptr2->max_bpp == modeptr->max_bpp) &&
                            (i < qmodes)))
                        {
                            // set skip flag and end loop
                            skip = 1;
                            i = (int)(queryinfo->mode_tables);
                        }

                        //
                        // reassign dot clock if bpp is less than compared
                        // one
                        //
                        if (modeptr2->max_bpp < modeptr->max_bpp)
                        {
                            modetable = modeptr2;
                        }
                    }
                }
            }

            // next search mode table
            qptr2 = (unsigned char *)(qptr2 + queryinfo->mode_table_size);
            modeptr2 = (PGL_modetable *)(qptr2);
        }

        // set skip flag if ROM color depth byte indicates no support
        switch(modeptr->max_bpp)
        {
            case 3: // 15 bpp
                if ((queryinfo->color_depth_support & COLORSUPPORT_15) != COLORSUPPORT_15)
                {
                    skip = 1;
                }
                break;

            case 4: // 16 bpp
                if ((queryinfo->color_depth_support & COLORSUPPORT_16) != COLORSUPPORT_16)
                {
                    skip = 1;
                }
                break;

            case 5: // 24 bpp
                if (((queryinfo->color_depth_support & COLORSUPPORT_24_RGB) != COLORSUPPORT_24_RGB) &&
                    ((queryinfo->color_depth_support & COLORSUPPORT_24_BGR) != COLORSUPPORT_24_BGR))
                {
                    skip = 1;
                }
                break;

            case 6: // 32 bpp
                if (((queryinfo->color_depth_support & COLORSUPPORT_32_RGBA) != COLORSUPPORT_32_RGBA) &&
                    ((queryinfo->color_depth_support & COLORSUPPORT_32_ARGB) != COLORSUPPORT_32_ARGB) &&
                    ((queryinfo->color_depth_support & COLORSUPPORT_32_BGRA) != COLORSUPPORT_32_BGRA) &&
                    ((queryinfo->color_depth_support & COLORSUPPORT_32_ABGR) != COLORSUPPORT_32_ABGR))
                {
                    skip = 1;
                }
                break;
        }

        // expand query modes into actual modes
        switch(modeptr->max_bpp)
        {
            case 1: // 4 bpp
            case 2: // 8 bpp
            case 3: // 15 bpp
            case 4: // 16 bpp
            case 5: // 24 bpp
            case 6: // 32 bpp
                if (skip == 0)
                {
                    bppselect = 0;
                    for (bppselect = 0; bppselect < (int)(modeptr->max_bpp); bppselect++)
                    {
                        mptr = (unsigned char *)(bufferptr + (nmodes * sizeof(PGL_modedef)));
                        modelist = (PGL_modedef *)(mptr);
                        modelist->xres = (int)(modeptr->xres);
                        modelist->yres = (int)(modeptr->yres);
                        modelist->bpp = bpplist[bppselect];
                        memcpy(&modelist->modetable, modetable, sizeof(PGL_modetable));
                        switch(modeptr->max_bpp)
                        {
                            case 5: // 24 bpp
                                // select color depth based on supported depths
                                if ((PGL_querydata.color_depth_support & COLORSUPPORT_24_RGB) == COLORSUPPORT_24_RGB)
                                {
                                    // RGB
                                    modelist->depth = DEPTH_24_RGB;
                                }
                                else if ((PGL_querydata.color_depth_support & COLORSUPPORT_24_BGR) == COLORSUPPORT_24_BGR)
                                {
                                    // BGR
                                    modelist->depth = DEPTH_24_BGR;
                                }
                                break;

                            case 6: // 32 bpp
                                // select color depth based on supported depths
                                if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_RGBA) == COLORSUPPORT_32_RGBA)
                                {
                                    // RGBa
                                    modelist->depth = DEPTH_32_RGBA;
                                }
                                else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_ARGB) == COLORSUPPORT_32_ARGB)
                                {
                                    // aRGB
                                    modelist->depth = DEPTH_32_ARGB;
                                }
                                else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_BGRA) == COLORSUPPORT_32_BGRA)
                                {
                                    // BGRa
                                    modelist->depth = DEPTH_32_BGRA;
                                }
                                else if ((PGL_querydata.color_depth_support & COLORSUPPORT_32_ABGR) == COLORSUPPORT_32_ABGR)
                                {
                                    // aBGR
                                    modelist->depth = DEPTH_32_ABGR;
                                }
                                break;

                            default:
                                modelist->depth = 0;
                                break;
                        }

                        modelist->mode_number = (int)(modeptr->mode_number);
                        nmodes++;
                        if (nmodes > totalmodes)
                        {
                            bppselect = (int)(modeptr->max_bpp);
                            qmodes = (int)(queryinfo->mode_tables);
                        }
                    }
                }
                break;
        }

        // next mode table
        qptr = (unsigned char *)(qptr + queryinfo->mode_table_size);
        modeptr = (PGL_modetable *)(qptr);
    }
    if (nmodes == 0)
    {
        free(bufferptr);
        return (NULL);
    }

    // deallocate query structure buffer
    free(queryptr);

    // assign pointer & fill available modes variable
    modelist = (PGL_modedef *)(bufferptr);
    *modes = nmodes;

    // return pointer to PGL available mode buffer
    // this buffer must be deallocated by calling PGL_unloadmodelist()
    return (modelist);
}

/* --------------------------------------------------------------------------
  PGL_unloadmodelist - deallocate buffer allocated in PGL_loadmodelist()

  This function should be called before exiting the application if a
  successful call was made to PGL_loadmodelist().
-------------------------------------------------------------------------- */
void PGL_unloadmodelist(PGL_modedef *modelist)
{
    if (modelist != NULL)
    {
        free((unsigned char *)(modelist));
    }
}

