/*==========================================================================
* OVRSCAN.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved *
*                                                                          *
* Functions for handling overscan size and colors.                         *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_enableoverscan - turn on overscan with given sizes and color

  The left and right widths are specified in pixels*8. The top and bottom
  widths are specified in lines. For 4 and 8 bpp modes, the color is
  specified as an index. For other modes, the appropriate R, G, B components
  must be given.
-------------------------------------------------------------------------- */
void PGL_enableoverscan(unsigned long color,
                        int leftwidth, int rightwidth,
                        int topwidth, int bottomwidth)
{
    unsigned long ovrclr;
    unsigned long red, green, blue;
    unsigned long rshift, gshift, bshift;
    unsigned long temp;
    int index;
    PGL_palette entry;

    // set overscan widths
    regw(OVR_WID_LEFT_RIGHT, ((unsigned long)rightwidth << 16) | leftwidth);
    regw(OVR_WID_TOP_BOTTOM, ((unsigned long)bottomwidth << 16) | topwidth);

    // setup overscan color for 4, 8 bpp modes
    if ((PGL_modecfg.bpp == 4) || (PGL_modecfg.bpp == 8))
    {
        // get color componets from palette using color indices
        if ((PGL_modecfg.dac_attribute == DAC_SP_ATI68860) ||
            (PGL_modecfg.dac_attribute == DAC_SP_TVP3026) ||
            (PGL_modecfg.dac_attribute == DAC_SP_IBM514) ||
            (PGL_modecfg.chip_type == CHIP_CT_ID) ||
            (PGL_modecfg.chip_type == CHIP_DT_ID))
        {
            // ensure palette registers are available
            iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);

            // Overscan color is an 8 bit value, palette color is a 6 bit value
            entry = PGL_getpalette((int)(color & 0xff));
            red = (unsigned long)(entry.red) << 2;
            green = (unsigned long)(entry.green) << 2;
            blue = (unsigned long)(entry.blue) << 2;

            // assign cursor color values for Mach64 Bedrock internal DACs
            if ((PGL_modecfg.chip_type == CHIP_CT_ID) ||
                (PGL_modecfg.chip_type == CHIP_DT_ID))
            {
                ovrclr = (red << 24) | (green << 16) | (blue << 8) | color;
            }
        }
        else // use color indices for standard DACs
        {
            ovrclr = (int)(color & 0xff);
        }
    }
    else // setup overscan color for 15, 16, 24, 32 bpp modes
    {
        // determine color component shifting values
        switch(PGL_modecfg.bpp)
        {
            case 16:
                if (PGL_modecfg.depth == 555)
                {
                    rshift = 3;
                    gshift = 3;
                    bshift = 3;
                }
                else  // 565 weight
                {
                    rshift = 3;
                    gshift = 2;
                    bshift = 3;
                }
                break;

            case 24:
            case 32:
                rshift = 0;
                gshift = 0;
                bshift = 0;
                break;
        }

        // get overscan color components
        red = PGL_getprimarycolor(RED, color) << rshift;
        green = PGL_getprimarycolor(GREEN, color) << gshift;
        blue = PGL_getprimarycolor(BLUE, color) << bshift;

        // special case for Brooktree DAC in 24 (BGR) bpp mode
        if ((PGL_modecfg.bpp == 24) &&
            (PGL_modecfg.depth == DEPTH_24_BGR) &&
            (PGL_querydata.dac_type == DAC_BT481))
        {
            // swap RED and BLUE components
            temp = blue;
            blue = red;
            red = temp;
        }

        // assign overscan color values for standard DACs
        if ((PGL_modecfg.dac_attribute != DAC_SP_ATI68860) &&
            (PGL_modecfg.dac_attribute != DAC_SP_TVP3026) &&
            (PGL_modecfg.dac_attribute != DAC_SP_IBM514))
        {
            ovrclr = (red << 24) | (green << 16) | (blue << 8);
        }
    }

    // write to hardware to set overscan for ATI68860,TVP3026,IBM514 DACs
    if ((PGL_modecfg.dac_attribute == DAC_SP_ATI68860) ||
        (PGL_modecfg.dac_attribute == DAC_SP_TVP3026) ||
        (PGL_modecfg.dac_attribute == DAC_SP_IBM514))
    {
        // special setup for ATI68860, TVP3026, and IBM514 DAC for overscan

        // access overscan color register
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);

        // select overscan color register
        switch(PGL_modecfg.dac_attribute)
        {
            case DAC_SP_ATI68860:
                // skip cursor colors 0 & 1
                iow8(ioDAC_REGS + DAC_W_INDEX, 2);
                index = DAC_DATA;
                break;

            case DAC_SP_TVP3026:
                // first color register on TVP3026 DAC is OVERSCAN
                iow8(ioDAC_REGS + DAC_W_INDEX, 0);
                index = DAC_DATA;
                break;

            case DAC_SP_IBM514:
                iow8(ioDAC_REGS + DAC_R_INDEX, 1);      // auto-increment
                iow8(ioDAC_REGS + DAC_W_INDEX, 0x60);
                iow8(ioDAC_REGS + DAC_DATA, 0);         // index = 0x60
                index = DAC_MASK;
                break;
        }

        // set overscan color - 8 bit values required
        iow8(ioDAC_REGS + index, (int)red);
        iow8(ioDAC_REGS + index, (int)green);
        iow8(ioDAC_REGS + index, (int)blue);

        // enable overscan
        switch(PGL_modecfg.dac_attribute)
        {
            case DAC_SP_ATI68860:
                iow8(ioGEN_TEST_CNTL, ior8(ioGEN_TEST_CNTL) | 0x20);
                iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 3);
                iow8(ioDAC_REGS + DAC_W_INDEX, ior8(ioDAC_REGS + DAC_W_INDEX) | 0x10);
                break;

            case DAC_SP_TVP3026:
                iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
                iow8(ioDAC_REGS + DAC_W_INDEX, 0x1d);
                iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 2);
                iow8(ioDAC_REGS + DAC_MASK, ior8(ioDAC_REGS + DAC_MASK) | 0x40);
                break;

            case DAC_SP_IBM514:
                iow8(ioGEN_TEST_CNTL, ior8(ioGEN_TEST_CNTL) | 0x20);
                break;
        }

        // return DAC register set to palette registers
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }
    else // write to hardware to set overscan color for standard DACs
    {
        regw(OVR_CLR, ovrclr);
    }
}

/* --------------------------------------------------------------------------
  PGL_disableoverscan - turn off overscan
-------------------------------------------------------------------------- */
void PGL_disableoverscan(void)
{
    // disable overscan on ATI68860 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_ATI68860)
    {
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 3);
        iow8(ioDAC_REGS + DAC_W_INDEX, ior8(ioDAC_REGS + DAC_W_INDEX) & 0xef);
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }

    // disable overscan on TVP3026 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_TVP3026)
    {
        iow8(ioGEN_TEST_CNTL, ior8(ioGEN_TEST_CNTL) & 0xdf);
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x1d);
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 2);
        iow8(ioDAC_REGS + DAC_MASK, ior8(ioDAC_REGS + DAC_MASK) & 0xbf);
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }

    // disable overscan on IBM514 DAC
    if (PGL_modecfg.dac_attribute == DAC_SP_IBM514)
    {
        iow8(ioGEN_TEST_CNTL, ior8(ioGEN_TEST_CNTL) & 0xdf);
        iow8(ioDAC_CNTL, (ior8(ioDAC_CNTL) & 0xfc) | 1);
        iow8(ioDAC_REGS + DAC_R_INDEX, 1);              // auto-increment
        iow8(ioDAC_REGS + DAC_W_INDEX, 0x60);
        iow8(ioDAC_REGS + DAC_DATA, 0);                 // index = 0x60
        iow8(ioDAC_REGS + DAC_MASK, 0);
        iow8(ioDAC_REGS + DAC_MASK, 0);
        iow8(ioDAC_REGS + DAC_MASK, 0);
        iow8(ioDAC_CNTL, ior8(ioDAC_CNTL) & 0xfc);
    }

    // set overscan widths to zero - this disables the overscan on non-
    // special DACs
    regw(OVR_WID_LEFT_RIGHT, 0);
    regw(OVR_WID_TOP_BOTTOM, 0);
}

