/*==========================================================================
* CURSOR.C - Copyright (c) 1993-94 ATI Technologies Inc. All rights reserved*
*                                                                          *
* Demo routines for hardware cursor used by PGLDEMO.EXE                    *
* ======================================================================== */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pgldemo.h"

/*
 *****************************************************************************
 */

/*
 * GLOBAL VARIABLES
 */

/*
 * EXTERNAL VARIABLES
 */

extern PGL_modeconfig  modeinfo;
extern PGL_queryheader queryinfo;

// ---- HARDWARE CURSOR DATA ----

unsigned int cursor8x8[8][1] =
{	
	0x0000,
	0x3ffc,
	0x355c,
	0x369c,
	0x369c,
	0x355c,
	0x3ffc,
	0x0000
};

unsigned int cursor16x16[16][2] =
{
	0x0000, 0x0000,
	0x0000, 0x0000,
	0xfff0, 0x0fff,
	0xfff0, 0x0fff,
	0x55f0, 0x0f55,
	0x55f0, 0x0f55,
	0xa5f0, 0x0f5a,
	0xa5f0, 0x0f5a,
	0xa5f0, 0x0f5a,
	0xa5f0, 0x0f5a,
	0x55f0, 0x0f55,
	0x55f0, 0x0f55,
	0xfff0, 0x0fff,
	0xfff0, 0x0fff,
	0x0000, 0x0000,
	0x0000, 0x0000
};

unsigned int cursor32x32[32][4] =
{
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0xaa55, 0x55aa, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0x5555, 0x5555, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0xff00, 0xffff, 0xffff, 0x00ff,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000
};

unsigned int cursor64x64[64][8] =
{
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0xaaaa, 0xaaaa, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000
};

unsigned int cursor8_b[8][1] =
{
	0x0077,
	0xaa77,
	0x0077,
	0xaa77,
	0xddaa,
	0xdd00,
	0xddaa,
	0xdd00,
};

unsigned int cursor16_b[16][2] =
{
	0x0000, 0x7777,
	0xaaaa, 0x7777,
	0x0000, 0x7777,
	0xaaaa, 0x7777,
	0x0000, 0x7777,
	0xaaaa, 0x7777,
	0x0000, 0x7777,
	0xaaaa, 0x7777,
	0xdddd, 0xaaaa,
	0xdddd, 0x0000,
	0xdddd, 0xaaaa,
	0xdddd, 0x0000,
	0xdddd, 0xaaaa,
	0xdddd, 0x0000,
	0xdddd, 0xaaaa,
	0xdddd, 0x0000,
};

unsigned int cursor32_b[32][4] =
{
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0x0000, 0x0000, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0x7777, 0x7777,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0x0000, 0x0000,
};

unsigned int cursor64_b[64][8] =
{
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0x0000, 0x0000, 0x0000, 0x0000, 0x7777, 0x7777, 0x7777, 0x7777,
	0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa, 0x7777, 0x7777, 0x7777, 0x7777,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
	0xdddd, 0xdddd, 0xdddd, 0xdddd, 0x0000, 0x0000, 0x0000, 0x0000,
};

/*---------------------------------------------------------------------------
*/
void GetButtonStatusAndMousePosition(int *pButtonStatus, int *pHorizontalPosition, int *pVerticalPosition)
{
	union REGS regs;

	regs.x.ax = 3;
	regs.x.bx = 0;
	int86( 0x33, &regs, &regs );
	*pButtonStatus = regs.x.bx;
	*pHorizontalPosition = regs.x.cx;
	*pVerticalPosition = regs.x.dx;
}

/*---------------------------------------------------------------------------
*/
void UpdateMouse(int *buttonstat, int *mouse_x, int *mouse_y)
{
    static int old_mouse_x;
    static int old_mouse_y;

    GetButtonStatusAndMousePosition(buttonstat, mouse_x, mouse_y);
	*mouse_x >>= 3;
	*mouse_y >>= 3;
    /* - the hardware cursor is forced to be on an even boundary for the  */
    /*   STG 170X DAC in high-refresh 24- & 32-bpp modes                  */
    if ((queryinfo.dac_type == DAC_STG1700) ||
        (queryinfo.dac_type == DAC_STG1702) ||
        (queryinfo.dac_type == DAC_STG1703))
    {
        if ((modeinfo.bpp == 24) || (modeinfo.bpp == 32))
        {
            *mouse_x &= 0xfffe;
        }
    }
	if (*mouse_x != old_mouse_x || *mouse_y != old_mouse_y)
	{
		old_mouse_x = *mouse_x;
		old_mouse_y = *mouse_y;
        PGL_sethwcursorpos(*mouse_x, *mouse_y);
	}
}

/*---------------------------------------------------------------------------
*/
void CenterMouse(PGL_modeconfig *modeinfo)
{
    int   button;
    int   x, y;
    union   REGS    regs;

    x = (modeinfo->xres/2) << 3;
    y = (modeinfo->yres/2) << 3;
    regs.x.ax = 4;
	regs.x.cx = x;
	regs.x.dx = y;
	int86( 0x33, &regs, &regs );
    UpdateMouse(&button, &x, &y);
    PGL_sethwcursorpos(x, y);
}

/*---------------------------------------------------------------------------
*/
int InitMouse(void)
{
    union REGS regs;
    int retval;

	regs.x.ax = 0;
	int86( 0x33, &regs, &regs );
	if (regs.x.ax == 0)
	{
		retval = 0;
	}
	else if (regs.x.ax & 0x8000)
	{
		PGL_enablehwcursor();
		retval = 1;
	}
	return retval;
}


/*---------------------------------------------------------------------------
*/
void ScaleMouse(int hot_x, int hot_y)
{
	union	REGS	regs;

    /* set mouse x limits */
    regs.x.ax = 0x0007;                         /* setting mouse x limits or */
	regs.x.cx = 0;								/* y limits < 0 will screw   */
    regs.x.dx = modeinfo.xres << 3;   /* up some older mouse       */
	int86( 0x33, &regs, &regs );				/* drivers                   */

    /* set mouse y limits */
	regs.x.ax = 0x0008;
	regs.x.cx = 0;
    regs.x.dx = modeinfo.yres << 3;
	int86( 0x33, &regs, &regs );

    /* set mickey-to-pixel ratio    */
	regs.x.ax = 0x000f;
    regs.x.cx = 1;
    regs.x.dx = 2;
	int86( 0x33, &regs, &regs );
}

/*---------------------------------------------------------------------------
*/

void hline(int xstart, int xend, int y, unsigned long color)
{
    unsigned int temp;
    
    if (xstart > xend)
    {
        temp = xstart;
        xstart = xend;
        xend = temp;
    }
    
    PGL_setfgcolor(color);
    PGL_drawrect(xstart, y, xend - xstart + 1, 1);
}

void vline(int ystart, int yend, int x, unsigned long color)
{
    unsigned int temp;
    
    if (ystart > yend)
    {
        temp = ystart;
        ystart = yend;
        yend = temp;
    }

    PGL_setfgcolor(color);
    PGL_drawrect(x, ystart, 1, yend - ystart + 1);
}

/* set palette register i to (r,g,b) */
void set_palette(unsigned int i, unsigned char r, 
                 unsigned char g, unsigned char b)
{
    PGL_palette entry;

    entry.red = r;
    entry.green = g;
    entry.blue = b;
    PGL_setpalette(i, entry);
}

void init_palette8(void)
{
    unsigned int i;
    unsigned char color;
    
    /* set the palette */
    i = 0;
    /* decreasing blue */
    for(color = 63; color > 0; color--)
    {

        set_palette(i, 0, 0, (unsigned char)(color));
        i++;
    }    
    set_palette(i++, 0, 0, 0);
    
    /* increasing green */
    for(color = 0; color < 64; color++)
    {
        set_palette(i, 0, (unsigned char)(color), 0);
        i++;
    }    
    
    /* decreasing red */
    for(color = 63; color > 0; color--)
    {
        set_palette(i, (unsigned char)(color), 0, 0);
        i++;
    }    
    set_palette(i++, 0, 0, 0);
    
    /* increasing gray */
    for(color = 0; color < 64; color++)
    {
        set_palette(i, (unsigned char)(color),
                       (unsigned char)(color),
                       (unsigned char)(color));
        i++;
    }    
}

void init_palette4(void)
{
    // setup 8 bpp first and then modify only the top & bottom 16 entries
    init_palette8();

    set_palette(0, 0, 0, 15);        /* increasing blue */
    set_palette(1, 0, 0, 31);
    set_palette(2, 0, 0, 46);
    set_palette(3, 0, 0, 63);
    set_palette(4, 0, 15, 0);        /* increasing green */
    set_palette(5, 0, 31, 0);
    set_palette(6, 0, 46, 0);
    set_palette(7, 0, 63, 0);
    set_palette(8, 15, 0, 0);        /* increasing red */
    set_palette(9, 31, 0, 0);
    set_palette(10, 46, 0, 0);
    set_palette(11, 63, 0, 0);
    set_palette(12, 15, 15, 15);      /* increasing gray */
    set_palette(13, 31, 31, 31);
    set_palette(14, 46, 46, 46);
    set_palette(15, 63, 63, 63);

    // compliment palette values of above
    set_palette(240, 63, 0, 0);
    set_palette(241, 46, 0, 0);
    set_palette(242, 31, 0, 0);
    set_palette(243, 15, 0, 0);
    set_palette(244, 63, 63, 63);
    set_palette(245, 46, 46, 46);
    set_palette(246, 31, 31, 31);
    set_palette(247, 15, 15, 15);
    set_palette(248, 0, 0, 63);
    set_palette(249, 0, 0, 46);
    set_palette(250, 0, 0, 31);
    set_palette(251, 0, 0, 15);
    set_palette(252, 0, 63, 0);
    set_palette(253, 0, 46, 0);
    set_palette(254, 0, 31, 0);
    set_palette(255, 0, 15, 0);
}

void pattern4(int xres, int yres)
{
    int xlength, ylength, i, pos;
    
    PGL_clearscreen(0, 0, xres, yres);
    init_palette4();
    
    xlength = xres/4;
    ylength = yres/4;
    pos = 0;
    for(i = 0; i < 4; i++)
    {
        // blue
        PGL_setfgcolor((unsigned long)i);
        PGL_drawrect(pos, 0, xlength, ylength);

        // green
        PGL_setfgcolor((unsigned long)(4+i));
        PGL_drawrect(pos, ylength, xlength, ylength);

        // red
        PGL_setfgcolor((unsigned long)(8+i));
        PGL_drawrect(pos, ylength*2, xlength, ylength);

        // gray
        PGL_setfgcolor((unsigned long)(12+i));
        PGL_drawrect(pos, ylength*3, xlength, ylength);

        pos = pos + xlength;
    }

    // draw WHITE border
    PGL_setfgcolor(15);
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

/* draw pattern for 8 bit color depth */   
void pattern8(int xres, int yres)
{
    int ylength, i;
    unsigned long color;
    
    PGL_clearscreen(0, 0, xres, yres);
    init_palette8();

    // set index 0 to black to remove BLUE palette artifact while drawing
    set_palette(0, 0, 0, 0);
    
    ylength = yres/4;
    for(i = 0; i < xres; i++)
    {
        color = (unsigned long)i;
        color = (color * 63)/(xres - 1);

        // blue
        PGL_setfgcolor(color);
        PGL_drawrect(i, 0, 1, ylength);

        // green
        PGL_setfgcolor(64+color);
        PGL_drawrect(i, ylength, 1, ylength);

        // red
        PGL_setfgcolor(128+color);
        PGL_drawrect(i, ylength*2, 1, ylength);

        // gray
        PGL_setfgcolor(192+color);
        PGL_drawrect(i, ylength*3, 1, ylength);
    }

    // set index 0 to bright blue
    set_palette(0, 0, 0, 63);

    // draw WHITE border
    PGL_setfgcolor(255);
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

/* draw pattern for 16 bit color depth using 555 color weighting */
void pattern15(int xres, int yres)
{
    int ylength, i;
    unsigned long color;
    unsigned int r, g, b, gray;
    
    PGL_clearscreen(0, 0, xres, yres);
    
    ylength = yres/4;

    for(i = 0; i < xres; i++)
    {
        color = (unsigned long)i;
        color = (color * 31)/(xres - 1);

        r = (unsigned int)(color << 10);
        g = (unsigned int)(color << 5);
        b = (unsigned int)(color);
        gray = r | g | b;

        // blue
        PGL_setfgcolor((unsigned long)(0x001f - b));
        PGL_drawrect(i, 0, 1, ylength);

        // green
        PGL_setfgcolor((unsigned long)(g));
        PGL_drawrect(i, ylength, 1, ylength);

        // red
        PGL_setfgcolor((unsigned long)(0xf800 - r));
        PGL_drawrect(i, ylength*2, 1, ylength);

        // gray
        PGL_setfgcolor((unsigned long)gray);
        PGL_drawrect(i, ylength*3, 1, ylength);
    }

    // draw WHITE border
    PGL_setfgcolor(0xffff);
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

/* draw pattern for 16 bit color depth using 565 color weighting */
void pattern16(int xres, int yres)
{
    int ylength, i;
    unsigned long color;
    unsigned int r, g, b, gray;
    
    PGL_clearscreen(0, 0, xres, yres);
    
    ylength = yres/4;

    for(i = 0; i < xres; i++)
    {
        color = (unsigned long)i;
        color = (color * 31)/(xres - 1);

        r = (unsigned int)(color << 11);
        g = (unsigned int)(color << 5);
        b = (unsigned int)(color);
        gray = r | (g << 1) | b;

        // blue
        PGL_setfgcolor((unsigned long)(0x001f - b));
        PGL_drawrect(i, 0, 1, ylength);

        // green
        PGL_setfgcolor((unsigned long)(g << 1));
        PGL_drawrect(i, ylength, 1, ylength);

        // red
        PGL_setfgcolor((unsigned long)(0xf800 - r));
        PGL_drawrect(i, ylength*2, 1, ylength);

        // gray
        PGL_setfgcolor((unsigned long)gray);
        PGL_drawrect(i, ylength*3, 1, ylength);
    }

    // draw WHITE border
    PGL_setfgcolor(0xffff);
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

/* draw pattern for 24 bit color depth using RGB or BGR color weighting

   depth: DEPTH_24_RGB (0) = RGB,
          DEPTH_24_BGR (1) = BGR
*/
void pattern24(int xres, int yres, int depth)
{
    int ylength, i;
    unsigned long color;
    unsigned long r, g, b, gray;

    PGL_clearscreen(0, 0, xres, yres);
    
    ylength = yres/4;

    for(i = 0; i < xres; i++)
    {
        color = (unsigned long)i;
        color = (color * 0xff)/(xres - 1);

        // RGB
        switch(depth)
        {
            case DEPTH_24_RGB:
                r = color << 16;
                g = color << 8;
                b = color;
                gray = r | g | b;
                PGL_setfgcolor(0x0000ff - b);           // blue (decreasing)
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green (increasing)
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0xff0000 - r);           // red (decreasing)
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray (increasing)
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;

            case DEPTH_24_BGR:
                r = color;
                g = color << 8;
                b = color << 16;
                gray = r | g | b;
                PGL_setfgcolor(0xff0000 - b);           // blue
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0x0000ff - r);           // red
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;
        }
    }

    // draw WHITE border
    PGL_setfgcolor(0xffffff);
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

/* draw pattern for 32 bit color depth using RGBa color weighting

   depth: DEPTH_32_RGBa (0) = RGBa,
          DEPTH_32_aBGR (1) = aBGR,
          DEPTH_32_aRGB (2) = aRGB,
          DEPTH_32_BGRa (3) = aBGR
*/
void pattern32(int xres, int yres, int depth)
{
    int ylength, i;
    unsigned long color;
    unsigned long r, g, b, gray;
    
    PGL_clearscreen(0, 0, xres, yres);
    
    ylength = yres/4;

    for(i = 0; i < xres; i++)
    {
        color = (unsigned long)i;
        color = (color * 0xff)/(xres - 1);

        switch(depth)
        {
            case DEPTH_32_RGBA:
                r = color << 24;
                g = color << 16;
                b = color << 8;
                gray = r | g | b;
                PGL_setfgcolor(0x0000ff00 - b);         // blue (decreasing)
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green (increasing)
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0xff000000 - r);         // red (decreasing)
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray (increasing)
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;

            case DEPTH_32_ABGR:
                r = color;
                g = color << 8;
                b = color << 16;
                gray = r | g | b;
                PGL_setfgcolor(0x00ff0000 - b);         // blue
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0x000000ff - r);         // red
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;

            case DEPTH_32_ARGB:
                r = color << 16;
                g = color << 8;
                b = color;
                gray = r | g | b;
                PGL_setfgcolor(0x000000ff - b);         // blue
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0x00ff0000 - r);         // red
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;

            case DEPTH_32_BGRA:
                r = color << 8;
                g = color << 16;
                b = color << 24;
                gray = r | g | b;
                PGL_setfgcolor(0xff000000 - b);         // blue
                PGL_drawrect(i, 0, 1, ylength);
                PGL_setfgcolor(g);                      // green
                PGL_drawrect(i, ylength, 1, ylength);
                PGL_setfgcolor(0x0000ff00 - r);         // red
                PGL_drawrect(i, ylength*2, 1, ylength);
                PGL_setfgcolor(gray);                   // gray
                PGL_drawrect(i, ylength*3, 1, ylength);
                break;
        }
    }
    
    // draw WHITE border
    switch(depth)
    {
        case DEPTH_32_RGBA:
        case DEPTH_32_BGRA: PGL_setfgcolor(0xffffff00); break;

        case DEPTH_32_ABGR:
        case DEPTH_32_ARGB: PGL_setfgcolor(0x00ffffff); break;
    }
    PGL_moveto(0, 0);
    PGL_drawto(xres-1, 0);
    PGL_drawto(xres-1, yres-1);
    PGL_drawto(0, yres-1);
    PGL_drawto(0, 0);
}

int CursorDemo(int type, int mouse)
{
    int buttonstat, mouse_x, mouse_y;
    unsigned long curcolor0, curcolor1;
    unsigned int *curdata;
    int wid;

    if (modeinfo.bpp == 4)
    {
        curcolor0 = 0x03;        // LIGHTBLUE in new palette
        curcolor1 = 0x07;        // LIGHTGREEN in new palette
    }
    else if (modeinfo.bpp == 8)
    {
        curcolor0 = 0x00;        // LIGHTBLUE in new palette
        curcolor1 = 0x7f;        // LIGHTGREEN in new palette
    }
    else
    {
        curcolor0 = PGL_getcolorcode(LIGHTBLUE);
        curcolor1 = PGL_getcolorcode(LIGHTGREEN);
    }

    switch(type)
    {
        case 0: curdata = (unsigned int *)cursor8x8;   wid = 8;  break;
        case 1: curdata = (unsigned int *)cursor16x16; wid = 16; break;
        case 2: curdata = (unsigned int *)cursor32x32; wid = 32; break;
        case 3: curdata = (unsigned int *)cursor64x64; wid = 64; break;
    }
    PGL_sethwcursor(modeinfo.yres, wid, wid, wid/2, wid/2,
                    curcolor0, curcolor1,
                    curdata);

    // check for mouse existence
    PGL_sethwcursorpos(modeinfo.xres/2, modeinfo.yres/2);
    PGL_enablehwcursor();
    if (mouse == 0)
    {
        // Wait until a key is pressed
        while (kbhit() == 0) ;
    }
    else
    {
        ScaleMouse(wid, wid);
        CenterMouse(&modeinfo);

        // Update mouse position until a key is pressed
        while (kbhit() == 0)
        {
            UpdateMouse(&buttonstat, &mouse_x, &mouse_y);
        }
    }
    getch();

    PGL_disablehwcursor();

    return (0);
}

