//==============================================================
//   8/12/99
//
//   Synchronous SRAM controller for PLX PCI 9054 mode C and J.
//   128K byte (32K x 32 bit) synchronous SRAM is used.
//   The memory map for the sync. SRAM is 2000_0000 - 2001_FFFFh.
//   A partial memory decode is used. The decode is only involved
//   address lines A31 to A28 (or A31-A29 and LD28 in J mode)
//
//	2/8/02
//	
//	Changed the "ready" signal to tree-state when there is no
//	access to the SRAM, Also, changed starting address of SRAM
//	from 2000_0000h to 0000_0000h.
//
//  3/14/02
//
//  Rewrited the local bus arbiter. Now the priority setting 
//  is 9054's LHOLD is higherst, LBR[1] is second,LBR[0] is
//  lowest.
//
//==============================================================

module SRAMCTR
	(
   
	// local bus signals

	CLK_50MHZ,        // clock to PCI9054, SRAM and controller 
	ADS_,             // address strobe from 9054
	BLAST_,           // burst last from 9054
	LBE_,             // byte enable from 9054
	LWDRD_,           // local bus read/write 
	ADDR_IN,          // local bus address inputs
	ADDR_4MSBS,       // local bus address A31 - A28
	READY_,           // ready signal to PLX PCI9054

	// address and control signals to synchronous SRAM

	SRAM_ADDR,        // address outputs to the sync. SRAM
	SRAMCS_,          // chip select to the SRAM
	SRAMOE_,          // output enable to the SRAM
	SRAM_BW_,         // byte enables in SRAM write cycle

	//bus arbitration

	LHOLD,            // bus hold request from PLX PCI9054
	LHOLDA,           // bus hold acknowledge
	LBR,              // two local bus request  
	LBG,              // two local bus grant 

	// chip selects

	CS_            	  // four chip select outputs
   
	);

	// port declarations

	output  [16:2]          SRAM_ADDR;
    output	                READY_;
	output                  SRAMCS_;
	output                  SRAMOE_;
	output  [3:0]           SRAM_BW_;
	output                  LHOLDA;
	output  [1:0]           LBG;
	output  [3:0]           CS_;

	input                   CLK_50MHZ;
	input                   ADS_, BLAST_;
	input   [3:0]           LBE_;
	input                   LWDRD_;
	input   [16:2]          ADDR_IN;
	input   [31:28]         ADDR_4MSBS;
	input                   LHOLD;
	input   [1:0]           LBR;
   
	reg     [16:2]          SRAM_ADDR;
	reg     [1:0]       	LBG;
	reg                     LHOLDA; 
	reg						SRAMCS_;
	reg						SRAMOE_;
	
	// internal veriables 

	reg     [3:0]           A31_28;
	reg		[2:0]			currentstate, nextstate;
	reg						oer;

	bufif0 (READY_,oer,oer);
	
	// chip selects  
	// Four most upper address lines, A31-A28, are used to  
	// generate four chip select signals for the board. They 
	// are CS[3:0] with address as
	//   
	//  CS_0: 4000_0000h
	//  CS_1: 5000_0000h
	//  CS_2: 6000_0000h
	//  CS_3: 7000_0000h
	

	wire  [3:0] CS_ = (ADDR_4MSBS == 4'b0100) ? 4'b1110:
			  (ADDR_4MSBS == 4'b0101) ? 4'b1101:
			  (ADDR_4MSBS == 4'b0110) ? 4'b1011:
			  (ADDR_4MSBS == 4'b0111) ? 4'b0111: 4'b1111;
   
	// byte enable encode for SRAM write cycles
	
	wire [3:0] SRAM_BW_ =({LWDRD_,A31_28}=='b1_0000)
						? LBE_[3:0] : 4'b1111;
			
	// store the upper address LA31 - LA28
   
	always @ (posedge CLK_50MHZ)
		if (LHOLD & !ADS_ & (ADDR_4MSBS==4'b0000))
			A31_28[3:0] = ADDR_4MSBS[31:28];
		else 			
			A31_28[3:0] = A31_28;
		
	// local bus arbitration

	always @(posedge CLK_50MHZ)
		if (LHOLD)
			LHOLDA = LHOLD;
		else
			LHOLDA = 0;
	always @ (posedge CLK_50MHZ)
		if (!LHOLD & LBR[1])
				LBG[1] = LBR[1];
		else
				LBG[1] = 0;
	always @ (posedge CLK_50MHZ)
		if (!LHOLD & !LBR[1] & LBR[0])
				LBG[0] = LBR[0];
		else
				LBG[0] = 0;

		
	// State definition
	parameter s0 = 4'b0000; // idle 
	parameter s1 = 4'b0001; // cycle start
	parameter s2 = 4'b0010; // single cycle wait state
	parameter s3 = 4'b0011; // single cycle last state
	parameter s4 = 4'b0100; // burst cycle wait state 
	parameter s5 = 4'b0101; // burst cycle repeat state 
	parameter s6 = 4'b0110; // burst cycle last state
	
	// SRAM address counter

		always @ (posedge CLK_50MHZ)	
			if (!ADS_)
				SRAM_ADDR[16:2] = ADDR_IN[16:2];
			else if (BLAST_ && !((currentstate == s1) && LWDRD_))
				SRAM_ADDR[12:2] = SRAM_ADDR[12:2] +1;
			else
				SRAM_ADDR[16:2] = SRAM_ADDR[16:2];


	//Next state logic

	always @ (ADS_ or BLAST_ or ADDR_4MSBS)
	casex (currentstate)
	
		s0: if (!ADS_ && (ADDR_4MSBS==4'b0000))
				nextstate = s1;
			else
				nextstate = s0;

		s1: if (!BLAST_)
				nextstate = s2;
			else if (BLAST_)
				nextstate = s4;
			else nextstate = s1;

		s2: 	nextstate = s3;

		s3:  if (!ADS_)
				nextstate = s1;
			 else
				nextstate = s0;
						
		s4:		nextstate = s5;
		
		s5:	if (BLAST_)
				nextstate = s5;
			else
				nextstate = s6;

		s6:	if (!ADS_)
				nextstate = s1;
			else
				nextstate = s0;
	endcase

	//output logic

	always @ (currentstate)
		casex(currentstate)

		s0: begin
				oer =1;
				SRAMCS_=1;
				SRAMOE_=1;
			end
		s1: begin
				oer =1;
				SRAMCS_=0;
				SRAMOE_=1;
			end
		s2:	begin
				oer=0;
				SRAMCS_=0;
				if (LWDRD_==0)
					SRAMOE_=0;
				else
					SRAMOE_=1;
			end
		s3:	begin
				oer=1;
				SRAMCS_=1;
				SRAMOE_=1;
			end
		s4:	begin
				oer=0;
				SRAMCS_=0;
				if (LWDRD_==0)
					SRAMOE_=0;
				else
					SRAMOE_=1;
			end
		s5:	begin
				oer=0;
				SRAMCS_=0;
				if (LWDRD_==0)
					SRAMOE_=0;
				else
					SRAMOE_=1;
			end

		s6:	begin
				oer=1;
				SRAMCS_=1;
				SRAMOE_=1;
			end

	endcase
	
	always @(posedge CLK_50MHZ)
		currentstate <= nextstate;


 endmodule

   













							  
