/*
  Adds a simulation of the 18S651 floppy disk controller. The uPD765 
  simulation is not complete but just enough to keep UT71 happy.


  Copyright 2016 David W. Schultz
  david.schultz@earthlink.net
  http://home.earthlink.net/~david.schultz

  Licensed under Creative Commons Attribution-NonCommercial-ShareAlike 3.0 
  license:

  http://creativecommons.org/licenses/by-nc-sa/3.0/us/


  The 18S651 includes, besides the uPD765, additional logic to manage and
  control the DMA system. It is controlled via the usual input and output
  commands. The ports are:

  uPD command/data - used to send commands and read responses
  uPD status       - read master status register

  DMA count        - write number of bytes to transfer
  control          - control operation of DMA transfers.

  EF3 is also used, it is driven by the INT pin of the uPD765

  This is not a high fidelity uPD765 simulation. Only enough of its operation
  is recreated to keep UT71 and MicroDOS happy.

  4 Oct. 2016 - CDSBIN revealed a problem with my code to avoid buffer
  overruns. The buffer it uses for directory sectors is at 0xFDD4. But after
  writing it doesn't reset R0 before performing a dummy read. This triggered
  the memory bounds check. So although I don't expect any reasonable program
  to need it, disk reads and writes will now wrap around memory.

  28 Nov. 2016 - Added file locking on the image files to help prevent image
  corruption of microdoswrite is used on an open image.
 */

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#include "1802sim.h"

#define SPCMD 3                    // uPD765 commands
#define RCCMD 7
#define SKCMD 15
#define RDCMD 0x46
#define WTCMD 0x45
#define SISCMD 8
#define INVCMD 0

#define DISK_SIZE (512*9*70)

static int mstatus = 0x80;                 // start ready to receive command
static int st0;
static int scount;
static int fds[4] =                        // file descriptors for image files
  { -1, -1, -1, -1 };
static int interrupt = 0;
static int dma_count, dma_control;
/*
  Not all commands are implemented and unimplemented commands are treated as
  invalid. This is a table of the number of bytes in the command packets.
 */
static int pcount[32] = {
  1,  // 0  - invalid
  1,  // 1  - invalid
  9,  // 2  - read track
  3,  // 3  - specify
  1,  // 4  - invalid
  9,  // 5  - write data
  9,  // 6  - read data
  2,  // 7  - recalibrate
  1,  // 8  - sense interrupt status
  1,  // 9  - invalid
  1,  // 10 - invalid
  1,  // 11 - invalid
  1,  // 12 - invalid
  1,  // 13 - invalid
  1,  // 14 - invalid
  3,  // 15 - seek 
  1,  // 16 - invalid
  1,  // 17 - invalid
  1,  // 18 - invalid
  1,  // 19 - invalid
  1,  // 20 - invalid
  1,  // 21 - invalid
  1,  // 22 - invalid
  1,  // 23 - invalid
  1,  // 24 - invalid
  1,  // 25 - invalid
  1,  // 26 - invalid
  1,  // 27 - invalid
  1,  // 28 - invalid
  1,  // 29 - invalid
  1,  // 30 - invalid
  1   // 31 - invalid
};
  
  
/*
  Each simulated drive needs to be connected to a file to simulate a
  floppy disk. This function handles that operation.
 */
void fdc_open_image(int drive, char *fname)
{
  int fd;

  if(drive < 0 || drive > 3)    // sanity check on drive #
    return;
  
  if(fds[drive] != -1)          // if open, close it
    close(fds[drive]);
  
  if((fd = open(fname, O_RDWR)) == -1)
      fprintf(stderr, "open of <%s> failed\n", fname);
  else
    {
      fds[drive] = fd;
      if(lockf(fds[drive], F_TLOCK, 0) == -1)
	{
	  close(fds[drive]);
	  fprintf(stderr, "File %s locked\nOpening read only.\n", fname);
	  fds[drive] = open(fname, O_RDONLY);
	}
    }
  
}

/*
  Close all locks and files.
 */
void fdc_termination(void)
{
  int i;
  
  for(i = 0; i < 4; i++)
    {
      if(fds[i] != -1)
        {
          lseek(fds[i], 0, 0);
          lockf(fds[i], F_ULOCK, 0);
          close(fds[i]);
        }
    }
}
/*
  Called by code for the B3 and BN3 instructions.
 */
int fdc_int(void)
{
  return interrupt;
}

static int cdata[16];  // command packet
static int pcn;

/*
  Transfer the data from the simulated file system. Data is transfered to/from
  the location at R[0].
 */
static void do_rdwr(int dir)
{
  int offset, drive, i;
  int fd;

  drive = cdata[1] & 3;                 // extract drive #
  if((fd = fds[drive]) == -1)
    {
      st0 = 0xc4 | drive;               // drive not ready error if not open
      scount = 1;
    }
  else
    {
      // Calculate location from cylinder and record number (head ignored)
      offset = (cdata[2]*9 + cdata[4]-1);
      /*
      printf("%c d:%d off:%d addr:%X cnt:%d cntl:%d\r\n", dir ? 'W' : 'R',
      	     drive, offset, R[0].w, dma_count, dma_control);
      */
      
      lseek(fd, offset*512, SEEK_SET);
      if(dir)           // disk write
	{
	  /*
	    Check to see if the transfer will cause R0 to wrap back
	    around to 0 and handle that case.
	   */
	  if(R[0].w > (64*1024 - dma_count))
	    {
	      i = 64*1024 - R[0].w;
	      write(fd, &sim_mem[R[0].w], i);
	      write(fd, &sim_mem[0], dma_count-i);
	    }
	  else
	    if(write(fd, &sim_mem[R[0].w], dma_count) != dma_count)
	      fprintf(stderr, "not all data written!\n");

	  R[0].w += dma_count;
	  R[0].w &= 0xffff;
	}
      else             // disk read
	{
	  /* The hardware can simulate data transfers for the FDC
	     without actually moving data. This is used to verify
	     that data has been written correctly. (CRC check)
	   */
	  if(dma_control == 3)   // actual transfer?
	    {
	      // Handle wrap around
	      if(R[0].w > (64*1024 - dma_count))
		{
		  i = 64*1024 - R[0].w;
		  read(fd, &sim_mem[R[0].w], i);
		  read(fd, &sim_mem[R[0].w], dma_count-i);
		}
	      else
		read(fd, &sim_mem[R[0].w], dma_count);
	      
	      R[0].w += dma_count;
	      R[0].w &= 0xffff;
	    }
	}
      st0 = 0 | drive;
      scount = 7;
    }
  mstatus = 0xd0;
  interrupt = 1;
}

/*
  Peform a command as specified by the data packet. Only a few commands
  are implemented. All others are invalid.
 */
static void do_cmd(void)
{
  switch(cdata[0])
    {
    case SPCMD:                  // specify
      //      printf("specify ");
      // Nothing to do
      break;
    case RCCMD:                  // recalibrate
      //      printf("recalibrate ");
      if(fds[cdata[1]&3] != -1)  // image file opened?
	{
	  pcn = 0;
	  st0 = 0x20 | (cdata[1]&3); // no, signal an error
	}
      else
	{
	  pcn = 0;
	  st0 = 0x40;
	}
      interrupt = 1;
      break;
    case SKCMD:                  // seek
      //      printf("seek ");
      // store some values in case this is followed by sense interrupt
      if(fds[cdata[1]&3] != -1)
	{
	  pcn = cdata[2];
	  st0 = 0x20 | (cdata[1]&3);
	}
      else
	{
	  pcn = 0;
	  st0 = 0x40;
	}
      interrupt = 1;
      break;
    case RDCMD:                  // read
      do_rdwr(0);
      break;
    case WTCMD:                  // write
      do_rdwr(1);
      break;
    case SISCMD:                 // sense interrupt status
      //      printf("sense ");
      interrupt = 1;
      mstatus = 0xd0;
      scount = 2;
      break;
    default:                 // Invalid command
      //      printf("invalid ");
      st0 = 0x80;
      scount = 1;
      mstatus = 0xc0;
      break;
    }
}

static int c_count = 0;
/*
  Data written to the command port is processed here.
 */
void fdc_command(int c)
{
  fflush(stdout);
  
  if(mstatus == 0x80)
    {
      cdata[c_count++] = c;

      if(c_count == pcount[cdata[0]&0x1f])
	{
	  do_cmd();
	  c_count = 0;
	}
    }
}

/*
  Reads from the data port are handled here.

  This is read to get the results of commands. UT71 only looks at the first
  byte (ST0). 

  ST0 -
      D7   - Interrupt code, normal termination is 00, but an invalid
      D6     command will result in 10.

      D5   - seek end, set after a seek completes
      D4   - equipment check
      D3   - not ready
      D2   - head address
      D1   - unit select
      D0
 */
/*
  UT71 only ever looks at ST0. So the rest don't matter, it is all just
  copies of ST0.
 */
int fdc_data(void)
{
  if((mstatus&0xc0) == 0xc0)
    {
      interrupt = 0;               // clear interrupt
      if(scount-- == 1) // last byte, change to ready for command
	mstatus = 0x80;

      return st0;
    }
  return 0;
}

/*
  Reads of the uPD765 master status port come here.

  This is complicated by its interaction with the command and data functions.

  The 5 least significant bits indicate if the FDC is busy with some task.

  Mostly UT71 tests the MRQ (master request) bit. But it also checks the 
  DIO (direction) bit in a couple of spots.

*/


int fdc_mstatus(void)
{
  return mstatus;
}

/*
  Writes to the DMA transfer counter.

  The hardware counter expects a count of the number of sectors (single
  density 128bytes/sector) to transfer.
 */

void fdc_set_dma(int count)
{
  dma_count = count*128;
}

/*
  Writes to the DMA control register.

  This tells the hardware what direction the transfer is in. We can figure
  that out from the command so that bit is ignored. It also has a bit used
  to cause dummy DMA cycles. The uPD765 is fed DMA acks but the 1802
  is not asked to generate actual DMA cycles.
 */
void fdc_control(int val)
{
  dma_control = val;
}
  
