/***************************************************************************\
* pie/hostos.c:  PIE host os interface                                      *
* Copyright (C) 1991 Advanced RISC Machines Limited. All rights reserved.   *
\***************************************************************************/

/*
 * RCS $Revision: 1.17 $
 * Checkin $Date: 1993/04/20 12:18:16 $
 * Revising $Author: hmeekings $
 */

#include <time.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef __STDC__
#  define unlink(s) remove(s)
#endif

#ifndef __STDC__
typedef char *VoidStar;
#endif

#ifndef FOPEN_MAX
#define FOPEN_MAX 256
#endif
#define UNIQUETEMPS 256
#define TARGETL_tmpnam 80

#ifdef __riscos
extern int _fisatty(FILE *);
#define isatty_(f) _fisatty(f)
#define EBADF (-1)
#define EMFILE (-1)
#else
#ifdef __ZTC__
#define isatty_(f) isatty((f)->_file)
#else
#ifdef macintosh
#include <ioctl.h>
#define isatty_(f) (~ioctl((f)->_file,FIOINTERACTIVE,NULL))
#else
#define isatty_(f) isatty(fileno(f))
#endif
#endif
#endif

#ifdef macintosh
#define FIXCRLF(t,buffer,size) \
  if (!(t & BINARY)) { \
     char *ptr ; \
     for (ptr = buffer ; ptr < (buffer + size) ; ptr++) \
        if (*ptr=='\n' || *ptr=='\r') \
           *ptr ^= 7 ; \
     }
#else
#define FIXCRLF(t,b,s)
#endif

typedef unsigned long ARMword;
#include "dbg_hif.h"
#include "pirdi.h"
#include "hostos.h"

#define rw_quantum 4096 /* number of bytes per host file r/w op */
#define rdp_quantum 64 /* max bytes per rdp read / write op. */
                       /* rdp_quantum must exactly divide rw_quantum */

static void fillbuffer(ARMword addr, char *buffer, int toread) {
  size_t offset = 0;
  while (toread != 0) {
    size_t bytes = toread > rdp_quantum ? rdp_quantum : toread;
    size_t bytes_copy = bytes;
    pisd_RDI_read(addr + offset, &buffer[offset], &bytes);
    toread -= bytes_copy; offset += bytes_copy;
  }
}

static char *GetString(SWIArgDesc *arg, char *buf, size_t len) {
  if (arg->type == swat_stringlit)
    return arg->s.s;
  else if (arg->n < len) {
    fillbuffer(arg->s.p, buf, (int)arg->n);
    buf[arg->n] = 0;
    return buf;
  } else
    return 0;
}

extern char* commandline ;
static ARMword savederrno;
static FILE *FileTable[FOPEN_MAX] = {NULL} ;
static char FileFlags[FOPEN_MAX] ;
static char *TempNames[UNIQUETEMPS] = {NULL} ;

#define NOOP 0
#define BINARY 1
#define READOP 2
#define WRITEOP 4

int HostOS_HandleSWI(ARMword swicode, SWIArgDesc *swiargs)
{
  ARMword addr, temp;
  char buffer[rw_quantum];

  switch (swicode) {
  case SWI_WriteC:
    pisd_hostif->writec(pisd_hostif->hostosarg, (int)swiargs[0].n);
    return 0;

  case SWI_Write0:
    { char *s = GetString(&swiargs[0], buffer, rw_quantum);
      int i;
      for (i = 0; i < swiargs[0].n; i++)
        pisd_hostif->writec(pisd_hostif->hostosarg, *s++);
      return 0;
    }

  case SWI_ReadC:
    swiargs[0].n = (ARMword)pisd_hostif->readc(pisd_hostif->hostosarg);
    return 1;

  case SWI_CLI:
    { char *s = GetString(&swiargs[0], buffer, rw_quantum);
      swiargs[0].n = (ARMword)system(s);
      savederrno = errno;
      return 1;
    }

  case SWI_GetEnv:
    { unsigned bytes = strlen(commandline) + 1 ;
      pisd_RDI_write(commandline, swiargs[0].n, &bytes);
      return 0;
    }

  case SWI_GetErrno:
    swiargs[0].n = savederrno;
    return 1;

  case SWI_Time:
    swiargs[0].n = (ARMword)time(NULL);
    return 1;

  case SWI_Remove:
    { char *s = GetString(&swiargs[0], buffer, rw_quantum);
      swiargs[0].n = unlink(s);
      savederrno = errno;
      return 1;
    }

  case SWI_Rename:
    { char *s1 = GetString(&swiargs[0], buffer, rw_quantum / 2);
      char *s2 = GetString(&swiargs[1], &buffer[rw_quantum / 2], rw_quantum / 2);
      swiargs[0].n = rename(s1, s2);
      savederrno = errno;
      return 1;
    }

  case SWI_Open:
    { static char* fmode[] = {"r","rb","r+","r+b",
                               "w","wb","w+","w+b",
                               "a","ab","a+","a+b",
                               "r","r","r","r"} /* last 4 are illegal */ ;
      unsigned type, temp;
      FILE *fptr ;
      char *s = GetString(&swiargs[0], buffer, rw_quantum);

      type = (unsigned)(swiargs[1].n & 15L);
      if (strcmp(s, ":tt") == 0 && (type == 0 || type == 1)) /* opening tty "r" */
        fptr = stdin;
      else if (strcmp(s, ":tt") == 0 && (type == 4 || type == 5)) /* opening tty "w" */
        fptr = stderr;
      else
        fptr = fopen(s, fmode[type]);

      swiargs[0].n = 0 ;
      if (fptr != NULL) {
        for (temp = 0 ; temp < FOPEN_MAX ; temp++)
          if (FileTable[temp] == NULL) {
            FileTable[temp] = fptr ;
            FileFlags[temp] = type & 1 ; /* preserve the binary bit */
            swiargs[0].n = (ARMword)temp + 1 ;
            break ;
            }
        if (swiargs[0].n == 0)
          savederrno = EMFILE ; /* too many open files */
        else
          savederrno = errno ;
        }
      else
        savederrno = errno ;
      return 1;
    }

  case SWI_Close:
    { FILE *fptr ;

      temp = swiargs[0].n ;
      if (temp == 0 || temp > FOPEN_MAX || FileTable[temp - 1] == 0) {
        savederrno = EBADF ;
        swiargs[0].n = -1L ;
        return(1) ;
        }
      temp-- ;
      fptr = FileTable[temp] ;
      FileTable[temp] = NULL ;
      swiargs[0].n = (fptr == stdin || fptr == stderr) ? 0 : fclose(fptr);
      savederrno = errno;
      return 1;
    }

  case SWI_Write:
    { size_t left = (size_t)swiargs[1].n;
      FILE *fptr ;
      unsigned type ;

      temp = swiargs[0].n ;
      if (temp == 0 || temp > FOPEN_MAX || FileTable[temp - 1] == 0) {
        savederrno = EBADF ;
        swiargs[0].n = -1L ;
        return(1) ;
        }

      temp-- ;
      fptr = FileTable[temp] ;
      type = FileFlags[temp] ;
      if (type & READOP)
        fseek(fptr,0,1) ;
      FileFlags[temp] = (type & BINARY) | WRITEOP;
      if (swiargs[1].type == swat_stringlit) {
        FIXCRLF(type, swiargs[1].s.s, left);
        if (fptr == stderr || fptr == stdout)
          left -= pisd_hostif->write(pisd_hostif->hostosarg, swiargs[1].s.s, left);
        else
          left -= fwrite(swiargs[1].s.s, 1, left, fptr);
      } else {
        ARMword addr = swiargs[1].s.p;
        while (left != 0) {
          size_t toread = left > rw_quantum ? rw_quantum : left;
          fillbuffer(addr, buffer, toread);
          FIXCRLF(type, buffer, toread);
          { size_t written = (fptr == stderr || fptr == stdout)
              ? pisd_hostif->write(pisd_hostif->hostosarg, buffer, toread)
              : fwrite(buffer, 1, toread, fptr);
            left -= written;
            if (written != toread) break;
          }
          addr += toread;
        }
      }
      swiargs[0].n = left;
      savederrno = errno;
      return 1;
    }

  case SWI_Read:
    { size_t left = (size_t)swiargs[2].n;
      ARMword addr = swiargs[1].n;
      FILE *fptr ;
      unsigned type ;

      temp = swiargs[0].n ;
      if (temp == 0 || temp > FOPEN_MAX || FileTable[temp - 1] == 0) {
        savederrno = EBADF ;
        swiargs[0].n = -1L ;
        return(1) ;
        }

      temp-- ;
      fptr = FileTable[temp] ;
      type = FileFlags[temp] ;
      if (type & WRITEOP)
         fseek(fptr,0,1) ;
      FileFlags[temp] = (type & BINARY) | READOP;
      while (left != 0) {
        size_t wanted;
        size_t got;
        if (isatty_(fptr)) {
          wanted = (left >= rw_quantum) ? rw_quantum : left + 1;
          if (pisd_hostif->gets(pisd_hostif->hostosarg, buffer, wanted) != 0)
            got = strlen(buffer);
          else
            got = 0;
          FIXCRLF(type, buffer, got);
          wanted--; /* 1 char used for terminating null */
        } else {
          wanted = (left >= rw_quantum)?rw_quantum:left;
          got = fread(buffer, 1, wanted, fptr);
          FIXCRLF(type, buffer, got);
          }

        { size_t towrite = got;
          size_t offset = 0;
          while (towrite != 0) {
            size_t bytes = towrite > rdp_quantum ? rdp_quantum : towrite;
            size_t bytes_copy = bytes;
            pisd_RDI_write(&buffer[offset], addr + offset, &bytes);
            offset += bytes_copy; towrite -= bytes_copy;
          }
        }
        addr += got; left -= got;
        if (got != wanted) break;
      }
      swiargs[0].n = left;
      savederrno = errno;
      return 1;
    }

  case SWI_Seek:
    {FILE *fptr ;

     if (swiargs[0].n == 0 || swiargs[0].n > FOPEN_MAX || FileTable[swiargs[0].n - 1] == 0) {
       savederrno = EBADF ;
       swiargs[0].n = -1L ;
       return(1) ;
       }
     fptr = FileTable[swiargs[0].n - 1] ;
     swiargs[0].n = fseek(fptr, (long)swiargs[1].n, 0);
     savederrno = errno;
     return 1;
     }

  case SWI_Flen:
    { FILE *fptr ;

      if (swiargs[0].n == 0 || swiargs[0].n > FOPEN_MAX || FileTable[swiargs[0].n - 1] == 0) {
        savederrno = EBADF ;
        swiargs[0].n = -1L ;
        return(1) ;
        }
      fptr = FileTable[swiargs[0].n - 1] ;
      addr = (ARMword)ftell(fptr);
      if (fseek(fptr, 0L, 2) < 0)
        swiargs[0].n = -1;
      else {
        swiargs[0].n = (ARMword)ftell(fptr);
        (void)fseek(fptr, addr, 1);
      }
      savederrno = errno;
      return 1;
    }

  case SWI_IsTTY:
    {FILE *fptr ;
     if (swiargs[0].n == 0 || swiargs[0].n > FOPEN_MAX || FileTable[swiargs[0].n - 1] == 0) {
       savederrno = EBADF ;
       swiargs[0].n = -1L ;
       return(1) ;
       }
     fptr = FileTable[swiargs[0].n - 1] ;
     swiargs[0].n = isatty_(fptr);
     savederrno = errno;
     return 1;
     }

  case SWI_TmpNam:
    { unsigned sig = (unsigned)swiargs[1].n & 0xff;
      char *p;
      unsigned l;

      if (TempNames[sig] == NULL) {
        if ((TempNames[sig] = (char *)malloc(L_tmpnam)) == NULL) {
          swiargs[0].n = 0;
          return 1;
          }
        (void)tmpnam(TempNames[sig]);
        }
      p = TempNames[sig];
      l = strlen(p) + 1;
      if (l <= TARGETL_tmpnam)
        pisd_RDI_write(p, swiargs[0].n, &l);
      else
        swiargs[0].n = 0;
      return 1;
    }
  default:
    return 0;
  }
}
