/*
 * Pre-processor scaffolding needed to make string.c compilable as a
 * stand alone shared string library.
 */

#define BYTESEX_EVEN   1    /* little-endian */  
/*#define BYTESEX_ODD    1  -- BIG-endian */

/* Include a copyable data image in the library */
#define DATA_IN_LIBRARY 1

/* interns.h needs to be a separate (dyummy) file because
 * string.c also includes it.
 */
#include "interns.h"

/*
 * #defines needed to enable compilation of each section of string.c.
 */
#define memcpy_c
#define memmove_c
#define memchr_c
#define memcmp_c
#define memset_c
#define strcat_c
#define strchr_c
#define strcmp_c
#define strcoll_c
#define strcpy_c
#define strcspn_c
#define strerror_c
#define strlen_c
#define strncat_c
#define strncmp_c
#define strncpy_c
#define strpbrk_c
#define strrchr_c
#define strspn_c
#define strstr_c
#ifndef DATA_IN_LIBRARY
#define strtok_c
#endif
#define strxfrm_c
#define _strerror_c

/*
 * Additions needed to make string.c compile to a stand alone object.
 */

#include <string.h>

/*
 * This function is referred to from _strerror(). We define it here:
 * (1) so that strlib depends on nothing else and 
 * (2) so you can see by calling strerror(42) that the shared version,
 *     not the standard library version, was called.
 */
static char *_hostos_error_string(unsigned int n, char *v)
{   char *s = v;
    int j;
    strcpy(s, "unknown shared string-library error 0x");
    s += strlen(s);
    for (j = 0;  j < 8;  ++ j)
    {   *s++ = "0123456789ABCDEF"[n >> 28];
        n <<= 4;
    }
    *s = 0;
    return v;
}


#ifdef DATA_IN_LIBRARY

/*
 * The implementation of strtok in the standard C library does not
 * support copying its data image. The reason is that it uses:
 *    static char nullstring[] = "";
 *    static char *saves1 = nullstring;
 * which would require relocation at run time, if copied.
 * The version below avoids this, but is otherwise identical.
 * (See the line below marked <use saved pointer>).
 */

#ifdef SAME_AS_STANDARD_LIBRARY
static char nullstring[] = "";
static char *saves1 = nullstring;
#else
static char *saves1 = NULL;
#endif

char *strtok(char *s1, const char *s2)
{   char *s0;
#ifdef SAME_AS_STANDARD_LIBRARY
    if (s1 == 0) s1 = saves1;                          /* use saved pointer */
#else
    if (s1 == 0) s1 = (saves1 == NULL) ? "" : saves1;  /* use saved pointer */
#endif
    if (*(s1 += strspn(s1,s2)) == 0) s0 = 0;             /* no tokens */
    else { s0 = s1;
           if (*(s1 += strcspn(s1,s2)) != 0) *s1++ = 0;  /* insert 0 if nec */
         }
    return (saves1 = s1, s0);
}

#endif
