/****************************************************************************/
/*     File:   pci2040dll.h                                                  */
/*                                                                          */
/*     Purpose: This is the public header file for the C54x EVM DLL         */
/*                                                                          */
/*     Copyright (c) 1998, DNA Enterprises, Inc.                            */
/****************************************************************************/
#ifndef _PCI2040_DLL_H_
#define _PCI2040_DLL_H_

#include "ddpci2040.h"     // board driver header file
#include "ppc54xdll.h"     // board driver header file

/****************************************************************************/
/* defines                                                                  */
/****************************************************************************/
 /* Memory space type definition */
#define     PROG_MEMORY          0
#define     DATA_MEMORY          1
#define     FLASH_MEMORY         2

#define     NO_FIFO_TIMEOUT      0

#define     MAX_PCI2040_DSPS     4
#define     HPI_SPACE_SIZE       0x2000
#define     HPI_REG_OFFSET       0x800

/***********************************************************************/
/*     DEFINE TBC REGISTERS INDICES                                    */
/***********************************************************************/
#define TBC_REG_CNTL0       (0x00>>2)
#define TBC_REG_CNTL1       (0x04>>2)
#define TBC_REG_CNTL2       (0x08>>2)
#define TBC_REG_CNTL3       (0x0C>>2)
#define TBC_REG_CNTL4       (0x10>>2)
#define TBC_REG_CNTL5       (0x14>>2)
#define TBC_REG_CNTL6       (0x18>>2)
#define TBC_REG_CNTL7       (0x1C>>2)
#define TBC_REG_CNTL8       (0x20>>2)
#define TBC_REG_CNTL9       (0x24>>2)
#define TBC_REG_MINOR_CMD   (0x28>>2)
#define TBC_REG_MAJOR_CMD   (0x2C>>2)
#define TBC_REG_CNT1_LOW    (0x30>>2)
#define TBC_REG_CNT1_HIGH   (0x34>>2)
#define TBC_REG_CNT2_LOW    (0x38>>2)
#define TBC_REG_CNT2_HIGH   (0x3C>>2)
#define TBC_REG_STATUS0     (0x40>>2)
#define TBC_REG_STATUS1     (0x44>>2)
#define TBC_REG_STATUS2     (0x48>>2)
#define TBC_REG_STATUS3     (0x4C>>2)
#define TBC_REG_CAPT_LOW    (0x50>>2)
#define TBC_REG_CAPT_HIGH   (0x54>>2)
#define TBC_REG_SERIAL_RD   (0x58>>2)
#define TBC_REG_SERIAL_WR   (0x5C>>2)

/***********************************************************************/
/*     DEFINE CPLD REGISTERS INDICES                                   */
/***********************************************************************/
#define CPLD_CTRL_OFFSET   0x00
#define CPLD_STAT1_OFFSET  0x04
#define CPLD_STAT2_OFFSET  0x08
#define CPLD_REV_OFFSET    0x0c
#define CPLD_SEM0_OFFSET   0x10
#define CPLD_SEM1_OFFSET   0x14

/***********************************************************************/
/*     DEFINE HPI REGISTERS INDICES                                    */
/***********************************************************************/
#define C54X_HPIC			0x00
#define C54X_HPIA			0x08
#define C54X_HPID_AUTOINC	0x04
#define C54X_HPID_NOAUTOINC	0x0c

/* fixup True/False defines */
#undef FALSE
#undef TRUE 
#define FALSE   ((BOOLEAN)0)
#define TRUE    ((BOOLEAN)1)

#define PCI2040_GLOBAL_EVENT_BASE_NAME "Pci2040MsgEventDev"

#ifndef INTDSP0
#define INTDSP0  (1<<0)     // Bit set if an interrupt has been generated by a device connected
#endif                     // to the HPI[0] interface. Software can set this bit for diagnostics.

#ifndef INTDSP1
#define INTDSP1  (1<<1)     // Bit set if an interrupt has been generated by a device connected
#endif                      // to the HPI[1] interface. Software can set this bit for diagnostics.

#ifndef INTDSP2
#define INTDSP2  (1<<2)     // Bit set if an interrupt has been generated by a device connected
#endif                      // to the HPI[2] interface. Software can set this bit for diagnostics.

#ifndef INTDSP3
#define INTDSP3  (1<<3)     // Bit set if an interrupt has been generated by a device connected
#endif                      // to the HPI[3] interface. Software can set this bit for diagnostics.

#ifndef GPINT
#define GPINT    (1<<26)    // Bit set if an interrupt has been generated by a device connected
#endif                      // to the GPINT# interface. Software can set this bit for diagnostics.

#ifndef INTGPIO2
#define INTGPIO2 (1<<27)    // Set when GPIO2Pin selects GPIO2 as an interrupt event input, and the
#endif                      // event type selected by the GPIO Interrupt Event Type Register occurs.

#ifndef INTGPIO3
#define INTGPIO3 (1<<28)    // Set when GPIO3Pin selects GPIO3 as an interrupt event input, and the
#endif                      // event type selected by the GPIO Interrupt Event Type Register occurs.

#ifndef GPERROR
#define GPERROR  (1<<29)    // Bit set upon serious error conditions on the GP interface, and
#endif                      // allows software to gracefully terminate communication with a GP device.
                            
#ifndef HPIERROR
#define HPIERROR (1<<30)    // Bit set upon serious error conditions on the HPI interface, and allows
#endif                      // software to gracefully terminate communication with an HPI device.
                            // This bit is the OR combination of the HPI errors in the HPI Error Report
                            // Register.

/****************************************************************************/
/* Structure used to get HPI mapped addresses                               */
/****************************************************************************/
   // Structure used to write PCI register
typedef struct _PCI2040_MAPPED_ADDRESS {
   PULONG   Address;     // Register to read/write
   ULONG    length;      // Data to read/write
} PCI2040_MAPPED_ADDRESS, *PPCI2040_MAPPED_ADDRESS;

typedef struct _PCI2040_INIT_DATA {
    PCI2040_DUMP_CONFIG_REGS pci_regs;
} PCI2040DLL_INIT_DATA, *PPCI2040DLL_INIT_DATA;

typedef struct _MemSpaceConfig {
   BOOL                  ovly;
   BOOL                  drom;
   BOOL                  mp_mc;
} MemSpaceConfig, *PMemSpaceConfig;

/****************************************************************************/
/* Enums                                                                    */
/****************************************************************************/
 // DSP board type definition
typedef enum {
    TYPE_UNKNOWN = 0,
    TYPE_PCI2040_C5410_EVM,
    TYPE_C5402_DSK
} EVMDSK54X_BOARD_TYPE, *PEVMDSK54X_BOARD_TYPE;

//board connection modes
typedef enum {
    EVMDSK54X_PCI_OPEN,
    EVMDSK54X_PARALLEL_OPEN
} EVMDSK54X_OPEN_TYPE, *PEVMDSK54X_OPEN_TYPEE;

typedef enum {
    ERASE_ALL = -1,
} EVMDSK54X_FLASH_CONTROL, *PEVMDSK54X_FLASH_CONTROL;

//DSP-to-HOST events
typedef enum {
    DSP0_INT = 0,
    DSP1_INT,
    DSP2_INT,
    DSP3_INT,
    GP_INT,
    GPIO2_INT,
    GPIO3_INT,
    HPIERROR_INT,
    HPIERROR0_INT,
    HPIERROR1_INT,
    HPIERROR2_INT,
    HPIERROR3_INT
} PCI2040_HOST_EVENT, *PPCI2040_HOST_EVENT;

//HOST-to-DSP events
typedef enum {
    NMI_INT = 0,
    DSP_INT,
    DSP_INT3
} PCI2040_DSP_EVENT, *PPCI2040_DSP_EVENT;

typedef enum {
    PCI2040_SEM0,
    PCI2040_SEM1
} PCI2040_SEM, PPCI2040_SEM;

typedef enum {
   CSR_BAR = 0,
   HPI_BAR,
   TBC_BAR,
   CPLD_BAR
} PCI2040_BAR, *PPCI2040_BAR;

/****************************************************************************/
/* typedefs                                                                 */
/****************************************************************************/
/* "printf" redirection function definition */
typedef void (PCI2040_PRINTF_FUNC)(char *);
typedef PCI2040_PRINTF_FUNC  *PPCI2040_PRINTF_FUNC;

typedef void (PCI2040_INPUT_FUNC)(char *);
typedef PCI2040_INPUT_FUNC  *PPCI2040_INPUT_FUNC;

typedef void (*FileIOFunc)(LPVOID hDevice, ULONG dspNum);

#undef DLL32_ENTRY
#ifdef evmdsk54xdll_c
    // export function, for building DLL
#define DLL32_ENTRY  __declspec(dllexport)
#else
    // import function, for using DLL
#define DLL32_ENTRY  __declspec(dllimport)
#endif

#ifdef __cplusplus

extern "C" {

#endif
    
/****************************************************************************/
/* all DLL function definitions                                             */
/****************************************************************************/

/*****************************************************************************
*  evmdsk54x_open()
*
*	 Description: 
*		Opens the board devices. Some functions included in the API requires an 
*		exclusive open so be sure to pass in a 1 for the exclusive flag.
*
*  Parameters:
*		board_index - corresponds to the board number for which the handle is 
*                 requested.
*		exclusive_flag - indicates exclusive ownership of the board.
*     openType - Specifies either a parallel or PCI connect. Note that the 
*                5402 DSK only supports a parallel conect. If a PCI connection 
*                is specified for the DSK an INVALID_HANDLE_VALUE will be returned
*
*  Return:
*		A handle to the board.
*
*******************************************************************************/
DLL32_ENTRY HANDLE evmdsk54x_open(int board_index, 
                                  EVMDSK54X_BOARD_TYPE boardType,
                                  EVMDSK54X_OPEN_TYPE openType,
                                  BOOL exclusive_flag);

/*****************************************************************************
*  evmdsk54x_init()
*
*	 Description: 
*		Initializes the board, specifically the montana device.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*                 requested.
*		pInitData - pointer to a PPCI2040DLL_INIT_DATA objects which has the 
*		configuration information. This can be a NULL, in which case the default 
*		configuration is used.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_pci_init(HANDLE hDevice, PPCI2040DLL_INIT_DATA pInitData);

/*****************************************************************************
*  evmdsk54x_close()
*
*	 Description: 
*		Closes the handle to the board. No communication with the board is possible 
*		after a call to this routine.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_close(HANDLE hDevice);

/*****************************************************************************
*  evmdsk54x_reset_board()
*
*	 Description: 
*		Resets the board to a known state.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_reset_board(HANDLE hDevice);

/*****************************************************************************
*  evmdsk54x_reset_dsp()
*
*	 Description: 
*		Holds the DSP device in reset.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		dspNum - specifies the DSP devioe. Up to 4 DSPs may be supported on the board.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_reset_dsp(HANDLE hDevice, ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_unreset_dsp()
*
*	 Description: 
*		Removes the DSP device from reset. The DSP will begin executing code if a 
*		COFF file is loaded.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		dspNum - specifies the DSP devioe. Up to 4 DSPs may be supported on the board.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_unreset_dsp(HANDLE hDevice, ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_board_info()
*
*	 Description: 
*		Retrieves the board type and revision id.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pBoardType - The board type.
*		pRevId - The revision identifier
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_board_info(HANDLE hDevice,
                                      PEVMDSK54X_BOARD_TYPE pBoardType, 
                                      PULONG pRevId);

/*****************************************************************************
*  evmdsk54x_get_configuration()
*
*	 Description: 
*		Retrieves the PCI configuration information.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pPciConfig - A pointer to a PPCI2040_DUMP_CONFIG_REGS object which contains
*		the PCI information.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_get_pci_configuration(HANDLE hDevice,
												 PPCI2040_DUMP_CONFIG_REGS pReg);

/*****************************************************************************
*  evmdsk54x_coff_load()
*
*	 Description: 
*		Loads a COFF file onto the board.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		lp_hpi - A pointer to the HPI map object returned from the call to 
*   evmdsk54x_hpi_open().
*		filename - The name of the COFF file.
*		verboseFlag - Display basic COFF file information via the standard output stream.
*   clear_bss_flag - Initialize all bss section variable to zero.
*   dump - Display all data via the standard output stream.
*   dspNum - Specifies the DSP to load the COFF file into.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_coff_load(HANDLE hDevice, 
                                   LPVOID lp_hpi,
                                   char *filename, 
                                   BOOL verboseFlag,
                                   BOOL clear_bss_flag, 
                                   BOOL dump_flag,
                                   ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_coff_display()
*
*	 Description: 
*		Displays only the COFF information for a file
*
*  Parameters:
*		filename - The name of the COFF file.
*   clear_bss_flag - Initialize all bss section variable to zero.
*   dump - Display all data via the standard output stream.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_coff_display(char *filename,
                                      BOOL clear_bss_flag, 
                                      BOOL dump_flag);

/*****************************************************************************
*  evmdsk54x_hpi_open()
*
*	 Description: 
*		Opens the host port interface (HPI) to the DSPs and guarantees mutual 
*		exclusion to the DSPs HPI as well as memory.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY LPVOID evmdsk54x_hpi_open(HANDLE hDevice);

/*****************************************************************************
*  evmdsk54x_hpi_close()
*
*	 Description: 
*		Closes the host port interface (HPI) to the DSPs.
*
*  Parameters:
* h_hpi_map - Hpi_map handle returned from the call to
* *				evmdsk54x_hpi_open.
*
*  Return:
*		TRUE - success.
*		FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_close(LPVOID h_hpi_map);

/*****************************************************************************
*  evmdsk54x_hpi_read()
*
*	 Description: 
*		Reads data from the DSP memory.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_buffer - Pointer to the buffer to read data into.
*		p_length - Number of 16 -bit words to read.
*		src_addr - DSP memory address to read from.
*		memSpace - Indicates whether to read from DATA memory or PROGRAM memory.
*   dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_read(LPVOID h_hpi_map, 
                                  PULONG p_buffer,
                                  PULONG p_length, 
                                  ULONG src_addr,
                                  int memSpace,
                                  ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_hpi_read_single()
*
*	 Description: 
*		Reads a single 16-bit value from the DSP memory.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_data - Pointer to the buffer to read data into.
*		src_addr - DSP memory address to read from.
*		memSpace - Indicates whether to read from DATA memory or PROGRAM memory.
*     dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*     FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_read_single(LPVOID h_hpi_map, 
                                  PULONG p_data,
                                  ULONG src_addr,
                                  int memSpace,
                                  ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_hpi_write()
*
*	 Description: 
*		Writes data to the DSP memory.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_buffer - Pointer to the buffer to read data into.
*		p_length - Number of 16 -bit words to write.
*		src_addr - DSP memory address to read from.
*		memSpace - Indicates whether to write to DATA memory or PROGRAM memory.
*   dspNum - Specifies the DSP
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_write(LPVOID h_hpi_map, 
                                   PULONG p_buffer,
                                   PULONG p_length, 
                                   ULONG dest_addr,
                                   int memSpace,
                                   ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_hpi_write_single()
*
*	 Description: 
*		Writes a single 16-bit value to the DSP memory.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_data - Pointer to the buffer to read data into.
*		dest_addr - DSP memory address to write to.
*		memSpace - Indicates whether to write to DATA memory or PROGRAM memory.
*     dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*     FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_write_single(LPVOID h_hpi_map, 
                                           PULONG p_data,
                                           ULONG dest_addr,
                                           int memSpace,
                                           ULONG dspNum);
                                  
/*****************************************************************************
*  evmdsk54x_hpi_fill()
*
*	 Description: 
*		Fills DSP memory with a single value.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		fill_value - 16-bit value to fill DSP memory with.
*		p_length - Number of 16 -bit words to write.
*		src_addr - DSP memory address to read from.
*		memSpace - Indicates whether to read from DATA memory or PROGRAM memory.
*           dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hpi_fill(LPVOID h_hpi_map, 
                                  USHORT fill_value,
                                  PULONG p_length, 
                                  ULONG dest_addr,
                                  int memSpace,
                                  ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_fifo_read()
*
*	 Description: 
*		Reads data from the fifo stream.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_buffer - Pointer to the buffer to read data into.
*		p_length - Number of 16-bit words to read.
*   dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_fifo_read(LPVOID h_hpi_map, 
                                  PULONG p_buffer,
                                  PULONG p_length, 
                                  ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_fifo_write()
*
*	 Description: 
*		Writes data to the fifo stream.
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*		p_buffer - Pointer to the buffer to write data from.
*		p_length - Number of 16-bit words to write.
*           dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_fifo_write(LPVOID h_hpi_map, 
                                    PULONG p_buffer,
                                    PULONG p_length, 
                                    ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_generate_int()
*
*	 Description: 
*		Interrupts the DSP device using either the CPLD or HPI depending on the 
*     desired interrupt..
*
*  Parameters:
*		h_hpi_map - A void pointer corresponding to the HPI map object.
*     dspNum - Specifies the DSP.
*     intr - The type of interrupt to generate to the DSP
* 
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_generate_int(LPVOID h_hpi_map, PCI2040_DSP_EVENT intr, ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_enable_int()
*
*	 Description: 
*		Enables the specified interrupt to the host from the board.
*
*  Parameters:
*		intMask - Mask for the interrupt.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_enable_int(LPVOID h_hpi_map, ULONG intMask);

/*****************************************************************************
*  evmdsk54x_disable_int()
*
*	 Description: 
*		disables the specified interrupt to the host from the board.
*
*  Parameters:
*		intMask - Mask for the interrupt.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_disable_int(LPVOID h_hpi_map, ULONG intMask);

/*****************************************************************************
*  evmdsk54x_hotswap_switch()
*
*	 Description: 
*		Returns the status of the hotswap switch.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pStat - Storage for the hotswap switch status
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hotswap_switch(HANDLE hDevice, PULONG pStat);

/*****************************************************************************
*  evmdsk54x_hotswap_led_on()
*
*	 Description: 
*		Turns on the hotswap switch LED.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hotswap_led_on(HANDLE hDevice);

/*****************************************************************************
*  evmdsk54x_hotswap_led_off()
*
*	 Description: 
*		Turns off the hotswap switch LED.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_hotswap_led_off(HANDLE hDevice);

/*****************************************************************************
*  evmdsk54x_printf_control()
*
*	 Description: 
*		Sets the function to be used for printf.
*
*  Parameters:
*		enable - Enables printf (1 - enable , 0 - disbale).
*		func - printf function
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_printf_control(BOOL enable,
                                        PPCI2040_PRINTF_FUNC func);

/*****************************************************************************
*  evmdsk54x_input_control()
*
*	 Description: 
*		Sets the function to be used for input.
*
*  Parameters:
*		func - input function
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_input_control(PPCI2040_INPUT_FUNC func);

/*****************************************************************************
*  evmdsk54x_dll_revision()
*
*	 Description: 
*		Sets the dll revision
*
*  Parameters:
*		pRevMajor - The revison major number.
*		pRevMinor - The revison minor number.
*		pBuildNum - The buil number.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_dll_revision(PULONG pRevMajor,
                                      PULONG pRevMinor,
                                      PULONG pBuildNum);

/*****************************************************************************
*  evmdsk54x_read_reg()
*
*	 Description: 
*		Modifies a Montana device (The PCI controller) register.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		preg - Pointer to the PCI20404_REG  object containing the new register value.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_read_pci_reg(HANDLE hDevice, PRegister  pReg);

/*****************************************************************************
*  evmdsk54x_write_reg()
*
*	 Description: 
*		Reads a Montana device (The PCI controller) register.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		preg - Pointer to the PCI20404_REG object to hold the read register value.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_write_pci_reg(HANDLE hDevice, PRegister  pReg);

/*****************************************************************************
*  evmdsk54x_dsp_present()
*
*	 Description: 
*		Determines if the specified DSP is present.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		dspNum - Specifies the DSP.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_dsp_present(HANDLE hDevice, ULONG dspNum);

/*****************************************************************************
*  evmdsk54x_mem_config()
*
*	 Description: 
*		Sets the memory configuration on the specified DSP.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pMem - Pointer to a PMemSpaceConfig object which contails the new OVLY, DROM,
*		and MP_MC flag values.
*
*  Return:
*		TRUE - success.
*   FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_mem_config(HANDLE hDevice, ULONG dspNum, PMemSpaceConfig pMem);

/*****************************************************************************
*  evmdsk54x_cpld_read_all()
*
*	 Description: 
*		Gets a dump of the CPLD registers.
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		regs - Pointer to array of shorts to hold register values.
*  Return:
*		TRUE - success.
*     FALSE - failure
*******************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_cpld_read_all(HANDLE hDevice, PUSHORT regs);


/*****************************************************************************
*  evmdsk54x_get_event_handle()
*
*	 Description: 
*		Returns the handle to the requested event
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		eventNUM - Event that monitoring is to be terminated.
*  Return:
*		event HANDLE - success.
*     NULL - failure
*******************************************************************************/
DLL32_ENTRY HANDLE evmdsk54x_get_event_handle(HANDLE hDevice, 
                                              PCI2040_HOST_EVENT eventNum);

/****************************************************************************
*evmdsk54x_read_cpld_reg()
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pReg - pointer to a PCI2040_REG that holds the read value.
*  The evmdsk54x_read_cpld_reg function reads a value from the CPLD register
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_read_cpld_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************
*evmdsk54x_write_cpld_reg()
*
*  The evmdsk54x_write_cpld_reg function writes a value to the CPLD register
*  
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pReg - pointer to a PCI2040_REG that holds the value to write
*
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_write_cpld_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_user_semaphore_get() - This function tries to acquire the        */
/*    specified user semaphore.                                             */
/*   sem will be filled with the semaphore state                           */
/*    Returns TRUE if acquired                                              */
/*            FALSE if semaphore is not acquired                            */
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_user_semaphore_get(HANDLE hDevice, PCI2040_SEM sem);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_user_semaphore_release() - This function releases the previously */
/*    acquired semaphore specified by sem.                                  */
/*                                                                          */
/*   returns FALSE if the semaphore has already been released.             */
/*            TRUE if operation is successful                               */
/*                                                                          */
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_user_semaphore_release(HANDLE hDevice, PCI2040_SEM sem);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_user_semaphore_wait() - This function waits indefinitely to      */
/*    acquire the semaphore specified by sem.                               */
/*                                                                          */
/*   returns FALSE if opreation fails                                      */
/*            TRUE if successful                                            */
/*                                                                          */
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_user_semaphore_wait(HANDLE hDevice, PCI2040_SEM sem);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_get_base_address() - This function returns the specified base    */
/*    address                               .                               */
/*                                                                          */
/*   returns FALSE if opreation fails                                      */
/*            TRUE if successful                                            */
/*                                                                          */
/****************************************************************************/
DLL32_ENTRY PULONG evmdsk54x_get_base_address(HANDLE hDevice, PCI2040_BAR bar_num);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_set_timeout() - The evmdsk54x_set_timeout function sets the    */
/* timeout for the fifo. Fifo access calls return with an error whenever    */
/* they block for a multiple of the specified timeout value.                */
/*                                                                          */
/*   returns FALSE if opreation fails                                      */
/*            TRUE if successful                                            */
/*                                                                          */
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_set_timeout(LPVOID pBd_, ULONG timeout);

/****************************************************************************/
/*                                                                          */
/* evmdsk54x_tbc_reset() - The evmdsk54x_tbc_reset function pulses the TBC  */
/*                       reset to reset the Test Bus Controller.            */
/*                                                                          */
/*   returns FALSE if opreation fails                                      */
/*            TRUE if successful                                            */
/*                                                                          */
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_tbc_reset(HANDLE hDevice);

/****************************************************************************
evmdsk54x_tbc_unreset()

  The evmdsk54x_tbc_unreset function removes the TBC from the reset state
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_tbc_unreset(HANDLE hDevice);

/****************************************************************************
evmdsk54x_read_tbc_reg()

  The evmdsk54x_read_reg function reads a value from the TBC device 
  register specified by the pointer pReg
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_read_tbc_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************
evmdsk54x_write_tbc_reg()

  The evmdsk54x_write_tbc_reg function writes a value to the TBC device 
  register specified by the pointer pReg
  
    Returns TRUE on succes
    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_write_tbc_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************
evmdsk54x_get_configuration()                                                       

  The evmdsk54x_get_configuration function retrieves the PCI configuration 
  information from the Montana device.
  
    The function returns a FALSE upon failure and TRUE on success
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_get_parallel_configuration(HANDLE hDevice,
                                           PDskConfig pReg);
/****************************************************************************
evmdsk54x_store_parallel_configuration()                                                       

  The evmdsk54x_store_parallel_configuration function updates the configuration
  of the DSK/EVM connected to the parallel port.
  
    The function returns a FALSE upon failure and TRUE on success
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_store_parallel_configuration(HANDLE hDevice,
                                           PDskConfig pReg);

/****************************************************************************
evmdsk54x_read_hpi_reg()

  The evmdsk54x_read_hpi_reg function writes a value to the HPI register
  specified by the reg.
  
    Returns TRUE on succes
    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_read_hpi_reg(LPVOID pBd_, PRegister pReg, ULONG dspNum);

/****************************************************************************
evmdsk54x_write_hpi_reg()

  The evmdsk54x_write_hpi_reg function writes a value to the HPI register
  specified by the reg.
  
    Returns TRUE on succes
    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_write_hpi_reg(LPVOID pBd_, PRegister pReg, ULONG dspNum);

/****************************************************************************
*evmdsk54x_read_cpld_reg()
*
*  Parameters:
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pReg - pointer to a PCI2040_REG that holds the read value.
*  The evmdsk54x_read_cpld_reg function reads a value from the CPLD register
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_read_cpld_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************
*evmdsk54x_write_cpld_reg()
*
*  The evmdsk54x_write_cpld_reg function writes a value to the CPLD register
*  
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		pReg - pointer to a PCI2040_REG that holds the value to write
*
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_write_cpld_reg(HANDLE hDevice, PRegister  pReg);

/****************************************************************************
*evmdsk54x_flash_access(HANDLE hDevice, BOOL enable)
*
*  The evmdsk54x_flash_access indicates that all memory accesses above 0x8000
*  will be to flash memory.
*  
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		enable - TRUE = enable flash access, FALSE = disable flash access
*
*    Returns TRUE on succes
*    FALSE on failure.
*
*  Note: It the caller's responsibilty to disable flash access when done 
*  accessing flash. Flash access is disabled by default.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_flash_access(HANDLE hDevice, BOOL enable);

/****************************************************************************
*evmdsk54x_flash_erase(HANDLE hDevice, EVMDSK54X_FLASH_CONTROL area)
*
*  The evmdsk54x_flash_erase erases a portion of flash memory
*  will be to flash memory.
*  
*		hDevice - Board handle returned from the call to evmdsk54x_open.
*		area - specifies portion of flash to erase
*
*    Returns TRUE on succes
*    FALSE on failure.
/****************************************************************************/
DLL32_ENTRY BOOL evmdsk54x_flash_erase(HANDLE hDevice, EVMDSK54X_FLASH_CONTROL area);

#ifdef __cplusplus
}

#endif

#endif // _PCI2040_DLL_H_
