//---------------------------------------------------------
// DSK_COFF.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//---------------------------------------------------------
//  The functions in this file convert DSK3A and COFF output files
//  into data streams suitable for the TMS320C3X DSK.  Depending on
//  the 'task' parameter these streams can be sent directly to the
//  DSK or a file and optionaly formatted for loading or bootloading
//---------------------------------------------------------
#include <conio.h>
#include <ctype.h>  // For toupper in HLL_SRCE_LINE test code section
#include <dos.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "dsk.h"

#define  DASM_EN  0
#if      DASM_EN
#include "ASSM_FUN.H"
#endif
char     DStrg[80];

ulong ENTRY;
char  section_name[20];
char  DASM_file[120]; // Current loaded DSK filename
char  HLL_file[120];  // High Level Language file (C source)
ulong section_start;
ulong section_offs;
int   CoffVer=0;
int   COFF_OS=-1;
ulong WSTATE = 7; // Default software wait state for DSP
ulong IOSTRB = 0x00F8;
ulong STRB0  = 0x10F8;
ulong STRB1  = 0x10F8;

MSGS data2target(ulong dat, ulong add, TASK tsk, char *msg, FILE *file);
MSGS Load_Hex_File(char *filename, TASK task);

char *DSKEXT = ".DSK";
char *OUTEXT = ".OUT";

fpos_t fpos;   // Last file pointer used speeds up finding next sect
FILE_HDR0 f_hdr0;
LINE_TBL huge HLL[HLL_limit];
int lastHLL;
//
// In WIN32 applications, integers are promoted to 32 bit values.
// Since COFF files have defiend bit fields, byte packing each element
// of an array, plus making sure each element of the structure is on a
// byte boundary must be forced.  This is done using '#pragma pack(1)'
//
// WIN32 defines a 16 bit integer as WORD, and 8 bit chars as BYTE
//

#if _WIN32 | __WIN32__
#else
  #define WORD  unsigned int  // Not definded for DOS
  #define BYTE  unsigned char // application builds
#endif

#pragma pack (1)
typedef struct
{
  BYTE         sname[8]; // 8 char name, or ptr to strg tbl
  long         svalu;    // symbol value
  WORD         sectn;    // section number of symbol
  WORD         stype;    // symbol type
  char         sclass;   // storage class
  char         s_aux;    // number of auxiliary entries (0 or 1)
}SYMB_TBL;

typedef union
{
  struct
  {
    char nused1[ 4];  //4
    long      fpos2;  //4
    char nused2[10];  //10 = 18
  }f;
  SYMB_TBL tbl;
}SYMS;
#pragma pack()
SYMS sx;
//-------------------------------------------------
union
  {
    A_18  a_18;
    A_18b a_18b;
    A_19  a_19;
    A_20  a_20;
    A_21  a_21;
    A_22  a_22;
    A_23  a_23;
    A_24  a_24;
    A_25  a_25;
    A_26  a_26;
  }aux;
//============================================================================
// Found Windows would GPF if the file being closed did not exist.  Replaced
// Replaced all fclose() functions and assigned all 'FILE *ptrs=NULL;'
//============================================================================
int my_fclose(FILE *file)
{
  if(file==NULL)
    return EOF;
  return(fclose(file));
}
//=================================================
// Symbol_Clear() clears previously loaded symbols
// by changing the last_ref symbol counter.
//=================================================
void Symbol_Clear(int level)
{
  if(level)                     // Load file - rewind references to 0
  {
    last_ref = post_boot_sym;
    symbol_ptr = post_boot_symbols;
  }
  else
  {
    last_ref = 0;               // Full reboot - rewind references to 0
    post_boot_sym = 0;
    symbol_ptr = symbols;
    post_boot_symbols = symbols;
  }
}
//--------------------------------------------------------------------
// Load_File is the handler for several tasks.  In the case of bootup
// the symbol table is cleared, followed by a symbol load.
//
// Enum task    Task to perform
// ---------    ---------------
// LOAD         loads file into DSK target
// BOOT         Boots file into DSK target
// FILE2HEX     Creates loadable ascii .HEX file
// LOADHEX      Load hex files
// BOOTHEX,     Bootload hex files
// DSK2COFF     Convert DSK file to COFF file
// MEM2COFF     Convert DSK memory to COFF file
// SLOAD        Loads symbols from the file
//-----------------------------------------------------
MSGS DLLEXTEND_EX Load_File(char *file,TASK task)
{
  char loadfilename[120];
  char *p;
  FILE *DASM_FILE;
  MSGS err;
  CoffVer=-1;                // Reset COFF version of last loaded file
  while(*file ==' ') file++; // Remove leading WS
  if(strlen(file) > 115) return FILE_LEN;
  strcpy(loadfilename,file);
  //
  // If extension is not supplied check for DSK3A file first
  //
  if(strstr(loadfilename,".")==NULL)
  { strcat(loadfilename,DSKEXT);
    if(access(loadfilename,0)!=0)
    {
      strcpy(strstr(loadfilename,"."),".out");
      if(access(loadfilename,0)!=0)
        strcpy(strstr(loadfilename,"."),".hex");
    }
  }
  // If the file does not exist, prompt the user with a message to
  // remind them to create the file.
  //
  p=loadfilename;
  for(;;)
  {
    if(*p== ' ') break;
    if(*p==  0 ) break;
    if(*p=='\t') break;
    p++;
  }
  *p = 0;
  if(access(loadfilename,0)!=0)
  {
     return ACCESS_ERR;
  }
  // Since file exists, write the name of the file to disk
  // such that other applications can see the name.  IE this
  // is a workaround that allows multiple applications access
  // to the name of the file currently being debugged.
  strupr(loadfilename);
  if(strexact("C3X.DSK",loadfilename)==0) // Dont update for kernel
  {
    DASM_FILE = fopen("DASMFILE.FIL","wt");
    if(DASM_FILE==NULL) return ACCESS_ERR;
    if(strstr(loadfilename,".OUT"))
      fprintf(DASM_FILE,"%s   Last COFF file loaded into DSK3D\n",loadfilename);
    else
      fprintf(DASM_FILE,"%s   Last DSK file loaded into DSK3D\n",loadfilename);
    my_fclose(DASM_FILE);
  }
  //
  *DASM_file = 0;     // Kill previous disassembler file name

  switch(task)
  {
    case BOOT : lastHLL=0; Symbol_Clear(0); break; // Rewind to pre boot level
    case SLOAD:
    case LOAD : lastHLL=0; Symbol_Clear(1); break; // Rewind to post boot level
    case BOOTHEX:
    case LOADHEX:lastHLL=0;Symbol_Clear(1);        // Rewind to post boot level
                p = strstr(loadfilename,".");
                strcpy(p,".hex");
                err = Load_Hex_File(loadfilename, task);
             // err=Load_File(loadfilename,SLOAD);
                return err;
    default   : break;
  }
  if((strstr(loadfilename,".HEX"))&&(task==LOAD))
    return(Load_Hex_File(loadfilename, task));

  if(task==DSK2COFF) return CVT2COFF(loadfilename,0,0, task);
  if((task == BOOT)&&(strstr(loadfilename,".HEX")!=NULL))
    return Load_Hex_File(loadfilename, BOOTHEX);
  if(strstr(loadfilename,".DSK")!=NULL) err = LOADDSKA(loadfilename, task);
  else                              err = LOADCOFF(loadfilename, task);
  return err;
}
//--------------------------------------------------------------------
// Save_Mem_HEX() saves a block of memory in COFF format
//--------------------------------------------------------------------
ulong STRB(ulong addr,int sext, int memw, int datasz)
{
  ulong val;
  val = (0x10F8 | (WSTATE << 5)) << 8; // Shift WSTATE to proper field
  for(;;)
  {
    if(addr<0x800000L) {val |= 0x64; break;} // STRB0
    if(addr<0x810000L) return val;           // Internal
    if(addr<0x830000L) {val |= 0x60; break;} // IOSTRB
    if(addr<0x880000L) return val;           // Internal
    if(addr<0x900000L) {val |= 0x64; break;} // STRB0
                        val |= 0x68; break;  // STRB1
  }
  if(sext      ) val |= 0x10000000L;
  if(datasz==16) val |= 0x01000000L;
  if(datasz==32) val |= 0x03000000L;
  if(memw  ==16) val |= 0x04000000L;
  if(memw  ==32) val |= 0x0C000000L;
  return val;
}
MSGS Save_Mem_HEX(char *file,ulong start,ulong length)
{
  MSGS err;
  FILE *out;
  TASK task = FILE2HEX;
  char *ptr;
  ulong temp;
  char boot_file[80];
  strcpy(boot_file,file);
  ptr = strstr(boot_file,".");
  if(ptr!=NULL) strcpy(ptr      ,".hex");
  else          strcat(boot_file,".hex");
  unlink(boot_file);
  if((out = fopen(boot_file,"wt"))==NULL)
  {
    my_fclose(out);
    return MEM_HEX;
  }
  if(TARGET==C31_DSK)  fprintf(out,"FILE2HEX\n");
  if(TARGET==C32_DSK)  fprintf(out,"FILE2HEX_C32\n");
  //------------------------------------------
  // write out data in one continuous section
  //------------------------------------------
  if((TARGET==C31_DSK)||(TARGET==C30_EVM))
  {
//  STRB0 = (STRB0&0xFF1F) || (WSTATE<<5);
    data2target(0x00000008L,0x00000000L,task,"Bus Width"            ,out);
//  data2target(0x000010F8L,0x00000000L,task,"Bus Control ws=7"     ,out);
    data2target(STRB0      ,0x00000000L,task,"Bus Control"          ,out);
    data2target(0x00000001L,0x00000000L,task,"Dummy Section size"   ,out);
    data2target(ENTRY      ,0x00000000L,task,"Dummy Start Address"  ,out);
    data2target(0x00000000L,0x00000000L,task,"Dummy Data"           ,out);
    data2target(length     ,0x00000000L,task,"Section Size"         ,out);
    data2target(start      ,0x00000000L,task,"Section start MEM2HEX",out);
  }
  if(TARGET==C32_DSK)
  {
    data2target(0x00000008L,0x00000000L,task,"Bus Width"            ,out);
    data2target(IOSTRB     ,0x00000000L,task,"IOSTRB"               ,out);
    data2target(STRB0      ,0x00000000L,task,"STRB0"                ,out);
    data2target(STRB1      ,0x00000000L,task,"STRB1"                ,out);
    data2target(0x00000001L,0x00000000L,task,"Dummy Section size"   ,out);
    data2target(ENTRY      ,0x00000000L,task,"Dummy Start Address"  ,out);
    data2target(STRB(ENTRY,0,0,0),0x00000000L,task,"Dummy STRBx ctrl",out);
    data2target(0x00000000L,0x00000000L,task,"Dummy Data"           ,out);
    data2target(length     ,0x00000000L,task,"Section Size"         ,out);
    data2target(start      ,0x00000000L,task,"Section start MEM2HEX",out);
    data2target(STRB(start,0,0,0),0x00000000L,task,"STRBx ctrl",out);
  }

  for(;length;length--)
  {
    if((err=getmem(start,1,&temp))!=NO_ERR) break;
    data2target(temp,0x00000000L,task,"",out);
    start++;
  }
  data2target(0x00000000L,0x00000000L,task,"Terminate Load",out);
  my_fclose(out);
  return err;
}
//--------------------------------------------------------------------
// Save_File is the file handler for several save file type tasks.
//-----------------------------------------------------
MSGS Save_File(char *file,ulong start, ulong length,TASK task)
{
  char filename[120];
  if(strlen(file) > 115) return FILE_LEN;
  strcpy(filename,file);
  //----------------------------------------------------------
  // If extension is not supplied append appropriate extension
  if(strstr(filename,".")==NULL)
  { switch(task)
    {
      case MEM2COFF: strcat(filename,".OUT"); break;
      case MEM2HEX : strcat(filename,".HEX"); break;
      default : return ACCESS_ERR;
    }
  }
  strupr(filename);
  switch(task)
  {
    case MEM2COFF: //err = Save_Mem_HEX (filename,start,length);
                   //if(err!=NO_ERR) return err;
                   strcpy(strstr(filename,"."),".hex");
                   //return(Load_File(filename,MEM2COFF));
                   return CVT2COFF(filename,start,length, task);
    case MEM2HEX : return(Save_Mem_HEX (filename,start,length));
    default      : break;
  }
  return ACCESS_ERR;
}

//----------------------------------------------------------------
// data2target sends data to the selected output device or file
//----------------------------------------------------------------
MSGS data2target(ulong data, ulong addr, TASK task, char *msg, FILE *hex_file)
{ switch(task)
  {
   case DSK2COFF:   fwrite(&data,4,1,hex_file);
                    return NO_ERR;
   case FILE2HEX:   fprintf(hex_file,"0x%08lx  %s\n",data,msg);
                    return NO_ERR;
//  case DSK2COFF : fwrite(lstfile,data);        // data goes to file
//                  return NO_ERR;
    case LOAD     : return putmem(addr,1,&data); // use kernel to download
    case BOOT     : if(TARGET==C31_DSK)
                    { if((addr==0x809800L)||(addr==0x809801L))
                        return BOOT_RNG;
                    }
                    return xmit_long(data);      // send directly to HPI
    case SLOAD    : return NO_ERR;               // No load required
  }
  return UNKN_ERR;
}

//
// loadsegments scans the DSK3A listing and returns the Nth segments
// name, start, end, and length
//
MSGS getsection(char *lst_file)
{
  FILE *lfile;
  char strg[120];
  char *e, *p1, *p2;
  if((lfile = fopen(lst_file,"rt"))==NULL) return OPEN_ERR;
  fsetpos(lfile, &fpos);
  for(int count=0;count<16384;count++) // 16K lines maximum
  {
    // All valid DSK3A files end with ">>>> END DSK" string
    if(fgets(strg,119, lfile) == NULL) {my_fclose(lfile); return d_EOF;}
    if(strstr(strg,">>>> END DSK"))    {my_fclose(lfile); return d_END;}
    if(   ((p1=strstr(strg,">>>> ENTRY"))!=NULL) )
//        ||((p1=strstr(strg,">>>> entry"))!=NULL) )
    { ENTRY = strtol(p1+11,&e,0);
    }
    if(   ((p1=strstr(strg,">>>> sect" ))!=NULL) )
//       || ((p1=strstr(strg,">>>> SECT" ))!=NULL) ) // Compatable <DSK3A
    { fgetpos(lfile, &fpos);
      p1+=  9; while(*p1 == ' ')p1++;// Find start of name string
      p2 = p1; while(*p2 != ' ')p2++;// Find end of name string
      *p2 = 0;                       // terminate name string
      strcpy(section_name,p1);       // copy section name
      e = p2+1;                      // pnt to 1st data col field
      section_start=strtol(e,&e,0);  // start
      section_offs =strtol(e,&e,0);  // end
      my_fclose(lfile);
      return NO_ERR;
    }
  }
  my_fclose(lfile);
  return d_filelength;
}
//-----------------------------------------------------
// LOADSEGMENTS()  Reads in all sections found in the
// DSK3A listing and add them to the SEG[] array.
//-----------------------------------------------------
MSGS loadsegments(char *lst_file)
{ FILE *lfile;
  char strg[120];
  char *e, *p1, *p2;
  int x=0;
  if((lfile = fopen(lst_file,"rt"))==NULL)
  { my_fclose(lfile);
    return FATAL_OPEN;
  }
  last_segment = 0;
  for(int count=0;count<16384;) // 16K lines maximum
  {
    if(fgets(strg,119, lfile) == NULL){ my_fclose(lfile); return NO_ERR; }

//strupr(strg);

    if(strstr(strg,"END DSK"))        { my_fclose(lfile); return NO_ERR; }
    if(   ((p1=strstr(strg,">>>> ENTRY"))!=NULL) )
//       || ((p1=strstr(strg,">>>> entry"))!=NULL)  )
    {
      ENTRY = strtol(p1+11,&e,0);
    }
    if(    ((p1=strstr(strg,">>>> sect"))!=NULL)  )
//        || ((p1=strstr(strg,">>>> SECT"))!=NULL)  )
    {
      p1+=  9; while(*p1 == ' ')p1++;// Find start of name string
      p2 = p1; while(*p2 != ' ')p2++;// Find end of name string
      *p2 = 0;                       // terminate name string
      strncpy(SEG[x].name,p1,16);    // copy section name
      e = p2+1;                      // pnt to 1st data col field
      SEG[x].start = strtol(e,&e,0); // start
      SEG[x].offs  = strtol(e,&e,0); // end
      SEG[x].length= strtol(e,&e,0); // end
      x++;
      last_segment++;
    }
  }
  my_fclose(lfile);
  return NO_ERR;
}
//----------------------------------------------------------------------
// write_seg_val() scans the DSK3A listing and outputs data belonging
// to the named section.  Depending on task the output data is sent to
// either a file, or directly to the DSK target.
//----------------------------------------------------------------------
MSGS write_seg_val(char *lst_file,TASK task,FILE *hex_file)
{
  FILE *lfile;
  char strg[120];
  MSGS err;
  int count_on = 0;
  ulong temp, addr;
  char *eptr;
  if((lfile = fopen(lst_file,"rt"))==NULL) return OPEN_ERR;
  for(int count=0;count<16384;) // 16K lines maximum
  {
    if(fgets(strg,119, lfile) == NULL)
    {
      my_fclose(lfile);
      return NO_ERR;
    }
    if(*strg == '>')  // DSK message field char used as accelerator
    {
      if(strstr(strg,"END DSK"))
      {
        my_fclose(lfile);
        return NO_ERR;
      }
    }
    else
    {
      if(strg[11] == '0')  // valid loadable value at address
      { if(count_on)
        { eptr = strg;
          addr = strtoul(eptr     ,&eptr,0);
          temp = strtoul(eptr     ,&eptr,0);

          #if DASM_EN
          BuildFastLook();
          strcpy(DStrg,">");
          get_label(DStrg+1,addr,0);
          strcat(DStrg,"          ");
          DStrg[10]=' ';
          Disasm(addr, temp, DStrg+11);
          #else
          sprintf(DStrg,"");
          #endif

          if((err=data2target(temp,addr,task,DStrg,hex_file))!=NO_ERR)
          { my_fclose(lfile); return err;}
        }
      }
      else  // directive, nocode, comment and other non-loadable lines
      {
        //
        // 0x00809802 directive          .START "CODE",0X809802
        // 0x00809802 directive          .SECT "CODE"
        // 0x00809802 directive          .ENTRY INIT
        //
        eptr = strstr(strg,"\"");  // Find start of section name
        if(eptr!=NULL)
        {
          *eptr = 0;
          strlwr(strg);  // Set lower case up to start of section name
          *eptr = '\"';
        }

        if(strstr(strg,".start")==NULL)
        {
          /*                                                       */
          /* If the section name is changed, turn off reading data */
          /*                                                       */
          if(   strstr(strg,".sect") || strstr(strg,".brstart")
             || strstr(strg,".text") || strstr(strg,".data"   ))
          {
            //
            // if the section name is changed, the line which is
            // being loaded should be a valid code line... IE
            // <directive> should be found in the second field
            eptr=strstr(strg,"directive");
            if((eptr <= strg+12) && (eptr != NULL))
            {
              if(strstr(strg,section_name)) count_on=1;
              else                          count_on=0;
            }
            /*
            //
            // if a .sect statement is commented out, the
            // 'nocode' attribute is used to not turn off
            // the current section load
            eptr=strstr(strg,"nocode");
            if((eptr==NULL)|| (eptr > strg+12))
            {
              if(strstr(strg,section_name)) count_on=1;
              else                         count_on=0;
            }
            */
          }
        }
      }
    }
  }
  my_fclose(lfile);
  return d_filelength;
}
//------------------------------------------------------------------
// read_symbols() scans through the DSK list file and fills the
// reference table with the values listed at the end of the file.
//------------------------------------------------------------------
MSGS read_syms(char *filename)
{
  FILE *lfile;
  MSGS err;
  char strg[120];
  char *p1, *p2;
  SYM_TYPE c;
  ulong value;
  int a;
  NUM_TYPE type;
  if((lfile = fopen(filename,"rt"))==NULL) return OPEN_ERR;

  for(int i=0;i<16384;i++)
  {
    if(fgets(strg,119, lfile) == NULL)      {my_fclose(lfile); return NO_ERR;}

//strupr(strg); Do not toupper this... will corrupt symbol names

    if((p1 = strstr(strg,"END DSK" ))!=NULL)
    { my_fclose(lfile); return NO_ERR;}
    if(   ((p1 = strstr(strg,">>>> ref"))!=NULL)  )
//      ||((p1 = strstr(strg,">>>> REF"))!=NULL)  )
    {
      p1 += 8; while(*p1 == ' ')p1++;// Find start of name string
      p2 = p1; while(*p2 != ' ')p2++;// Find end of name string
      *p2++ = 0;                     // terminate name string
      value =      strtol(p2,&p2,0);
      switch((int) strtol(p2,&p2,0))
      {
        default:
        case 1: type = NT_INTEGER ; break;
        case 2: type = NT_FLOAT   ; break;
        case 3: type = NT_SLONG   ; break;
        case 4: type = NT_ULONG   ; break;
        case 5: type = NT_QFORM   ; break;
      }
      a     = (int)strtol(p2,&p2,0);
      switch(a)
      {
         case  0: c = SYM_VAR; break;
         case  1: c = SYM_GLBL; break;
         case  2: c = SYM_DEF; break;
         case  3: c = SYM_LLBL; break;
         default: c = SYM_VAR; break;
      }
      //if(scnt > MAX_SYMBOLS) {my_fcloseall(); return SYM_LOAD;}
      switch(err=xref_add(p1,value,type,c))
      {
        case MULT_REF:
                       xref_mod2(p1,value,type,c);
                       break; // Notify user of multiple references?
        case NO_ERR  : break;
        default: my_fclose(lfile);
                 return err;
      }
    }
  }
  my_fclose(lfile);
  return NO_ERR;
}
//-------------------------------------------------------------------------
// LOADDSKA reads a DSK3A file and either loads the DSK directly or creates
// a file.  The file is of text formatted hex values (not EPROM formated).
// A post HEX->ROMFILE converter would be required for non DSK use.
//-------------------------------------------------------------------------
MSGS LOADDSKA(char *lst_file, TASK task)
{
  FILE *hex_file;
  MSGS err = NO_ERR;
  char buf[80];
  char boot_file[120];
  char *ptr;
  ulong length;
  int i;
  //---------------------------------------------------------
  // Verify that the DSK file is a legal file before going on
  // If valid, the first line will contain >>>>
  if((hex_file = fopen(lst_file,"rb"))==NULL) return OPEN_ERR;
  fgets(buf,20,hex_file);
  ptr = strstr(buf,">>>> DSK3A C3x");
  my_fclose(hex_file);
  if(ptr!=buf) return DSK_FILE_ERR;
  //---------------------------------------------
  strcpy(DASM_file,lst_file);  // Current disassembler file name
  if(task == SLOAD)
  {
    return(read_syms(lst_file));
  }
  else
  {
    //----------------------------------------------------------
    // Open output file to send conversion to (HEX is used for fast loads)
    //----------------------------------------------------------
    if(task==FILE2HEX)
    {
      strcpy(boot_file,lst_file);
      ptr = strstr(boot_file,".");
      strcpy(ptr+1,"hex");
      unlink(boot_file);
      if((hex_file = fopen(boot_file,"wt"))==NULL)
      {
        my_fclose(hex_file);
        return DSK_HEX;
      }
      if(TARGET==C31_DSK)  fprintf(hex_file,"FILE2HEX\n");
      if(TARGET==C32_DSK)  fprintf(hex_file,"FILE2HEX_C32\n");
    }
    //---------------------------------
    // write out data for each section
    //---------------------------------
    fpos = 0;                 // reset the section file ptr
    for(i=0; i<32; i++)   // Load maximum of 32 section
    { // get entry, sect name, start and end
      if((err=getsection(lst_file))==d_END) break;// Exit no more sections
      if(err == d_EOF)
        {my_fclose(hex_file); return DSK_FILE_ERR;}
      //-----------------------------------------------
      // Create boot header before first output section
      //-----------------------------------------------
      if(i==0)
      { if((task==BOOT)||(task==FILE2HEX))
        { if(task==BOOT)
          {
            err=DSK_reset();
            switch(err)
            {
              case NO_ERR:
              case EXT_MSG_OK: break;
              default : return err;
            }
            if(TARGET==C31_DSK)
            {
             xmit_byte(0x08);
             xmit_byte(0x00);
            }
            else
             data2target(0x00000008L,0x00000000L,task,"Bus Width",hex_file);
          }
          else
          {
            data2target(0x00000008L,0x00000000L,task,"Bus Width",hex_file);
          }
  if(TARGET==C31_DSK)
  {
//  data2target(0x000010F8L,0x00000000L,task,"Bus Control ws=7"   ,hex_file);
    data2target(STRB0      ,0x00000000L,task,"Bus Control ws=7"   ,hex_file);
    data2target(0x00000001L,0x00000000L,task,"Dummy Section size" ,hex_file);
    data2target(      ENTRY,0x00000000L,task,"Dummy Start Address",hex_file);
    err=
    data2target(0x00000000L,0x00000000L,task,"Dummy Data"         ,hex_file);
  }
  if(TARGET==C32_DSK)
  {
  data2target(IOSTRB     ,0x00000000L,task,"IOSTRB"               ,hex_file);
  data2target(STRB0      ,0x00000000L,task,"STRB0"                ,hex_file);
  data2target(STRB1      ,0x00000000L,task,"STRB1"                ,hex_file);
  data2target(0x00000001L,0x00000000L,task,"Dummy Section size"   ,hex_file);
  data2target(ENTRY      ,0x00000000L,task,"Dummy Start Address"  ,hex_file);
  data2target(STRB(ENTRY,0,0,0),0x00000000L,task,"Dummy STRBx ctrl",hex_file);
  data2target(0x00000000L,0x00000000L,task,"Dummy Data"           ,hex_file);
  }
          if(err!=NO_ERR) {my_fclose(hex_file); return err;}
        }
      }
      length = section_offs-section_start;
      if(length != 0)     // If section length=0, do not convert
      { if(task!=LOAD)    // kernel LOAD's do not need section header info
        {
          data2target(length,0x0L,task,"Section Size",hex_file);
          sprintf(buf,"Start Address '%s'",section_name);
          data2target(section_start,0x0L,task,buf,hex_file);
          if(TARGET==C32_DSK)
data2target(STRB(section_start,0,0,0),0x0L,task,"STRBx ctrl",hex_file);
        //data2target(section_start,0x0L,task,"Start Address NAME",hex_file);
        }
        if((err=write_seg_val(lst_file, task,hex_file))!=NO_ERR)
        {my_fclose(hex_file); return err;}
      }
    }
    // Zero length section terminates Bootload and Load files
    if(task!=LOAD)
      data2target(0x00000000L,0x00000000L,task,"Terminate Load",hex_file);
    else { PC_Appli=ENTRY; err = putmem(CTXT_PTR+PC,1,&ENTRY);}
  }
  if(err == d_END) err = NO_ERR;
  my_fclose(hex_file);
  return err; //NO_ERR;
}
//--------------------------------------------------------------
// Load_Hex_File() load hexfile (*.HEX) formatted files into
// the DSK.  Both kernel loading and bootloading are supported
//--------------------------------------------------------------
MSGS Load_Hex_File(char *filename, TASK task)
{
  #define slen  120
  FILE *hex_file;
  MSGS err;
  char strg[slen];
  char *eptr;
  ulong length;
  ulong l, addr;
  int x;
  int HEXTYPE;
  //----------------------------------------------------------
  // Open hex output file to be used for fast loads
  //----------------------------------------------------------
  if((hex_file = fopen(filename,"rb"))==NULL)
  {
    my_fclose(hex_file);
    return FATAL_OPEN;
  }
  //-----------------------------------------------------------------------
  // From a bootable immage a file can be simply downloaded after reset
  // using xmit_long() till the file is complete, or by reading the block
  // transfer information and then using the putmem() command.  The first
  // option is suitable for standalone applications while the second for
  // use with the communications kernel (previously loaded).
  // If the file is FILE2HEX  read the header info and use block transfers
  //-----------------------------------------------------------------------
  fgets(strg,slen,hex_file);
  if((eptr=strstr(strg,"\r"))!=NULL) *eptr=0;
  if((eptr=strstr(strg,"\n"))!=NULL) *eptr=0;
  //
  // 'FILE2HEX' substring is in the first line of a HEX file.
  //
  if(strstr(strg,"FILE2HEX")==0)
  { my_fclose(hex_file);
    return OPEN_ERR;  // error MSG needs to be more descriptive
  }
  if(strexact(strg,"FILE2HEX_C32")) HEXTYPE = C32_DSK;
  else                              HEXTYPE = C31_DSK;
  //----------------------------------------------------------
  // For boot, check to see if the target and file types match
  //----------------------------------------------------------
  if(task==BOOTHEX)
  {
    if(TARGET!=HEXTYPE) {my_fclose(hex_file);return HEXMATCH;}
    //
    DSK_reset();
    if(TARGET==C31_DSK)        // The C31 DSK bootloader does not read
    { xmit_byte(0x08);           // all 32 bits of an EPROM during bootload.
      xmit_byte(0x00);           // Only 16 lsb's are sent to the target
      fgets(strg,slen,hex_file); // First line (EPROM width) is bypassed
    }
    for(x=0;x<10000;x++) // 10K words is realistic upper limit for DSK
    {
      if(fgets(strg,slen,hex_file)==NULL) break;
      l = strtoul(strg,&eptr,0);
      err = xmit_long(l);
      if(err!=NO_ERR)
      {
        my_fclose(hex_file);
        return err;
      }
    }
    my_fclose(hex_file);
    if(TARGET!=HEXTYPE) return HEXMATCH;
    return NO_ERR;
  }
  //
  // else (only other case) is LOADHEX
  // NOTE: This is a smart loader. Both C31_DSK or C32_DSK HEX formats work
  //
  switch(HEXTYPE)
  {
    case C31_DSK: fgets(strg,slen,hex_file);  //strip memwidth
                  fgets(strg,slen,hex_file);  //strip bus control
                  break;
    case C32_DSK: fgets(strg,slen,hex_file);  //strip memwidth
                  fgets(strg,slen,hex_file);  //strip IOSTRB control
                  fgets(strg,slen,hex_file);  //strip STRB0 control
                  fgets(strg,slen,hex_file);  //strip STRB1 control
                  break;
  }
  for(x=0;x<16;x++) // realistic upper limit of 16 sections
  {
    fgets(strg,slen,hex_file); length=strtoul(strg,&eptr,0);//hdr: how long
    if(length==0) break;
    if(length>10000)
    {
      my_fclose(hex_file);
      return OPEN_ERR;
    }
    fgets(strg,slen,hex_file); addr =strtoul(strg,&eptr,0); //hdr: where to
    switch(HEXTYPE)
    {
      case C31_DSK: break;
      case C32_DSK: fgets(strg,slen,hex_file);  //strip STRBx control
                    break;
    }
    if(x==0)
    {
      ENTRY = addr;
    }
    //
    // Send putmem header W/O data
    //
    xmit_long(XWRIT);  // write command
    xmit_long(length); // data packet length
    xmit_long(addr);   // dest addr
    xmit_long(1);      // dest indx
    for(;length>0;length--)
    {
      fgets(strg,slen,hex_file);   // Read data word from file
      l = strtoul(strg,&eptr,0); // convert to binary
      err = xmit_long(l);        // send data
      if(err!=NO_ERR) break;     // exit if there is a problem
      //putmem(addr++,1,&l);
    }
  }
  SEG[current_seg].offs = ENTRY;
  PC_Appli=ENTRY;
  err = putmem(CTXT_PTR+PC,1,&ENTRY);
  my_fclose(hex_file);
  if(TARGET!=HEXTYPE) return HEXMATCH;
  return err;
}
//***************************************************************
// This function reads COFF file section headers and moves the
// data into either the target or the output file depending on task.
// Note that inlining here will not speed up transfers since the
// core inner loop is unaffected.  IE better to save code space!
//***************************************************************
ulong Read_Data(ushort COFF_magic, FILE *cfile)
{
  ulong data;
  uint idata;
  switch(COFF_magic)
  { case MAGIC_FX : fread(&idata,2,1,cfile); data = idata; break;
    case MAGIC_FL : fread(& data,4,1,cfile); break;
    default:  break;
  }
  return data;
}
MSGS Read_Secn
  (ulong addr,  ulong length, FILE *cfile, ushort COFF_magic,
   TASK task,   int cr_cinit, FILE *hex_file)

{ MSGS err;
  ulong data;
  long clength;
  int crcnt = 0;
  char buf[80];
  do
  { for(;length>0;length--)        // read the data stream
    {
      data = Read_Data(COFF_magic,cfile);
      if(cr_cinit) // if cinit is set then a direct load to memory is needed
      {         // The cinit table format is as shown
                //
                // ulong length
                // ulong address
                // ulong data_0
                // ulong data_1
                // ulong data_2
                //   etc...
                // ulong -1        Terminates cinit
        if(data == 0) return NO_ERR;  // No more .cinit to load
        if((task==FILE2HEX) || (task==BOOT))
        {
          sprintf(buf,"Section Size  cinit[%d]",crcnt++);
          if((err=data2target(data,addr,task,buf,hex_file))!=NO_ERR) return err;
          length--;
        }
        clength = data;
        data = Read_Data(COFF_magic,cfile);
        if((task==FILE2HEX) || (task==BOOT))
        {
          sprintf(buf,"Start Address");
          if((err=data2target(data,addr,task,buf,hex_file))!=NO_ERR) return err;
          length--;
        }
        addr = data;
        for(;clength>0;clength--)        // read the data stream
        {
          data = Read_Data(COFF_magic,cfile);
          if((err=data2target(data,addr,task,"",hex_file))!=NO_ERR)
                return err;
          length--;
          addr++;
        }
      }
      else
      {
        #if DASM_EN
        BuildFastLook();
        sprintf(DStrg,">%08lx ",addr);
        get_label(DStrg+10,addr,0);
        strcat(DStrg,"          ");
        DStrg[19]=' ';
        Disasm(addr, data, DStrg+20);
        #else
        sprintf(DStrg,"");
        #endif
        err=data2target(data,addr,task,DStrg,hex_file);
        if(err!=NO_ERR) return err;
      }
      sprintf(buf,"%08lx\n",data); // Print to files
      addr++;
    }
  }while(length >0);
  return NO_ERR;
}
//-----------------------------------------------------
// Load Symbols from string table if >=8 (offset into)
//              or   direct from Symbol block
//-------------------------------------------------
MSGS LoadLongName(FILE *cfile, char *buf, fpos_t fp2)
{                              // filepos ptr to original value
  fpos_t filepos;//, fp2;
  if(fgetpos(cfile,&filepos))    return ACCESS_ERR;
//  fp2 = s.f.fpos2;
  fp2 += (f_hdr0.symb + sizeof(A_18)*f_hdr0.nsym);
  if(fsetpos(cfile,&fp2))        return ACCESS_ERR;
  if(fgets(buf,20, cfile)==NULL) return ACCESS_ERR;
  if(fsetpos(cfile,&filepos))    return ACCESS_ERR;
  return NO_ERR;
}
//-----------------------------------------------------
MSGS LoadSymbol(FILE *cfile, char *buf, SYMS sb, int chop)
{
  MSGS err;
  if(sb.tbl.sname[0]==0)
  {
    err = LoadLongName(cfile,buf,sb.f.fpos2);
  }
  else
  {
    strncpy(buf,(char *)sb.tbl.sname,18);
    if(chop) buf[8]=0;
    err = NO_ERR;
  }
  return err;
}
//-----------------------------------------------------
void concat(char *s1, char *s2)
{
  if(*s2)
  {
    strcat(s1,":");
    strcat(s1, s2);
  }
}
char FCN_name[80];
//------------------------------------------------------------------
// Filter the COFF data types into valid DSK types
// Next realease? convert DSK data types to COFF format
//-------------------------------------------------------------------
MSGS xxref_add(char *buf1,ulong svalu,int Basic,SYM_TYPE SYM_GLBL)
{
  NUM_TYPE NUM_XXX;
  switch(Basic)
  { default       : NUM_XXX = NT_INTEGER; break;
    case  T_FLOAT :
    case  T_DOUBLE: NUM_XXX = NT_FLOAT  ; break;
  }
  return(xref_add(buf1,svalu,NUM_XXX,SYM_GLBL));
}
//--------------------------------------------------
char buf1[80];
MSGS Load_COFF_sym(FILE *cfile,SYMS sb, int *count)
{
  MSGS err;
  char buf2[80];
  #define Basic        ( sb.tbl.stype & 0xf)
  #define Derived1     ((sb.tbl.stype >> 4) & 3)
  #define Aux          ( sb.tbl.s_aux )
  int Storage;
  Storage = ( sb.tbl.sclass);
  //--------------------------------------------------
  buf1[0]=0; buf2[0]=0;
  switch(Aux)
  {
    case  0: if((err=LoadSymbol(cfile,buf1,sb,1))!=NO_ERR) return err;
             break;
    case  1: fread(&aux.a_18,sizeof(A_18),1,cfile); *count+=1;
             if(feof(cfile)) return FORMAT_ERR;
             if((err=LoadSymbol(cfile,buf1,sb,1))!=NO_ERR) return err;
             break;
    default: return FORMAT_ERR;
  }
  //--------------------------------------------------
  switch(Storage)
  {
    case SC_FILE: if(Aux)
                  {
                    if((err=LoadSymbol(cfile,buf2,*(SYMS *)&aux,0))!=NO_ERR)
                    return err;
                    strcpy(HLL_file,buf2);
                  }
                  else strcpy(HLL_file,buf1);
                  break;
    case SC_EXT : switch(Derived1) // Filter which names are to be appended
                  {
                    case DT_FCN: strcpy(FCN_name,buf1);
                                 concat(buf1,HLL_file);
                    default: break;
                  }
                  err=xxref_add(buf1,sb.tbl.svalu,Basic,SYM_GLBL);
                  break;

    case SC_AUTO: switch(Derived1)
                  {
                    case DT_FCN: concat(buf1,HLL_file); break;
                    case DT_NON: concat(buf1,FCN_name); break;
                    default: break;
                  }
                  err=xxref_add(buf1,sb.tbl.svalu,Basic,SYM_LLBL);
                  break;
    default:
    case SC_NULL   :
    case SC_STAT   :
    case SC_REG    :
    case SC_EXTDEF :
    case SC_LABEL  :
    case SC_ULABEL :
    case SC_MOS    :
    case SC_ARG    :
    case SC_STRTAG :
    case SC_MOU    :
    case SC_UNTAG  :
    case SC_TPDEF  :
    case SC_USTATIC:
    case SC_ENTAG  :
    case SC_MOE    :
    case SC_REGPARM:
    case SC_FIELD  :
    case SC_UEXT   :
    case SC_STATLAB:
    case SC_EXTLAB :
    case SC_BLOCK  :
    case SC_FCN    :
    case SC_EOS    :
    case SC_LINE   :
                     buf1[1] = '0';
                     break;
  }
  //--------------------------------------------------------
  switch(err)
  { case MULT_REF: err = NO_ERR; break; // Notify multiple references?
    default:
    case NO_ERR  : break;
  }
  return err;
}
ulong bss_start;
MSGS Secn_Load(SECT_HDR1 s_hdrx, TASK task, FILE *hex_file, FILE *cfile)
{
  MSGS err;
  char buf[80];
  if((s_hdrx.fptr)&&(s_hdrx.size))
  {
    // If this is the .cinit section, and the -cr model was
    // used, each cinit 'block' becomes a seperate section
    if(strcmp(s_hdrx.name,".cinit")==0&&(s_hdrx.flag&0x10))
    {
      sprintf(buf,"Start Address '.cinit->.bss -cr model");
      s_hdrx.radd = bss_start;
    }
    else
    { // Send section size and start address
      data2target(s_hdrx.size,0x0L,task,"Section Size",hex_file);
      sprintf(buf,"Start Address '%.8s'",s_hdrx.name);
      data2target(s_hdrx.radd,0x0L,task,buf,hex_file);
    }
    if(TARGET==C32_DSK)
      data2target(STRB(s_hdrx.radd,0,0,0),0x0L,task,"STRBx ctrl",hex_file);
    if(strcmp(s_hdrx.name,".cinit")==0&&(s_hdrx.flag&0x10))
    {
  // .cinit needs to be directly loaded to the .bss section.  The form
  // .cinit address and length needs to be read by the
  // loader, which then directly places the data into the .bss memory
      s_hdrx.radd = bss_start;
      if(fseek(cfile, s_hdrx.fptr, SEEK_SET)) return ACCESS_ERR;
      err=Read_Secn(s_hdrx.radd,s_hdrx.size,cfile,f_hdr0.mgc,task,1,hex_file);
      if(err!=NO_ERR) return err;
    }
    else
    {
      if(fseek(cfile, s_hdrx.fptr, SEEK_SET)) return ACCESS_ERR;
      err=Read_Secn(s_hdrx.radd,s_hdrx.size,cfile,f_hdr0.mgc,task,0,hex_file);
      if(err !=NO_ERR) return err;
    }
  }
  return NO_ERR;
}
MSGS LOADCOFF(char *infilename,TASK task)
{
  FILE *cfile;
  FILE *hex_file=NULL;
  fpos_t filepos, fp2;
  uint i;
  int x;
  int dummy=0;
  long yl;
  uint fn_baseline;
  char filename[16];
//  char buf[80];
  char boot_file[120];
  char *ptr;
//  unsigned long bss_start;
  MSGS err;
  OPTN_HDR o_hdr;
  SECT_HDR0 s_hdr0[MAXSECTIONS]; // Coff ver 0 info can be safely
  SECT_HDR1 s_hdrx[MAXSECTIONS]; // copied to ver 1 or 2 structures
  LINE_TBL le;
  strcpy(filename,infilename);
  if(!strstr(filename,".")) strcat(filename,OUTEXT);
  if(access(filename,0)) return ACCESS_ERR;
  if((cfile = fopen(filename,"rb")) == NULL) return OPEN_ERR;
  //----------------------------------------------------------
  if(task==FILE2HEX)
  {
    strcpy(boot_file,infilename);
    ptr = strstr(boot_file,".");
    strcpy(ptr+1,"hex");
    unlink(boot_file);
    if((hex_file = fopen(boot_file,"wt"))==NULL)
    {
      my_fclose(cfile);
      my_fclose(hex_file);
      return COFF_HEX;
    }
    if(TARGET==C31_DSK)  fprintf(hex_file,"FILE2HEX\n");
    if(TARGET==C32_DSK)  fprintf(hex_file,"FILE2HEX_C32\n");
  }
  //----------------------------------------------------------
  fread(&f_hdr0,sizeof(FILE_HDR0),1,cfile);         // read file header
  /*
     To simplify reading the file header, the only difference is that
     the length is 2 bytes longer and the magic number has been moved
     to the last 2 bytes, not the first 2.  The first two indicate
     if the fileheader is an old version 0 or a newer version 1 or 2

     Since the size and position of other sub elements did not change,
     only the old file header structure is needed.

     This section of code identifies the file as floating point or
     fixed by the magic number, which can be in either position.

     Also, the type of OS that created the file can be determined by
     reading the magic number.  If the OS reading the file (debugger
     code) does not match the OS that created the file the high and
     low bytes will be swapped.

     Since at this time there is no provision to unswap the bytes,
     all non-OS matching files are returned as an INV_COFF_MGC error.
  */
  COFF_OS = DOS_OS;
  CoffVer = 0x0;
  switch(f_hdr0.mgc)
  {
    // If f_hdr0.mgc = 0x92 or 0x93 file header is for version 0 COFF
    //    0x92 = fixed point
    //    0x93 = float point
    case MAGIC_FL: break;
    case MAGIC_FX: return FIX_COFF;
    // if f_hdr0.mgc = 0xC0 file header is for version 1 COFF
    //    target ID type is located in next 2 bytes
    //    0x92 = fixed point
    //    0x93 = float point
    case 0x00C5  : CoffVer++;  // Coff version 5
    case 0x00C4  : CoffVer++;  // Coff version 4
    case 0x00C3  : CoffVer++;  // Coff version 3
    case 0x00C2  : CoffVer++;  // Coff version 2
    case 0x00C1  : CoffVer++;  // Coff version 1
    case 0x00C0  :             // Coff version 0
                   fread(&f_hdr0.mgc,sizeof(f_hdr0.mgc),1,cfile);
                   break;
    default      : COFF_OS = INV_COFF_MGC;
                   my_fclose(cfile);
                   my_fclose(hex_file);
                   return INV_COFF_MGC;
  }
  if(f_hdr0.mgc != MAGIC_FL) return FORMAT_ERR;
  strcpy(DASM_file,infilename);
  if(f_hdr0.sect  > MAXSECTIONS)
  {
    my_fclose(cfile);
    my_fclose(hex_file);
    return MAX_SECTN;
  }

  /* The OS must already compatible but the assy language tools
     indicate that this is where the OS comparison is made */
  if(f_hdr0.optn == 28)
  { fread(&o_hdr,sizeof(OPTN_HDR),1,cfile);   // read optional header
    switch(o_hdr.mgc)
    {
      case  0x108: COFF_OS =  DOS_OS; break; // Which OS type created file
      case  0x801: COFF_OS = UNIX_OS; break;
      default:     my_fclose(cfile);
                   my_fclose(hex_file);
                   return BAD_OPTN_HDR;
    }
  }
  else
  { if(f_hdr0.optn != 0)
    {
      my_fclose(cfile);
      my_fclose(hex_file);
      return  BAD_OPTN_HDR;
    }
  }

  ENTRY = o_hdr.entry; // Added for ROM support

  for(i=0; i<f_hdr0.sect; i++)                  // read each section header
  {
    if(CoffVer == 0)
    {
      fread(&s_hdr0[i],sizeof(SECT_HDR0),1,cfile);// Read old structure
   // s_hdrx[i].     = s_hdr0[i].     ;           // then copy to new form
      strncpy(s_hdrx[i].name,s_hdr0[i].name,8);
      s_hdrx[i].radd = s_hdr0[i].radd;
      s_hdrx[i].vadd = s_hdr0[i].vadd;
      s_hdrx[i].size = s_hdr0[i].size;
      s_hdrx[i].fptr = s_hdr0[i].fptr;
      s_hdrx[i].rent = s_hdr0[i].rent;
      s_hdrx[i].lent = s_hdr0[i].lent;
      s_hdrx[i].nrel = s_hdr0[i].nrel;
      s_hdrx[i].nlin = s_hdr0[i].nlin;
      s_hdrx[i].flag = s_hdr0[i].flag;
      s_hdrx[i].resv = s_hdr0[i].resv;
      s_hdrx[i].page = s_hdr0[i].page;
    }
    else // If new version read directly into s_hdrx structure
      fread(&s_hdrx[i],sizeof(SECT_HDR1),1,cfile);
    // the starting address of .bss is kept in the event of a
    // direct load of the cinit section (-CR model)
    if(strcmp(s_hdrx[i].name,".bss")==0) bss_start = s_hdrx[i].radd;
  }
  //
  // Create bootfile header
  if((task == FILE2HEX)||(task == BOOT))
  {
    if(task==BOOT)
    {
    // First word sets byte size download.
    // NOTE: The first 32 bit word is NOT read from the boot image table
    // and entirely transferred since the last two bytes are skipped
    // over in the bootloader code... (not a problem for external EPROM)
      DSK_reset();
      if(TARGET==C31_DSK)
      {
        xmit_byte(0x8);
        xmit_byte(0x0);
      }
      else
        data2target(0x00000008L,0x0L,task,"Bus Width"   ,hex_file);
    }
    else
      data2target(0x00000008L,0x0L,task,"Bus Width"   ,hex_file);

    if(TARGET==C31_DSK)
    {
  //  data2target(0x000010F8L,0x0L,task,"Bus Control ws=7"   ,hex_file);
      data2target(STRB0      ,0x0L,task,"Bus Control ws=7"   ,hex_file);
      data2target(0x00000001L,0x0L,task,"Dummy Section Size" ,hex_file);
      data2target(o_hdr.entry,0x0L,task,"Dummy Start Address",hex_file);
      data2target(0x00000000L,0x0L,task,"Dummy Data"         ,hex_file);
    }
    if(TARGET==C32_DSK)
    {
      data2target(IOSTRB     ,0x0L,task,"IOSTRB"               ,hex_file);
      data2target(STRB0      ,0x0L,task,"STRB0"                ,hex_file);
      data2target(STRB1      ,0x0L,task,"STRB1"                ,hex_file);
      data2target(0x00000001L,0x0L,task,"Dummy Section Size"   ,hex_file);
      data2target(ENTRY      ,0x0L,task,"Dummy Start Address"  ,hex_file);
     data2target(STRB(ENTRY,0,0,0),0x0L,task,"Dummy STRBx ctrl",hex_file);
      data2target(0x00000000L,0x0L,task,"Dummy Data"           ,hex_file);
    }
  }
  for(i=0; i<f_hdr0.sect; i++)                  // download each section
  {
    err = Secn_Load(s_hdrx[i],task,hex_file,cfile);
    if(err!=NO_ERR)
    {
      my_fclose(cfile);
      my_fclose(hex_file);
      return err;
    }
    /*
    if((s_hdrx[i].fptr)&&(s_hdrx[i].size))
    {
      // Send section size and start address
      data2target(s_hdrx[i].size,0x0L,task,"Section Size",hex_file);
      sprintf(buf,"Start Address '%.8s'",s_hdrx[i].name); //section_name);
      if(strcmp(s_hdrx[i].name,".cinit")==0&&(s_hdrx[i].flag&0x10))
      {
        sprintf(buf,"Start Address '.cinit->.bss -cr model");
        s_hdrx[i].radd = bss_start;
      }
      data2target(s_hdrx[i].radd,0x0L,task,buf,hex_file);
      if(TARGET==C32_DSK)
        data2target(STRB(s_hdrx[i].radd,0,0,0),0x0L,task,"STRBx ctrl",hex_file);
      if(strcmp(s_hdrx[i].name,".cinit")==0&&(s_hdrx[i].flag&0x10))// -CR
      {
        if(s_hdrx[i].nrel != 0)                // smart copy .cinit -->.bss
        {                                     // !data
          if(fseek(cfile, s_hdrx[i].fptr, SEEK_SET))
          { my_fclose(cfile); my_fclose(hex_file); return ACCESS_ERR; }
          err=Read_Secn(s_hdrx[i].radd,s_hdrx[i].size,cfile,f_hdr0.mgc,task,0,hex_file);
          if(err!=NO_ERR){my_fclose(cfile);my_fclose(hex_file);return err;}
        }
        else
        {
      // .cinit needs to be directly loaded to the .bss section.  The form
      // .cinit address and length needs to be read by the
      // loader, which then directly places the data into the .bss memory
          s_hdrx[i].radd = bss_start;
          if(fseek(cfile, s_hdrx[i].fptr, SEEK_SET))
          {my_fclose(cfile);my_fclose(hex_file);return ACCESS_ERR;}
          err=Read_Secn(s_hdrx[i].radd,s_hdrx[i].size,cfile,f_hdr0.mgc,task,1,hex_file);
          if(err!=NO_ERR){my_fclose(cfile);my_fclose(hex_file);return err;}
        }
      }
      else
      {
        if(fseek(cfile, s_hdrx[i].fptr, SEEK_SET))
        {my_fclose(cfile);my_fclose(hex_file);return ACCESS_ERR;}
        err=Read_Secn(s_hdrx[i].radd,s_hdrx[i].size,cfile,f_hdr0.mgc,task,0,hex_file);
        if(err !=NO_ERR){my_fclose(cfile);my_fclose(hex_file);return err;}
      }
    }
    */
  }
  data2target(0x00000000L,0x00000000L,task,"Terminate Load",hex_file);
  //--------------------------------------------------------
  // Load Line information into the HLL line cross
  // reference table.  Note that the name of each function
  // is loaded at this time
  //--------------------------------------------------------
  xref_add(">>__FUNCTION__<<",0,NT_INTEGER,SYM_VAR); // SYM_VAR was 0
  lastHLL = 0;
  HLL_file[0] = 0;
  if((f_hdr0.flag & 4)==0) // Load line info if not stripped from file
  { // Read line info for each section in order
//    xxy = 0;
    for(x=0;x<f_hdr0.sect;x++)
    {  // Use the already loaded section header to set the
       // file pointer to the first line info header
       fp2 = s_hdrx[x].lent;
       if(fsetpos(cfile,&fp2)) return ACCESS_ERR;
       //
       fn_baseline = 0;
       for(yl=s_hdrx[x].nlin;yl>0;yl--)
       {
         // Read each line entry which may contain either a file ptr
         // to a symbol entry in the symbol list or actual line
         // information.  If the entry is for a symbol, load the symbol.
         //
         fread(&le,sizeof(LINE_TBL),1,cfile);
         if(le.sptr.zero2 == 0) // First line of a function entry ?
         {
           if(fgetpos(cfile,&filepos)) return ACCESS_ERR; // keep original ptr
           fp2 = f_hdr0.symb;
           fp2 = fp2 + sizeof(SYMB_TBL) * le.sptr.sym_offs;
           if(fsetpos(cfile,&fp2)) return ACCESS_ERR;
           fread(&sx.tbl,sizeof(SYMB_TBL),1,cfile);
           if(feof(cfile)) return ACCESS_ERR;
           err = Load_COFF_sym(cfile,sx,&dummy);  // Load function symbol
           //
           // The next locations in the symbol table define the
           // base address of the function, if followed by a .bf
           //
           fread(&sx.tbl,sizeof(SYMB_TBL),1,cfile);
           if(strexact((char *)sx.tbl.sname,".bf"))
           {
             fread(&aux,sizeof(A_25),1,cfile);
             fn_baseline = aux.a_25.cline;
           }
           if(err!=NO_ERR) return err;
           if(fsetpos(cfile,&filepos))
             return ACCESS_ERR;
           // Add to the HLL list the reference number to the symbol
           // which was just added.  Then set the line_no to zero
           lastHLL++;
           if(lastHLL >= HLL_limit) return HLL_LIMIT;
           HLL[lastHLL].sptr.sym_offs = last_ref-1;
           HLL[lastHLL].sptr.zero2    = 0;
           // Next add the line start information for the function name
           // which was just added.  IE the value of the symbol
           lastHLL++;
           if(lastHLL >= HLL_limit) return HLL_LIMIT;
           HLL[lastHLL].xref.DSP_addr = SYM[last_ref-1].value;
           HLL[lastHLL].xref.HLL_line = fn_baseline;
         }
         else // Add line info to xref table
         {
           lastHLL++;
           if(lastHLL >= HLL_limit) return HLL_LIMIT; // ACCESS_ERR;
           HLL[lastHLL] = le;
           HLL[lastHLL].xref.HLL_line += fn_baseline;
         }
       }
    }
  }
  xref_add(">>__SYMBOL:FILE__<<",0,NT_INTEGER,SYM_VAR);
  //---------------------------------------------------------------
  // Load symbols relating to functions.  Symbol information
  // which is non-line referencable are not loaded.  IOW definitions
  // for structure elements, constants etc... that are local to
  // a function are not loaded (they are temporary stack vars).
  //----------------------------------------------------------------
  HLL_file[0] = 0;
  if(fseek(cfile, f_hdr0.symb, SEEK_SET))
  {
    my_fclose(cfile);
    my_fclose(hex_file);
    return ACCESS_ERR;
  }
  for(x=0; x < (int)f_hdr0.nsym; x++)
  {
    //-----------------------------------------------------
    // Determine if symbol is to be added to symbol table
    //-----------------------------------------------------
    if(feof(cfile)) break;
    fread(&sx.tbl,sizeof(SYMB_TBL),1,cfile);
    err = Load_COFF_sym(cfile,sx,&x);
    if(err!=NO_ERR) return err;
  }
  my_fclose(cfile);
  my_fclose(hex_file);
  if(task == LOAD)
  { PC_Appli = o_hdr.entry;
    return putmem(CTXT_PTR+PC,1,&PC_Appli);
  }
  return NO_ERR;
}
//
// The CVT2COFF() format converter creates version 1 COFF file structures.
// To create COFF version 2 structures the file and section headers need
// to be defined as type FILE_HDR1 and SECT_HDR1 respectively.  The COFF
// loader will read in either type.
//
#define COFF_VER 0 // Old products should always be able to load ver 0
//
MSGS CVT2COFF (char *lst_file, ulong st, ulong ln, TASK task)
{
  MSGS err;
  FILE *cfile;
  char *ptr;
  char coff_file[16];
  ulong start;
  ulong length;
  ulong data_block_start;
  ulong temp=0;
  uint i;
  fpos_t filepos;
  OPTN_HDR o_hdr;
#if COFF_VER == 0
  FILE_HDR0 f_hdr;              // Use for COFF ver 0 output
  SECT_HDR0 s_hdr[MAXSECTIONS]; //
#endif
#if COFF_VER == 1
  FILE_HDR1 f_hdr;              // Use for COFF ver 1
  SECT_HDR1 s_hdr[MAXSECTIONS]; // Note Use of same variable name
#endif
#if COFF_VER == 2
  FILE_HDR1 f_hdr;              // Use for COFF ver 2
  SECT_HDR1 s_hdr[MAXSECTIONS]; // Note Use of same variable name
#endif
  //----------------------------------------------------------
  // Open input/output files
  //----------------------------------------------------------
  start  = st;
  length = ln;
  strcpy(coff_file,lst_file);
  ptr = strstr(coff_file,".");
  strcpy(ptr+1,"out");
  unlink(coff_file);
  if((cfile = fopen(coff_file,"wb"))==NULL)
  { my_fclose(cfile);
    return FATAL_OPEN;
  }
  //-----------------------------------------------------------
  // Reread in the information stored in the DSK file
  //-----------------------------------------------------------
  // Read DSK3A symbol table
  if(task==DSK2COFF)
    if((err=loadsegments(lst_file))!=NO_ERR)
      { my_fclose(cfile); return err;}
  if(task==MEM2COFF)
  {
    last_segment = 1;
    strcpy(SEG[0].name,"MEM2COFF");
    SEG[0].start = start;
    SEG[0].length= length;
  }
  //-----------------------------------------------------------
  // Fill in FILE_HDR structure and then write it out
  //-----------------------------------------------------------
  f_hdr.mgc  = MAGIC_FL;              // coff magic num 0x93 (0x92==fixed pt)
  f_hdr.sect = (ushort)last_segment;  // number of sections in file

//  time
//Fill this in?
  f_hdr.date = 0;                     // time and date stamp

  f_hdr.symb = 0;                     // file pointer to symbol table
  f_hdr.nsym = last_ref-post_boot_sym;// number of symbols in symbol table
  f_hdr.optn = 28;                    // length of optional header
  f_hdr.flag = 0;                     // Flags
  fwrite(&f_hdr,sizeof(FILE_HDR0),1,cfile);  // write file header
//fwrite(&f_hdr,sizeof(FILE_HDR1),1,cfile);  // COFF ver 2 output
#if COFF_VER == 0
  f_hdr.mgc  = MAGIC_FL;     // Version 0 coff
#endif
#if COFF_VER == 1
  f_hdr.mgc  = 0xC1;         // Version 1 coff (Never released)
  f_hdr.mgc2 = MAGIC_FL;
#endif
#if COFF_VER == 2
  f_hdr.mgc  = 0xC2;         // Version 2 coff
  f_hdr.mgc2 = MAGIC_FL;
#endif
  //-----------------------------------------------------------
  // Fill in OPTN_HDR structure and then write it out
  //-----------------------------------------------------------
  o_hdr.mgc    = 0x108;  // options magic number 0x108
  o_hdr.vers   = 0x100;  // version stamp
  o_hdr.text_sz= 0    ;  // size (in words) of .text
  o_hdr.data_sz= 0    ;  // size (in words) of .data
  o_hdr.bss_sz = 0    ;  // size (in bits) of .bss
  o_hdr.entry  = ENTRY;  // entry point address
  o_hdr.text_ad= 0    ;  // beginning address of .text
  o_hdr.data_ad= 0    ;  // beginning address of .data
  fwrite(&o_hdr,sizeof(OPTN_HDR),1,cfile);  // write file header
  //-----------------------------------------------------------
  // Fill in SECT_HDR structures and then write them out
  // immediately followed by the section data
  //-----------------------------------------------------------
  // Calculate file position for the first data block
  //
  fgetpos(cfile,&filepos);
  data_block_start = (ulong)(filepos + f_hdr.sect * sizeof(SECT_HDR0));
//data_block_start = filepos + f_hdr.sect * sizeof(SECT_HDR1); // COFF ver 2
  for(i=0;i<f_hdr.sect;i++)                 // write each section header
  {
    strncpy(s_hdr[i].name,SEG[i].name,8);     // Section name
    s_hdr[i].radd = SEG[i].start;             // runtime address
    s_hdr[i].vadd = SEG[i].start;             // virtual address (load addr)
//  s_hdr[i].size = sizeofseg(i,lst_file);    // section size
//  s_hdr[i].size = SEG[i].offs - SEG[i].start; // section size
    s_hdr[i].size = SEG[i].length;            // section size
    s_hdr[i].fptr = data_block_start;
    s_hdr[i].rent = 0;                        // file ptr to reloc entries
    s_hdr[i].lent = 0;                        // file ptr to line num entries
    s_hdr[i].nrel = 0;                        // number of relocation entries
    s_hdr[i].nlin = 0;                        // number of line num entries
    s_hdr[i].flag = 0;                        // flags
    s_hdr[i].resv = 0;                        // reserved
    s_hdr[i].page = 0;                        // memory page number
    fwrite(&s_hdr[i],sizeof(SECT_HDR0),1,cfile);
//  fwrite(&s_hdr1[i],sizeof(SECT_HDR1),1,cfile); // COFF Ver 2 output
    data_block_start += (4*s_hdr[i].size);    // start of next data block
    fgetpos(cfile,&filepos);                   // file ptr to raw data
  }
  //---------------------------------
  // write out data for section
  //---------------------------------
  for(i=0; i<f_hdr.sect; i++)             // write each section header
  {
    strcpy(section_name,SEG[i].name);
    if(task==DSK2COFF) write_seg_val(lst_file,task,cfile);
    if(task==MEM2COFF)
    {
      for(;length;length--)
      {
        if((err=getmem(start,1,&temp))!=NO_ERR)
          {my_fclose(cfile); return err;}
        fwrite(&temp,4,1,cfile);
        start++;
      }
    }
  }
  //----------------------------------------------------------------
  // Write out symbol table
  // NOTE: Symbols that are defined in the kernel (such as 'stack')
  //       may have only been updated, but since they appear in the
  //       symbol list after the kernel init, they wont get put into
  //       the COFF file symbol list
  //-----------------------------------------------------------------
  fgetpos(cfile,&filepos); // filepos is at the start of the symbol table
  //
  int syms = 0;
  int x1;
  if(ln==0)  {x1=post_boot_sym;}// ln=-1;} // file convert
  else       x1=0;             // mem convert
  for(; x1 < last_ref; x1++)
  {
    //if(x >= last_ref) break;
    switch(SYM[x1].symt)    // if addressab le, treat as a label
    {
      case  SYM_VAR : sx.tbl.sclass = SC_EXT   ; break;
      case  SYM_GLBL: sx.tbl.sclass = SC_LABEL ; break;
      default       : sx.tbl.sclass = SC_EXT   ; break;
    }

    if(SYM[x1].symt == SYM_GLBL) // Addressable symbols only
    {
 // ???
 //     if((SYM[x1].value > (long)st)&&(SYM[x1].value < (long)(st+ln)))
 //
      {
        strncpy((char *)sx.tbl.sname,SYM[x1].sptr,8);//8 char or ptr to strg tbl
        sx.tbl.svalu = SYM[x1].value;        // symbol value
        sx.tbl.sectn = 1;                    // DSK section number of symbol
        sx.tbl.stype = 5;                    // symbol type (long int default)
//      sx.tbl.sclass= s_class;              // DSK store class = static label
        sx.tbl.s_aux = 0;                    // DSK number of aux entries = 0
        fwrite(&sx.tbl,sizeof(SYMB_TBL),1,cfile);  // write symbol block
        syms++;
      }
    }
  }
  //--------------------------------------------
  // Go back and set symbol table file pointer
  //--------------------------------------------
  #define FHDRSYMB_OFFS  8
  fseek(cfile,FHDRSYMB_OFFS,SEEK_SET);  // File offeset up to f_hdr.symb
  f_hdr.symb = (ulong)filepos;
  f_hdr.nsym = syms;
  fwrite(&f_hdr.symb,sizeof(f_hdr.symb),1,cfile);
  fwrite(&f_hdr.nsym,sizeof(f_hdr.nsym),1,cfile);
  my_fclose(cfile);
  return NO_ERR;
}

