//---------------------------------------------------------
// TARGET.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996, 1997
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//---------------------------------------------------------
#include <dos.h>
#include <conio.h>
#include <stdio.h>
#include "dsk.h"
//#include "errormsg.h"
#if __DSKWINAPP    // MSVC, Borland C++ Windows
#else
#include <bios.h>  // Borland DOS
#endif

ulong PC_Appli;
ulong dT_Test_NOP;
int swap_en = 0;    // Used for forcing all interrupts to singlestep
//
// Non DOS application environments (Borland) do not support the delay()
// function.  This function would normaly use the system timer to achieve
// 1 mS accuracy.  However, it can be reasonably emulated using the I/O
// channel read rate of 2uS/access.
//
void mydelay(int ms)
{
  int t;
  for(;ms>0;ms--)
    for(t=0;t<500;t++)
#if _WIN32
		_inp(0x378);
#else
		inportb(0x378);  // Reading the printer data port is safe
#endif
}
//********************************************************************
// fgetmem()/fputmem() are wrappers around getmem/putmem.  These
// functions convert floating point arrays to and from the native
// host IEEE float format to the TMS float format
//*********************************************************************/
MSGS DLLEXTEND_EX fgetmem(ulong addr, ulong length, float *data)
{
  MSGS err;
  int i;
  ulong temp;
  if((err = getmem(addr,length,(ulong *)data)  )!=NO_ERR) return err;
  for(i=0;i<(long)length;i++)
  {
    temp = *((long *)&data[i]);
    data[i] = TMS_IEEE(temp);
  }
  return NO_ERR;
}
MSGS DLLEXTEND_EX fputmem(ulong addr, ulong length, float *data)
{
  MSGS err;   // To save malloc space, IEEE<->TMS conversions
  int i;      // are done in place in both directions (no temp needed)
  ulong temp;
  for(i=0;i<(long)length;i++)
  {
    temp = IEEE_TMS(data[i]);
    data[i] = *((float *)&temp);
  }
  err = putmem(addr,length,(ulong *)data);
  for(i=0;i<(long)length;i++)
  {
    temp = *((long *)&data[i]);
    data[i] = TMS_IEEE(temp);
  }
  return err;
}
//********************************************************************
// getmem() reads a block of size 'length' from the DSK memory 'addr'
// to the hosts local memory 'data'
//*********************************************************************/
MSGS DLLEXTEND_EX getmem(ulong addr, ulong length, ulong *data)
{
  int i;
  ulong CMD[4];
  long t;
  MSGS err;
  if(( addr         & 0xF00000L) == 0xF00000L) return HOST_PORT;
  if(((addr+length) & 0xF00000L) == 0xF00000L) return HOST_PORT;
  CMD[0] = XREAD;     // Read command
  CMD[1] = length;    // data packet length
  CMD[2] = addr;      // srce addr
  CMD[3] = 1;         // srce indx
  disable_Mtask();
  for(i=0;i<4;i++)
    if((err = xmit_long(CMD[i]))!=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  if(FAST_FLAG)
  {
    t = MAX_WAIT;  // DSP should be ready to xfr with no pre-synch
    MAX_WAIT = 0;
  }
  for(;length>0;length--)
    if((err = recv_long(data++))!=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  if(FAST_FLAG)
    MAX_WAIT = t;
  enable_Mtask();
  return NO_ERR;
}
//********************************************************************
// putmem() writes a block of size 'length' to the DSK memory 'addr'
// from the hosts local memory 'data'
//*********************************************************************/
MSGS DLLEXTEND_EX putmem(ulong addr,ulong length,ulong *data)
{
  int i;
  ulong CMD[4];
  MSGS err;
  long t;
  if(( addr         & 0xF00000L) == 0xF00000L) return HOST_PORT;
  if(((addr+length) & 0xF00000L) == 0xF00000L) return HOST_PORT;
  CMD[0] = XWRIT;  // write command
  CMD[1] = length; // data packet length
  CMD[2] = addr;   // dest addr
  CMD[3] = 1;      // dest indx
  disable_Mtask();
  for(i=0;i<4;i++)
    if((err=xmit_long(CMD[i])) !=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  if(FAST_FLAG)
  {
    t = MAX_WAIT;  // DSP should be ready to xfr with no pre-synch
    MAX_WAIT = 0;
  }
  for(;length>0;length--)
    if((err=xmit_long(*data++))!=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  if(FAST_FLAG)
    MAX_WAIT = t;
  enable_Mtask();
  return NO_ERR;
}
//---------------------------------------------------------------
// SSTEP_CPU() causes the DSK to restore the context, execute
// one opcode, save the context and then go back to the command
// (spin) mode.  The host application can then download the
// new context for display.
//
// If the swap_en flag is enabled the vector table corresponding
// to higher level interrupts is swapped with branch instructions
// that point all of the higher level interrupts back to the
// single step instruction.  This is used cases where the instruction
// being singlestepped enables interrupts with a higher priority
// level than the singlestep routine,
//---------------------------------------------------------------
/*
MSGS SSTEP_CPU_OLD(void)
{
  int t;
  ulong CMD;
  MSGS err;
  CMD = XSSTEP;
  disable_Mtask();
  if((err=xmit_long(CMD))!=NO_ERR)
  {
    enable_Mtask();
    return err;
  }
  for(t=500;t>0;t--)
  {
    if(HPI_ACK()) break;
    delay(1);
  }
  if(t==0)
  {
    enable_Mtask();
    return RECV_ERR; // SSTEP synch failed, INT2 recovery failed
  }
  err = recv_long(&CMD); // Should always return zero if step successful
  enable_Mtask();
  if(CMD!=0)          // Rev 1.02 and later always returns 0
    return COM_ERR;
  return err;
}
*/

MSGS DLLEXTEND_EX SSTEP_CPU(void)
{
  #define VL 7
  int t,T;
  ulong CMD;
  ulong ZERO=0;
  ulong VECT_BUFFR[VL];
  /* B  XINT1 */
  ulong VECT_SSTEP[]=
  {
    0x6a000005L, /* INT0   B  XINT1                        */
    0x6a000004L, /* INT1   B  XINT1                        */
    0x6a00FFD9L, /* INT2   B  ????    HOST PORT INT MODIFY */
    0x6a000002L, /* INT3   B  XINT1   AND PUT BACK IS OK   */
    0x6a000001L, /* XINT0  B  XINT1                        */
    0x6a000000L, /* RINT0  B  XINT1                        */
    0x6a000000L, /* XINT1  B  XINT1                        */
  };
  MSGS err;
  CMD = XSSTEP;
  /***************************************
   Swap out the higher level INT routines compared to the
   SSTEP interrupt (XINT0) with a branch to XINT0.  This
   will cause the timing to be off by several cycles as the
   ints cause a double branch (detect INT activity?)
   *******************************************************/
   if(swap_en)
   {
     getmem(VECTLOC,VL,VECT_BUFFR);  /* get high level branch table */
     switch(TARGET)
     {
       case C32_DSK:
       VECT_SSTEP[0] = VECT_BUFFR[6];    /* All vectors are to XINT1 */
       VECT_SSTEP[1] = VECT_BUFFR[6];
       VECT_SSTEP[2] = VECT_BUFFR[6];
       VECT_SSTEP[3] = VECT_BUFFR[6];
       VECT_SSTEP[4] = VECT_BUFFR[6];
       VECT_SSTEP[5] = VECT_BUFFR[6];
       VECT_SSTEP[6] = VECT_BUFFR[6];
       break;
       default:
       case C31_DSK:
       VECT_SSTEP[2] = VECT_BUFFR[2];    /* dont modify INT2        */
       VECT_SSTEP[6] = VECT_BUFFR[6];    /* or XINT1                */
       break;
     }
     putmem(VECTLOC,VL,VECT_SSTEP);      /* replace with B to XINT1 */
   }
  /****************************************/
  disable_Mtask();
  if((err=xmit_long(CMD))!=NO_ERR)
  { enable_Mtask();
    return err;
  }
  for(t=500;t>0;t--)
  {
    if(HPI_ACK()) break;
    mydelay(1);
  }
  if(t==0)
  { /* Attempt to reaquire communication via HPI strb low */
    HPI_STRB(0);
    for(t=500;t>0;t--)
    {
      if(HPI_ACK())
      { //---------------------------------------------------
        // err = ???? // SSTEP synch lost, try INT2 recovery
        //---------------------------------------------------
        T = No_MTask; // Dont release timeslot here...
        No_MTask=1;
        HALT_CPU();   // Save the context as if SSTEP worked
        CMD = 0;
        err = Recover1;
        putmem(CTXT_PTR+IE,1,&ZERO);  // Needed?
        No_MTask=T;
        break;
      }
      mydelay(1);
    }
    if(t==0)
    { enable_Mtask();
      return Recover2; // SSTEP synch failed, INT2 recovery failed
    }
  }
  else
    err = recv_long(&CMD);
  enable_Mtask();
  if(CMD!=0)          // Rev 1.02 and later always returns 0
    return COM_ERR;
  /***************************************/
  if(swap_en)
    putmem(VECTLOC,VL,VECT_BUFFR);  /* restore higher INT vects */
  /***************************************/
  return err;
}

//----------------------------------------------------------
// RUN_CPU() causes the DSK to restore the contents of the
// context to the DSP and then begin execution at the
// current defined PC.
//
// NOTE: If a breakpoint is encountered at any time that
//       the DSP is running the context is saved and the
//       CPU falls back into command (spin) mode.
//-----------------------------------------------------------
MSGS DLLEXTEND_EX RUN_CPU(void)
{
  ulong CMD;
  MSGS err;
  disable_Mtask();
  CMD = XRUNF;          // command
  err = xmit_long(CMD);
  mydelay(20);            // Allow time for context restore
  enable_Mtask();
  return err;
}
//------------------------------------------------------------
// HALT_CPU() Halts the DSK and then waits for a synchronization
// word.  This command places the CPU in a spin loop where it
// is waiting for a command.  Do not re-issue this command
// while the DSK is halted as this will cause the context to
// be overwritten with the state of the processor while it
// is in the command 'spin' loop.
//
// NOTE:
//   In some cases a longer timeout may be required for the
//   initial transmission of "xmit_long(CMD)".  This can be
//   accomplished by driving the strobe line low, followed
//   by a acknowlwdge wait loop.  The xmit_long() function
//   can then be safely executed.
//------------------------------------------------------------
MSGS DLLEXTEND_EX HALT_CPU(void)
{
  int t;
  ulong CMD;
  MSGS err;
  CMD = XHALT;                // Write link command
  disable_Mtask();
  if((err=xmit_long(CMD))!=NO_ERR)
  {
    enable_Mtask();
    return err;
  }
  for(t=500;t>0;t--)     // Allow time for context save in slow DSK
  { if(HPI_ACK()) break;
    mydelay(1);
  }
  if(t==0)
  {
    enable_Mtask();
    return RECV_ERR;
  }
  err = recv_long(&CMD);
  enable_Mtask();
  return err;
}
//-----------------------------------------------------
// GET_DEBUG_CTXT() returns the address of the CPU
// CPU context save area.  This address is later
// used by the debugger and other applications to
// retrieve the contents of the DSP's registers.
//
// NOTE: This function DOES NOT cause the DSP context
//       to be updated.  To update the context, a
//       CPU_HALT() should be sent to the DSK.  Also
//       do not send more than one CPU_HALT() as this
//       will cause the context to be overwritten by
//       the halt state of the processor.
//-----------------------------------------------------

MSGS DLLEXTEND_EX GET_DEBUG_CTXT(void)
{
  ulong CMD;
  MSGS err;
  CMD = XCTXT;                // Write linke command
  disable_Mtask();
  if((err=xmit_long(CMD))!=NO_ERR)
  {
    enable_Mtask();
    return(err);
  }
  if((err=recv_long(&CTXT_PTR))!=NO_ERR)
  {
    enable_Mtask();
    return(err);
  }
  enable_Mtask();
  return err;
}
//----------------------------------------------------------------------
// By testing the timing of a known opcode in the kernel (INT0 branch)
// the correct difference for any memory space can be calculated.
//----------------------------------------------------------------------
MSGS DLLEXTEND_EX SSTEP_NOP(void)
{
  MSGS err;
  int temp;
  ulong temp1;
  ulong temp2,temp3;
//static int SSTEP_NOP_DONE = 0;     // Do only one time
//if(SSTEP_NOP_DONE) return NO_ERR;  //
//SSTEP_NOP_DONE = 1;                //
  if(TARGET==C30_EVM)
  {
    dT_Test_NOP=0x100; return NO_ERR;
  }
  if((err=getmem(CTXT_PTR+   PC,1,&temp1))!=NO_ERR) return err;
  if((err=getmem(CTXT_PTR+T1DIF,1,&temp2))!=NO_ERR) return err;
  temp3 = STEPLOC; // Safe place to singlestep (no reg modification)
  if((err=putmem(CTXT_PTR+   PC,1,&temp3))!=NO_ERR) return err;
  temp = swap_en;
  swap_en = 1;
  if((err=SSTEP_CPU())!=NO_ERR) {swap_en = temp; return err;}
  swap_en = temp;

  if((err=getmem(CTXT_PTR+T1DIF,1,&(ulong)dT_Test_NOP))!=NO_ERR)
    return err;
//dT_Test_NOP -= 4;  // if STEPLOC code is a 4 cycle branch
  dT_Test_NOP -= 1;  // if STEPLOC code is a 1 cycle instruction

  if((err=putmem(CTXT_PTR+   PC,1,&temp1))!=NO_ERR) return err;
  if((err=putmem(CTXT_PTR+T1DIF,1,&temp2))!=NO_ERR) return err;
//ref_mod("PC",DASMBGN);
  return NO_ERR;
}
