/*=======================================================================
  DSK_BODE.C     (Uses Sliding FFT technology)
  Keith Larson
  TMS320 DSP Applications
  (C) Copyright 1996,1997,1998
  Texas Instruments Incorporated

  This is unsupported freeware with no implied warranties or
  liabilities.  See the C3x DSK disclaimer document for details
 ========================================================================*/
#include "C3MMR.H"
#include <math.h>
/*===================================================================
  Constants for functions and setup
 ====================================================================*/
/*efine  S0gctrl 00E970300h */       /* Sport, noninverted clkx/clkr     */
#define  S0gctrl 0x0E973300          /* Sport,    inverted clkx/clkr     */
#define  S0xctrl 0x00000111          /*                                  */
#define  S0rctrl 0x00000111          /*                                  */
#define  bigval  000010000h          /* Used in overflow mode saturation */

#define  TA    3
#define  TB   24
#define  RA    3
#define  RB   24

#define  SFFTSIZEn 900           /* Sample Window length (FFT size)      */
#define  BIN_START 16            /* Start computing SFFT at this bin     */
#define  BIN_END   17            /* End computing SFFT at this bin       */
#define  ANGLE     90.0          /* Filter reconstruction angle (degrees)*/
/*========================================================================
  PARAMETERS BELOW THIS LINE ARE COMPUTED FROM THE INFORMATION ABOVE.
          THERE IS NO NEED TO MODIFY ANYTHING BELOW THIS POINT
  =======================================================================*/
#define  BIN_LEN   (BIN_END-BIN_START)/* Filter length in bins           */
int      SFFTBINS = BIN_LEN+2;        /* Bins to calculate               */
#define  pi        3.14159265         /* Useful in making apple pie      */
#define  w         2.0*pi/SFFTSIZEn   /* angle = F * 2*pi/Fs             */
#define  K1        0.99995            /* K1 is slightly less than 1.0    */
/*========================================================================
  If the input parameters won't work, generate a descriptive error for
  the user letting them know what to look for and maybe fix
 =========================================================================*/
#if (BIN_LEN < 1)
  APP MESSAGE: Calculated BIN_LEN must be >1
#endif
/*==================================================================*/
void ST_STUB  (void   );
float Input   (void   );
void  Output  (float  );
void  prog_AIC(int    );
void  AIC_INIT(void   );
float diff    (float f);
float SFFT_C  (float f); /*    C version 15 cyc/bin */
float SFFT_A  (float f); /* Assy version  7 cyc/bin */
float siggen  (void   );
/*************************************************/
volatile long A_LOC   = ((TA<<9)+(RA<<2)+0);
volatile long B_LOC   = ((TB<<9)+(RB<<2)+2);
volatile long C_LOC   = 3;
volatile long T_LOC   = 2;
volatile long TRG_LOC = 0;
volatile long SZ_LOC  = SFFTSIZEn; /* max allocated SFFT size     */
volatile long SFFTSIZE= SFFTSIZEn; /* Runtime size can be changed */
/*************************************************/
typedef struct CMPLX
{
  float R;
  float I;
}cmplx;
/*************************************************/
cmplx TW  [BIN_LEN+2];
cmplx BIN [BIN_LEN+2];
float BUF [SFFTSIZEn];
float SFFT_RL;
float SFFT_IM;
float REAL_VEC;
float IMAG_VEC;
float K2     = .95000; /* K1^N */
float Scale  = 2.0/SFFTSIZEn;
float Rsig = 1.0;
float Isig = 0.0;
int   Tbase = (int)&TW;
int   Bbase = (int)&BIN;
float WRsig;
float WIsig;
volatile int xdata    = 0;
volatile int free_run = 0;
volatile int RRDY     = 0;
/*===================================================================
  Initialize runtime constants and arrays
  ===================================================================*/
void init_arrays(void)
{
  int x;
  float n;
  n = BIN_START;
  for(x=0;x<SFFTBINS;x++)
  {
    TW[x].R = K1*cos(n*w);   /* R/I phase or twiddle coefficients */
    TW[x].I = K1*sin(n*w);
    BIN[x].R = 0.0;
    BIN[x].I = 0.0;
    n+=1.0;
  }
  REAL_VEC = -cos(pi*ANGLE/180.0); /* filtered REAL scale factor         */
  IMAG_VEC = -sin(pi*ANGLE/180.0); /* filtered IMAG scale factor         */
/*K2 = pow(K1,SFFTSIZE);*/
  K2 = K1;
  for(x=0;x<SFFTSIZE;x++)     /* Slow but very little code space used */
    K2 *= K1;
  WRsig = cos((BIN_START+1.0)*w);
  WIsig = sin((BIN_START+1.0)*w);
  Scale  = 2.0/SFFTSIZE;
}
/********************************************************/
float fback(float);
/*
asm("_fback                    ; Value is in F0 (float F0)              ");
asm("__qsqrt lsh   1,R0        ; Fast sqrt of numbers ~1.000 calculated ");
asm("        ash   -2,R0       ; by signed shifting the mantissa right  ");
asm("        and   @_qmsk,R0   ; Clear sign bit                         ");
asm("__qinv                    ;  >> EXP bits are all 0 or all 1 <<     ");
asm("        subrf 2.0,R0      ; 1's compliment all bits using floating ");
asm("        rets              ; point identity of value close to 1.000 ");
asm("_qmsk   .word  07FFFFFFFh ;                                        ");
*/
asm("_fback                    ; Value is in F0                 ");
asm("__qsqrt lsh   1,R0        ; Fast sqrt of numbers ~1.000    ");
asm("        ash   -2,R0       ; are calcualted by shifting     ");
asm("        lsh   1,R0        ; the mantissa. The EXP field    ");
asm("        lsh   -1,R0       ; is unchanged                   ");
asm("__qinv                    ;                                ");
asm("        stf   R0,@_qtmp   ; Fast inversion is done by      ");
asm("        ldi   @_qtmp,R0   ; inverting all bits except      ");
asm("        xor   @_qmsk,R0   ; the sign bit.                  ");
asm("        sti   R0,@_qtmp   ;                                ");
asm("        ldf   @_qtmp,R0   ;                                ");
asm("        rets              ;                                ");
asm("_qtmp   .float  0         ;                                ");
asm("_qmsk   .word  0FF7FFFFFh ;                                ");

float siggen(void)
{
  float f;
  f    = Rsig*WRsig - Isig*WIsig;
  Isig = Isig*WRsig + Rsig*WIsig;
  Rsig = f;
  /*                                                               */
  /* Feedback accuracy depends on the method used.  In assembler,  */
  /* a rotating vector magnitude = 1.0+/-smallerr performance is   */
  /* improved using hardware tricks                                */
  /*                                                               */
  /*         f=1 / sqrt(R*R + I*I);                                */
  /*         f=2 - sqrt(R*R + I*I);                                */
  /*         f=2 - (R*R+I*I);                                      */
  /*                                                               */
  f = fback((Rsig*Rsig) + (Isig*Isig));
  Rsig *= f;
  Isig *= f;
  f= Rsig * 32750.0; /* Leave a few bits of headroom for error feedback */
  return f;
}
/********************************************************/
void main(void)
{
   float f;
   int x;



   init_arrays();
   ST_STUB();
   for(;;)
   {
     /* Slow but very little code space used */
     K2 = K1;
     SFFTSIZE = SZ_LOC;
     for(x=0;x<SFFTSIZE;x++) K2 *= K1;
     Scale  = 2.0/SFFTSIZE;

     for(x=0;x<SFFTSIZE;x++)
     {
       BUF[x] = 0;
     }
     for(x=0;x<SFFTBINS;x++)
     {
       BIN[x].R=0;
       BIN[x].I=0;
     }
     WRsig = TW[1].R;
     WIsig = TW[1].I;

     *T0_ctrl = 0;         /* Halt TIM0           */
     *T0_count= 0;         /* Set counts to 0     */
     *T0_prd  = T_LOC;     /* Set period          */
     *T0_ctrl = 0x2C1;     /* Restart timer       */
     free_run = 0;
     AIC_INIT();
     free_run = 1;
     TRG_LOC = 0;
     asm("  ldi   0F4h,IE     ");  /*   Enable XINT/RINT/INT2             */
     asm("  andn  00Fh,IF     ");
     //
     // The first receive interrupt RINT is imminent so take
     // care of it now.  If not, the long time that it takes to
     // do the SFFT could cause an underrun.  In addition, with
     // the AIC being shutdown and reprogrammed, a startup
     // glitch will occur.  To avoid seeing this glitch, the host
     // needs to wait a few samples longer than the actual window
     // length before stopping the SFFT and reading the results.
     //
     f = Input();
     *S0_xdata = xdata;
     asm(" OR 02000h,ST  ");

     for(;;)
     {
       asm(" STI R0,@080A000h ");
       asm(" ;idle            ");  /* Wait for Receive Interrupt     */

       while(RRDY==0);
       RRDY = 0;

       f = Input();                /* Put ADC sample in delay buffer */
       f = diff(f);
       f = SFFT_A(f);
       f = siggen();
       Output(f);                  /* Output result                  */
       asm(" STI R0,@080AFFFh");
       if(TRG_LOC) break;          /* Re-start with new parameters   */
     }                             /* downloaded from host           */
   }
}
/*===================================================================
  The ADC data is read and buffered here
 ====================================================================*/
float Input(void)
{
  int   x;
  float f;
  x = *S0_rdata;     /*get ADC data                         */
  x = x >> 16;       /* Sign extend previous sample in MSBs */
  f = x;             /* Convert the ADC data to float       */
  return f;
}
float diff(float f)
{
  float f2;
  static int bufptr = 0;
  f2 = f - (K2 * BUF[bufptr]);
  BUF[bufptr] = f;
  bufptr++;
  if(bufptr >= SFFTSIZE) bufptr = 0;
  return f2;
}

/*===================================================================
  The output section is written for both Spectrum analyzer output
  as well as REAL/IMAG filter sum outputs
 ====================================================================*/
void Output(float f)
{
  long x;
/* f *=512; */ /* Use this to look at weak signals */
  if(f>  32767) f =  32767;
  if(f< -32768) f = -32768;
  x = f;
  x &= 0xFFFC;
  xdata = x;
/*
  *S0_xdata = x; */

}
/*===================================================================
  The startup stub is used during initialization only and can be
  overwritten by the stack or data after initialization is complete.
  Note: A DSK or RTOS communications kernel may also use the stack.
  In this case be sure to not put the stack here during debug.
 ====================================================================*/
void ST_STUB(void)
{
   *T0_ctrl = 0;         /* Halt TIM0           */
   *T0_count= 0;         /* Set counts to 0     */
   *T0_prd  = T_LOC;     /* Set period          */
   *T0_ctrl = 0x2C1;     /* Restart both timers */
   /* - - - - - - - - - - - - - - - - - - - - - */
   *S0_xctrl = S0xctrl;  /* transmit control    */
   *S0_rctrl = S0rctrl;  /* receive  control    */
   *S0_xdata =       0;  /* DXR data value      */
   *S0_gctrl = S0gctrl;  /* global control      */
  /*AIC_INIT();*/
}
/*================================================
  This function initializes the AIC
 ================================================*/
void AIC_INIT(void)
{
  asm("  andn  034h,IF  ");
  asm("  ldi   004h,IE  ");  /* Enable only INT2 */
  *S0_xdata = 0;
  asm(" rpts  0040h     ");
  asm(" ldi   2,IOF     ");  /* XF0=0 resets AIC */
  asm(" ldi   6,IOF     ");  /* XF0=1 runs AIC   */
  asm(" rpts  040h      ");
  asm(" nop             ");
  asm(" andn  034h,IF   ");
  asm(" ldi   014h,IE   ");  /* Enable only XINT interrupt */
  /*- - - - - - - - - - - -*/
  prog_AIC(C_LOC   );      /* program control register */
  prog_AIC(0xFFFC  );      /* Program the AIC to be real slow */
  prog_AIC(0xFFFC|2);      /* Program the AIC to be real slow */
  prog_AIC(B_LOC   );      /* Bump up the Fs to final rate    */
  prog_AIC(A_LOC   );      /* smaller divisors sent first     */
  asm("  or  080h,ST");    /* Use the overflow mode for fast saturate */
}
/*===================================================================
  prog_AIC is used to transmit new timing configurations to the AIC.
  If you single step this routine, the AIC timing will be corrupted
  causing AIC programming to fail.
  STEP OVER THIS ROUTINE USING THE F10 FUNCTION STEP
 ====================================================================*/
void prog_AIC(int xmit2)
{
  int x;
  *S0_xdata =     0; asm(" idle "); /* Pre transmit a safe value       */
  *S0_xdata =     3; asm(" idle "); /* Request 2ndy xmit               */
  *S0_xdata = xmit2; asm(" idle "); /* Send register porgram value     */
  *S0_xdata =     0; asm(" idle "); /* Leave with a safe value         */
  x = *S0_rdata;                    /* Fix rcvr underrun by dummy read */
}

void c_int05(void)
{
  if(free_run)
    *S0_xdata = xdata;
}
void c_int06(void)
{
  RRDY = 1;
}

/*===================================================================
  Install the XINT/RINT ISR branch vectors
 ====================================================================*/
  asm(" .sect  \"BRTBL\" ; secondary branch table ");
  asm(" b   _c_int05        ; XINT0                  ");
  asm(" b   _c_int06        ; RINT0                  ");
  asm(" .text               ; Back to normal compile ");

/*===================================================================
  Recreate the C31 vector table in the bootrom if using an EVM
 ====================================================================*/
  asm(" .sect  \"VECTS\"   "); /* secondary branch table  */
  asm(" .word  0000045h    ");  /* Reset                  */
  asm(" .word  0809FC1h    ");  /* INT0                   */
  asm(" .word  0809FC2h    ");  /* INT1                   */
  asm(" .word  0809FC3h    ");  /* INT2  (Debug Host int) */
  asm(" .word  0809FC4h    ");  /* INT3                   */
  asm(" .word  0809FC5h    ");  /* XINT0                  */
  asm(" .word  0809FC6h    ");  /* RINT0                  */
  asm(" .word  0809FC7h    ");  /* XINT1 (Debug SSTEP)    */
  asm(" .word  0809FC8h    ");  /* RINT1                  */
  asm(" .word  0809FC9h    ");  /* TINT0                  */
  asm(" .word  0809FCAh    ");  /* TINT1                  */
  asm(" .word  0809FCBh    ");  /* DMA                    */
  asm(" .text               ");
/*===================================================================*/

