//----------------------------------------------------------------------
// DSK_DTMF.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//----------------------------------------------------------------------
// DSKDTMF.ASM is a precision signal generator application which uses
// a table  of complex vectors to create multiple precision sine wave
// oscillations.  The name DSKDTMF is derived from the host control
// program which is designed to generate DTMF tone pairs when the
// numeric keys 0-9 are pressed.  Otherwise this application is a
// precision sine wave signal generator.
//
// NOTE: DSKSG.ASM uses a table lookup scheme with interpolation in both
//       time and level.  DSKSG.ASM and can be used for complex waveforms
//
// Sine Generation
// ---------------
// DSKDTMF.ASM generates each precision sine wave by rotating (a complex
// multiplication) a complex vector pair with a magnitude of 1.0 at
// 2*PI*Freq radians per second.
//
// The vector rotation is produced by a second unity magnitude vector
// pair which defines the rotation rate.  The precision of frequency
// control is therefor determined by the precision of the vector
// accumulation rate which uses 32 bit floating point arithmetic.
// The value of the complex rotation vector rate pair is therefor
// determined by the sampling rate of the DAC and the desired frequency.
//
// Fout = Fdac * Radians/sample
//
// RATE->REAL = cos(Radians/sample)
// RATE->IMAG = sin(Radians/sample)
//
// To prevent the magnitude from decaying or growing over time, the
// magnitude of the vector is controlled by calculating the difference
// of the desired magnitude (1.0) and the actual magnitude (R^2+I^2).
// The difference is then used as a feedback top control the magnitude
// achieving a very high precision.
//
// The output amplitude of each sine wave is then controlled by a scale
// factor which is multiplied into either the real or imaginary component.
//
// NOTE: Both a REAL _and_ IMAGINARY component is available for each
//       sine wave component.  This may be useful for some applications!
//--------------------------------------------------------------
//  File name     Location
//  DSK_DTMF.CPP  DSK3APPS
//  DRIVER.CPP    COMMON
//  DSKKCOFF.CPP  COMMON
//  ERRORMSG.CPP  COMMON
//  OBJECT.CPP    COMMON
//  SYMBOLS.CPP   COMMON
//  TARGET.CPP    COMMON
//  TMSFLOAT..PP  COMMON
//  TEXTWIN.CPP   COMMON
//----------------------------------------------------------
#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <bios.h>
#include "DSK.H"
#include "DSK_COFF.H"
#include "C3XMMRS.H"
#include "keydef.h"
#include "TMSFLOAT.H"
#include "math.h"

#define MSG_BOX   0x809C00L
#define A_BOX     0x809C01L
#define B_BOX     0x809C02L
#define C_BOX     0x809C03L
#define LOAD_FLAG 0x809C04L  // write a non-zero to reload AIC

#define SPARE     0x809C05L
#define NOISELVL  0x809C06L
#define NOISESCL  0x809C07L
#define OFF_LVL   0x809C08L
#define MUTE      0x809C09L

#define V_ENBL    0x809C0AL  // Vector enable
#define R_ACC     0x809C0BL  // Phase accumulation rate R/I vector pair
#define I_ACC     0x809C0CL  //
#define R_DATA    0x809C0DL  // Present data R/I data vector pair
#define I_DATA    0x809C0EL  //
#define A_LVL     0x809C0FL
#define FRQNCY    0x809C10L
#define PI        3.141592654
#define last_cell        16  // up to 12 terms can fit the screen
// Pick a DSP cycle time H1/H3 clock rate
//
//float  THx=50e-9;         // 40 Mhz
float  THx=40e-9;           // 50 Mhz
//float  THx=33e-9;         // 60 Mhz
//float THx = 25e-9;        // 80 Mhz
//float THx = 20e-9;        // 100 Mhz

#define A_REG  ((TA<<9)+(RA<<2)+0) // A divisors.. set SCF rate
#define B_REG  ((TB<<9)+(RB<<2)+2) // B divisors

typedef struct
{
  float V_enbl    ; // Non-zero enables this vector
  float R_acc     ; // R/I rotation rate vector pair... Ra^2+Ia^2 = 1.0
  float I_acc     ;
  float R_data    ; // R/I present data vector pair...  Rd^2+Id^2 = 1.0
  float I_data    ;
  float Ampli     ; // Amplitude level multiplier
  float Freq      ; // NOTE: This element is NOT replicated on DSK
}Harmonic_def;

Harmonic_def H[last_cell+2];
char  DSK_APP    []="DSKDTMF.DSK";
char  DSK_EXE_APP[]="DSK_DTMF.EXE";
int   TA        =     8; // DAC divisors
int   TB        =    20; //
int   RA        =    10; // ADC divisors
int   RB        =    14; //
int   C_REG     =  0x03; // AIC control register bits
int   Cell      =     0; // Cell component being edited
int   DTMF      =     0;
float DAC_scale =     1; // scale factor takes non-linearities into account
float LoF       =     1;
float HiF       =     1;
float mute      =   1.0;
float V_offset  =   0.0;
float V_rmsrand =   0.0;
float Fsr       =1000.0;
float Fsx       =1000.0;
ulong T0_prdv   =     2;
ulong ZERO =0x00000000L; // 32 bit long zero
ulong FZERO=0x80000000L; // Floating point zero

void binsprintf       (char *s,int val);
void update_list      (void);
int  check_key        (void);
MSGS update_DSK_params(void);

MSGS  putmemf(ulong DEST,float FVAL)
{
  ulong temp;
  temp = IEEE_TMS(FVAL);
  return putmem(DEST,1,&temp);
}

void main(void)
{
  int reset_flag = 0;
  MSGS err;
  clrscr();
  Scan_Command_line(DSK_EXE_APP);
  clrscr();
  //***********************************************//
  // The outer loop initializes the DSK on entry   //
  // or if the application fails                   //
  //***********************************************//
  for(;;)
  {
    for(;;)
    { if(Init_Communication(10000) == NO_ERR) break;
      if(kbhit()) exit(0);
    }
    HALT_CPU(); // Halt previously running apps code
    if((err=Load_File(DSK_APP,LOAD))!=NO_ERR)
    { printf("%s %s\n",DSK_APP,Error_Strg(err));
      exit(0);
    }
    RUN_CPU();
    clrscr();
    check_key();
    err = update_DSK_params();
    //***********************************************************//
    // The inner loop is repeated until a keyboard hit exits the //
    // application or the application reports an error and needs //
    // to be reinitialized.                                      //
    //***********************************************************//
    for(;err==NO_ERR;)
    { for(;;)
      { // If there is a keystroke waiting or if the last command turned
        // on one of the DTMF tone pairs, then go back to check_key.  If
        // DTMF is set check_key then waits for any key press to clear
        // and then shuts off the tone.
        //
        //  NOTE: Because of the keyboards typematic rate and lack of
        //        keyboard stack control (unless written in ugly assembly
        //        code) the keyup/keydown control is not very good.  So far
        //        this is the best I could come up with using the C language.
        //
        if(kbhit() || DTMF)
        {
          reset_flag = check_key();
          if((err=update_DSK_params())!=NO_ERR) break;
          break;
        }
        delay(1);
      }
      if(reset_flag || (err!=NO_ERR)) break;
    }
    printf("Communications are being reinitialized");
    DSK_reset();
  }
}

int inline clipint(int x, int min, int max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}

long inline cliplong(long x, long min, long max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}

void update_list(void)
{
  int x;
  //
  // modify each harmonic Frequency, Amplitude, start phase
  //
  #define ampli H[x].Ampli
  #define freq  H[x].Freq
  #define enbl  H[x].V_enbl
  Fsx = 1/(2*TA*TB*(2*THx * T0_prdv));
  if(C_REG & 0x20) Fsr = Fsx;
  else
  Fsr = 1/(2*RA*RB*(2*THx * T0_prdv));
  x = 0;
  if(DTMF)
  {
    x=0; freq=LoF; ampli=1; enbl = 1;
    x=1; freq=HiF; ampli=1; enbl = 1;
    x=2;
  }
  for(x=0;x<last_cell;x++)
  {
    if(ampli > 100) ampli = 100; if(ampli < -100) ampli = -100;
    if(freq  > 1e9) freq  = 1e9; if(freq  < -1e9) freq  = -1e9;
    H[x].R_acc = cos(2*PI*freq / Fsx);  // phase accumulation rate
    H[x].I_acc = sin(2*PI*freq / Fsx);
    H[x].R_data = 1.0;                  // phase start angle
    H[x].I_data = 0.0;
    H[x].Ampli  = ampli;                // signal amplitude
  }
  if(V_offset  >  100) V_offset =  100;
  if(V_offset  < -100) V_offset = -100;
}
void get_float(float *f, char *s)
{
  float ftemp;
  char strng[40];
  printf("%s=%7.8f ",s,*f);
  scanf(" %20s",&strng);
  ftemp = atof(strng);
  *f = ftemp;
}
int check_key(void)
{
  char strng[64];
  int key=0, x,y,xc,yc,n;
  int shift_key = 0;
  //
  // If the last keystroke caused a DTMF signal to be generated
  // wait for the key to be released before moving on.  Note
  // that the keyboards typematic rate is a potential problem here
  // and that the minimum DTMF period is 100mS
  //
  if(DTMF)
  {
    delay(100);                      // 100 mS minimum
    H[0].Ampli = 0;
    H[1].Ampli = 0;
  }
  DTMF=0;
  if(kbhit())
  {
    if(bioskey(2) & 3)  shift_key = 1;
    key = bioskey(0) & 0xFF00;
  }
  n = Cell-2; // Array offset is cursor pos - 2
  if(key)
  {
    switch(key)
    {
      //============== Quit (with run optional), Reset ======//
      case _R  : return 1;  // Return reset (break) flag
      case _Q  : clrscr();
                 _setcursortype(_NORMALCURSOR);
                 printf("Quit [Y,N]?");
                 if(toupper(getch())!='Y') break;
                 printf("\n\nKill signal generator on exit [Y,N]?");
                 if(toupper(getch())=='Y') DSK_reset();
                 exit(0);
                 break;
      case _M :  if(mute!=0.0) mute=0.0; else mute=1.0; break;
      //============== Timer period setup ===================//
      case _F9 : T0_prdv -=1; break;
      case _F10: T0_prdv +=1; break;
      //============== Modify DAC paramerters ===============//
      case _F1 : TA++; break;
      case _F2 : TA--; break;
      case _F3 : TB++; break;
      case _F4 : TB--; break;
      //============== Modify ADC paramerters ===============//
      case _SF1: RA++; break;
      case _SF2: RA--; break;
      case _SF3: RB++; break;
      case _SF4: RB--; break;
      //========= Select term in equation to modify =========//
      case _Dn : if(Cell >=last_cell-1) break;
                 Cell++;          // Enable next cell
                 n++;
                 H[n].V_enbl = 1.0;
                 break;
      case _Up : Cell--;
                 if(Cell< 0) Cell = 0; break;
      //=========== Amplitude level adjustment ===============//
      case _F7 : H[n].Ampli = (H[n].Ampli * 1.1) + .0001; break;
      case _F8 : H[n].Ampli = (H[n].Ampli / 1.1) - .0001; break;
      case _A:   switch(Cell)
                 { case 0:  get_float(&V_offset  ,"Voff"); break;
                   case 1:  get_float(&V_rmsrand ,"Vrand"); break;
                   default: get_float(&H[n].Ampli,"A"); break;
                 }
                 break;
      //============== Term delete (no frequency)============//
      case _Del: if(n<0) break;
                 Cell = n+2;
                 H[n].V_enbl = 0.0;
                 n--;
                 Cell--;
                 break;
      //============== Frequency adjustment ==================//
      case _F5 : if(n<0) break; H[n].Freq=(H[n].Freq*1.1)+.0001; break;
      case _F6 : if(n<0) break; H[n].Freq=(H[n].Freq/1.1)-.0001; break;
      case _F  : if(n<0) break; get_float(&H[n].Freq,"F");       break;
      //============== AIC control register bit set ==========//
      case _1: if(shift_key) C_REG = C_REG ^ 0x80;        // bit 7
               else         {LoF=697.0; HiF=1209.0; DTMF=1;} break;
      case _2: if(shift_key) C_REG = C_REG ^ 0x40;        // bit 6
               else         {LoF=697.0; HiF=1336.0; DTMF=1;} break;
      case _3: if(shift_key) C_REG = C_REG ^ 0x20;        // bit 5
               else         {LoF=697.0; HiF=1477.0; DTMF=1;} break;
      case _4: if(shift_key) C_REG = C_REG ^ 0x10;        // bit 4
               else         {LoF=770.0; HiF=1209.0; DTMF=1;} break;
      case _5: if(shift_key) C_REG = C_REG ^ 0x08;        // bit 3
               else         {LoF=770.0; HiF=1336.0; DTMF=1;} break;
      case _6: if(shift_key) C_REG = C_REG ^ 0x04;        // bit 2
               else         {LoF=770.0; HiF=1477.0; DTMF=1;} break;
  //  case _1  : LoF=697.0; HiF=1209.0; DTMF=1; break;
  //  case _2  : LoF=697.0; HiF=1336.0; DTMF=1; break;
  //  case _3  : LoF=697.0; HiF=1477.0; DTMF=1; break;
  //  case _4  : LoF=770.0; HiF=1209.0; DTMF=1; break;
  //  case _5  : LoF=770.0; HiF=1336.0; DTMF=1; break;
  //  case _6  : LoF=770.0; HiF=1477.0; DTMF=1; break;
      case _7  : LoF=852.0; HiF=1209.0; DTMF=1; break;
      case _8  : LoF=852.0; HiF=1336.0; DTMF=1; break;
      case _9  : LoF=852.0; HiF=1477.0; DTMF=1; break;
      case _0  : LoF=941.0; HiF=1336.0; DTMF=1; break;
  //  case _prd: LoF=941.0; HiF=1209.0; DTMF=1; break;
  //  case _pnd: LoF=941.0; HiF=1477.0; DTMF=1; break;
  //  case _A  : LoF=697.0; HiF=1633.0; DTMF=1; break;
  //  case _B  : LoF=770.0; HiF=1633.0; DTMF=1; break;
  //  case _C  : LoF=852.0; HiF=1633.0; DTMF=1; break;
  //  case _D  : LoF=941.0; HiF=1633.0; DTMF=1; break;

      //============== Default - do nothing  =================//
      default : return 0;
    }
    TA      = clipint (TA     ,  3, 31); // Clip the AIC timer registers
    TB      = clipint (TB     , 12, 63); // to values that typicly work
    RA      = clipint (RA     ,  3, 31);
    RB      = clipint (RB     , 12, 63);
    T0_prdv = cliplong(T0_prdv,  1, 64);
  }
  clrscr();
  update_list();

  if(update_DSK_params()!=NO_ERR) return 1; // reload parameters
  //
  binsprintf(strng,C_REG);
  for(y=2;y<9;y++)
  {
    gotoxy(72,y);
    printf("");
  }
  gotoxy(1,1);
  if(mute!=0.0)
  printf(
"ͻ\n"
"  F1:F2   TA  = %02d  (xmit=dac)      Fdac=%7.2f\n"
"  F3:F4   TB  = %02d                  Fadc=%7.2f\n"
"  SF1:SF2 RA  = %02d  (recv=adc)      shift [1-6] set AIC ctrl\n"
"  SF3:SF4 RB  = %02d                  keys  [1-6] set DTMF Hi/Lo\n"
"  F9:F10  TIM0= %02ld  (timer 0 )      %sb AIC ctrl word\n"
"͹\n"
" (Q)uit (R)eset (M)ute         [F5:F6](F)requency  [F7:F8](A)mplitude\n"
"ͼ\n",
  TA,Fsx,TB,Fsr,RA,RB,T0_prdv,strng);
  else
  printf(
"ͻ\n"
"  F1:F2   TA  = %02d  (xmit=dac)      Fdac=%7.2f\n"
"  F3:F3   TB  = %02d                  Fadc=%7.2f\n"
"  SF1:SF2 RA  = %02d  (recv=adc)      shift [1-6] set AIC ctrl\n"
"  SF3:SF4 RB  = %02d                  keys  [1-6] set DTMF Hi/Lo\n"
"  F9:F10  TIM0= %02ld  (timer 0 )      %sb AIC ctrl word\n"
"͹\n"
" (Q)uit (R)eset un(M)ute       [F5:F6](F)requency  [F7:F8](A)mplitude\n"
"ͼ\n",
    TA,Fsx,TB,Fsr,RA,RB,T0_prdv,strng);
  //TA,TB ,strng,RA,RB ,T0_prdv,Fsx          );

  for(x=0;x<last_cell;x++)
  {
    n = x-2;
    if(n > 8)
      if(n<1) break;
    if(x==Cell) printf(">>");
    else         printf("  ");
    switch(x)
    {
      case  0: printf("   %7.6f Vdc offset    ",V_offset); break;
      case  1: printf(" + %7.6f Vac random    ",V_rmsrand); break;
      default: printf(" + %7.6f * sin(%7.6f)",  H[n].Ampli,H[n].Freq);
               break;
    }
    gotoxy(38,wherey());
    if(x==Cell)
    {
      printf("Edit> ");        // remember x,y local for
      xc=wherex();yc=wherey(); // for cursor placement
    }
    if(n>=0)
      if(H[n+1].V_enbl ==0.0) break;
    printf("\n");
  }
  gotoxy(xc,yc);
  return 0;
}

MSGS update_DSK_params(void)
{
  MSGS err = NO_ERR;
  ulong TEMP;
  ulong DSPdata[20];
  long Sz;
  int x;
  ulong LONGPRD = 64;
  delay(1);
  if(T0_prdv > 1)  DAC_scale = 3.08;
  else
  { // When the AIC clock reference is run past the designed 10 MHz
    // limit the loweer diviser values produce internal timings which
    // cause non-linearities in both the DAC and ADC.  The following
    // table partially compensates for these non-linearities
    switch(TA)
    {
      case  3: DAC_scale = 2.50; break; case  4: DAC_scale = 2.05; break;
      case  5: DAC_scale = 2.70; break; case  6: DAC_scale = 3.18; break;
      case  7: DAC_scale = 3.18; break; case  8: DAC_scale = 3.10; break;
      default: DAC_scale = 3.08; break;
    }
  }
  // When stopping the AIC for a long time put zero in the DXR and
  // slow down the timers for a safe re-start.  This effectively
  // allows the DSK to continue servicing the serial port even when
  // the host hangs the interface for long periods of time.
//  TEMP = 0;
//  if((err=putmem(LOAD_FLAG,1,&TEMP))!=NO_ERR) return err;
  if((err=putmem(S0_xdata,1,    &ZERO))!=NO_ERR) return err;
  if((err=putmem(T0_prd  ,1, &LONGPRD))!=NO_ERR) return err;
  if((err=putmem(T0_count,1,    &ZERO))!=NO_ERR) return err;
  if((err=putmem(T1_prd  ,1, &LONGPRD))!=NO_ERR) return err;
  if((err=putmem(T1_count,1,    &ZERO))!=NO_ERR) return err;
  //
  // The AIC can be programmed to achieve sample rates >20,000Hz, usualy
  // with some degradation in performance.  To do this, the AIC is reset
  // and then programmed to run at a low rate.  By then loading the slower
  // of the two A or B regs first (usualy B since B is limited to >13), the
  // sampling rate can be safely 'bumped up' without squeezing out the next
  // secondary transfer.  Normal and 2ndy request timings are shown below.
  //
  //      |<-----------------  1/Fs ------------------>|
  // Fsx --________________-----------------------------________________----
  // Dx  --xxxxxxxxxxxxxx00-----------------------------xxxxxxxxxxxxxx00----
  //       14b DAC data  |                              14b DAC data
  //                     +--- No 2ndy request
  //
  //
  //      |<------------------  1/Fs ----------------->|<----- New Timing---
  //                       |4 SCLKs |
  // Fsx --________________----------________________---________________----
  // Dx  --xxxxxxxxxxxxxx11----------xxxxxxxxxxxxxxRR---xxxxxxxxxxxxxx00----
  //       14b DAC data  |           2ndy value    |    14b DAC data
  //                     |                         |
  //                     +--- request for 2ndy     +--- register to load
  //
  //
  // NOTE: Higher sampling rates can result in eratic AIC behaviour that
  //       cannot be predicted from one device to another, over temperature
  //       etc... Use these setups for evaluation only.
  //
  if(TB<TA)
  { TEMP = A_REG; if((err=putmem(A_BOX,1,&TEMP))!=NO_ERR) return err;
    TEMP = B_REG; if((err=putmem(B_BOX,1,&TEMP))!=NO_ERR) return err;
  }
  else
  { TEMP = B_REG; if((err=putmem(A_BOX,1,&TEMP))!=NO_ERR) return err;
    TEMP = A_REG; if((err=putmem(B_BOX,1,&TEMP))!=NO_ERR) return err;
  }
  TEMP = C_REG; if((err=putmem(C_BOX,1,&TEMP))!=NO_ERR) return err;
  //
  // Update the DC offset and noise values.  These values are kept in
  // the hosts harmonic list in positions 0,1 so they can be easily edited.
  // Their location in the DSK application is however not in the hamrmonic
  // list.  In other words, look at the putmemf(dest,fvalue) destination,
  // it is not the same as those used for updating the harmonic list.
  //
  if((err=putmemf(OFF_LVL ,V_offset *30500.0/DAC_scale))!=NO_ERR) return err;
  if((err=putmemf(NOISELVL,V_rmsrand ))!=NO_ERR) return err;//DC Offset
  if((err=putmemf(MUTE    ,mute      ))!=NO_ERR) return err;//DC Offset
  if((err=putmem(S0_xdata,1,&ZERO    ))!=NO_ERR) return err;
  Sz = sizeof(H[0])/4;  // elements in structure
  for(x=0;x<last_cell;x++)
  {
    // On each loop reload the DXR with zero
//  if((err=putmem(S0_xdata,1,    &ZERO))!=NO_ERR) return err;
    //
    // Load the harmonic list into the DSK.  Each structure tells the
    // DSK application the start frequency and amplitude and if the
    // particular harmonic is enabled.  Note that there are two complex
    // R/I pair vectors to define each harmonics rotation rate (frequency)
    // and current vector position.  The other two elements tell the
    // application wether or not the harmonic is to be enabled and the
    // amplitude at which to sum it into the total signal.
    //
    DSPdata[0] = IEEE_TMS(H[x].V_enbl);
    DSPdata[1] = IEEE_TMS(H[x].R_acc );
    DSPdata[2] = IEEE_TMS(H[x].I_acc );
    DSPdata[3] = IEEE_TMS(H[x].R_data);
    DSPdata[4] = IEEE_TMS(H[x].I_data);
    DSPdata[5] = IEEE_TMS(H[x].Ampli/DAC_scale);
    DSPdata[6] = IEEE_TMS(H[x].Freq);
    putmem(V_ENBL+x*Sz,7,DSPdata);
    /*
    if((err=putmemf(V_ENBL+x*Sz,H[x].V_enbl         ))!=NO_ERR) return err;
    if((err=putmemf(R_ACC +x*Sz,H[x].R_acc          ))!=NO_ERR) return err;
    if((err=putmemf(I_ACC +x*Sz,H[x].I_acc          ))!=NO_ERR) return err;
    if((err=putmemf(R_DATA+x*Sz,H[x].R_data         ))!=NO_ERR) return err;
    if((err=putmemf(I_DATA+x*Sz,H[x].I_data         ))!=NO_ERR) return err;
    if((err=putmemf(A_LVL +x*Sz,H[x].Ampli/DAC_scale))!=NO_ERR) return err;
    if((err=putmemf(FRQNCY+x*Sz,H[x].Freq           ))!=NO_ERR) return err;
    */
    if(H[x].V_enbl==0.0) break;
  }
  if((err=putmem(T0_prd  ,1, &T0_prdv))!=NO_ERR) return err;
  if((err=putmem(T0_count,1,    &ZERO))!=NO_ERR) return err;
  if((err=putmem(T1_prd  ,1, &T0_prdv))!=NO_ERR) return err;
  if((err=putmem(T1_count,1,    &ZERO))!=NO_ERR) return err;
  //
  // A non-zero LOAD_FLAG signals DSK to restart the
  // application with the new runtime parameters
  //
  if((err=putmem(LOAD_FLAG,1,&TEMP))!=NO_ERR) return err;
  return err;
}

void binsprintf(char *s,int val)
{
  char *p;
  unsigned int t;
  p = s;
  t = 0x80;  // scan 8 bits
  for(;t>0;t=t>>1)
  {
    if(val & t) *p++ = '1';
    else        *p++ = '0';
  }
  *p=0;
}
