//------------------------------------------------------------
// DSK_SG.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//---------------------------------------------------------------
// DSKSG.ASM is a precision signal generator application which uses
// a wave table lookup to create multiple precision signals.  The
// table lookup scheme uses both interpolation in time and level to
// achieve precision in frequency and in level
//
// NOTE: Please see DSKDTMF.ASM for generating ultra-precise 'pure'
//       sine waves.  This application will even produce both the
//       sine and cosine (complex conjugate) at the same time.
//
// Sine Generation
// ---------------
// DSKSG.ASM generates a sine wave when the SINE_SG.ASM file has been
// assembled and loaded into the signal lookup table.
//
// The output waveform is produced by assuming that the table length
// of N=128 samples is 2*PI radians.  The desired frequency (radian
// accumulation rate) is then used to determine both the whole step
// step values and also a residual step which is used in a first order
// interpolation (slope and intercept).
//
// NOTE: This technique produces 32 bit floating point frequency accuracy
//       with excellent amplitude precision for signals that fit the
//       interpolation and sampling criteria.  For a sine wave, linear
//       interpolation will result in nearly 2x the bits of precision
//       used in the lookup, or about 14 bits in a 7 bit (128 element) table.
//
// The output amplitude of each signal wave is then controlled by a scale
// factor.
//--------------------------------------------------------------
//  File name     Location
//  DSK_SG.CPP    DSK3APPS
//  DRIVER.CPP    COMMON
//  DSK_COFF.CPP  COMMON
//  ERRORMSG.CPP  COMMON
//  OBJECT.CPP    COMMON
//  SYMBOLS.CPP   COMMON
//  TARGET.CPP    COMMON
//  TMSFLOAT.CPP  COMMON
//----------------------------------------------
#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <bios.h>
#include "DSK.H"
#include "DSK_COFF.H"
#include "C3XMMRS.H"
#include "keydef.h"
#include "TMSFLOAT.H"
#include "math.h"


#define TBLSIZE 128.0
#define PI 3.141592654

#define MSG_BOX   0x809C00L
#define A_BOX     0x809C01L
#define B_BOX     0x809C02L
#define C_BOX     0x809C03L
#define LOAD_FLAG 0x809C04L  // write a non-zero to reload AIC

#define SPARE     0x809C05L
#define NOISELVL  0x809C06L
#define NOISESCL  0x809C07L
#define OFF_LVL   0x809C08L

#define PHASEx    0x809C09L  // Phase accumulation rate
#define OFFSx     0x809C0AL  // Current phase offset into lookup table
#define A_LVLx    0x809C0BL  // Amplitude of this term
#define last_cell 16     // up to 16 terms are allowed
#define THx 40e-9            // DSP cycle time H1/H3 clock rate
#define A_REG  ((TA <<9)+(RA <<2)+0) // A divisors.. set SCF rate
#define B_REG  ((TB <<9)+(RB <<2)+2) // B divisors
//----------------------------------------
typedef struct
{
  float frequency ;
  float phase_accu;
  float phase_strt;
  float amplitude ;
}Harmonic_def;
//----------------------------------------
Harmonic_def H[last_cell];
ulong T0_prdv =  2;
ulong ZERO =0x00000000L; // 32 bit long zero
ulong FZERO=0x80000000L; // Floating point zero
float DAC_scale;         // fudge factor for gain/non-linearities
float Fsr=1000.0;
float Fsx=1000.0;
int   hm =0;              // current harmonic component
int   TA   =   8;         // DAC divisors
int   TB   =  20;         //
int   RA   =  10;         // ADC divisors
int   RB   =  14;         //
int   C_REG=0x03;         // control
//----------------------------------------
int  check_key(void);
MSGS update_application(void);
void binsprintf(char *s,int val);
char DSK_APP[]    ="DSKSG.DSK";
char DSK_EXE_APP[]="DSK_SG.EXE";
//----------------------------------------
MSGS  putmemf(ulong DEST,float FVAL)
{
  ulong temp;
  temp = IEEE_TMS(FVAL);
  return putmem(DEST,1,&temp);
}
//----------------------------------------
void main(void)
{
  int New_Params = 0, reset_flag = 0;
  MSGS err;
  clrscr();
  Scan_Command_line(DSK_EXE_APP);
  clrscr();
  //***********************************************//
  // The outer loop initializes the DSK on entry   //
  // or if the application fails                   //
  //***********************************************//
  for(;;)
  { for(;;)
    { if(Init_Communication(10000) == NO_ERR) break;
      if(kbhit()) exit(0);
    }
    HALT_CPU(); // Halt previously running apps code
    if((err=Load_File(DSK_APP,LOAD))!=NO_ERR)
    { printf("%s %s\n",DSK_APP,Error_Strg(err));
      exit(0);
    }
    RUN_CPU();
    if((err=Load_File("SINE_SG.DSK",LOAD))!=NO_ERR)
    { printf("%s %s\n","SINE_SG.DSK",Error_Strg(err));
      exit(0);
    }
    check_key();  // Check key will update parameters even if no keypress
    clrscr();
    if((err=update_application())!=NO_ERR) break;
    //***********************************************************//
    // The inner loop is repeated until a keyboard hit exits the //
    // application or the application reports an error and needs //
    // to be reinitialized                                       //
    //***********************************************************//
    check_key();   // by putting a dummy character in the buffer
    for(;;)
    {
      reset_flag = 0;
      for(;;)                 // for DSK to stop with full buffer
      {
        if(kbhit())
        {
          reset_flag = check_key();
          New_Params=1;
          break;
        }
        delay(1);
      }
      if(reset_flag) break;
      //
      // If a key was pressed, update the AIC setup
      if(New_Params)
      {
       if((err=update_application())!=NO_ERR) break;
      }
      New_Params = 0;
    }
    printf("Communications are being reinitialized");
    DSK_reset();
  }
}

int inline clipint(int x, int min, int max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}

long inline cliplong(long x, long min, long max)
{
  if(x<min) return min;
  if(x>max) return max;
  return x;
}


int check_key(void)
{
  char strng[64];
  int key=0, x,y,xc,yc;
  float ftemp;

  if(kbhit()) key = bioskey(0) & 0xFF00;
  H[0].frequency = 1;  // DC offset and noise terms are never turned off
  H[1].frequency = 1;  // Set H[x].frequency structure element to non zero
  if(key)
  {
    switch(key)
    {
      //============== Quit (with run optional), Reset ======//
      case _R  : return 1;  // Return reset (break) flag
      case _Q  : clrscr();
                 _setcursortype(_NORMALCURSOR);
                 printf("Kill signal generator on exit [Y,N]?");
                 if(toupper(getch())=='Y') DSK_reset();
                 exit(0);
                 break;
      //============== Timer period setup ===================//
      case _F9 : T0_prdv -=1; break;
      case _F10: T0_prdv +=1; break;
      //============== Modify DAC paramerters ===============//
      case _F1 : TA++; break;
      case _F2 : TA--; break;
      case _F3 : TB++; break;
      case _F4 : TB--; break;
      //============== Modify ADC paramerters ===============//
      case _SF1: RA++; break;
      case _SF2: RA--; break;
      case _SF3: RB++; break;
      case _SF4: RB--; break;
      //========= Select term in equation to modify =========//
      case _Dn : hm++; if(hm>last_cell-1) hm = last_cell-1;
                       if(H[hm-1].frequency==0.0) hm--;
                       break;
      case _Up : hm--; if(hm<              0) hm =               0; break;
      //=========== Amplitude level adjustment ===============//
      case _F5 : H[hm].amplitude = (H[hm].amplitude * 1.1) + .0001; break;
      case _F6 : H[hm].amplitude = (H[hm].amplitude / 1.1) - .0001; break;
      case _A:   printf(" A=%7.8f ", H[hm].amplitude);
                 scanf(" %20s",&strng);
                 ftemp = atof(strng);
              // ftemp = fabs(ftemp);
                 H[hm].amplitude = ftemp;
                 break;
      //============== Term delete (no frequency)============//
      case _Del: H[hm].frequency = 0.0; break;
      //============== Frequency adjustment ==================//
      case _F7 : H[hm].frequency = (H[hm].frequency * 1.1) + .0001; break;
      case _F8 : H[hm].frequency = (H[hm].frequency / 1.1) - .0001; break;
      case _F:   printf(" F=%7.8f ", H[hm].frequency);
                 scanf(" %20s",&strng);
                 ftemp = atof(strng);
             //  ftemp = fabs(ftemp);
                 H[hm].frequency = ftemp;
                 break;
      //============== AIC control register bit set ==========//
      case _1  : C_REG = C_REG ^ 0x80; break; // bit 7
      case _2  : C_REG = C_REG ^ 0x40; break; // bit 6
      case _3  : C_REG = C_REG ^ 0x20; break; // bit 5
      case _4  : C_REG = C_REG ^ 0x10; break; // bit 4
      case _5  : C_REG = C_REG ^ 0x08; break; // bit 3
      case _6  : C_REG = C_REG ^ 0x04; break; // bit 2
      //============== Select waverform file =================//
      case _7  : if(Load_File("SINE_SG.DSK",LOAD)!=NO_ERR) return 1; break;
      case _8  : if(Load_File("RAMP_SG.DSK",LOAD)!=NO_ERR) return 1; break;
      case _9  : if(Load_File("SAWT_SG.DSK",LOAD)!=NO_ERR) return 1; break;
      case _0  : if(Load_File("RAND_SG.DSK",LOAD)!=NO_ERR) return 1; break;

      default : return 0;
    }
    TA      = clipint (TA     ,  3, 31); // Clip the AIC timer registers
    TB      = clipint (TB     , 12, 63); // to values that typicly work
    RA      = clipint (RA     ,  3, 31);
    RB      = clipint (RB     , 12, 63);
    T0_prdv = cliplong(T0_prdv,  1, 64);
  }
  clrscr();
  //
  // modify each harmonic Frequency, Amplitude, start phase
  //
  #define ampli H[x].amplitude
  #define freq  H[x].frequency
  Fsx     = 1/(2*TA*TB*(2*THx * T0_prdv));
  for(x=2;x<last_cell;x++)
  {
    if(ampli > 100) ampli = 100; if(ampli < -100) ampli = -100;
    if(freq  > 1e9) freq  = 1e9; if(freq  < -1e9) freq  = -1e9;
    H[x].frequency  = freq;
    H[x].phase_accu = freq * TBLSIZE / Fsx; // phase accumulation rate
    H[x].phase_strt = 0.0;                  // phase start angle
    H[x].amplitude  = ampli;                // signal amplitude
  }

  if(H[0].amplitude  >  100) H[0].amplitude  =  100;
  if(H[0].amplitude  < -100) H[0].amplitude  = -100;
  if(T0_prdv > 1)
    DAC_scale = 3.08;
  else
  { //
    // When the AIC clock reference is run past the designed 10 MHz
    // limit the lowest diviser values produce internal timings which
    // cause non-linearities in both the DAC and ADC.  However, in some
    // cases these configurations can be helpful in that they can provide
    // >20 Khz bandpass at the DAC output.  The following table partially
    // compensates for these non-linearities
    //
    // NOTE: The DAC non-linearities which are produced appear to cause a
    //       moderate amount of logarithmic compression.  To view this
    //       compression, set DSK_SG to produce a low frequency sawtooth
    //       or triangle wave and then use an oscilliscope with good
    //       vertical and horizontal linearity.
    //
    switch(TA)
    {
      case  3: DAC_scale = 2.50; break;
      case  4: DAC_scale = 2.05; break;
      case  5: DAC_scale = 2.70; break;
      case  6: DAC_scale = 3.18; break;
      case  7: DAC_scale = 3.18; break;
      case  8: DAC_scale = 3.10; break;
      default: DAC_scale = 3.08; break;
    }
  }
  //
  binsprintf(strng,C_REG);
  for(y=2;y<9;y++)
  {
    gotoxy(72,y);
    printf("");
  }
  gotoxy(1,1);
  printf(
"ͻ\n"
"  F1:F2   TA  = %02d  (xmit=dac)       [1-6] set AIC ctrl bits\n"
"  F3:F3   TB  = %02d                   %s\n"
"  SF1:SF2 RA  = %02d  (recv=adc)      7,8,9,0 Keys change waveform\n"
"  SF3:SF4 RB  = %02d                  Sine, Ramp, Triangle, Rand\n"
"  F9:F10  TIM0= %02ld  (timer 0 )       Fdac=%7.2f\n"
"͹\n"
" (Q)uit (R)eset               [F5:F6](F)requency   [F7:F8](A)mplitude\n"
"ͼ\n",
TA,TB,strng,RA,RB,T0_prdv,Fsx);

  for(x=0;x<last_cell;x++)
  {
    if(x==hm) printf(">>");
    else      printf("  ");
    switch(x)
    {
      case  0: printf("   %7.6f Vdc offset    ",H[0].amplitude); break;
      case  1: printf(" + %7.6f Vac random    ",H[1].amplitude); break;
      default: printf(" + %7.6f * sin(%7.6f)",
                        H[x].amplitude,H[x].frequency);
               break;
    }
    gotoxy(38,wherey());
    if(x==hm)
    {
      printf("Edit>");    // remember x,y local for
      xc=wherex();yc=wherey(); // for cursor placement
    }

    if(H[x].frequency ==0.0) break;
    printf("\n");
  }
  gotoxy(xc,yc);
  return 0;
}

MSGS update_application(void)
{
  MSGS err = NO_ERR;
  ulong TEMP;
  int x, XH;
  delay(50);
  if((err=putmem(T0_prd  ,1, &T0_prdv))!=NO_ERR) return err;
  if((err=putmem(T0_count,1,    &ZERO))!=NO_ERR) return err;
  if((err=putmem(T1_prd  ,1, &T0_prdv))!=NO_ERR) return err;
  if((err=putmem(T1_count,1,    &ZERO))!=NO_ERR) return err;
  if(TB<TA)
  { TEMP = A_REG; if((err=putmem(A_BOX,1,&TEMP))!=NO_ERR) return err;
    TEMP = B_REG; if((err=putmem(B_BOX,1,&TEMP))!=NO_ERR) return err;
  }
  else
  { TEMP = B_REG; if((err=putmem(A_BOX,1,&TEMP))!=NO_ERR) return err;
    TEMP = A_REG; if((err=putmem(B_BOX,1,&TEMP))!=NO_ERR) return err;
  }
  TEMP = C_REG; if((err=putmem(C_BOX,1,&TEMP))!=NO_ERR) return err;

  //
  // Update frequency component values
  if((err=putmemf(OFF_LVL ,H[0].amplitude *30500.0/DAC_scale))!=NO_ERR) return err;
  if((err=putmemf(NOISELVL,H[1].amplitude))!=NO_ERR) return err;//DC Offset
  for(x=2;x<last_cell;x++)
  {
   // Load the Frequency, Amplitude, start phase into the DSK
   XH = x - 2;
   if((err=putmemf(PHASEx+XH*3,H[x].phase_accu         ))!=NO_ERR) return err;
   if((err=putmemf(OFFSx +XH*3,H[x].phase_strt         ))!=NO_ERR) return err;
   if((err=putmemf(A_LVLx+XH*3,H[x].amplitude/DAC_scale))!=NO_ERR) return err;
  }
  // non-zero LOAD_FLAG signals update
  if((err=putmem(LOAD_FLAG,1,&TEMP))!=NO_ERR) return err;
  return err;
}

void binsprintf(char *s,int val)
{
  char *p;
  unsigned int t;
  p = s;
  t = 0x80;  // scan 8 bits
  for(;t>0;t=t>>1)
  {
    if(val & t) *p++ = '1';
    else        *p++ = '0';
  }
  *p=0;
}
