//------------------------------------------------------------
// DSK_WAV.CPP
// Keith Larson
// TMS320 DSP Applications
// (c) Copyright 1995, 1996
// Texas Instruments Incorporated
//
// This is unsupported freeware code with no implied warranties or
// liabilities.  See the disclaimer document for details
//------------------------------------------------------------
//  DSK_WAV plays and records *.WAV files with the C3x DSK
//
//  Use: DSK_WAV [options] <- standard DSK options
//                            User I/F used to play and record files
//
//  The project file should include the following
//
//    DSK_WAV.CPP   This file
//    DRIVER.CPP    Low level printer port drivers
//    TARGET.CPP    DSK Command level
//    OBJECT.CPP    Application setup routines
//    DSK_COFF.CPP  DSK and COFF file loaders and other utils
//    ERRORMSG.CPP  Messages used for most function returns
//    SYMBOLS.CPP   Symbol tables (needed to link DSK_COFF)
//    TEXTWIN.CPP   DOS level text window drivers
//    HARDWARE.CPP  Command line help message
//                  (Built from HARDWARE.HLP source)
//
//  Borland C++ version 3.1 setup (use defaults if not shown)
//
//    Compiler
//     Memory Model      - Large        (objects reused in DSK3D link)
//     Processor         - 80286        (old PC support)
//     Floating Point    - Emulation
//     Code Gen          - DOS Standard (works in Win DOS prompt box)
//     Calling convention- C
//     Optimization      - Speed        (code reduction is not significant)
//
//    Linker
//     Output              - DOS Standard EXE   (Not a windows app!)
//     Libraries           - Graphics
//     Object Windows Lib  - None
//     Container Class     - None
//     Standard Runtime Lib- Static
//
//  NOTE: Other than using '//' comments and other simple C++ features
//        this code follows ANSI C.  The C++ compiler is used primarily
//        for convenience as well as its performance and advanced error
//        checking and warnings.
//
//        An 80286 output is used for DSK users who are using old PC's
//        for automated 'smart' data collection boxes.
//
//  Windows 3.x/95
//        This is a DOS text window executable which uses the timelsice
//        managment interrupt hooks to make the application more efficient
//        at swapping in and out as a Windows task.
//
// 9/23/98 DSK3WAV.EXE no longer exits if THEFORCE.WAV is not found
//         on startup.  Allows creation of a default WAV file.
//------------------------------------------------------------
#include <dos.h>
#include <bios.h>
#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dsk.h"
#include "errormsg.h"
#include "dsk_coff.h"
#include "keydef.h"
/*---------------------------------------------------------------
   Read WAV file and send to DSK
----------------------------------------------------------------*/
#define WAV_START         3   /* start WAVE (xfer now via DMA)  */
#define DSK_WAVCMD 0x809C00L  /* Location of WAV command buffer */
/*--------------------------------------------------------------*/
ulong TA       = 4; /* AIC TA diviser to be programmed into DSK */
ulong TB       =18; /* AIC TB diviser to be programmed into DSK */
ulong TIM0     = 1; /* TIM0 period to be programmed into DSK    */
int   bsize    = 1; /* Chunk size to be read from wave file     */
int   AutoSet  = 1; /* Flag for enabling Auto set of Fs         */
ulong DTYPE    =32; /* Data type sent to DSK 8,16,32 stereo etc */
uchar wavbuf[4096]; /* char data buffer for file readin         */
ulong WAVDAT[1024]; /* data buffer for DSK data output          */
ulong WAVCMD  [64]; /* data buffer for DSK command output       */
char  FileName  []= "THEFORCE.WAV\0  Default Wave file to play  ";
int WRITE_EN    =0;
/*--------------------------------------------------------------
  RIFF/WAVE header
  --------------------------------------------------------------*/
struct WAV_HDR
{                 // Sz
  char  RIFF[ 4]; // 4 - RIFF block      "RIFF"
  ulong F_leng  ; // 4                   Filesize
  char  WAVE[ 4]; // 4                   "WAVE"
  char  fmt [ 4]; // 4 - format block    "fmt " -> fmt block
  ulong blen    ; // 4   leng of block   0x10 length of format block
  uint  ftag    ; // 2 -\ block data     format tag 1=MS PCM
  uint  ch      ; // 2  | (blen bytes)   channels (1=mono 2=stereo)
  ulong fs1     ; // 4  |                samples/sec
  ulong byte_sec; // 4  |                bytes/sec
  uint  byte_blk; // 2  |                bytes/block
  uint  bit_samp; // 2 -/                bits/sample
  char  data[4] ; // 4 -- data block     char "data"
  ulong D_leng  ; // 4 -- leng of block  Filesize - Headersize
}wh;
/*--------------------------------------------------------------
  Enumerated modes for wavefile playback
  --------------------------------------------------------------*/
enum WavMode
{
  CHAR,      /* Mono    8 bit */
  SCHAR,     /* Stereo  8 bit */
  INT,       /* Mono   16 bit */
  SINT       /* Stereo 16 bit */
} Mode;
//-----------------------------------------------------------
// Send user defined command start to the target
//-----------------------------------------------------------
MSGS USERCMD1(ulong CMD)
{
  MSGS err=NO_ERR;
  if((err=xmit_long(XSPARE))!=NO_ERR) return err; // Function
  if((err=xmit_long(   CMD))!=NO_ERR) return err; // Sub function or 1st arg
  return err;
}
MSGS USERCMD0(void)
{
  MSGS err=NO_ERR;
  if((err=xmit_long(XSPARE))!=NO_ERR) return err; // Function
  return err;
}
/*********************************************************************
   sendbuf() and recvbuf() read and write blocks of size 'length' to
   and from the DSK transferring from the hosts local memory at 'data'
 *********************************************************************/
MSGS sendbuf(ulong length,ulong *data)
{
  MSGS err;
  HPI_STRB(0);              // Drop the strobe to initiate a transfer
  for(long x=0;x<100000L;x++)  // 800 mS
  {
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
  }
  for(;length>0;length--)
    if((err=xmit_long(*data++))!=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  return NO_ERR;
}

MSGS recvbuf(ulong length,ulong *data)
{
  MSGS err;
  HPI_STRB(0);              // Drop the strobe to initiate a transfer
  for(long x=0;x<100000L;x++)  // 800mS uS
  {
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
    if(HPI_ACK()==1) break;
  }
  for(;length>0;length--)
    if((err=recv_long(data++))!=NO_ERR)
    {
      enable_Mtask();
      return err;
    }
  return NO_ERR;
}

/*--------------------------------------------------------------
  Loads the RIFF/WAV header information
  -------------------------------------------------------------*/
MSGS LoadRIFF(char *name)
{
  FILE *stream;
  stream = NULL;
  if((stream=fopen(name,"rb"))==NULL) return OPEN_ERR;
  fread(&wh,sizeof(WAV_HDR),1,stream);
  /* Make sure this is a RIFF/WAVE file  */
  if(strstr(wh.RIFF,"RIFF")==0) {fclose(stream); return OPEN_ERR;}
  if(strstr(wh.WAVE,"WAVE")==0) {fclose(stream); return OPEN_ERR;}
  /* Determine the playback mode         */
  switch(wh.bit_samp)
  { case 16: switch(wh.ch)
             { case 2: Mode = SINT; bsize=4; break;
               case 1: Mode =  INT; bsize=2; break;
             }
             break;
    default:
    case  8: switch(wh.ch)
             {
               case 2: Mode = SCHAR; bsize=2; break;
               case 1: Mode =  CHAR; bsize=1; break;
             }
             break;
  }
  fclose(stream);
  return NO_ERR;
}
/*---------------------------------------------------------------*/
/* This routine takes the data from the WAV file and packs it as */
/* integers to be played back by the DSK.  In the case of        */
/* partial blocks the end is zero padded                         */
/*---------------------------------------------------------------*/
void WAVpack(int count,int max, WavMode mode, ulong *ulptr)
{
  int x;
  long t1;
  int *i;
  int   *iptr;
  uchar *ucptr;
  ucptr = wavbuf;  // Raw data image from disk
  iptr = (int *)ucptr;
  i    = (int *)ulptr;
  switch(mode)
  {
    case  CHAR: for(x=0;x<count;x++) *i++ = (*ucptr++ -128)<<8; break;
    case  INT : for(x=0;x<count;x++) *i++ = *iptr++;            break;
    case SCHAR: for(x=0;x<  count;x++)
                { t1  = (*ucptr++ -128)<<8; // Sum (Mix) L/R channels
                  t1 += (*ucptr++ -128)<<8;
                  *i++ = (int)(t1 >> 1);
                }
                break;
    case  SINT: for(x=0;x<  count;x++)
                { t1 = (int)*iptr++;        // Sum (Mix) L/R channels
                  t1+= (int)*iptr++;
                  *i++ = (int)(t1 >> 1);
                } break;
    default   : break;
  }
  for(;x<max;x++) *i++ = 0;
}

/*------------------------------------------------------------*/
/* Open the WAVE/RIFF file and send it to the DSK using the   */
/* installed DSK wave play function                           */
/*------------------------------------------------------------*/
MSGS sendwav(int length, int PLAY)
{
  MSGS err;
  length = length >> 1;  // Length is now in longs (2 int samples/long)
  WAVCMD[2] = length;
  disable_Mtask();
  //
  // Send command packet
  //
  if((err=sendbuf(     8,WAVCMD))!=NO_ERR) return err;
  //
  // Send or receive data
  //
  if(PLAY) { if((err=sendbuf(length,WAVDAT))!=NO_ERR) return err;}
  else     { if((err=recvbuf(length,WAVDAT))!=NO_ERR) return err;}
  enable_Mtask();
  return NO_ERR;
}
/*------------------------------------------------------------------*/
/* Use a maximum packet size of 512 samples for on-chip C31 use.    */
/* If your DSK system can handle larger packets (external SRAM) you */
/* will need increase/move the buffer allocation in DSKWAV.ASM      */
/* as well as buffer size in this program                           */
/* Small packet sizes can also cause distortion because of the      */
/* time required for a windows multitask switch.  Run strictly as   */
/* a DOS application, very small buffers can be used, up until the  */
/* point that the host CPU cannot keep up.  IE buffer clashing while*/
/* reading the disk, printer port speed etc...                      */
/*------------------------------------------------------------------*/
int  PCKTSZ     =       256;  /* 512 samples=1024 bytes in DMA rcv buffer */
long MAX_RECORD = 10000000L;  /* 10 Mbytes max record size                */
MSGS WAV2DSK(char *name,int PLAY)
{
  FILE *stream;
  long l;
  long x;
  long c;
  long Fs;
  MSGS err;
  //
  // Set the packet size depending on the 4/8 bit readback mode.
  // The packet buffer on the DSK is hardcoded to 1024 positions
  //
  if(WSHIFT == -8)PCKTSZ = 512;// 512 16 bit samples fit 1024 byte buffer
  else            PCKTSZ = 256;// 256 16 bit samples fit 1024 nible buffer
  /*---------------------------------------------------------------
     Initialize the WAVE playback header
   ----------------------------------------------------------------*/
  if(PLAY)
    if((err=LoadRIFF(name))!=NO_ERR) return err;
  if(AutoSet)
  { if(PLAY==0)               // If record and sutoset mode,
    {                         // pick a standard Fs value for the header
      wh.fs1 = 12.5E6/(TIM0*2.0*TA*TB);
      if((wh.fs1 > 25000.0)                    ) wh.fs1 = 44100.0;
      if((wh.fs1 > 12000.0)&&(wh.fs1 < 25000.0)) wh.fs1 = 22050.0;
      if(                    (wh.fs1 < 12000.0)) wh.fs1 = 11025.0;
    }
    TA =   4;
    TB =  18;
    TIM0= 2.0*(44100.0*1.05)/(float)wh.fs1; /* 1.05 is a rounding factor */
  }
  /*---------------------------------------------------------------*/
  //
  // Initialize the WAVE setup command string
  //
  WAVCMD[ 0] = XSPARE             ;// User defined DSK command follows
  WAVCMD[ 1] = PLAY               ;// Sub command RECORD=0, PLAY!=0
  WAVCMD[ 2] = PCKTSZ             ;// Buffer length in 32 bit words
  WAVCMD[ 3] = (TA<<9)|(TA<<2)|0  ;// AIC setup values for A and B regs
  WAVCMD[ 4] = (TB<<9)|(TB<<2)|2  ;//
  WAVCMD[ 5] = TIM0               ;// TIMER 0 for AIC master clock
  WAVCMD[ 6] = DTYPE              ;// Data type (only packed 16 bit now)
  WAVCMD[ 7] = 1                  ;// 1=program AIC and timer on first go
  /*---------------------------------------------------------------*/
  WAVpack(0,PCKTSZ,Mode,WAVDAT); // Send a leading zero packet

  l = 0;
  //--------------------------------------------------------
  // A time sample is defined as a monophonic sample that
  // is created from monophonic or stereo 8/16 bit data
  //--------------------------------------------------------
  if(PLAY)
  {
    clrscr();
    err = sendwav(PCKTSZ,PLAY);
    printf(name);
    gotoxy(1,1);
    cprintf("Playing %s ->",name);
    if((stream=fopen(name,"rb"))==NULL) return OPEN_ERR;
    fread(&wh,sizeof(WAV_HDR),1,stream);
    wh.D_leng /= bsize; // convert to number of time samples to read
    for(x=0;x < wh.D_leng; x+= PCKTSZ) // Read in PCKTSZ time sample chunks
    {
      gotoxy(40,1);
      cprintf("%ld bytes",x*bsize);
      l += PCKTSZ;
      if(l>wh.D_leng) c = PCKTSZ-(l-wh.D_leng); // l = cumulative time samples
      else            c = PCKTSZ;               // c = samples to read from file
      c = fread(wavbuf,bsize,(int)c, stream);// read time samples
      WAVpack((int)c,PCKTSZ,Mode,WAVDAT);       // pack into monophonic samples
      if((err=sendwav(PCKTSZ,PLAY))!=NO_ERR) break;  //
      if(c==0) break;                        //
      if(kbhit()) break;
      WAVCMD[ 7] = 0; // Stop reprogramming the AIC and timer
    }
  }
  else /* Record */
  { clrscr();
    gotoxy(1,1);
    if(WRITE_EN==0)
    { cprintf("ERROR - WRITE MODE IS NOT ENABLED FOR %s!\r\n",name);
      getch();
      return NO_ERR;
    }
    err = sendwav(PCKTSZ,PLAY);
    cprintf("Recording to %s ->",name);
    if((stream=fopen(name,"wb"))==NULL) return OPEN_ERR;
   /*-----------------------------------------------------------*/
   /* Initialize WAV/RIFF header information                    */
   /*-----------------------------------------------------------*/
    Fs = 12.5E6/(TIM0*2.0*TA*TB);
//  Ft = 12.5e6/TIM0;

    strncpy     (wh.RIFF,"RIFF",4);
    strncpy     (wh.WAVE,"WAVE",4);
    strncpy     (wh.fmt ,"fmt ",4);
    strncpy     (wh.data,"data",4);
    wh.F_leng   = sizeof(wh)+MAX_RECORD; //  4  Filesize
    wh.blen     =                    16; //  4  0x10 length of block
    wh.ftag     =                     1; //  2  format tag 1=MS PCM
    wh.ch       =                     1; //  2  channels (1=mono 2=stereo)
//  wh.fs1      =                22050L; //  4  samples/sec
//  wh.byte_sec =                44100L; //  4  bytes/sec

    wh.fs1      =              Fs      ; //  4  samples/sec
    wh.byte_sec =              Fs*2    ; //  4  bytes/sec

    wh.byte_blk =                     2; //  2  bytes/block
    wh.bit_samp =                    16; //  2  bits/sample
    wh.D_leng   =            MAX_RECORD; //  4  data size in bytes
    /*-----------------------------------------------------------*/
    l = wh.D_leng/2;                     //  Use for counting longs
    fwrite(&wh,sizeof(WAV_HDR),1,stream);// Write header
    bsize = 2;                           // Monophonic 16 bit data
    /*-----------------------------------------------------------*/
    for(x=0;x < l; x+= PCKTSZ)           // Write PCKTSZ samples
    {
      gotoxy(40,1);
      cprintf("%ld bytes",x+x);
      if((err=sendwav(PCKTSZ,PLAY))!=NO_ERR) break; //
      c = fwrite(WAVDAT,PCKTSZ*bsize,1, stream);    // GET ADC samples
      if(c==0) break;
      if(kbhit()) break;
      WAVCMD[ 7] = 0;     // Stop reprogramming the AIC and timer
    }
    /*--------------------------------------------------------------*/
    /* Go back and adjust the header for the actual recorded length */
    /*--------------------------------------------------------------*/
    fseek(stream,0,SEEK_SET);
    x *= bsize;
    wh.D_leng = x + sizeof(WAV_HDR);
    wh.F_leng = x;
    fwrite(&wh,sizeof(WAV_HDR),1,stream);  // re-write header
  }
  /*--------------------------------------------------------------*/
  fclose(stream);
  WAVCMD[ 1] = 1;
  if(err!=NO_ERR) return err;
  WAVpack(0,PCKTSZ,Mode,WAVDAT);          // Send a trailing zero packet
  err = sendwav(PCKTSZ,1);
  clrscr();
  return err;
}
/*-----------------------------------------------------------------
   Record audio with DSK and send to wav file
-------------------------------------------------------------------*/
MSGS DSK2WAV(char *name)
{
  printf(name);
  return NO_ERR;
}
/*-----------------------------------------------------------------
   Act on user keystrokes
-------------------------------------------------------------------*/
MSGS check_key(void)
{
  FILE *stream;
  int key;
  float SampleRate = 20000.0;
  MSGS err;
  char *p;
  key = bioskey(0) & 0xFF00;
  switch(key)
  {
    case  _Q  : clrscr();
                _setcursortype(_NORMALCURSOR);
                exit(0);        // Quit
    case  _pls: SampleRate*=1.1; break;   // Slow down screen update rate
    case  _mns: SampleRate/=1.1; break;   // SampleRate up
    case    _A: AutoSet ^=1; break;
    case   _F1:   TA--; break;
    case   _F2:   TA++; break;
    case   _F3:   TB--; break;
    case   _F4:   TB++; break;
    case   _F5: TIM0--; break;
    case   _F6: TIM0++; break;
    case  _W  : clrscr();
                WRITE_EN ^=1;
                stream = fopen(FileName,"rb");
                if((stream != NULL) && (WRITE_EN))
                {
                  printf("Allow overwrite? [Y/N]:");
                  if(toupper(getch())=='Y') WRITE_EN = 1;
                  else                      WRITE_EN = 0;
                }
                fclose(stream);
                break;
    case  _F  : WRITE_EN = 0;
                clrscr();
                cprintf("New filename : ");
                scanf("%s",FileName);
                strupr(FileName);
                //-----------------------------------
                // Append .HEX to file... *.HEX only!
                //-----------------------------------
                if((p=strstr(FileName,"."))!=NULL) strcpy(p,".WAV");
                else  strcat(FileName,".WAV");
                clrscr();
                stream = fopen(FileName,"rb");
                if(stream == NULL)
                {
                  printf("%s does not exist\r\n",FileName);
                  printf("Create new or allow overwrite? [Y/N]:");
                  if(toupper(getch())=='Y') WRITE_EN = 1;
                  else                      WRITE_EN = 0;
                  if(stream==NULL) break;
                }
                else               printf("%s found\r\n",FileName);
                fread(&wh,sizeof(WAV_HDR),1, stream); // Read in the header
                fclose(stream);
                clrscr();
            //  break;
    case  _P  : WRITE_EN=0;
                printf("\rWAV->DSK play!          \n");
                if((err=WAV2DSK(FileName,1))!=NO_ERR)
                { clrscr();
                  printf("%s",Error_Strg(err));
                  getch();
                }
                clrscr();
                break;
    case  _R  : printf("\rDSK->WAV record!        \nHit any key to stop\n");
                if((err=WAV2DSK(FileName,0))!=NO_ERR)
                { clrscr();
                  printf("%s",Error_Strg(err));
                  getch();
                }
                clrscr();
                break;
    case  _ESC: clrscr();
                Pulse_Init();
                strcpy(EXTMSG,
                        "You hit the ESC key to reset DSK_WAV.EXE\n\n"
                        "This message string is user defined and then\n"
                        "accessed using an 'Extended Message'.\n"
                        "Look for this string in DSK_WAV.CPP\n\n"
                        "Hit any key");
                return EXT_MSG_OK;
                //return COM_ERR;
    default   : break;
  }
  if(TA   <  4)   TA= 4; if(TA   > 31)   TA=31;
  if(TB   < 14)   TB=14; if(TB   > 63)   TB=63;
  if(TIM0 <  1) TIM0= 1; if(TIM0 > 20) TIM0=20;
  if(SampleRate <   1000.0) SampleRate =   1000.0;
  if(SampleRate > 100000.0) SampleRate = 100000.0;
  return NO_ERR;
}
char *ONOFF(int sw)
{
  if(sw) return "ON  ";
  else   return "OFF ";
}
//--------------------------------------------
// whirly() lets a user visualy see that a
// function is being called repeatedly
//--------------------------------------------
void whirly(int x,int y)
{
  static char   whir = 0;
  int ox, oy;
  ox = wherex();
  oy = wherey();
  gotoxy(x,y);
  switch(whir++)
  { default: whir=0;
    case 0: printf( "/"); break;
    case 1: printf( "-"); break;
    case 2: printf("\\"); break;
    case 3: printf( "|"); break;
    case 4: printf( "/"); break;
    case 5: printf( "-"); break;
    case 6: printf("\\"); break;
    case 7: printf( "|"); break;
  }
  gotoxy(ox,oy);
}
/*-----------------------------------------------------------------
     Main application starts here
-------------------------------------------------------------------*/
void main(void)
{
  FILE *stream;
  float Fs, Ft;
  int Default_Wav = 1;
  MSGS err;           // Enumerated messages
  /*-------------------------------------------------------------*/
  /* The following line inserts DSK3 specific command line       */
  /* options into the help string before being printed. The help */
  /* string itself has enough WS chars to allow this.            */
  /*                                                             */
  /*         No application specific options are added           */
  /*-------------------------------------------------------------*/
  Edit_Help_Msg("DSK_WAV","","");
  Scan_Command_line("DSK_WAV"); // Command line options, LPT1, LPT2 etc...
  Detect_Windows();
  if(kbhit()) getch();          // Clear pending keystrokes
  //--------------------------------------------------------
  // The outer loop runs forever until the task is complete
  // or the user has hit a key to terminate the loader
  Pulse_Init();                            // Kill previous applications.
  clrscr();
  if((stream = fopen(FileName,"rb"))==NULL)
  { printf("%s does not exist",FileName);
    //while(!kbhit());
    getch();  // Need to remove keystroke... else Init_Communication() fails
    Default_Wav = 0;
    // Create default WAV file
  }
  else
  {
    fread(&wh,sizeof(WAV_HDR),1, stream);    // Read in the header
    fclose(stream);
  }
  for(;;)
  { Init_Communication(1000);
    clrscr();
    HALT_CPU();                            // Put C31 into spin0 mode
    if((err=Load_File("DSKWAV",LOAD))!=NO_ERR) // Load the app file
    { // Print the appropriate error message if an error occurs
      printf("\r\n%s",Error_Strg(err));
      getch();
      exit(0);
    }
    RUN_CPU();            // run the app file
    if(Default_Wav)
      err = WAV2DSK(FileName,1);    // Play the default RIFF/WAV file
    //
    // Now that the app file is downloaded and running,
    // sit in a loop waiting for commands from the user.
    //
    for(;;)
    {
      while(!kbhit())
      {
        _setcursortype(_NOCURSOR);
        delay(50);
        gotoxy(1,1);
        cprintf(
"[F]ile   [P]lay   [R]ecord   [W]rite_en   [A]utoset AIC   [ESC]reset\r\n"
"--------------------------------------------------------------------\r\n");
    gotoxy( 1, 3); cprintf("File        : %s" ,       FileName);
    gotoxy( 1, 4); cprintf("WRITE_EN    : %s" ,ONOFF(WRITE_EN));
    gotoxy( 1, 5); cprintf("Autoset     : %s" ,ONOFF( AutoSet));
    gotoxy( 1, 6); cprintf("PP_mode     : %ld",        -WSHIFT);

    gotoxy(40, 3); cprintf("File leng   : %ld", wh.F_leng  );
    gotoxy(40, 4); cprintf("format tag  : %d" , wh.ftag    );
    gotoxy(40, 5); cprintf("Channels    : %d" , wh.ch      );
    gotoxy(40, 6); cprintf("fs          : %ld", wh.fs1     );
    gotoxy(40, 7); cprintf("bytes/sample: %d" , wh.byte_blk);
    gotoxy(40, 8); cprintf("bits/channel: %d" , wh.bit_samp);
    gotoxy(40, 9); cprintf("Data length : %ld", wh.D_leng  );
    gotoxy( 1,10); cprintf(
"--------------------------------------------------------------------\r\n");
   Fs = 12.5E6/(TIM0*2.0*TA*TB);
   Ft = 12.5e6/TIM0;
   gotoxy(1,11);
   cprintf(
   "F ADC/DAC= %7.2f Khz    F_keys -> F1-F2  F3-F4  F5-F6\r\n"
   "F TCLK0  = %7.2f Khz              TA:%02ld  TB:%02ld  TIM0:%02ld   \r\n"
   "--------------------------------------------------------------------\r\n",
   Fs/1000.0,Ft/1000.0,TA,TB,TIM0);
   gotoxy( 1,19); cprintf(
   " Autoset sets the AIC sampling rate according to the Fs in the wave\r\n"
   " file header.  Autoset in record mode will narrow the possible\r\n"
   " sampling rates to the standard rates of 11,22, or 44 Khz.\r\n\r\n"
   " Autoset must be off to adjust TA, TB and TIM0");
   gotoxy(1,15);
        cprintf("Waiting for command :");
        whirly(wherex(),wherey());
        Release_TSlice();
      };                                    // Exit for key stroke
      _setcursortype(_NORMALCURSOR);
      err = check_key();                    // Act on user keyhits
      if(err!=NO_ERR)
      { clrscr();
        printf("\r\n%s",Error_Strg(err));   // print error message
        //while(!kbhit());                  // Display error until keyhit
        getch();
        break;               // Need to remove keystroke
      }
    }
  }
}
