;=======================================================================
SFFTSIZE  .set     512                ; Sample Window length (FFT size)
BIN_START .set     100                ; Start computing SFFT at this bin
BIN_END   .set     200                ; End computing SFFT at this bin
;- - - - - - - - - - - - - - - - - - -
ANGLE     .set     00.0               ; Filter reconstruction angle (degrees)
;- - - - - - - - - - - - - - - - - - -
SPECT_EN  .set     0                  ; Enable spectrum analyzer output
RATE      .set     1                  ; Write display points RATE times each
;- - - - - - - - - - - - - - - - - - -
TIM0_prd  .set     4                  ; AIC reference clock is TIM0
TA        .set     6                  ; DAC setup
TB        .set     30                 ;
RA        .set     6                  ; ADC setup
RB        .set     30                 ;
;=======================================================================
; PARAMETERS BELOW THIS LINE ARE COMPUTED FROM THE INFORMATION
; ABOVE.  THERE IS NO NEED TO MODIFY ANYTHING BELOW THIS POINT
;=======================================================================
BIN_LEN   .set     BIN_END-BIN_START  ; Filter length in bins
SFFTBINS  .set     BIN_LEN+1          ;
N         .set     SFFTSIZE           ; 'N' used as shorthand for SFFTSIZE
TR        .set     0                  ; Real twiddle offset in each cell
TI        .set     1                  ; Imag
DR        .set     0                  ; Real data offset in each cell
DI        .set     1                  ; Imag
RIBINSIZE .set     2                  ; Size of R/I element pair
pi        .set     3.14159265         ; Useful in making apple pie
w         .set     2.0*pi/N           ; angle = F * 2*pi/Fs
OVM       .set     0x80               ; Use overflow mode to saturate results
;=======================================================================
; If the input parameters won't work, generate a descriptive error
; for the user letting them know what to look for and maybe fix
;=======================================================================
      .if (BIN_LEN < 1)
APP MESSAGE: Calculated BIN_LEN must be >1
      .endif
      .if  ((SFFTBINS*4) + SFFTSIZE) > (0xE40-0x800)
APP MESSAGE: The Fbin and data storage buffers are too big for the DSK
      .endif
;====================================================================;
; The SFFT twiddles, data, and input buffer arrays are allocated     ;
; to be placed into RAM0 to avoid bus conflicts with program fetching;
;====================================================================;
          .include "C3XMMRS.ASM"      ;
          .start   "DATA",0x809800    ; Data arrays are placed at start of RAM0
          .sect    "DATA"             ;
TWIDCOEF   ;------------------------- ;
n         .set     BIN_START          ;
          .loop    SFFTBINS           ; R/I phase or twiddle coefficients
          .float   K1*cos(n*w)        ;
          .float   K1*sin(n*w)        ;
n         .sdef    n+1.0              ; next 'n'
          .endloop                    ;
SFFTDATA  ;---------------------------;
          .loop    SFFTBINS           ; R/I frequency bin data
          .float   0,0                ;   Pre-Zeroing bin data removes
          .endloop                    ;   startup glitches
BUF       ;---------------------------;
          .loop    N/2                ; N samples of ADC input delay data
          .float   0,0                ;
          .endloop                    ;
;====================================================================;
; The application code begins here, beginning with constants that    ;
; are used in various routines.                                      ;
;====================================================================;
Tbase     .word    TWIDCOEF           ; Location of twiddle coefficients
Bbase     .word    SFFTDATA           ; Location of R/I SFFT Bin data
CircAddr  .word    BUF                ; Current pointer into sample data
BUFSTART  .word    BUF                ; Start address of sample data
BUFEND    .word    BUF+N              ; End address of sample data
OutBin    .float   0                  ; Current spectrum analyzer bin
MAX       .float   32000.0            ; Used synch pulse and scaling
          ;- - - - - - - - - - - - - -
A_REG     .word    (TA<<9)+(RA<<2)+0  ; Packed AIC register values
B_REG     .word    (TB<<9)+(RB<<2)+2  ;
C_REG     .word    00000011b          ;
;0gctrl   .word    0x0E970300         ; Sport setup, noninverted clkx/clkr
S0gctrl   .word    0x0E973300         ; Sport setup,    inverted clkx/clkr
S0xctrl   .word    0x00000111         ;
S0rctrl   .word    0x00000111         ;
NewMnsOld .word    0                  ;
K1        .set     0.99995            ; Use a value slightly less than 1.0
K2        .float   pow(K1,N)          ; K1^N oldest sample scale factor
FILTEROUT .float   0.0                ; Temp storage for SFFT filter output
Scale     .float   2.0/N              ; SFFT growth scale factor
REAL_VEC  .float  -cos(pi*ANGLE/180.0); filtered REAL scale factor
IMAG_VEC  .float  -sin(pi*ANGLE/180.0); filtered IMAG scale factor
FLOG2SC   .float   pow(2.0,-24.0)     ; Scale factor for log2 calculations
bigval    .word    0x00010000         ; Used in overflow mode saturation
;====================================================================;
; The main loop consists of waiting for a new ADC sample.            ;
; When an receive interrupt occurs, the new data is loaded into the  ;
; data delay line buffer, followed by the SFFT and output routines.  ;
; Four dummy writes to the external bus have been added in the main  ;
; loop to allow real time benchmarking of the three functions using  ;
; and oscilloscope to monitor the address bus LSB's                  ;
;====================================================================;
          .start   "CODE",0x809E40    ; Start in last 512 words of RAM0
          .sect    "CODE"             ;  (also includes DSK kernel)
main      ldi      0xE4,IE            ;   Enable XINT/RINT/INT2
          idle                        ;   Wait for Receive Interrupt
          ;- - - - - - - - - - - - - -
          ldi      @S0_rdata,R0       ; The first interrupt occurs shortly
          ldi      0,R0               ; after AIC init is complete, which
          sti      R0,@S0_xdata       ; will not leave enough time for SFFT
          ;- - - - - - - - - - - - - -
loop      idle                        ;   Wait for Receive Interrupt
          sti      R0,@0x80A000       ;<1
          call     Input              ;   Put ADC sample in delay buffer
          sti      R0,@0x80AF03       ;<2
          call     SFFT               ;   Calculate SFFT
          sti      R0,@0x80AF0F       ;<3
          call     Output             ;   Output result
          sti      R0,@0x80AF3F       ;<4
          b        loop               ;   Loop back and do forever
;====================================================================;
; The ADC data is read and buffered here                             ;
;====================================================================;
Input     ldi      @S0_rdata,R0       ; get ADC data
          ash      -16,R0             ; Sign extend previous sample in MSB's
          float    R0,R0              ; Convert the ADC data to float
          ldi      @CircAddr,AR0      ; Load present circ buf address
          ldf      *AR0,R7            ; Multiply by 'K2' for bin stability
          mpyf     @K2,R7             ;  (see text)
          stf      R0,*AR0++          ;
          cmpi     @BUFEND,AR0        ; If at end of buffer, point to start
          ldige    @BUFSTART,AR0      ;
          subrf    R0,R7              ; R7 =  X[-N] - X[0]
          sti      AR0,@CircAddr      ; save new 'circular' modified ptr
          stf      R7,@NewMnsOld      ;
          rets                        ;
;====================================================================;
; The forward and reverse SFFT are calculated within this one loop   ;
; The loop itself is unrolled to achieve an inner loop cycle count   ;
; of 7 cycles per bin calculation.  The inner loop contains both the ;
; REAL and IMAG filter summations, so if the output is for spectrum  ;
; analysis or only one filter sum is required, one or both summations;
; can be removed giving an inner loop speed of 6 cycles/bin          ;
;====================================================================;
SFFT      ldi      @Tbase,AR0         ; R/I twiddle ptr
          ldi      @Bbase,AR1         ; R/I SFFT array ptr
          ldi      @Bbase,AR2         ; SFFT output (usualy in place)
          ldi      SFFTBINS-1,RC      ; Number of bins to calculate
          ldi      RIBINSIZE,IR0      ; Size of R/I pair in array
          ldf      @NewMnsOld,R7      ; R7 = (New - K2*Old)
          ;- - - - - - - - - - - - - -
          ldf      0,R4               ; Zero the REAL filter sum
          ldf      0,R5               ; Zero the IMAG filter sum
          ;- - - - - - - - - - - - - - - -
          mpyf3 *+AR0(TR),*+AR1(DR)    ,R0 ; TR*DR <- unroll from main loop
          rptb  EndSFFT                    ;
          ;- - - - - - - - - - - - - - - -
Loop      mpyf3 *+AR0(TR)  ,*+AR1(DI)  ,R1 ; TR*DI
          mpyf3 *+AR0(TI)  ,*+AR1(DI)  ,R0 ; TI*DI
       || addf3 R7,R0                  ,R3 ; (TR*DR + DELTA)
          mpyf3 *+AR0(TI)  ,*+AR1(DR)  ,R0 ; TI*DR
       || subf3 R0,R3                  ,R3 ; TR*DR - TI*DI + DELTA
          mpyf3 *++AR0(IR0),*++AR1(IR0),R0 ; TR*DR (used in next loop)
       || addf3 R1,R0                  ,R2 ; TR*DI + TI*DR
          stf   R2,*+AR2(DI)               ; Save the new Fbin values
       || stf   R3,*AR2++(IR0)             ;
          ;- - - - - - - - - - - - - - - -
          subf3    R4,R3,R4       ;REAL sum; sum'=R-sum  alternates sign of
EndSFFT   subf3    R5,R2,R5       ;IMAG sum; raised cosine window coeficients
          ;-----------------------------------------------------------
          ; For raised cosine window filters the endpoint bin values
          ; are scaled to 1/2 relative to the pass bins
          ;-----------------------------------------------------------
          addf     R4,R4              ; Double inner +/-1 sum loop
          addf     R5,R5              ;
          subf     R3,R4              ; Subtract endpoints at 50%
          subf     R2,R5              ;
          ldi      @Bbase,AR1         ; ptr to start of R/I SFFT array
          ldf      *+AR1(DI),R2       ;
       || ldf      *+AR1(DR),R3       ;
          .if      SFFTBINS&1         ; If the loop count was odd, the
          mpyf     -1,R4              ; +,-,+,- sum result is negative
          mpyf     -1,R5              ;
          .endif                      ;
          addf     R3,R4              ;
          addf     R2,R5              ;
          ;-----------------------------------------------------------
          ; When the SFFT is finished, the REAL/IMAG sums are scaled
          ; accordingly for the desired output phase angle.  A 'growth'
          ; scale factor is also applied since the summation occurs
          ; over N data points.
          ;-----------------------------------------------------------
ExitSFFT  mpyf     @REAL_VEC,R4       ; Rotate to desired output phase
          mpyf     @IMAG_VEC,R5       ;
          addf3    R4,R5,R0           ; Sum the R/I into a REAL output
          mpyf     @Scale,R0          ; inverse of N/2 growth

 ;mpyf   8,R0

          stf      R0,@FILTEROUT      ;
          rets                        ;
;====================================================================;
; The output section is written for both Spectrum analyzer output    ;
; as well as REAL/IMAG filter sum outputs                            ;
;====================================================================;
Output:   .if      SPECT_EN=0       ; If SPECT_EN=0 (disable) output either
          ldf     @FILTEROUT,R0     ; Output REAL/IMAG bin sum
          .else                     ;
          ;----------------------------------------------------
          ; The Spectrum analyzer output section is bypassed
          ; if the spectrum analyzer is not enabled
          ;----------------------------------------------------
          ldf      @OutBin,R0       ; Point to next output bin
          addf     1.0/RATE,R0      ; increment analyzer output pointer
          cmpf     BIN_LEN,R0       ;
          ldfge    0,R0             ;
          stf      R0,@OutBin       ;
          fix      R0,R0            ;
          bzd      Out              ;
          mpyi     RIBINSIZE,R0     ; Fbins are 2 words (R/I) per bin
          ldfz     @MAX,R0          ; If at base Fbin 0 Hz, output a synch
          ldi      @Bbase,AR0       ;
          subi     2,AR0            ; point to output bin-1 to perform
          addi     R0,AR0           ; -.5,1.0,-.5 convolutional window
          ;- - - - - - - - - - - - -
          ldf      *+AR0(DI+0),R0   ; Perform convolutional window filter
       || ldf      *+AR0(DR+0),R2   ; on the R/I pairs for this output
          addf     *+AR0(DI+4),R0   ;
          addf     *+AR0(DR+4),R2   ;
          mpyf     -0.5,R0          ; Scaling coefficient for -1,+1 bins
          mpyf     -0.5,R2          ;
          addf     *+AR0(DI+2),R0   ;
          addf     *+AR0(DR+2),R2   ;
          ;- - - - - - - - - - - - -
          mpyf     R0,R0            ; Calculate REAL^2 + IMAG^2 magnitude
          mpyf     R2,R2            ;
          addf     R2,R0            ;
          call     FLOG2            ; Convert to log2(), then scale
          mpyf     32,R0            ; and shift for best display
          mpyf     32,R0            ;
          subf     @MAX,R0          ;
          ;- - - - - - - - - - - - -
          .endif                    ;
Out       fix      R0,R0            ; Convert to integer DAC output
          mpyi     @bigval,R0       ; Use Overflow mode ALU saturation
          ash      -16,R0           ;
          andn     3,R0             ; Do not request a 2nd xmit
          sti      R0,@S0_xdata     ; Output DAC value to serial port
          rets                      ;
;====================================================================;
; FLOG2() Ultra Fast LOG2 function                                   ;
; computes log2(R0) and returns e8/s1/m4 accuracy float value in R0  ;
;====================================================================;
FLOG2:    cmpf     0.0,R0           ; Exit if value is <= Zero
          ldfle    -1,R0            ; if x<=0 return -1 (error)
          retsle                    ; return if X<=0
          lsh      1,R0             ; Concatenate mantissa to exponent
          pushf    R0               ; Convert 'fast log' to int, then float
          pop      R0               ; Value is accurate but scaled by 2^24
          float    R0,R0            ;
          mpyf     @FLOG2SC,R0      ; Mpy by scale factor
          rets                      ;
;====================================================================;
; The startup stub is used during initialization only and can be     ;
; overwritten by the stack or data after initialization is complete. ;
; Note: A DSK or RTOS communications kernel may also use the stack.  ;
; In this case be sure to not put the stack here during debug.       ;
;====================================================================;
          .entry   ST_STUB          ; Debugger starts here
ST_STUB   ldp      T0_ctrl          ; Use kernel data page and stack
          ldi      @stack,SP        ;
          ldi      0,R0             ; Halt TIM0 & TIM1
          sti      R0,@T0_ctrl      ;
          sti      R0,@T0_count     ; Set counts to 0
          ldi      TIM0_prd,R0      ; Set period
          sti      R0,@T0_prd       ;
          ldi      0x2C1,R0         ; Restart both timers
          sti      R0,@T0_ctrl      ;
          ;- - - - - - - - - - - - -
          ldi      @S0xctrl,R0      ;
          sti      R0,@S0_xctrl     ; transmit control
          ldi      @S0rctrl,R0      ;
          sti      R0,@S0_rctrl     ; receive control
          ldi      0,R0             ;
          sti      R0,@S0_xdata     ; DXR data value
          ldi      @S0gctrl,R0      ; Setup serial port
          sti      R0,@S0_gctrl     ; global control
;====================================================================;
; This section of code initializes the AIC                           ;
;====================================================================;
AIC_INIT  LDI      0x10,IE          ; Enable only XINT interrupt
          andn     0x34,IF          ;
          ldi      0,R0             ;
          sti      R0,@S0_xdata     ;
          RPTS     0x040            ;
          LDI      2,IOF            ; XF0=0 resets AIC
          rpts     0x40             ;
          LDI      6,IOF            ; XF0=1 runs AIC
          ;- - - - - - - - - - - - -
          ldi      @C_REG,R0        ; Setup control register
          call     prog_AIC         ;
          ldi      0xfffc  ,R0      ; Program the AIC to be real slow
          call     prog_AIC         ;
          ldi      0xfffc|2,R0      ;
          call     prog_AIC         ;
          ldi      @B_REG,R0        ; Bump up the Fs to final rate
          call     prog_AIC         ; (smaller divisors should be sent last)
          ldi      @A_REG,R0        ;
          call     prog_AIC         ;
          or       OVM,ST           ; Use the overflow mode for fast saturate
          b        main             ; the DRR before going to the main loop
;====================================================================;
; prog_AIC is used to transmit new timing configurations to the AIC. ;
; If you single step this routine, the AIC timing will be corrupted  ;
; causing AIC programming to fail.                                   ;
; STEP OVER THIS ROUTINE USING THE F10 FUNCTION STEP                 ;
;====================================================================;
prog_AIC  ldi      @S0_xdata,R1     ; Use original DXR data during 2 ndy
          sti      R1,@S0_xdata     ;
          idle                      ;
          ldi      @S0_xdata,R1     ; Use original DXR data during 2 ndy
          or       3,R1             ; Request 2 ndy XMIT
          sti      R1,@S0_xdata     ;
          idle                      ;
          sti      R0,@S0_xdata     ; Send register value
          idle                      ;
          andn     3,R1             ;
          sti      R1,@S0_xdata     ; Leave with original safe value in DXR
          ;- - - - - - - - - - - - -
          ldi      @S0_rdata,R0     ; Fix receiver underrun by dummy read
          rets                      ;
;====================================================================;
; By placing the stack at the end of the users runtime code, the     ;
; maximum space is made available for applications.  Essentialy once ;
; used initialization code or data can be reclaimed after it is used.;
; However, use this configuration for debug purposes                 ;
;====================================================================;
          .start   "STACK",$        ; This is a reminder to put the stack
          .sect    "STACK"          ; stack in a safe place.  $ places
stack     .word    stack            ; section at the current assy address
;====================================================================;
; Install the XINT/RINT ISR branch vectors                           ;
;====================================================================;
          .start "SP0VECTS",0x809FC5; Place ISR returns directly into
          .sect  "SP0VECTS"         ; secondary branch table
          reti                      ; XINT0
          reti                      ; RINT0
