;*********************************************************************
;
; Filename:  PARSE.ASM
;
; Author:  Chris Buehler, Jonas Keating
; Date:  10/11/95
; Last revision:  12/16/95
;
;*********************************************************************
;
; Description:  Contains routines:
;	         1) parse_data
;
; Algorithm:  1) Loads the address of the appropriate parse function
;	         (parse_data96, parse_data144, or parse_data192) into
;		 the ACC and branch to that address.
;	      2) Turns the datastream (32, 48, or 64 bits) into the
;		 necessary chunks.
;
;*********************************************************************
;
; Usage:  call parse_data
;
; Inputs:  bitstream, parse_addr
;
; Outputs:  ibits, qbits, r0
;
;*********************************************************************
;
; Comments:  Requires that parse_addr be set to the correct parse
;	     function for the current speed.
;
;*********************************************************************

  .mmregs
  .text

  .include macros.inc

  .global parse_data,parse_data96,parse_data144,parse_data192
  .global bitstream,r0,ibits,qbits,temp3
  .global parse_addr,datastart


parse_data:
  ldpk	parse_addr
  lacl	parse_addr
  ldpk	datastart
  bacc

parse_data96:
;[III III III III RRRR RRRRRRRRRRRRRRRR]
;[333 222 111 000 0000 0000000000000000]
;[--------------------|----------------]
;[---bitstream+1------|---bitstream----]

  lacc	bitstream+1,16		; Load bitstream+1 into high ACC
  and	#0Fh,16			; Mask off all but last 4 bits of r0
  or	bitstream		; Pull in the lower 16 bits of r0
  sach	r0+1			; Store high part
  sacl	r0			;   and low part of r0
  lar	AR0,#(ibits+3)		; Load AR0 with TOP of ibits
  lacc	bitstream+1,3		; Load to get ibits[3] at edge of high ACC
  and	#7,16			; this comment left intentionally blank
  sach	*-			; Store to ibits[3]
  lacc	bitstream+1,6		; Load to get ibits[2] at edge of high ACC
  and	#7,16			; Mask off to only get ibits[2]
  sach	*-			; Store ibits[2]
  lacc	bitstream+1,9		; Load to get ibits[1]
  and 	#7,16			; Mask off to get only ibits[1]
  sach  *-			; Store to ibits[1]
  lacc	bitstream+1,12		; Load to get ibits[0]
  and	#7,16			; Mask off to get only ibits[0]
  sach	*-			; Store ibits[0]
  ret

parse_data144:
;[Q Q III Q Q III Q Q III Q Q III RRRRRRRRRRRR RRRRRRRRRRRRRRRR]
;[7 6 333 5 4 222 3 2 111 1 0 000 000000000000 0000000000000000]
;[-------------------------|------------------|----------------]
;[------bitstream+2--------|---bitstream+1----|---bitstream----]

  sar	AR1,temp3		; Store AR1
  lacc	bitstream+1,16		; Load bitstream+1 into high ACC
  and	#0FFFh,16		; AND to get only 12 bits
  or	bitstream		; OR in bitstream for a total of 28 bits
  sach	r0+1			; Store to r0+1
  sacl	r0			;   and r0
  lar	AR0,#ibits	       	; AR0 is a pointer to ibits - SHOULD BE ACTIVE
  lar	AR1,#qbits		; AR1 is a pointer to qbits
  lacc	bitstream+1,4		; Shift to get the 3 bits you want in high half of ACC
  and	#7,16			; AND to get only desired 3 bits
  sach	*+,AR1			; Store to to ibits[0] and set qbits active
  lacc	bitstream+1,1		; Load to get qbits[0] at edge of high ACC
  and 	#1,16			; Mask off to get only qbits[0]
  sach	*+			; Store it to qbits[0]
  lacc	bitstream+2,16		; Load to get qbits[1] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[1]
  sach	*+,AR0			; Store it to qbits[1] and set ibits active
  lacc	bitstream+2,15		; Load to get ibits[1] at edge of high ACC
  and	#7,16			; Mask off to get only ibits[1]
  sach	*+,AR1			; Store it to ibits[1] and set qbits ative
  lacc	bitstream+2,12		; Load to get qbits[2] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[2]
  sach	*+			; Store it to qbits[2]
  lacc	bitstream+2,11		; Load to get qbits[3] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[3]
  sach	*+,AR0			; Store it to qbits[3] and set ibits active
  lacc	bitstream+2,10		; Load to get ibits[2] at edge of high ACC
  and	#7,16			; Mask off to get only ibits[2]
  sach	*+,AR1			; Store it to ibits[2] and set qbits ative
  lacc	bitstream+2,7		; Load to get qbits[4] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[4]
  sach	*+			; Store it to qbits[4]
  lacc	bitstream+2,6		; Load to get qbits[5] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[5]
  sach	*+,AR0			; Store it to qbits[5] and set ibits active
  lacc	bitstream+2,5		; Load to get ibits[3] at edge of high ACC
  and	#7,16			; Mask off to get only ibits[3]
  sach	*+,AR1			; Store it to ibits[3] and set qbits ative
  lacc	bitstream+2,2		; Load to get qbits[6] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[6]
  sach	*+			; Store it to qbits[6]
  lacc	bitstream+2,1		; Load to get qbits[7] at edge of high ACC
  and	#1,16			; Mask off to get only qbits[7] (just to be sure)
  sach	*+			; Store it to qbits[7]
  lar	AR1,temp3		; Restore AR1
  ret

parse_data192:
;[QQQ QQQ III QQQ QQQ I II QQQ QQQ III QQQ QQ Q III RRRRRRRRRRRR RRRRRRRRRRRRRRRR]
;[777 666 333 555 444 2 22 333 222 111 111 00 0 000 000000000000 0000000000000000]
;[---------------------|---------------------|------------------|----------------]
;[-----bitstream+3-----|----bitstream+2------|---bitstream+1----|---bitstream----]

  sar	AR1,temp3		; Store AR1
  lacc	bitstream+1,16		; Load bitstream+1 into high ACC
  and	#0FFFh,16		; AND to get only 12 bits
  or	bitstream		; OR in bitstream for a total of 28 bits
  sach	r0+1			; Store to r0+1
  sacl	r0			;   and r0
  lar	AR0,#ibits	       	; AR0 is a pointer to ibits - SHOULD BE ACTIVE
  lar	AR1,#qbits		; AR1 is a pointer to qbits
  lacc	bitstream+1,4		; Shift to get the 3 bits you want in high half of ACC
  and	#7,16			; AND to get only desired 3 bits
  sach	*+,AR1			; Store it to ibits[0] and set qbits active
  lacc	bitstream+2,16	        ; Qbits[0] lies across bitstream+1
  and	#3,16			;   and bitstream+2, so we must load
  or	bitstream+1		;   both words and shift to get the
  sfl				;   result into high ACC
  sach	*+			; Store result to qbits[0]
  lacc	bitstream+2,14		; Load to get qbits[1] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[1]
  sach  *+,AR0			; Store it to qbits[1] and set ibits to active
  lacc	bitstream+2,11		; Load to get ibits[1] at edge of high ACC
  and	#7,16			; Mask off to get only ibits[1]
  sach	*+,AR1			; Store it to ibits[1] and set qbits to active
  lacc  bitstream+2,8		; Load to get qbits[2] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[2]
  sach	*+			; Store it to qbits[2]
  lacc	bitstream+2,5		; Load to get qbits[3] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[3]
  sach	*+,AR0			; Store it to qbits[3] and set ibits to active
  lacc	bitstream+3,16		; Ibits[2] lies across bitstream+2
  and	#1,16			;   and bitstream+3, but we do something diferent
  or	bitstream+2		;   this time.  We would need two SFLs to get the
  bsar	14			;   3 bits in the high ACC, so we barrel shift right
  sacl	*+,AR1			;   and store the low ACC -- setting qbits to active
  lacc	bitstream+3,15		; Load to get qbits[4] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[4]
  sach	*+			; Store it to qbits[4]
  lacc	bitstream+3,12		; Load to get qbits[5] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[5]
  sach	*+,AR0			; Store it to qbits[5] and set ibits to active
  lacc	bitstream+3,9		; Load to get ibits[3] at edge of high ACC
  and	#7,16			; Mask off to get only ibits[3]
  sach	*+,AR1			; Store it to ibits[3] and set qbits to active
  lacc	bitstream+3,6		; Load to get qbits[6] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[6]
  sach	*+			; Store it to qbits[6]
  lacc	bitstream+3,3		; Load to get qbits[7] at edge of high ACC
  and	#7,16			; Mask off to get only qbits[7] (to make sure nothing from high ACC appears)
  sach	*+			; Store it to qbits[7]
  lar	AR1,temp3		; Restore AR1
  ret