;*********************************************************************
;
; Filename:  QAM_MOD.ASM
;
; Author:  Chris Buehler, Hemanth "Thank you, come again" Sampath
; Date:  8/15/95
; Last revision:  10/11/95
;
;*********************************************************************
;
; Description: QAM_mod:
;	       Modulates points using QAM for output through
;	       the AIC.
;	       QAM_init:
;	       Initializes variables and AR registers for QAM_mod.
;	       QAM_start:
;	       Starts interrupts.
;	       QAM_wait:
;	       Waits for the interrupt routine to catch up with
;	       the main program.
;
; Algorithm:   Filters the output points with passband raised cosine
;              shaping filters.
;
;
;*********************************************************************
;
; Usage:  Point current (x,y) pair into xx and xy, call QAM_mod then
;         QAM_wait.  Repeat for all output points.
;
; Inputs:  xx and xy
;
; Outputs:  analog waveform
;
;*********************************************************************
;
; Comments:  Thse routines implement the QAM modulation used for the entire
;            transmitter.
;
;*********************************************************************
  .mmregs
  .include macros.inc

ntaps  .set	8

  .global	QAM_mod
  .global	QAM_init
  .global	QAM_start
  .global	QAM_wait
  .global	rot1
  .global	rot2
  .global	rot3
  .global       datastart
  .global       xx, xy, xx_7, xy_7,tx,ty
  .global	n
  .global	rot_table
  .global       outbufferx, outbuffery
  .global       outbuffer, endbuffer
  .global	testxf, tcont

  .global	points,pointadd

  .text

QAM_init:
  clrc CNF
;  setc  CNF
  lar	AR7,#outbuffer          ; initialize global AR regs
  lar   AR5,#4			;   AR7 = AR3 = output buffer
  lar	AR3,#(outbuffer+4)		;   AR5 = interrupt counter = 4
  splk  #endbuffer, ARCR        ; set ARCR to end of main output buffer
  ret

QAM_start:
  zap
  poll_xmit
  ldp	#0
  splk  #0021h, IMR
  smmr  IFR,IFR
  clrc  INTM
  ldpk	datastart
  ret

QAM_wait:
  mar	*,AR7
  cmpr	2
  nop
  nop
  bcnd  no_wrap_ar7, NTC
  lar	AR7, #outbuffer

no_wrap_ar7:
  mar *,AR5

wait_int:                      ; AR5 is used by the interrupt routine
  banz  wait_int,*,AR5         ; as soon as AR5 =0, we can put the next

  lar   AR5, #4                ; load in the mailbox.
;  ldpk  testxf
  lar   AR0,#testxf            ; every baud toggle XF bit for sync signal
  mar   *,AR0
  sst   #1,*
  xpl   #010h,*
  lst   #1, *
  ret

cby_init:
  ldp   #0
  splk  #xy, CBER1
  splk  #xy_7, CBSR1
  lar   AR2, #xy_7
  splk  #0ah, CBCR
  lar   AR2, #xy_7
  ldpk  datastart
  ret

cbx_init:
  ldp   #0
  splk  #xx, CBER2
  splk  #xx_7, CBSR2
  lar	AR2, #xx_7
  splk  #0a0h, CBCR
  lar   AR2, #xx_7
  ldpk  datastart
  ret

rot1:
  lacc	xy			; get yn
  sacb				;   save it in accb
  lacc	xx			; get xn
  neg				;   negate it
  sacl	xy			;   put it in yn
  lacb                          ; reload yn
  sacl	xx                      ;   put it in xn
  ret				; now xn' = yn and yn' = -xn

rot2:
  lacc	xy			; get yn
  neg				;   negate it
  sacl	xy			;   put it in yn
  lacc	xx			; get xn
  neg				;   negate it
  sacl	xx			;   put it in xn
  ret				; now xn' = -xn and yn' = -yn

rot3:
  lacc	xx			; get xn
  sacb				;   save it in accb
  lacc	xy			; get yn
  neg				;   negate it
  sacl	xx			;   put it in xn
  lacb				; reload xn
  sacl	xy			;   put it in yn
  ret				; now xn' = -yn and yn' = xn

; QAM_mod
; generates output sample from xn amd yn arrays and stores it in
; outbuffer
; algorithm:
;   rotate (xn,yn) depending on value of n:
;      n = 0 -> no rotation
;      n = 1 -> (y,-x)
;      n = 2 -> (-x,-y)
;      n = 3 -> (-y,x)
;
;   modulate the rotated constellation points with the passband
;     shaping filters resulting in two waveforms
;   subtract the two waveforms to produce final output waveform
;
; register conventions and assumptions:
; AR0* = temporary pointer calculations
; AR1  = unused
; AR2* = circular buffer data pointer (yn and xn)
; AR3  = interrupt routine output buffer pointer
; AR4* = temporary output buffer pointer
; AR5  = interrupt counter
; AR6* = temporary output buffer pointer
; AR7* = main routine output buffer pointer
;       AR7 is assumed to be properly initialized
;
; * indicates the register is used in QAM_mod

QAM_mod:
  ldpk	pointadd
  lar	AR0,pointadd
  lacl	pointadd
  add	#2
  sacl	pointadd
  and	#0FFh
  bcnd	go_on,NEQ
  splk	#(pointadd-2),pointadd

go_on:
  ldpk	datastart
  mar   *,AR0
  lacl	xx			; save in (tx,ty) for output purposes
  sacl	tx
  sacl  *+
  lacl	xy
  sacl	ty
  sacl  *
  lacc	n
  and	#3			; make sure doesn't go above 3

  bcnd	rot_done, EQ		; branch if n=0, no rotation
				; else branch to proper rot_ function
  add	#rot_table
  samm	AR0			; use AR0 for temp pointer calculation
  mar	*,AR0			; make AR0 the current aux. reg.
  nop				; wait for write to AR0 to finish
  lacc	*			; load the address of rot_  function
  cala  			; execute proper rot_ function

rot_done:
  ldp	#0
  splk	#0,CBCR			; clear all circular buffers
  ldpk	datastart
  call cbx_init, AR2		; initialize x circular buffer
				;   AR2 now points to the xn-data
				;   and is the current aux. reg.
  lar	AR4, #outbufferx         ; AR4 now points to the temp xn-outbuffer

				; calculate the 4 output values for x
  rptz   #(ntaps-1)
  mac   xilt_co0,*-
  apac
  mar   *,AR4
  sach  *+,AR2

  rptz   #(ntaps-1)
  mac   xilt_co1,*-
  apac
  mar   *,AR4
  sach  *+,AR2

  rptz   #(ntaps-1)
  mac   xilt_co2,*-
  apac
  mar   *,AR4
  sach  *+, AR2

  rptz   #(ntaps-1)
  macd  xilt_co3,*-
  apac
  mar   *,AR4
  sach  *+  			; now outbufferx has 4 output samples

  ldp	#0
  splk	#0,CBCR
  ldpk	datastart		; clear all circular buffers
  call cby_init, AR2		; initialize y circular buffer
				;   AR2 now points to the yn-data
				;   and is the current aux. reg.
  lar	AR4, #outbuffery         ; AR4 now points to the temp yn-outbuffer

				; calculate the 4 output values for y
  rptz   #(ntaps-1)
  mac   yilt_co0,*-
  apac
  mar   *,AR4
  sach  *+,AR2

  rptz   #(ntaps-1)
  mac   yilt_co1,*-
  apac
  mar   *,AR4
  sach  *+,AR2

  rptz   #(ntaps-1)
  mac   yilt_co2,*-
  apac
  mar   *,AR4
  sach  *+, AR2

  rptz   #(ntaps-1)
  macd  yilt_co3,*-
  apac
  mar   *,AR4
  sach  *+,AR6			; now outbuffery has 4 output samples

  lar	AR4, #outbufferx 	; reload the outbuffer pointers to get
  lar	AR6, #outbuffery	;  for the subtraction
  mar	*,AR6			; make AR6 current

  ldp	#0
  splk #3,BRCR          	; perform 4 subtractions to get final output
  ldpk  datastart
  nop
  rptb #(subtract_blk -1)        ;  waveform
  lacc *+, AR4                  ; load from x-waveform, make AR4 current
  sub *+, AR7                   ; subtract y-wavefrom, make AR7 current
  sacl *+, AR6			; store result in main output buffer

subtract_blk:
  lacl	n
  add	#1
  and	#7
  sacl	n
  ret

; multiplied by cos(wt) (GOOD)
xilt_co0:   .word   0,    0,     0,  32767,      0,      0,      0,   0
xilt_co1:   .word 323, -555, -4306,  11186,  -8358,   1003,    692, -61
xilt_co2:   .word 233, 1055, -3486, -14216, -14216,  -3486,   1055, 233
xilt_co3:   .word -61,  692,  1003,  -8358,  11186,  -4306,   -555, 323

; multiplied by sin(wt) (GOOD)
yilt_co0:   .word    0,     0,     0,     0,      0,     0,    0,    0
yilt_co1:   .word -134, -1341,  1783, 27005,   3461,  2422, -286, -147
yilt_co2:   .word  233, -1055, -3486, 14216, -14216,  3486, 1055, -233
yilt_co3:   .word  147,   286, -2422, -3461, -27005, -1783, 1341,  134

  .end
