/*********************************************************************
;
; Filename: TRANS96A.CPP
;
;
; Author: Chris Buehler, Jonas Keating
; Date: 9/2/95
; Last revision: 10/26/95
;
;*********************************************************************
;
; Description: A C++ simulation of encoding data from a bitstream
;      	       of 1's and 0's into hexidecimal values representing
;	       those that would be modulated to form analog output
;	       to the phone line.  The output can be decoded by using
;	       RECV96B.EXE.
;
;*********************************************************************
;
; Usage: TRANS96A < infile.dat > outfile.dat
;
;*********************************************************************/

#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>

#define M 6 			// 6 rings
#define bits 32			// 32 bits/mapping frame
#define K 20			// 20 bits -> shell mapper
#define Q 0
#define TWOtotheQ 1
#define W 1			// scale factor w = 1 for b < 56
#define WMASK 0x3FF             // mask for w = 1
#define XOFF    3

typedef struct
{
  long int real, imag;
} complex;

typedef unsigned long int ulong;
typedef unsigned int uint;

void init_tables(void);
void show_tables(void);

ulong getdata(void);
void parse_data96(void);
void shell_map(void);
void compute_rotation_factorZ(void);
void rotate90_cw(void);
void get_initial_point(void);
void precoder_filter_real(void);
void precoder_filter_imag(void);
void round_precoder_output(void);
void quantize96(void);
void complex_sum(void);
void complex_diff(void);
void compute_c0(void);
void compute_u0(void);
void get_binary_subset_label(void);
void QAM_mod(void);
void compute_rotation_factorW(void);
void write4dig(unsigned int word);

				// shell mapper tables g2 - z8
ulong g2[4*(M-1) + 1];
ulong g4[8*(M-1) + 1];
ulong g8[8*(M-1) + 1];
ulong z8[8*(M-1) + 1];

long int A, B, C, D, E, F, G, H,temp;
long int r1, r2, r3, r4, r5;


//mapping frame counters
//   m counts 4D symbols
int m=0;
//   n counts 2D symbols
int n = 0;

//mapping frame parser output
ulong r0 = 0;
uint qbits[8];
uint ibits[4];
				// unrotated points
int vx=0, vy=0;
				// rotated points
int ux=0,uy=0;

// need to remember some old x values
// in C program:  x[0]  x[1]  x[2]  x[3] x[4] x[5] x[6] x[7] x[8] x[9] x[10]
// in transmitter:x(-3) x[-2] x[-1] x[0] x[1] x[2] x[3] x[4] x[5] x[6] x[7]

int xx[8+XOFF],xy[8+XOFF];
int yy=0, yx=0;
int pr=0, pi=0;
int c_old_r=0, c_old_i=0, c_new_r=0, c_new_i=0;
uint s_early=0, s_late=0;
uint c0=0;
uint y0=0;
uint v0=0;
uint u0=0;
uint y1234=0;
int z=0;
int w=0;
long int ACC=0;
long int ACCB=0;
complex h[3];
// convolutional encoder state variable (4 bit)
uint state = 0x0;
				// super constellation points
int coords[2*240];
uint conv_enc_input[8][8];
uint conv_enc_tab[16][4];
// 32 bit data buffer
// MSB								LSB
//|I(3,3)-I(1,3)|I(3,2)-I(1,2)|I(3,1)-I(1,1)|I(3,0)-I(1,0)|S(k)-S(1)|
// 9600 baud - no uncoded Q bits
ulong bitstream = 0;

// stored (x0,y0) (x1,y1)...

int mjk[8];

void main(void)
{
  init_tables();

  //  get 32 bits of new data to encode
  while((bitstream = getdata()) != EOF)
  {
	n = 0;
	// parse data into S bits, Q bits, and I bits
	parse_data96();
	//  generate 8 mapping shells
	shell_map();

	for(m=0; m< 4; m++)
	{
	  get_initial_point();
	  compute_rotation_factorZ();
	  ACC = z;
	  rotate90_cw();
	  complex_sum();
	  complex_diff();
	  get_binary_subset_label();
	  s_early = ACC;
	  QAM_mod();
	  // onto next 2D symbol
	  n++;

	  precoder_filter_real();
	  round_precoder_output();
	  pr = ACC;
	  precoder_filter_imag();
	  round_precoder_output();
	  pi = ACC;

	  c_old_r = c_new_r;
	  c_old_i = c_new_i;
	  ACC = pr;
	  quantize96();
	  c_new_r = ACC;
	  ACC = pi;
	  quantize96();
	  c_new_i = ACC;

	  compute_c0();
	  compute_u0();
	  compute_rotation_factorW();
	  get_initial_point();
	  ACC = w;
	  rotate90_cw();
	  complex_sum();
	  complex_diff();
	  get_binary_subset_label();
	  s_late = ACC;
	  QAM_mod();
	  // get ready for next symbol
	  n++;
	  precoder_filter_real();
	  round_precoder_output();
	  pr = ACC;
	  precoder_filter_imag();
	  round_precoder_output();
	  pi = ACC;

	  c_old_r = c_new_r;
	  c_old_i = c_new_i;
	  ACC = pr;
	  quantize96();
	  c_new_r = ACC;
	  ACC = pi;
	  quantize96();
	  c_new_i = ACC;

	  y1234 = conv_enc_input[s_early][s_late];
	  y0 = (conv_enc_tab[state][y1234&3])&1;
	  state = (conv_enc_tab[state][y1234&3])>>1;
	}
	// at the end of 8 2D symbols, need to reset some buffers
	//   these values are remembered from one mapping frame to the next

	xx[XOFF-3] = xx[XOFF+5];
	xx[XOFF-2] = xx[XOFF+6];
	xx[XOFF-1] = xx[XOFF+7];
	xy[XOFF-3] = xy[XOFF+5];
	xy[XOFF-2] = xy[XOFF+6];
	xy[XOFF-1] = xy[XOFF+7];
  }
}


// simulate the QAM modulator - output the point to stdout
void QAM_mod(void)
{
//  printf("%d %d\n",xx[XOFF+n],xy[XOFF+n]);
  write4dig(xx[XOFF+n]);
  write4dig(xy[XOFF+n]);

}

void write4dig(unsigned int word)
{
  if(word < 0x1000L)
	printf("0");
  if(word < 0x0100L)
	printf("0");
  if(word < 0x0010L)
	printf("0");
  printf("%x\n",word);
}

// determines the 3 bits subset label for a particular constellation point
//   subset label is stored j2j1j0
//   point p is assumed to be in the format 15:1
//   the 1 fractional bit is the 1/2
void get_binary_subset_label(void)
{
  uint j0,j1,j2;

  // Dr. Tretter's formula assumes that the lattice points
  // are shifted by .5,.5 ( and are (.5,.5) (1.5,1.5) etc...
  // ours are actually 2*his, so we'll divide by 2 to fix this
  // algorithm.. i hope

  // x0 = x-.5   y0 = y-.5
  yx = yx - 1;
  yy = yy - 1;
  // divide our coords by two, should give us Dr. Tretter's
  yx >>= 1;
  yy >>= 1;
  // j0 = mod(x0+y0,2)  that is, the remainder after dividing by 2
  //    or, the lsb

  j0 = (yx + yy)&1;

  // j1 = mod(x0,2) = lsb of x0

  j1 = yx&1;

  // now want (x1,y1) = (x0,y0) - j1(1,1) - j0(0,1)
  //                    ~~~~~~~~~~~~2~~~~~~~~~~~~~~
  // p.x = (p.x - j1)>>1;

  yx = yx>>1;
  yy = (yy - j1 - j0)>>1;

  // now j2 = mod(x1+y1,2)^(j0*!j1)

  j2 = ((yx+yy)&1)^(j0&(!j1));

  ACC = (j2<<2) + (j1<<1) + j0;
}

// compute the u0 bit as the exclusive OR of y0 (the trellis encoder output),
//  c0 (modulo encoder), and v0 (the bit inversion pattern)

void compute_u0(void)
{
  // ^ is XOR in C
  u0 = (y0^c0^v0)&1;
}


// compute c0
void compute_c0(void)
{
  long int sum1,sum2;

  sum1 = (c_old_r+c_old_i)>>1;
  sum2 = (c_new_r+c_new_i)>>1;
  c0 = ((sum1^sum2)&1);
}

void quantize96(void)
{
  // p stored in 9:7 format.  must quantize to multiple of 2w, w=1,2
  //     this routine is specific to 9600, it assumes 2w = 2!!!
  //this routine outputs integer values = 16:0
  // now it also works for negative numbers properly

  ACCB = ACC;
  ACC = (labs(ACC) + 0x07F)>>7;

  // we must mask out the 1st bit because we are rounding to the 2nd bit
  //  all numbers should be even coming out of quantize96
  ACC &= 0xFFFFFFFEL;
  if(ACCB < 0)
	ACC = -ACC;
}


void precoder_filter_real()
{
  // h's are stored 2:14    = 1 sign bit, 1 magnitude bit, 14 fractional
  // x's are stored  9:7    = 1 sign bit, 8 magnitude bits, 7 fractional
  //    not sure about x's
  // so product is 11:21    - 2 sign bits (?) 9 magnitude, 21 fractional
  int p;
  ACC = 0;

  for(p=1; p < 4; p++)
  {
	ACC += xx[XOFF + n - p]*h[p-1].real -
		xy[XOFF + n - p]*h[p-1].imag;
  }
}

void precoder_filter_imag()
{
  // h's are stored 2:14    = 1 sign bit, 1 magnitude bit, 14 fractional
  // x's are stored  9:7    = 1 sign bit, 8 magnitude bits, 7 fractional
  //    not sure about x's
  // so product is 11:21    - 2 sign bits (?) 9 magnitude, 21 fractional
  int p;
  ACC = 0;

  for(p=1; p < 4; p++)
  {
	ACC += xy[XOFF + n - p]*h[p-1].real +
		xx[XOFF + n - p]*h[p-1].imag;
  }
}

void round_precoder_output(void)
{
  // rounding 11:21 number format to 9:7 = 1 sign,8 magn, 7 fractional

  ACCB = ACC;
  ACC = (labs(ACC) + 0x01FFFL)>>14;

  if(ACCB < 0)
	ACC = -ACC;
}

void complex_sum(void)
{
  yx = ux + c_new_r;
  yy = uy + c_new_i;
}

void complex_diff(void)
{
  xx[XOFF+n] = (yx<<7) - pr;
  xy[XOFF+n] = (yy<<7) - pi;
}

void parse_data96()
{
  int i,j;

   //9600 specific

  r0 = (bitstream & 0xFFFFFUL);
  // no uncoded bits
  for(i = 0; i < 8; i++)
	qbits[i] = 0;

  // extract 4 'i' fields
  bitstream >>= 20;
  ibits[0] = bitstream & 7;
  bitstream >>=3;
  ibits[1] = bitstream & 7;
  bitstream >>=3;
  ibits[2] = bitstream & 7;
  bitstream >>=3;
  ibits[3] = bitstream & 7;
}


// rotates a point pt by rot_factor*90 degrees
//  90 degree rotations are performed by negating and swapping coords

void rotate90_cw()
{
  switch(ACC)
  {
	case 0:  ux = vx;
			 uy = vy;
			 break;
	case 1:  ux = vy;
			 uy = -vx;
			 break;
	case 2:  ux = -vx;
			 uy = -vy;
			 break;
	case 3:  ux = -vy;
			 uy = vx;
			 break;
  }
}

void compute_rotation_factorW(void)
{
  // mask out lsb
  w = (z + ((ibits[m]&1)<<1) + u0)&3;
}

//  ibits are in the format:  i3i2i1
//   shift out the i1 bit to get:  i3i2
//   add to the old z, AND it with ..0011 to get mod-4 addition

void compute_rotation_factorZ(void)
{
//  z = (z + (ibits[m]>>1)&3);
  z = (z+(ibits[m]>>1))&3;
}

// get the initial unrotated constellation point from the table 'coords'
//
// point = 2^q + qbits
// 'coords' is stored as follows: coords[0], coords[1], coords[2],coords[3]
//                                   x0         y0         x1        y1 ...

void get_initial_point()
{
  int map_index = (mjk[n]*TWOtotheQ + qbits[n])<<1;

  vx = coords[map_index];
  vy = coords[map_index + 1];
}

void shell_map()
{

  A = 0;
  while(z8[A] <= r0)
  {A++;}
  A--;

  r1 = r0-z8[A];

  B = -1;
  while(r1 >= 0)
  {
	B++;
	r1 -= g4[B]*g4[A-B];
  }
  r1 += g4[B]*g4[A-B];

  r2 = r1 % g4[B];
  r3 = (r1-r2)/g4[B];

  r4 = r2;
  C = -1;
  while(r4 >= 0)
  {
	C++;
	r4 -= g2[C]*g2[B-C];
  }
  r4 += g2[C]*g2[B-C];

  r5 = r3;
  D=-1;
  while(r5 >= 0)
  {
	D++;
	r5 -= g2[D]*g2[A-B-D];
  }
  r5 += g2[D]*g2[A-B-D];

  E = r4 % g2[C];
  F = (r4-E)/g2[C];

  G = r5 % g2[D];
  H = (r5-G)/g2[D];

  if(C < M)
  {
	mjk[0] = E;
	mjk[1] = C-mjk[0];
  }
  else
  {
	mjk[1] = M-1-E;
	mjk[0] = C-mjk[1];
  }

  if(B-C<M)
  {
	mjk[2] = F;
	mjk[3] = B-C-mjk[2];
  }
  else
  {
	mjk[3] = M-1-F;
	mjk[2] = B-C-mjk[3];
  }

  if(D<M)
  {
	mjk[4] = G;
	mjk[5] = D-mjk[4];
  }
  else
  {
	mjk[5] = M-1-G;
	mjk[4] = D-mjk[5];
  }

  if(A-B-D < M)
  {
	mjk[6] = H;
	mjk[7] = A-B-D-mjk[6];
  }
  else
  {
	mjk[7] = M-1-H;
	mjk[6] = A-B-D-mjk[7];
  }

}

void show_tables(void)
{
  int p,j;

  cout << "g2:\t\t.word ";
  for(p = 0; p <= 2*(M-1); p++)
	cout << g2[p] << ((p==2*(M-1))?"":", ");
  cout << "\n";
  cout << "g4:\t\t.word ";
  for(p = 0; p <= 4*(M-1); p++)
	cout << g4[p] << ((p==4*(M-1))?"":", ");
  cout << "\n";
//  cout << "g8:\n";
//  for(p = 0; p <= 8*(M-1); p++)
//	cout << g8[p] << "\n";
//  cout << "\n";
  cout << "z8:";
  for(p = 0; p <= 8*(M-1); p++)
//    printf("%ld\n",z8[p]);
	cout << "\t\t.long " << z8[p] << "  \t; z8[" << p << "]\n";
  cout << "\n";
//int coords[2*240];
//uint conv_enc_input[8][8];
//uint conv_enc_tab[16][4];
  cout << "conv_enc_input:";
  for(p = 0; p < 8; p++)
    for(j = 0; j < 8; j++)
      cout << "\t\t.word " << conv_enc_input[p][j] << "  \t; c_e_i[" << p << "][" << j << "]\n";
  cout << "\nconv_enc_tab:";
  for(p = 0; p < 16; p++)
    for(j = 0; j < 4; j++)
      cout << "\t\t.word " << conv_enc_tab[p][j] << "  \t; c_e_t[" << p << "][" << j << "]\n";
  cout << "\ncoords:";
  for(p=0; p < 480; p++)
  {
    cout << (((p%10)==0) ? "\t.word " : "");
    cout.width(3);
    cout << coords[p] << (((p%10)==9) ? "\n":", ");
  }
}


ulong getdata(void)
{
  long int ch;
  long int bitstream;

  if(!(ch = getc(stdin)))
	return EOF;
  else
	bitstream = ch;
  if(!(ch = getc(stdin)))
	return bitstream;
  else
	bitstream |= (ch<<8);
  if(!(ch = getc(stdin)))
	return bitstream;
  else
	bitstream |= (ch<<16);
  if(!(ch = getc(stdin)))
	return bitstream;
  else
	bitstream |= (ch<<24);
  return bitstream;
}

void init_tables(void)
{
  long int i,k,sum;
  // init g2
  for(i = 0; i <= 2*(M-1); i++)
	g2[i] = M - abs(i-M+1);
  for(i = 2*(M-1)+1; i <= 4*(M-1); i++)
	g2[i] = 0;

  //init g4
  for(i = 0; i <= 4*(M-1); i++)
  {
	sum = 0;
	for(k = 0; k<=i; k++)
	  sum += g2[k]*g2[i-k];
	g4[i] = sum;
  }
  for(i = 4*(M-1)+1; i<=8*(M-1); i++)
	g4[i] = 0;

  // init g8
  for(i = 0; i <= 8*(M-1); i++)
  {
	sum = 0;
	for(k = 0; k <= i; k++)
	  sum += g4[k]*g4[i-k];
	g8[i] = sum;
  }

  for(i = 0; i <= 8*(M-1); i++)
  {
	sum = 0;
	for(k = 0; k <= i-1; k++)
	  sum += g8[k];
	z8[i] = sum;
  }
  #include "table.h"
  #include "conv_tab.h"
  #include "conv_tb2.h"

  for(i=0; i<11; i++)
  {
	xx[i] = 0;
	xy[i] = 0;
  }

  for(i=0; i<3;i++)
  {
	h[i].imag = 0;
  }
//  h[0].real = -24576;  // -1.5
//  h[1].real = 12288;   // .75
//  h[2].real = -2048;   // -.125
//  h[0].real = -4915;  // -.3
//  h[1].real = 492;   // .03
//  h[2].real = -164;   // -.001
//  h[0].real = -2458;  // -.15
//  h[1].real = 123;   // .0075
//  h[2].real = -2;   // -.000125
  h[0].real = -300;
  h[1].real = 42;
  h[2].real = -1;
//  h[0].real = 0;
//  h[1].real = 0;
//  h[2].real = 0;
}
