/*********************************************************************
;
; Filename: TRANSMIT.C
;
;
; Author: Chris Buehler, Jonas Keating
; Date: 10/20/95
; Last revision: 12/13/95
;
;*********************************************************************
;
; Description:  A DOS front end to control the transmitter.  Gives
;		ability to start, stop, and reset the transmitter at
;		all three different speeds.  Also enables transmission
;		of data and zooming in and out on constelation points.
;
;*********************************************************************
;
; Usage:  Load tran.out into the EVM via EVM5X and 'runf' it (allow
;	  it to run free.  Then quit EVM5X, and run TRANSMIT.EXE.
;	  There should be no result yet, but F1, F2, and F3 will
;	  initiate connections at 9600, 14400, and 19200 bps
;	  respectively.  F10 will close the connections.  Once a
;	  a connection has been established, any data typed into
;	  the TRANSMIT program will be displayed on both the
;	  transmitter and receiver side.  For a more in depth
;	  description of feature, consult the README.TXT file in this
;	  directory.
;
;*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include <string.h>

#define PAUSELEN 10
#define MEMLEN  512 //128
#define DIVFACTOR 32 //8

#define SENDXY 		0
#define READH		2
#define WRITEH		3
#define CONST_4 	4
#define CONST_16 	5
#define CONST_24 	6
#define SCALEDOWN	7
#define SCALEUP		8
#define START96		9
#define START144	10
#define START192	11
#define DIE		12
#define SENDBITS	13
#define ENDBITS		14
#define GETBITS		15


void setmode(int);
void putpix(int,int,int);
void quadplot(int,int,int);
void clearscr(void);
void send(int,int);
void waitfordata(void);
void waitforcommand(void);
void sendcommand(int);

unsigned int     CONT_REG     = (unsigned int )(0x240 + 0x804);
unsigned int     CMND_REG     = (unsigned int )(0x240 + 0x802);
unsigned int     DATA_REG     = (unsigned int )(0x240 + 0x800);
unsigned int     TDATAINT     = (unsigned int ) 0x0002;
unsigned int     TCMNDINT     = (unsigned int ) 0x0001;

int hs[6];

void main(void)
{
  int data, i;
  int x=0,y=0;
  int xx[MEMLEN],yy[MEMLEN];
  int ch=0;
  int scale = 3,speed = 0,on_hook=0;
  unsigned char bitstream[8];
  int bytecount = 0;

  for(i=0; i < MEMLEN; i++)
  {
    xx[i] = 0;
    yy[i] = 0;
  }

  setmode(19);
  send(CONT_REG,0xC200);

  while(ch!=27)
  {
    if(kbhit())
    {
      ch = getch();
      switch(ch)
      {
	/* control character hit (alt-_ or F-key) */
	case 0 :ch = getch();
		switch(ch)
		{
		  case 'h' : clearscr();
			     waitfordata();
			     send(CMND_REG,READH);
			     for(i=0; i < 6; i++)
			     {
			       waitfordata();
			       x = inport(DATA_REG);
			       printf("%d ",x);
			     }
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 'p' : waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case '1' : waitfordata();
			     send(CMND_REG,CONST_4);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case '2' : waitfordata();
			     send(CMND_REG,CONST_16);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case '3' : waitfordata();
			     send(CMND_REG,CONST_24);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 59  : waitfordata();
			     send(CMND_REG,START96);
			     speed = 2;
			     on_hook = 1;
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 60  : waitfordata();
			     send(CMND_REG,START144);
			     speed = 3;
			     on_hook = 1;
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 61  : waitfordata();
			     send(CMND_REG,START192);
			     speed = 4;
			     on_hook = 1;
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 68  : waitfordata();
			     send(CMND_REG,DIE);
			     speed = 0;
			     on_hook = 0;
			     bytecount=0;
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 'w' : waitfordata();
			     send(CMND_REG,WRITEH);
			     waitfordata();
			     printf("enter new coeffs: ");
			     scanf("%d %d %d %d %d %d",&hs[0],&hs[1],&hs[2],&hs[3],&hs[4],&hs[5]);
			     clearscr();
			     for(i=0;i<6;i++)
			     {
			       send(DATA_REG,hs[i]);
			       waitfordata();
			     }
			     send(CMND_REG,SENDXY);
			     break;
		  case ',' : waitfordata();
			     send(CMND_REG,SCALEUP);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case '.' : waitfordata();
			     send(CMND_REG,SCALEDOWN);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 80  : scale++;
			     break;
		  case 72  : if(scale > 0) scale--;
			     break;
		  case 31  : waitfordata();
			     send(CMND_REG,SENDBITS);
			     waitfordata();
			     printf("enter %d 16 bit values: ",speed);
			     for(i = 0; i < speed; i++)
			     {
			       scanf("%d",bitstream+i);
			     }
			     for(i = 0; i < speed; i++)
			     {
			       send(DATA_REG,bitstream[i]);
			       waitfordata();
			     }
			     send(CMND_REG,ENDBITS);
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;
		  case 34 : waitfordata();
			     send(CMND_REG,GETBITS);
			     for(i=0; i < speed; i++)
			     {
			       waitfordata();
			       x = inport(DATA_REG);
			       printf("%04X ",x);
			     }
			     waitfordata();
			     send(CMND_REG,SENDXY);
			     break;

		  default  : break;
		}
		break;
	/* non-control character hit */
	default : if(on_hook)
		  {  /* if on send read character */
		    printf("%c",ch);
		    bitstream[bytecount] = ch;
		    bytecount++;
		    if(ch == 13)
		    {
		      if(bytecount != speed*2)
		      {
			bitstream[bytecount] = 10;
			bytecount++;
		      }
		    }
		    if(bytecount == speed*2)  /* speed == # words sent at a time*/
		    {
		      bytecount = 0;
		      waitfordata();
		      send(CMND_REG,SENDBITS);
		      data = 1;
		      while(data)
		      {
			waitfordata();
			data = inport(DATA_REG);
		      }
		      waitfordata();
		      for(i = 0; i < speed; i++)
		      {
			data = bitstream[2*i]|((int)bitstream[2*i+1]<<8);
			send(DATA_REG,data);
			/* printf(" [%04x] ",data); */
			waitfordata();
		      }
		      send(CMND_REG,ENDBITS);
		      waitfordata();
		      send(CMND_REG,SENDXY);
		    }
		  }
		 break;
      }
    }

    if (on_hook)
    {
      waitfordata();
      x = inport(DATA_REG);
      waitfordata();
      y = inport(DATA_REG);
    }

      if(x > 0)
	x >>= scale;
      else
      {
	x = -x;
	x >>=scale;
	x = -x;
      }
      if(y > 0)
	y >>= scale;
      else
      {
	y = -y;
	y >>=scale;
	y = -y;
      }
      xx[0] = x;
      yy[0] = y;
      for(i = 0; i < MEMLEN; i++)
      {
	if(yy[i])
	  quadplot(xx[i], yy[i],30-i/DIVFACTOR+1);
	else
	{
	  quadplot(xx[i], yy[i],4);
	}

      }
      quadplot(xx[MEMLEN-1],yy[MEMLEN-1],0);
      for(i = MEMLEN-1; i >0; i--)
      {
	xx[i] = xx[i-1];
	yy[i] = yy[i-1];
      }
  }

  setmode(3);
  printf("Thank you for flying the friendly skies.\n");

}

void waitfordata(void)
{
  while(!(inport(CONT_REG) & TDATAINT)&&!kbhit());
}

void waitforcommand(void)
{
  printf("c");
  while(!(inport(CONT_REG) & TCMNDINT)&&!kbhit());
  printf("C");
}

void putpix(int x, int y, int color)
{
  char far *vidmem = (char far *) 0xA0000000L;

  *(vidmem + y*320 + x) = color;
}

void clearscr(void)
{
  char far *vidmem = (char far *) 0xA0000000L;
  _fmemset(vidmem,0,64000L);
}

void quadplot(int x, int y, int color)
{
  x += 160;
  y = 100-y;
  if((x > 0) && (x < 318) && (y>0) && (y < 198))
    putpix(x,y,color);
}

void setmode( int mode)
{
  asm mov  AX, mode
  asm int  0x10

}

void send(int reg, int data)
{
  outport(reg,data);
}

void sendcommand(int command)
{
  int test=-1;
  outport(CMND_REG,command);
  waitfordata();
  outport(CMND_REG,command);
  while(test != command)
  {
    test = inport(DATA_REG);
    waitfordata();
  }
}
