;*H*************************************************************************
; TI Proprietary Information - Internal Data
; Copyright (c) 1993-1999, Texas Instruments Incorporated
; All rights reserved
;***************************************************************************
;
; DESCRIPTION:
;
;     Truely awesome monitor program
;       for C25LP HLL Real-Time Debugger.
;
;     TI Proprietary Information - Internal Data.
;     Copyright (c) 1993-1997, Texas Instruments Incorporated.
;
;     This real-time monitor was developed by H.Roland Hoar.
;
; STATUS:
;
;     This monitor version is dated 24th of June 1997.
;     It is intended for use with HLL Debugger versions "1.19" or later.
;     The MON_RELEASE mnemonic has the hexadecimal value '1019'.
;
;     This document has page feed characters spaced for 66 lines per page.
;     Its page alignment is adjusted to ensure
;         attractive two-page-per-sheet printing.
;
;***************************************************************************
;
               .title "Monitor program for C25LP HLL Real-Time Debugger."
               .include "c200mnrt.i" ; Include conditional assembly options.
               .mmregs               ; Include standard register mnemonics.
               .length 66            ; Set the page length for the listing.
;
               .page 
;***************************************************************************
;
; EXTERNAL FUNCTIONS:
;
;      MON_SE_CNFG    SE device configuration routine.
;      MON_RT_CNFG    RT device configuration routine.
;      MON_EINTR      Interrupt vector table macro "mon_eintr_vecs".
;      MON_ETRAP      Interrupt vector table macro "mon_etrap_vecs".
;
; INTERNAL FUNCTIONS:
;
;      MON_TRAP_ENTRY Monitor emulation trap service routine.
;      MON_INTR_ENTRY Monitor emulation interrupt service routine.
;      MON_ENTRY      Entry routine shared by traps and interrupts.
;      MON_WAIT       Wait to read a debugger message.
;      MON_EVALUATE   Evaluate the type of debugger message.
;      SENDCMND, RECVCMND, DOAWRITE, SWCHOPRT, MAYBERUN:
;                     These functions, labelled "_ENTR" and "_EXIT",
;                     respond to the type and value of the message.
;                     They implement the core of monitor state machine.
;      MON_UNUSED     Process a debugger unused command.
;      MON_ADRSWR     Process a debugger write address command.
;      MON_PMRD       Process a debugger read program memory command.
;      MON_PMWR       Process a debugger write program memory command.
;      MON_DMRD       Process a debugger read data memory command.
;      MON_DMWR       Process a debugger write data memory command.
;      MON_IORD       Process a debugger read io memory command.
;      MON_IOWR       Process a debugger write io memory command.
;      MON_EXECUTE, MON_EXIT, MON_QUICKGO:
;                     Monitor exit routines to return to the application.
;
;***************************************************************************
;
               .page 
;***************************************************************************
; REALTIME MNEMONICS
;***************************************************************************

;---------------------------------------------------------------------------
; Define the PRM_CMND register masks and bits.
;---------------------------------------------------------------------------
CMNDMASK       .set   00007h ; Mask for command opcode field.
;
CMND_NONE      .set   00000h ; Address value read command opcode.
CMND_ADRS      .set   00001h ; Address value write command opcode.
CMND_PMRD      .set   00002h ; Program memory read command opcode.
CMND_PMWR      .set   00003h ; Program memory write command opcode.
CMND_DMRD      .set   00004h ; Data memory read command opcode.
CMND_DMWR      .set   00005h ; Data memory write command opcode.
CMND_IORD      .set   00006h ; IO memory read command opcode.
CMND_IOWR      .set   00007h ; IO memory write command opcode.
;
RORWMASK       .set   00001h ; Mask to distinguish read/write commands.
PORDMASK       .set   00004h ; Mask to distinguish program/data commands.
;
AINCMASK       .set   00008h ; Mask for address increment option.
;
STOPMASK       .set   00010h ; Mask for stop-mode flag.
SNGLMASK       .set   00020h ; Mask for single-step flag.
EVERMASK       .set   00040h ; Mask for forever-run flag.
;
MAPSMASK       .set   00080h ; Mask for map register usage option.
;
B_RORWMASK     .set      00h ; Bit number of the read/write command mask.
B_PORDMASK     .set      02h ; Bit number of the program/data command mask.
;
B_AINCMASK     .set      03h ; Bit number of the address increment option.
;
B_STOPMASK     .set      04h ; Bit number of the stop-mode flag.
B_SNGLMASK     .set      05h ; Bit number of the single-step flag.
B_EVERMASK     .set      06h ; Bit number of the forever-run flag.
;
B_MAPSMASK     .set      07h ; Bit number of the map register usage option.
;
               .page 
;---------------------------------------------------------------------------
; Define the MON_STATUS register masks and bits.
;---------------------------------------------------------------------------
ATN_OK         .set   00000h ; Value for no outstanding status.
ATN_RESET      .set   00001h ; Mask for reset status.
ATN_SWBP       .set   0000Ch ; Mask for software breakpoint status.
ATN_TRAP       .set   00080h ; Mask for trap error.
;
;---------------------------------------------------------------------------
; Define the ST0 register masks used for manipulating public bits.
;---------------------------------------------------------------------------
USR_INT_DSB    .set   00200h ; Mask for ST0 user interrupt mode bit.
;
;---------------------------------------------------------------------------
; Define the ST1 register masks used for manipulating hidden bits.
;---------------------------------------------------------------------------
SNG_STP_DSB    .set   00020h ; Mask for ST1 single step mode disable.
GLB_INT_ENB    .set   00080h ; Mask for ST1 global interrupt enable.
;
;---------------------------------------------------------------------------
; Define the MON_AR value to be used (choice of AR7-0 register)
;---------------------------------------------------------------------------
MON_AR         .set   MON_AR_VALUE & 00007h ;
;
;---------------------------------------------------------------------------
; Define the IFR register masks and bits.
;---------------------------------------------------------------------------
MON_EMUMSK     .set   00040h ; Mask for emulation interrupt.
;
B_EMUMSK       .set      06h ; Bit number of emulation interrupt.
;
;---------------------------------------------------------------------------
; Define the IMR register masks and bits.
;---------------------------------------------------------------------------
               .eval  0, MON_USER_INT
               .eval  (MON_USER_INT1 << 0) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT2 << 1) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT3 << 2) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT4 << 3) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT5 << 4) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT6 << 5) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT7 << 6) | MON_USER_INT, MON_USER_INT
               .eval  (MON_USER_INT8 << 7) | MON_USER_INT, MON_USER_INT
MON_INT_MSK    .set   MON_USER_INT
;
               .page 
;---------------------------------------------------------------------------
; Define the analysis and message passing control registers addresses.
;
; MSG_DATA is used to access data via the message passing register.
; MSG_CMND is used to access command via the message passing register.
; MSG_STAT is used to check message status.
; MSG_CNTL is used to configure analysis and message passing.
;---------------------------------------------------------------------------
MSG_DATA       .set     1Ch  ; Analysis data register.
MSG_CMND       .set     1Dh  ; Analysis command register.
MSG_STAT       .set     1Eh  ; Analysis status register.
MSG_CNTL       .set     1Fh  ; Analysis control register.
;
;---------------------------------------------------------------------------
; Define the MSG_STAT register masks and bits.
;---------------------------------------------------------------------------
STAT_XFER      .set     01h  ; Mask for good transfer.
STAT_RFUL      .set     02h  ; Mask for read buffer full.
STAT_CMND      .set     04h  ; Mask for command type message.
STAT_WFUL      .set     08h  ; Mask for write buffer full.
STAT_ANYBP     .set     10h  ; Mask for message/breakpoint.
STAT_STRP      .set     20h  ; Mask for strap mode.
STAT_FUNC      .set     40h  ; Mask for functional mode.
;
B_STATXFER     .set       0  ; Bit number of good transfer.
B_STATRFUL     .set       1  ; Bit number of read buffer full.
B_STATCMND     .set       2  ; Bit number of command type message.
B_STATWFUL     .set       3  ; Bit number of write buffer full.
B_STATANYBP    .set       4  ; Bit number of message/breakpoint.
B_STATSTRP     .set       5  ; Bit number of strap mode.
B_STATFUNC     .set       6  ; Bit number of functional mode.
;
               .page 
;---------------------------------------------------------------------------
; Define the MSG_CNTL register masks and bits.
;---------------------------------------------------------------------------
ANA_TRPON      .set    001h  ; Trap On.
                             ; When set, enables analysis.
ANA_UBP0       .set    002h  ; User Breakpoint Bit #0.
ANA_UBP1       .set    004h  ; User Breakpoint Bit #1.
                             ; 00 => reserved.
                             ; 01 => create load mechanism.
                             ; 10 => analysis in wait state,
                             ;       disallow write to message registers.
                             ; 11 => analysis in run state,
                             ;       allow write to message registers.
ANA_MSGFLG     .set    008h  ; Message Flag.
                             ; Used as simple flag from monitor to debugger.
ANA_SWITCH     .set    010h  ; Message Switch.
                             ; Toggling this bit high then low
                             ; generates an event to the JTM.
ANA_SPOFF      .set    020h  ; Suspend Off.
                             ; When set, allows access the message passing
                             ; registers when not in ETRAP routine.
ANA_TPOFF      .set    040h  ; Analysis NMI Off.
                             ; When set, disables the analysis NMI.
;
B_ANATRPON     .set       0  ; Bit number of trap on.
B_ANAUBP0      .set       1  ; Bit number of user breakpoint bit 0.
B_ANAUBP1      .set       2  ; Bit number of user breakpoint bit 1.
B_ANAMSGFLG    .set       3  ; Bit number of message flag.
B_ANASWITCH    .set       4  ; Bit number of message switch.
B_ANASPOFF     .set       5  ; Bit number of suspend off.
B_ANATPOFF     .set       6  ; Bit number of trap off.
;
;---------------------------------------------------------------------------
; Define the MSG_CNTL register values that are to be actually used.
;---------------------------------------------------------------------------
ANA_DEFAULT    .set ANA_TPOFF | ANA_SPOFF | ANA_UBP1 | ANA_UBP0 | ANA_TRPON
ANA_SWLO       .set ANA_DEFAULT | 000h ; ANA_SWITCH low.
ANA_SWHI       .set ANA_DEFAULT | 010h ; ANA_SWITCH high.
;
               .page 
;---------------------------------------------------------------------------
; Define the MON_OPTION1 mnemonic.
;---------------------------------------------------------------------------
               .eval  0, OPTION1
               .eval  (MON_INT_MSK  << 00h) | OPTION1, OPTION1 ; uses 8 bits
;
               .eval  (0            << 08h) | OPTION1, OPTION1 ; unused
               .eval  (0            << 09h) | OPTION1, OPTION1 ; unused
               .eval  (MON_VECTOR   << 0Ah) | OPTION1, OPTION1 ; uses 1 bit
               .eval  (MON_INT_ALTR << 0Bh) | OPTION1, OPTION1 ; uses 1 bit
;
               .eval  (MON_RGST << 0Ch) | OPTION1, OPTION1 ; uses 1 bit
               .eval  (MON_AR   << 0Dh) | OPTION1, OPTION1 ; uses 3 bits
               ;eval  (MON_AR   << 0Eh) | OPTION1, OPTION1 ; uses 3 bits
               ;eval  (MON_AR   << 0Fh) | OPTION1, OPTION1 ; uses 3 bits
MON_OPTION1    .set   OPTION1
;
;---------------------------------------------------------------------------
; Define the MON_OPTION2 mnemonic.
;---------------------------------------------------------------------------
               .eval  0, OPTION2
               .eval  MON_IO_MAP_USE    | OPTION2, OPTION2 ; uses 2 bits
               ;eval  MON_IO_MAP_USE    | OPTION2, OPTION2 ; uses 2 bits
               .eval  MON_AUTO_INC      | OPTION2, OPTION2 ; uses 2 bits
               ;eval  MON_AUTO_INC      | OPTION2, OPTION2 ; uses 2 bits
;
               .eval  MON_ERROR_CHECK   | OPTION2, OPTION2 ; uses 2 bits
               ;eval  MON_ERROR_CHECK   | OPTION2, OPTION2 ; uses 2 bits
               .eval  MON_RESET_MESG    | OPTION2, OPTION2 ; uses 2 bits
               ;eval  MON_RESET_MESG    | OPTION2, OPTION2 ; uses 2 bits
;
               .eval  MON_CORE          | OPTION2, OPTION2
               .eval  MON_FULL          | OPTION2, OPTION2
               .eval  MON_SE_CNFG_USE   | OPTION2, OPTION2
               .eval  MON_WAIT_USE      | OPTION2, OPTION2
;
               .eval  MON_WAIT_STOP     | OPTION2, OPTION2
               .eval  MON_WAIT_SNGL     | OPTION2, OPTION2
               .eval  MON_WAIT_EVER     | OPTION2, OPTION2
               .eval  (0        << 0Fh) | OPTION2, OPTION2 ; unused
MON_OPTION2    .set   OPTION2
;
               .page 
;---------------------------------------------------------------------------
; Define the MON_OPTION3 mnemonic.
;---------------------------------------------------------------------------
               .eval  0, OPTION3
               .eval  MON_PMAP_PAGE     | OPTION3, OPTION3 ; uses 2 bits
               ;eval  MON_PMAP_PAGE     | OPTION3, OPTION3 ; uses 2 bits
               .eval  MON_XTND_PROG     | OPTION3, OPTION3
               .eval  (0        << 03h) | OPTION3, OPTION3 ; unused
;
MON_OPTION3    .set   OPTION3
;
;---------------------------------------------------------------------------
; Define the MON_OPTION4 mnemonic.
;---------------------------------------------------------------------------
               .eval  0, OPTION4
               .eval  MON_DMAP_PAGE     | OPTION4, OPTION4 ; uses 2 bits
               ;eval  MON_DMAP_PAGE     | OPTION4, OPTION4 ; uses 2 bits
               .eval  MON_XTND_DATA     | OPTION4, OPTION4
               .eval  (0        << 03h) | OPTION4, OPTION4 ; unused
;
MON_OPTION4    .set   OPTION4
;
               .page 
;***************************************************************************
; REALTIME SECTIONS
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define the table of addresses for the
; external commands used by the HLL Debugger.
;
; This table contain one address parameter per external command.
; It is the start point of the routine that implements the command.
;
; Note that "_A" signifies an address table mnemonic.
; Note that "_C" signifies an state machine command mnemonic.
; Note that "_S" signifies an state machine state mnemonic.
;---------------------------------------------------------------------------
EXTERNAL_TABLE .sect "mon_main"
;
UNUSED_A       .word MON_UNUSED    ; Unused Command.
;
A2MNRT_A       .word MON_ADRSWR    ; Write Monitor Address
;
PM2HOST_A      .word MON_PMRD      ; Read Program Memory
;
PM2MNRT_A      .word MON_PMWR      ; Write Program Memory
;
DM2HOST_A      .word MON_DMRD      ; Read Data Memory
;
DM2MNRT_A      .word MON_DMWR      ; Write Data Memory
                .if (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
IO2HOST_A      .word MON_IORD      ; Read IO Memory
;
IO2MNRT_A      .word MON_IOWR      ; Write IO Memory
               .endif ; (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
;
;---------------------------------------------------------------------------
; Define the set of external commands used by the HLL Debugger.
;
; These command values intentionally increment by ones.
;---------------------------------------------------------------------------
UNUSED_C       .set  UNUSED_A  - EXTERNAL_TABLE ;
;
A2MNRT_C       .set  A2MNRT_A  - EXTERNAL_TABLE ;
;
PM2HOST_C      .set  PM2HOST_A - EXTERNAL_TABLE ;
;
PM2MNRT_C      .set  PM2MNRT_A - EXTERNAL_TABLE ;
;
DM2HOST_C      .set  DM2HOST_A - EXTERNAL_TABLE ;
;
DM2MNRT_C      .set  DM2MNRT_A - EXTERNAL_TABLE ;
               .if (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
IO2HOST_C      .set  IO2HOST_A - EXTERNAL_TABLE ;
;
IO2MNRT_C      .set  IO2MNRT_A - EXTERNAL_TABLE ;
               .endif ; (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
;
               .page 
;---------------------------------------------------------------------------
; Define the table of addresses for the routines that implement
; the internal commands used by the monitor itself.
;
; This table contain two address parameters per internal command.
; The start point for entry into the routine is the second parameter.
; The return point for exit from the routine is the first parameter.
; The reverse order is used to shave a few words from the monitor's size.
; 
; Note that "_A" signifies an address table mnemonic.
; Note that "_C" signifies an state machine command mnemonic.
; Note that "_S" signifies an state machine state mnemonic.
;---------------------------------------------------------------------------
INTERNAL_TABLE .sect "mon_main"
;
SENDCMND_A     .word MON_WAIT      ; Monitor send command
               .word SENDCMND_ENTR ; and send status values.
;
RECVCMND_A     .word RECVCMND_EXIT ; Monitor receive command
               .word RECVCMND_ENTR ; and use a read command.
;
DOAWRITE_A     .word DOAWRITE_EXIT ; Use a write command.
               .word NOACTION_ENTR ; 
;
SWCHOPRT_A     .word MON_WAIT      ;
               .word SWCHOPRT_ENTR ; Operate the message switch.
;
MAYBERUN_A     .word MAYBERUN_EXIT ;
               .word NOACTION_ENTR ; Possibly exit the monitor.
;
;---------------------------------------------------------------------------
; Define the set of internal commands used by the monitor itself.
;
; These command values intentionally increment by twos.
;---------------------------------------------------------------------------
SENDCMND_C     .set  SENDCMND_A - INTERNAL_TABLE ;
;
RECVCMND_C     .set  RECVCMND_A - INTERNAL_TABLE ;
;
DOAWRITE_C     .set  DOAWRITE_A - INTERNAL_TABLE ;
;
SWCHOPRT_C     .set  SWCHOPRT_A - INTERNAL_TABLE ;
;
MAYBERUN_C     .set  MAYBERUN_A - INTERNAL_TABLE ;
;
               .page 
;---------------------------------------------------------------------------
; Define the four monitor states and the possible actions taken during
; each monitor state, based on four permutations of the status of the
; MSG_DATA and MSG_CMND buffers. The status of the buffers is taken from
; the STAT_WFUL, STAT_CMD and STAT_RFUL bits in the MSG_STAT register.
;
; Each five line ".eval" defines a monitor state and its possible actions.
;
; Line 1 declares the mnemonic for the state.
; Line 2 is nibble 3, the buffer has something from the monitor to the host.
; Line 3 is nibble 2, the buffer has a command to the monitor.
; Line 4 is nibble 1, the buffer has a datum to the monitor.
; Line 5 is nibble 0, the buffer is empty.
;
; Note that "_A" signifies an table entry address mnemonic.
; Note that "_C" signifies an state machine command mnemonic.
; Note that "_S" signifies an state machine state mnemonic.
;---------------------------------------------------------------------------
;
;---------------------------------------------------------------------------
; Define the AFTERREAD_S state.  This is the monitor state after
; accepting an external read command from the HLL Debugger.
;---------------------------------------------------------------------------
               .eval 0, AFTERREAD_S
               .eval (SWCHOPRT_C << 12) | AFTERREAD_S, AFTERREAD_S
               .eval (RECVCMND_C <<  8) | AFTERREAD_S, AFTERREAD_S
               .eval (SWCHOPRT_C <<  4) | AFTERREAD_S, AFTERREAD_S
               .eval (MAYBERUN_C      ) | AFTERREAD_S, AFTERREAD_S
;
;---------------------------------------------------------------------------
; Define the AFTERWRIT_S state.  This is the monitor state after
; accepting an external write command from the HLL Debugger.
;---------------------------------------------------------------------------
               .eval 0, AFTERWRIT_S
               .eval (SWCHOPRT_C << 12) | AFTERWRIT_S, AFTERWRIT_S
               .eval (RECVCMND_C <<  8) | AFTERWRIT_S, AFTERWRIT_S
               .eval (DOAWRITE_C <<  4) | AFTERWRIT_S, AFTERWRIT_S
               .eval (MAYBERUN_C      ) | AFTERWRIT_S, AFTERWRIT_S
;
;---------------------------------------------------------------------------
; Define the AFTEREROR_S state.  This is the monitor state after an error
; has occurred, when all external commands from the HLL Debugger are
; discarded, until a status message is passed to the HLL debugger.
;---------------------------------------------------------------------------
               .eval 0, AFTEREROR_S
               .eval (SWCHOPRT_C << 12) | AFTEREROR_S, AFTEREROR_S
               .eval (SWCHOPRT_C <<  8) | AFTEREROR_S, AFTEREROR_S
               .eval (SWCHOPRT_C <<  4) | AFTEREROR_S, AFTEREROR_S
               .eval (SENDCMND_C      ) | AFTEREROR_S, AFTEREROR_S
;
;---------------------------------------------------------------------------
; Define the AFTERSTAT_S state.  This is the monitor state after an error
; has occurred and a status message has been passed to the HLL debugger.
; The monitor is ready to accept an external command from the HLL Debugger.
;---------------------------------------------------------------------------
               .eval 0, AFTERSTAT_S
               .eval (SWCHOPRT_C << 12) | AFTERSTAT_S, AFTERSTAT_S
               .eval (RECVCMND_C <<  8) | AFTERSTAT_S, AFTERSTAT_S
               .eval (SWCHOPRT_C <<  4) | AFTERSTAT_S, AFTERSTAT_S
               .eval (SENDCMND_C      ) | AFTERSTAT_S, AFTERSTAT_S
;
               .page 
;***************************************************************************
; This page is intentionally left blank.
;***************************************************************************
;
               .page 
;***************************************************************************
; REALTIME SECTIONS
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define the "mon_pge0" section in data memory.
;
; Note that "mon_pge0" is defined in the link command file.
; It can be put anywhere in page zero data memory to suit the application.
;---------------------------------------------------------------------------
PRM_CMND       .usect  "mon_pge0", 1  ; Message command parameter.
PRM_ADRS       .usect  "mon_pge0", 1  ; Message base address parameter.
               .if (MON_XTND_EITHER == MON_XTND_ON)
PRM_XTND       .usect  "mon_pge0", 1  ; Message extended address parameter.
               .else  ; (MON_XTND_EITHER == MON_XTND_OFF)
PRM_XTND       .set     0             ; Unused.
               .endif ; (MON_XTND_EITHER)
;
MON_MACHINE    .usect  "mon_pge0", 1  ; Monitor state machine value.
MON_STATUS     .usect  "mon_pge0", 1  ; Monitor status.
MON_TRAP_VEC   .usect  "mon_pge0", 1  ; Remappable vector 0x0026 address.
;
               .if (MON_XTND_EITHER == MON_XTND_ON)
MON_TWO_SAVE   .usect  "mon_pge0", 1  ; Saved bits from IMR/ST0 register.
;              .asg     MON_TWO_SAVE, MON_IMR_SAVE ; This is in "c200mnrt.i".
;              .asg     MON_TWO_SAVE, MON_ST0_SAVE ; This is in "c200mnrt.i".
               .else  ; (MON_XTND_EITHER == MON_XTND_OFF)
MON_TWO_SAVE   .set     0             ; Unused.
MON_IMR_SAVE   .usect  "mon_pge0", 1  ; Saved bits from IMR register.
MON_ST0_SAVE   .usect  "mon_pge0", 1  ; Saved bits from ST0 register.
               .endif ; (MON_XTND_EITHER)
;
               .if (MON_WAIT_USE == MON_WAIT_USE_YES)
MON_WAIT_B2    .usect  "mon_pge0", 1  ; Stop-sign flag.
               .else  ; (MON_WAIT_USE == MON_WAIT_USE_NOT)
MON_WAIT_B2    .set     0             ; Unused.
               .endif ; (MON_WAIT_USE)
;
MON_TEMP0      .usect  "mon_pge0", 1  ; General purpose temporary storage.
MON_TEMP1      .usect  "mon_pge0", 1  ; General purpose temporary storage.
MON_TEMP2      .usect  "mon_pge0", 1  ; General purpose temporary storage.
MON_TEMP3      .usect  "mon_pge0", 1  ; General purpose temporary storage.
MON_TEMP4      .usect  "mon_pge0", 1  ; General purpose temporary storage.
               .if (MON_XTND_EITHER == MON_XTND_ON)
MON_TEMP5      .usect  "mon_pge0", 1  ; General purpose temporary storage.
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
;
MON_TEMP_PC    .set MON_TEMP0 ; Temporary for PC in ETRAP & EINTR macros.
MON_TEMP_ST0   .set MON_TEMP1 ; Temporary for ST0 in ETRAP & EINTR macros.
MON_TEMP_ST1   .set MON_TEMP2 ; Temporary for ST1 in ETRAP & EINTR macros.
;
MON_TEMP_ACCL  .set MON_TEMP3 ; Temporary for ACCL/H after monitor entry,
MON_TEMP_ACCH  .set MON_TEMP4 ; inside EINTR macro and after ETRAP macro.
;
ENTR_ADRS      .set MON_TEMP3 ; Temporary for the entry and exit addresses
EXIT_ADRS      .set MON_TEMP4 ; used by the internal action routines.
;
               .if (MON_XTND_EITHER == MON_XTND_ON)
XTND_ADRS      .set MON_TEMP5 ; Temporary for PMAP/DMAP register values.
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
;
MON_PGE0_DP    .set PRM_CMND  ; DP value for "mon_pge0" section.
;
               .page 
;---------------------------------------------------------------------------
; Define the "mon_rgst" section in data memory.
;
; Note that "mon_rgst" is defined in the link command file.
; It can be put anywhere in data memory to suit the application.
;
; The ordering is defined by the HLL Debugger and must *not* be altered.
;---------------------------------------------------------------------------
MON_PC         .usect  "mon_rgst", 1  ; 0x00 program counter.
MON_ST0        .usect  "mon_rgst", 1  ; 0x01 status register 0.
MON_ST1        .usect  "mon_rgst", 1  ; 0x02 status register 1.
MON_ACCL       .usect  "mon_rgst", 1  ; 0x03 accumulator low.
MON_ACCH       .usect  "mon_rgst", 1  ; 0x04 accumulator high.
;
MON_PLO        .usect  "mon_rgst", 1  ; 0x05 product register low.
MON_PHI        .usect  "mon_rgst", 1  ; 0x06 product register high.
MON_TREG       .usect  "mon_rgst", 1  ; 0x07 temporary register.
MON_TOS        .usect  "mon_rgst", 1  ; 0x08 top of stack.
;
AUXREG0        .usect  "mon_rgst", 1  ; 0x09 auxiliary register 0.
AUXREG1        .usect  "mon_rgst", 1  ; 0x0a auxiliary register 1.
AUXREG2        .usect  "mon_rgst", 1  ; 0x0b auxiliary register 2.
AUXREG3        .usect  "mon_rgst", 1  ; 0x0c auxiliary register 3.
AUXREG4        .usect  "mon_rgst", 1  ; 0x0d auxiliary register 4.
AUXREG5        .usect  "mon_rgst", 1  ; 0x0e auxiliary register 5.
AUXREG6        .usect  "mon_rgst", 1  ; 0x0f auxiliary register 6.
AUXREG7        .usect  "mon_rgst", 1  ; 0x10 auxiliary register 7.
;
               .if (MON_XTND_EITHER == MON_XTND_ON)
MON_PMAP       .usect  "mon_rgst", 1  ; 0x11 program map register.
MON_DMAP       .usect  "mon_rgst", 1  ; 0x12 data map register.
               .else  ; (MON_XTND_EITHER == MON_XTND_OFF)
MON_PMAP       .set     0             ; Unused.
MON_DMAP       .set     0             ; Unused.
               .endif ; (MON_XTND_EITHER)
;
MON_RGST_DP    .set MON_PC            ; DP value for "mon_rgst" section.
;
               .page 
;---------------------------------------------------------------------------
; Define the "mon_ioio" section in program memory that is used as part
; of the MON_IO_MAP_USE_RAM option so as to access the IO memory space.
;
; Note that "mon_ioio" is defined in the link command file.
; It can be put anywhere in RAM program memory to suit the application.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
MON_IO_RUN     .usect   "mon_ioio", 3   ; RAM program memory.
MON_IORD       .set MON_IO_RUN
MON_IOWR       .set MON_IO_RUN
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
;
;---------------------------------------------------------------------------
; Define the "mon_b0p0" and "mon_b0p1" sections in program and data memory
; that are used as part of the MON_IO_MAP_USE_ROM option so as to access
; the IO memory space.
;
; Note that "mon_b0p0" and "mon_b0p0" are defined in the link command file.
; They can be put anywhere in Bank #0 to suit the application.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
MON_B0_RUN     .usect   "mon_b0p0", 3   ; Bank #0 high program memory.
MON_B0_LOAD    .usect   "mon_b0p1", 3   ; Bank #0 low data memory.
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
;
               .page 
;***************************************************************************
; This page is intentionally left blank.
;***************************************************************************
;
               .page 
;***************************************************************************
; MISCELLANEOUS TABLES AND CODE FOR THE SECTION "MON_MAIN"
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define the monitor table of constant values in "mon_main" program
; memory and the related mnemonics MON_RELEASE and MON_POINTER that
; the HLL debugger expects mnemonics to be located at MON_ROOT
; (addresses 0x002C and 0x002D in program memory).
;---------------------------------------------------------------------------
               .sect "mon_main"
FIX_ETRAP      mon_etrap            ;
FIX_ESTOP      mon_estop            ;
FIX_REGISTER   .word  MON_RGST_DP   ;
FIX_OPTION1    .word  MON_OPTION1   ;
FIX_OPTION2    .word  MON_OPTION2   ;
FIX_PAGE_ZRO   .word  MON_PGE0_DP   ;
FIX_OPTION3    .word  MON_OPTION3   ;
FIX_OPTION4    .word  MON_OPTION4   ;
FIX_PMAP_ADDR  .word  MON_PMAP_ADDR ;
FIX_DMAP_ADDR  .word  MON_DMAP_ADDR ;
;
MON_RELEASE    .set   01019h      ; The release number of this monitor.
MON_POINTER    .set   FIX_ETRAP   ; Pointer to the table of constant values.
;
MON_START      .set   FIX_ETRAP   ; These two mnemonics are no longer used,
MON_STOP       .set   FIX_ESTOP   ; they exist for backwards compatibility.
;
;---------------------------------------------------------------------------
; Define the monitor table of SE device's memory map
; configuration values in "mon_main" program memory.
;
; This table is used by the MON_SE_CNFG routine to configure
; the SE device's memory map after a reset has occured.
;---------------------------------------------------------------------------
               .if (MON_SE_CNFG_USE == MON_SE_CNFG_USE_YES)
MON_SE_VALU    .sect "mon_main"
               .word  00000H ; 07f0 disable loading
               .word  08000H ; 07f0 enable loading
               .word  0A514H ; 07f1
               .word  00000H ; 07f2
               .word  00000H ; 07f3
               .word  00000H ; 07f4
               .word  00000H ; 07f5
               .word  00000H ; 07f6
               .word  00000H ; 07f7
               .word  00000H ; 07f8
               .word  02000H ; 07f9
               .word  08C88H ; 07fa
               .word  00000H ; 07fb
               .word  00000H ; 07fc
               .word  00000H ; 07fd
               .word  0FFFFH ; 07fe
               .word  0FFFFH ; 07ff
;
MON_SE_ADRS    .set   007F0H ; Address of memory configuration registers.
               .endif ; (MON_SE_CNFG_USE == MON_SE_CNFG_USE_YES)
;
               .page 
;---------------------------------------------------------------------------
; Define the MON_EINTR_VECS and MON_ETRAP_VECS macros to be located
; in "mon_main" program memory.
;
; If the expression "MON_VECTOR == MON_VECTOR_BRANCH" is true then the
; monitor is configured to have its MON_EINTR_VECS and MON_ETRAP_VECS
; macros placed directly in the monitor code. The corresponding entries
; in the interrupt vector table at the bottom of program memory contain
; the first of a series of branch instructions that must eventually lead
; to these macros. Read the description of the MON_VECTOR conditional
; assembly option in the monitor's include file for a detailed
; explanation of what this is all about.
;---------------------------------------------------------------------------
               .if (MON_VECTOR == MON_VECTOR_BRANCH)
               .sect "mon_main"
MON_EINTR      mon_eintr_vecs        ; This occupies fourteen words.
MON_ETRAP      mon_etrap_vecs        ; This occupies eight words.
               .endif
;
               .page 
;***************************************************************************
; REALTIME MACROS
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define MON_SAVE_FULL macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_save_full  .macro
               .if (MON_RGST == MON_RGST_ANY)
               LDP    #MON_RGST_DP  ; Set DP to the register table.
               .endif
;
               POPD    MON_TOS      ; save TOS
               PSHD    MON_TOS      ;
;
               SPM     0            ;
               SPL     MON_PLO      ; save PLO
               SPH     MON_PHI      ; save PHI
               MPYK    1            ;
               SPL     MON_TREG     ; save TREG
               LT      MON_PLO      ;
               MPYK    1            ; restore PLO
               LPH     MON_PHI      ; restore PHI
               LT      MON_TREG     ; restore TREG
;
               SAR     AR0, AUXREG0 ; save AR0
               SAR     AR1, AUXREG1 ; save AR1
               SAR     AR2, AUXREG2 ; save AR2
               SAR     AR3, AUXREG3 ; save AR3
               SAR     AR4, AUXREG4 ; save AR4
               SAR     AR5, AUXREG5 ; save AR5
               SAR     AR6, AUXREG6 ; save AR6
               SAR     AR7, AUXREG7 ; save AR7
;
               .if (MON_XTND_PROG == MON_XTND_PROG_ON)
               .if (MON_PMAP_PAGE == MON_PMAP_PAGE_PROG)
               LALK    #MON_PMAP_ADDR           ;
               TBLR    MON_PMAP                 ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_DATA)
               BLDD    #MON_PMAP_ADDR, MON_PMAP ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_IO)
               IN      MON_PMAP, MON_PMAP_ADDR  ;
               .endif ; (MON_PMAP_PAGE)
               .endif ; (MON_XTND_PROG)
;
               .if (MON_XTND_DATA == MON_XTND_DATA_ON)
               .if (MON_DMAP_PAGE == MON_DMAP_PAGE_PROG)
               LALK    #MON_DMAP_ADDR           ;
               TBLR    MON_DMAP                 ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_DATA)
               BLDD    #MON_DMAP_ADDR, MON_DMAP ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_IO)
               IN      MON_DMAP, MON_DMAP_ADDR  ;
               .endif ; (MON_DMAP_PAGE)
               .endif ; (MON_XTND_DATA)
;
               .if (MON_RGST == MON_RGST_ANY)
               LDP    #MON_PGE0_DP  ; Set DP to page zero.
               .endif
               .endm  ; mon_save_full
;
               .page 
;---------------------------------------------------------------------------
; Define MON_REST_FULL macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_rest_full  .macro
               POP                    ;
               PSHD    MON_TOS        ; restore TOS
;
               LT      MON_PLO        ;
               MPYK    1              ; restore PLO
               LPH     MON_PHI        ; restore PHI
               LT      MON_TREG       ; restore TREG
;
               LAR     AR0, AUXREG0   ; restore AR0
               LAR     AR1, AUXREG1   ; restore AR1
               LAR     AR2, AUXREG2   ; restore AR2
               LAR     AR3, AUXREG3   ; restore AR3
               LAR     AR4, AUXREG4   ; restore AR4
               LAR     AR5, AUXREG5   ; restore AR5
               LAR     AR6, AUXREG6   ; restore AR6
               LAR     AR7, AUXREG7   ; restore AR7
;
               .if (MON_XTND_PROG == MON_XTND_PROG_ON)
               .if (MON_PMAP_PAGE == MON_PMAP_PAGE_PROG)
               LALK   #MON_PMAP_ADDR           ;
               TBLW   MON_PMAP                 ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_DATA)
               BLDD   MON_PMAP, #MON_PMAP_ADDR ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_IO)
               OUT    MON_PMAP, MON_PMAP_ADDR  ;
               .endif ; (MON_PMAP_PAGE)
               .endif ; (MON_XTND_PROG)
;
               .if (MON_XTND_DATA == MON_XTND_DATA_ON)
               .if (MON_DMAP_PAGE == MON_DMAP_PAGE_PROG)
               LALK   #MON_DMAP_ADDR           ;
               TBLW   MON_DMAP                 ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_DATA)
               BLDD   MON_DMAP, #MON_DMAP_ADDR ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_IO)
               OUT    MON_DMAP, MON_DMAP_ADDR  ;
               .endif ; (MON_DMAP_PAGE}
               .endif ; (MON_XTND_DATA)
               .endm  ; mon_rest_full
;
               .page 
;---------------------------------------------------------------------------
; Define MON_LOAD_NEW macro.
; Save the appropriate MON_PMAP or MON_DMAP value into XTND_ADRS.
; Load the new PRM_XTND value value into the appropriate map register.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_load_new   .macro
               .if (MON_XTND_EITHER == MON_XTND_ON)
               BIT    PRM_CMND, 15-B_MAPSMASK   ;
               BBZ    LOAD_NEW_DONE             ;
;
               .if (MON_XTND_BOTH == MON_XTND_ON)
               BIT    PRM_CMND, 15-B_PORDMASK   ;
               BBNZ   LOAD_NEW_DATA             ;
LOAD_NEW_PROG                                   ;
               .endif ; (MON_XTND_BOTH == MON_XTND_ON)
;
               .if (MON_XTND_PROG == MON_XTND_PROG_ON)
               .if (MON_PMAP_PAGE == MON_PMAP_PAGE_PROG)
               LALK   #MON_PMAP_ADDR            ;
               TBLR   XTND_ADRS                 ;
               TBLW   PRM_XTND                  ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_DATA)
               BLDD   #MON_PMAP_ADDR, XTND_ADRS ;
               BLDD   PRM_XTND, #MON_PMAP_ADDR  ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_IO)
               IN     XTND_ADRS, MON_PMAP_ADDR  ;
               OUT    PRM_XTND, MON_PMAP_ADDR   ;
               .endif ; (MON_PMAP_PAGE)
               .endif ; (MON_XTND_PROG == MON_XTND_PROG_ON)
;
               .if (MON_XTND_BOTH == MON_XTND_ON)
               B      LOAD_NEW_DONE             ;
LOAD_NEW_DATA                                   ;
               .endif ; (MON_XTND_BOTH == MON_XTND_ON)
;
               .if (MON_XTND_DATA == MON_XTND_DATA_ON)
               .if (MON_DMAP_PAGE == MON_DMAP_PAGE_PROG)
               LALK   #MON_DMAP_ADDR            ;
               TBLR   XTND_ADRS                 ;
               TBLW   PRM_XTND                  ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_DATA)
               BLDD   #MON_DMAP_ADDR, XTND_ADRS ;
               BLDD   PRM_XTND, #MON_DMAP_ADDR  ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_IO)
               IN     XTND_ADRS, MON_DMAP_ADDR  ;
               OUT    PRM_XTND, MON_DMAP_ADDR   ;
               .endif ; (MON_DMAP_PAGE)
               .endif ; (MON_XTND_DATA == MON_XTND_DATA_ON)
;
LOAD_NEW_DONE
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
               .endm  ; mon_load_new
;
               .page 
;---------------------------------------------------------------------------
; Define MON_LOAD_OLD macro.
; Load the old MON_PMAP or MON_DMAP value into the appropriate map register.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_load_old   .macro
               .if (MON_XTND_EITHER == MON_XTND_ON)
               BIT    PRM_CMND, 15-B_MAPSMASK   ;
               BBZ    LOAD_OLD_DONE             ;
;
               .if (MON_XTND_BOTH == MON_XTND_ON)
               BIT    PRM_CMND, 15-B_PORDMASK   ;
               BBNZ   LOAD_OLD_DATA             ;
LOAD_OLD_PROG                                   ;
               .endif ; (MON_XTND_BOTH == MON_XTND_ON)
;
               .if (MON_XTND_PROG == MON_XTND_PROG_ON)
               .if (MON_PMAP_PAGE == MON_PMAP_PAGE_PROG)
               LALK   #MON_PMAP_ADDR            ;
               TBLW   XTND_ADRS                 ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_DATA)
               BLDD   XTND_ADRS, #MON_PMAP_ADDR ;
               .elseif (MON_PMAP_PAGE == MON_PMAP_PAGE_IO)
               OUT    XTND_ADRS, MON_PMAP_ADDR  ;
               .endif ; (MON_PMAP_PAGE)
               .endif ; (MON_XTND_PROG == MON_XTND_PROG_ON)
;
               .if (MON_XTND_BOTH == MON_XTND_ON)
               B      LOAD_OLD_DONE             ;
LOAD_OLD_DATA                                   ;
               .endif ; (MON_XTND_BOTH == MON_XTND_ON)
;
               .if (MON_XTND_DATA == MON_XTND_DATA_ON)
               .if (MON_DMAP_PAGE == MON_DMAP_PAGE_PROG)
               LALK   #MON_DMAP_ADDR            ;
               TBLW   XTND_ADRS                 ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_DATA)
               BLDD   XTND_ADRS, #MON_DMAP_ADDR ;
               .elseif (MON_DMAP_PAGE == MON_DMAP_PAGE_IO)
               OUT    XTND_ADRS, MON_DMAP_ADDR  ;
               .endif ; (MON_DMAP_PAGE}
               .endif ; (MON_XTND_DATA == MON_XTND_DATA_ON)
;
LOAD_OLD_DONE
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
               .endm  ; mon_load_old
;
               .page 
;---------------------------------------------------------------------------
; Define MON_SAVE_CORE macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_save_core  .macro
               .if (MON_RGST == MON_RGST_ANY)
               BLDD  MON_TEMP_PC,   #MON_PC   ; Save PC.
               BLDD  MON_TEMP_ST0,  #MON_ST0  ; Save ST0.
               BLDD  MON_TEMP_ST1,  #MON_ST1  ; Save ST1.
               BLDD  MON_TEMP_ACCL, #MON_ACCL ; Save ACCL.
               BLDD  MON_TEMP_ACCH, #MON_ACCH ; Save ACCH.
               .else ; (MON_RGST == MON_RGST_ZRO)
               PSHD  MON_TEMP_PC  ;
               POPD  MON_PC       ; Save PC.
               PSHD  MON_TEMP_ST0 ;
               POPD  MON_ST0      ; Save ST0.
               PSHD  MON_TEMP_ST1 ;
               POPD  MON_ST1      ; Save ST1.
               ;                  ;
               ;                  ; Already saved ACCL.
               ;                  ;
               ;                  ; Already saved ACCH.
               .endif ; (MON_RGST)
               .endm ; mon_save_core
;
;---------------------------------------------------------------------------
; Define MON_ETRAP_LOAD macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_etrap_load .macro VALU
               SPLK  #VALU, MON_TRAP_VEC
               .endm
;
;---------------------------------------------------------------------------
; Define MON_ETRAP_NULL macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_etrap_null .macro VALU
               SPLK  #NULL_ETRP?, MON_TRAP_VEC
               mon_etrap
NULL_ETRP?     SPLK  #VALU, MON_TRAP_VEC
               .endm
;
;---------------------------------------------------------------------------
; Define MON_ERET_NULL macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_eret_null  .macro VALU
               LALK   #VALU
               PUSH
               mon_eret
               .endm
;
               .page 
;---------------------------------------------------------------------------
; Define the MON_WAIT macro.
; This macro should not be modified.
;---------------------------------------------------------------------------
mon_wait       .macro
;
               .if (MON_WAIT_USE == MON_WAIT_USE_YES)
               .var   WAIT_MASK
               .asg   0, WAIT_MASK
;
               .if (MON_WAIT_STOP == MON_WAIT_STOP_YES)
               .eval  STOPMASK | WAIT_MASK, WAIT_MASK
               .endif
               .if (MON_WAIT_SNGL == MON_WAIT_SNGL_YES)
               .eval  SNGLMASK | WAIT_MASK, WAIT_MASK
               .endif
               .if (MON_WAIT_EVER == MON_WAIT_EVER_YES)
               .eval  EVERMASK | WAIT_MASK, WAIT_MASK
               .endif
               .if (MON_WAIT_AUTO == MON_WAIT_AUTO_YES)
               .eval  AUTOMASK | WAIT_MASK, WAIT_MASK
               .endif
;
               .if (0 != WAIT_MASK)
               LACK   #WAIT_MASK      ; Short immediate is ok.
               AND    PRM_CMND        ;
               BZ     NOT_WAIT?       ;
               mon_wait_test          ;
NOT_WAIT?      .endif ; (0 != WAIT_MASK)
;
               .endif ; (MON_WAIT_USE == MON_WAIT_USE_YES)
;
               .endm  ; mon_wait
;
;---------------------------------------------------------------------------
; Define the MON_WAIT_TEST macro.
; The default code in this macro is very generic, so is likely usable as is.
; It waits for an interrupt driven routine to signal permission to stop,
; step or run by a placing a non-zero value in location MON_WAIT_B2.
; This is a global symbol referring to a location in data memory page zero
; that is reserved for the sole use of this macro.
; If can be modified by the user as required.
;---------------------------------------------------------------------------
mon_wait_test  .macro
               SPLK   #0, MON_WAIT_B2 ;
WAIT_TEST?     LACL   MON_WAIT_B2     ;
               BZ     WAIT_TEST?      ;
               .endm ; mon_wait_test
;
               .page 
;***************************************************************************
; REALTIME OPCODES
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define the C2xx opcode mnemonics that are used as part of
; the MON_IO_MAP_USE_RAM and MON_IO_MAP_USE_ROM options.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
OPCODE_ERET    .set   0be96h ;
                             ; Return Opcode in "ERET"
DP_MASK        .set   0007fh ;
                             ; DP field mask for IN/OUT opcodes.
OPCODE_IN      .set  (0af00h & ~DP_MASK) | (MSG_DATA & DP_MASK) 
                             ; IO Read Opcode "IN MSG_DATA, PA0"
OPCODE_OUT     .set  (00c00h & ~DP_MASK) | (MSG_DATA & DP_MASK)
                             ; IO Write Opcode "OUT MSG_DATA, PA0"
               .endif ; (MON_IO_MAP_USE != MON_IO_MAP_USE_NOT)
;
               .page 
;***************************************************************************
; This page is intentionally left blank.
;***************************************************************************
;
               .page 
;***************************************************************************
; DEBUG MACROS
;***************************************************************************
;
;---------------------------------------------------------------------------
; Define MON_DBG_SAVE macro.
; This macro is used to sample various status bits on monitor entry.
;
; Configure the monitor to save (or not save) the values of the four
; registers ST0, ST1, MSG_STAT and MSG_BP in data memory locations 0x70
; to 0x73. The current and prior values of ST1(3-2) and MSG_STAT(4) are
; stored in locations 0x74 to 0x77.
;
; The results of using mon_dbg_save are:            ST1(3-2)   MSG_STAT(4)
;
; monitor initialisation                              2             0
; step-n                                              1             0
; run-n                                               1             0
; update            (during run with updating)        3             1
; halt               (after run with updating)        3             1
; halt            (after run without updating)        3             1
; hard breakpoint (after run without updating)        3             1
; soft breakpoint (after run without updating)        2             0
; idle     (after step/run-n without updating)        0             0
;---------------------------------------------------------------------------
               .if (MON_DEBUG)
mon_dbg_save   .macro
               LDP    #MON_PGE0_DP    ; Set DP to page zero.
               BLDD   #MON_ST0, 070h  ; Save ST0.
               BLDD   #MON_ST1, 071h  ; Save ST1.
               BLDD   #MSG_STAT, 072h ; Save MSG_STAT.
               BLDD   #MSG_CNTL, 073h ; Save MSG_CNTL.
;
               LACL   070h
               ANDK   #000Ch
               RPTK   #(2-1)
               SFRD
               BLDD   #074h, 076h ; Save prior ST1(3-2).
               SACL   074h        ; Save current ST1(3-2).
;
               LACL   072h
               ANDK   #0010h
               RPTK   #(4-1)
               SFR
               BLDD   #075h, 077h ; Save prior MSG_STAT(4).
               SACL   075h        ; Save current MSG_STAT(4).
               .endm
               .endif ; (MON_DEBUG)
;
               .page 
;---------------------------------------------------------------------------
; Define MON_DBG_START macro.
; This macro is used to start a C209 timer so as to benchmark
; the HLL Debugger updating memory or register values during a run command.
;---------------------------------------------------------------------------
               .if (MON_DEBUG)
mon_dbg_start  .macro
               LDP    #MON_PGE0_DP  ; Set DP to page zero.
               NOP                  ;
               SPLK   #0FFFFh, 07Eh ;
               OUT    07Eh, 0FFFDh  ; set the 16 bit reload value
               SPLK   #00C20h, 07Eh ;
               OUT    07Eh, 0FFFCh  ; set the 4 bit reload value
                                    ; set the free-run, load and start bits
               .endm
               .endif ; (MON_DEBUG)
;
;---------------------------------------------------------------------------
; Define MON_DBG_STOP macro.
; This macro is used to stop a C209 timer so as to benchmark
; the HLL Debugger updating memory or register values during a run command.
;---------------------------------------------------------------------------
               .if (MON_DEBUG)
mon_dbg_stop   .macro
               LDP    #MON_PGE0_DP ; Set DP to page zero.
               NOP                 ;
               IN     07Fh, 0FFFEh ; read the 16 bit timer value
               LACL   07Fh         ;
               NEG                 ;
               SACL   07Fh         ;
               .endm
               .endif ; (MON_DEBUG)
;
               .page 
;***************************************************************************
; Name: The MON_SE_CNFG routine
;
; Description
;    Routine to configure the SE Device.
;
;    This routine will over-write the contents of ACCL, ACCH, ST0, ST1.
;    This routine will temporarily use an extra level of stack.
;
; 1. Initialise the SE's sixteen memory map configuration registers.
;
;***************************************************************************
MON_SE_CNFG    .sect "mon_main"
;---------------------------------------------------------------------------
; This RET instruction is here in case MON_SE_CNFG is excluded
; by conditional assembly but still CALL'ed in the application code.
; Attempts to reuse existing MON_ERET instructions incovered problems
; when hardware resets occured during a RUNF.
;---------------------------------------------------------------------------
               .if (MON_SE_CNFG_USE == MON_SE_CNFG_USE_NOT)
               RET
               .endif ; (MON_SE_CNFG_USE == MON_SE_CNFG_USE_NOT)
;
;---------------------------------------------------------------------------
; Set the ANA_TRPON bit in MSG_CNTL before any other analysis access.
;
; This code places appropriate values in the sixteen memory mapped
; configuration buffers. During this load sequence the most
; significant bit in the least significant buffer must be set.
; All this must be in ETRAP/ERET protected code.
;---------------------------------------------------------------------------
               .if (MON_SE_CNFG_USE == MON_SE_CNFG_USE_YES)
               LDP    #MON_PGE0_DP         ; Set DP to page zero.
               SPLK   #ANA_TRPON, MSG_CNTL ; Important initialisation.
;
               mon_etrap_load $+3          ; Enter fully protected mode
               mon_etrap                   ; with a null ETRAP.
;
               .if (MON_RESET_MESG == MON_RESET_MESG_FLOW)
               SPLK   #ANA_SWLO, MSG_CNTL  ; Set the message switch low.
               .endif ; (MON_RESET_MESG == MON_RESET_MESG_FLOW)
;
               LALK   #(MON_SE_VALU + 1)
                                   ; Set ACCL near to the table's low end.
               LAR    MON_AR, #MON_SE_ADRS
                                   ; Set MON_AR to the buffer's low end.
               RPTK   #(16 - 1)    ;
               TBLR   *+           ;
;
               SUBK   #1           ; Set ACCL to the table's low end.
               SBRK   #16          ; Set MON_AR to the buffer's low end.
               TBLR   *            ;
;
               mon_eret            ; Exit fully protected mode and
                                   ; return to the application code.
;
               .endif ; (MON_SE_CNFG_USE == MON_SE_CNFG_USE_YES)
;
               .page 
;***************************************************************************
; This page is intentionally left blank.
;***************************************************************************
;
               .page 
;***************************************************************************
; Name: The MON_RT_CNFG routine
;
; Description
;    Routine to configure the RT Monitor.
;
;    This routine will over-write the contents of ACCL, ACCH, ST0, ST1.
;    This routine will temporarily use an extra level of stack.
;
; 1. Initialise the monitor and send a reset status to the HLL Debugger.
;
;***************************************************************************
MON_RT_CNFG    .sect "mon_main"
;---------------------------------------------------------------------------
; Do the initialisation of the analaysis logic.
; Set the ANA_TRPON bit in MSG_CNTL before any other analysis access.
; Do an access in fully protected mode to initialise the rest of MSG_CNTL.
; If MON_SE_CNFG is used then it does the analysis initialisation instead.
;---------------------------------------------------------------------------
               LDP    #MON_PGE0_DP           ; Set DP to page zero.
;
               .if (MON_SE_CNFG_USE == MON_SE_CNFG_USE_NOT)
               SPLK   #ANA_TRPON, MSG_CNTL   ; Important initialisation.
;
;               mon_etrap_load $+3          ; Enter fully protected mode
;               mon_etrap                   ; with a null ETRAP.
               .if (MON_RESET_MESG == MON_RESET_MESG_FLOW)
               mon_etrap_load $+3            ; Enter fully protected mode
               mon_etrap                     ; with a null ETRAP.
;
               SPLK   #ANA_SWLO, MSG_CNTL    ; Set the message switch low.
               .endif ; (MON_RESET_MESG == MON_RESET_MESG_FLOW)
               .endif ; (MON_SE_CNFG_USE == MON_SE_CNFG_USE_NOT)
;
;---------------------------------------------------------------------------
; Initialise everything that is not self-initialising in the monitor.
;---------------------------------------------------------------------------
               mon_etrap_load MON_TRAP_ENTRY ;
               SPLK   #ATN_OK, MON_STATUS    ;
;
               .page 
;---------------------------------------------------------------------------
; Set the monitor status to indicate a reset.
;
; After a HLL Debugger "RESET" command and before entering realtime with
; the HLL Debugger "REALTIME" command, the PC value must reach MON_GO.
; This may be done by using the "GO MON_GO" command, or by setting a
; hardware breakpoint at MON_GO and using the "RUN" command.
;---------------------------------------------------------------------------
MON_GO         SPLK   #ATN_RESET, MON_STATUS ;
;
;---------------------------------------------------------------------------
; Enter the monitor.  If the HLL Debugger is executing the RUNF command
; or is physically dis-connected, then the monitor will exit as soon as
; it sees the STAT_STRAP bit in MSG_STAT is zero.
;
; If the HLL Debugger has the application code halted in realtime,
; then the PC value will become MON_RESET after a hardware reset occurs.
;---------------------------------------------------------------------------
               .if (MON_RESET_MESG == MON_RESET_MESG_DEFAULT)
               mon_etrap                     ;
MON_RESET      NOP                           ;
               .endif ; (MON_RESET_MESG == MON_RESET_MESG_DEFAULT)
;
;---------------------------------------------------------------------------
; Do a RET or MON_ERET to return to the calling program.
;---------------------------------------------------------------------------
               .if (MON_SE_CNFG_USE != MON_SE_CNFG_USE_NOT)
               RET                           ;
               .elseif (MON_RESET_MESG != MON_RESET_MESG_FLOW)
               RET                           ;
               .else ; (MON_SE_CNFG_USE_NOT && MON_RESET_MESG_FLOW)
               mon_eret                      ;
               .endif
;
               .page 
;***************************************************************************
; Name: The MON_TRAP_ENTRY / MON_INTR_ENTRY / MON_ENTRY routines
;
; Description
;    Emulation trap and interrupt service routines.
;
; 1. If MON_TRAP_ENTRY then do the following:
;    Copy ACCL & ACCH to page zero temporary storage.
;    Disable the emulation interrupt and selected user interrupt masks.
;    Disable the single step function.
;    Initialise the message passing logic.
;
; 2. Then if MON_TRAP_ENTRY or
;            MON_INTR_ENTRY do the following:
;    If monitor entered from MON_INTR_ENTRY then interrupts now enabled.
;    Possibly copy PC & ST1-0 from temporary storage to the register image.
;    Possibly copy ACCL & ACCH from temporary storage to the register image.
;    Possibly copy the full register set exactly to the register image.
;    Set the emulation vector's to MON_ENTRY while in the monitor.
;    If monitor entered from MON_TRAP_ENTRY then exit ETRAP protected mode.
;
; 3. Then do the following in MON_ENTRY:
;    Possibly copy PC & ST1-0 from temporary storage to the register image.
;    Possibly copy ACCL & ACCH from temporary storage to the register image.
;    Possibly copy the full register set lazily to the register image.
;    Calculate which interrupts will be enabled when the monitor exits.
;    Wait for the message passing logi to be in its "MPSD FUNC" state.
;    Test for the occurance of software breakpoints.
;    Enter the monitor at its unchanged or after-error states.
;
; Input     PRM_CMND     = Preserved command.
;           PRM_ADRS     = Preserved address.
;           MON_TEMP_PC  = Temporarily saved user PC value.
;           MON_TEMP_ST0 = Temporarily saved user ST0 value.
;           MON_TEMP_ST1 = Temporarily saved user ST1 value.
;
;***************************************************************************
;
               .page 
MON_TRAP_ENTRY .sect "mon_main"
;---------------------------------------------------------------------------
; Save the accumalator in temporary storage.
;---------------------------------------------------------------------------
               .if (MON_RGST == MON_RGST_ANY)
               SACL  MON_TEMP_ACCL ;
               SACH  MON_TEMP_ACCH ;
               .else ; (MON_RGST == MON_RGST_ZRO)
               SACL  MON_ACCL      ;
               SACH  MON_ACCH      ;
               .endif ; (MON_RGST)
;
;---------------------------------------------------------------------------
; Disable the emulation interrupt and selected other interrupts
; so that their service routines may be breakpointed or single stepped.
;---------------------------------------------------------------------------
               LACL  IMR               ;
               SACL  MON_IMR_SAVE      ;
               ANDK  #(MON_INT_MSK)    ;
               SACL  IMR               ;
;
;---------------------------------------------------------------------------
; Possibly clear the Interrupt Mode bit (INTM) in Status Register #0.
; This is done just after we manipulate the IMR.
;---------------------------------------------------------------------------
               .if (MON_INT_ALTR == MON_INT_ALTR_ENB)
               EINT                    ; Enable interrupts.
               .endif ; (MON_INT_ALTR == MON_INT_ALTR_ENB)
;
;---------------------------------------------------------------------------
; Disable the emulation single step bit in case we were single stepping
; or single running. This must be after the ETRAP so the status emulation
; bits are accessable but before MON_TEMP_ST1 is saved to the register
; image and before an ERET triggers single step again.
;---------------------------------------------------------------------------
               LACK   #(SNG_STP_DSB | GLB_INT_ENB); Short immediate is ok.
               OR     MON_TEMP_ST1     ;
               SACL   MON_TEMP_ST1     ;
               LST    #1, MON_TEMP_ST1 ; Restore ST1
;
;---------------------------------------------------------------------------
; The first part of initialising the message passing logic was setting the
; ANA_TRPON bit in MSG_CNTL before entering ETRAP/ERET protected mode.
; That was done in MON_SE_CNFG or MON_RT_CNFG. The second part is done
; here. It consists of writing the initialisaton value while in ETRAP/ERET
; protected mode, but before actually using the message passing logic.
;---------------------------------------------------------------------------
               SPLK    #ANA_SWLO, MSG_CNTL ; Set the message switch low.
;
               .page 
MON_INTR_ENTRY .sect "mon_main"
;---------------------------------------------------------------------------
; Save the final value of INTM after the EINTs in the MON_EINTR_VECS macro
; and MON_ETRAP_ENTRY code.  This is not the ideal location to save INTM.
; It would be better done in interrupt disabled code immediately after the
; EINTs in both the macro and code.  Doing it here avoids changing the size
; of the macro. It also keeps B2 ram usage within 32 words when extended
; addressing is enabled, by sharing MON_ST0_SAVE and MON_IMR_SAVE.
;---------------------------------------------------------------------------
               .if (MON_XTND_EITHER == MON_XTND_OFF)
               SST    #0, MON_ST0_SAVE ; Save ST0
               .else ; (MON_XTND_EITHER == MON_XTND_ON)
               SST    #0, MON_MACHINE  ; Use of MON_MACHINE as a temporary
               LALK   #USR_INT_DSB     ; location is ok. Later on it is
               AND    MON_MACHINE      ; always loaded with "AFTEREROR_S".
               OR     MON_ST0_SAVE     ; MON_ST0_SAVE and MON_IMR_SAVE
               SACL   MON_ST0_SAVE     ; are a shared memory location.
               .endif ; (MON_XTND_EITHER)
;
;---------------------------------------------------------------------------
; This is experimental error checking code not used in this release.
;---------------------------------------------------------------------------
               .if (MON_ERROR_CHECK != MON_ERROR_CHECK_NONE)
               LACK   #ATN_TRAP        ; Short immediate is ok.
               ADD    MON_STATUS       ;
               SACL   MON_STATUS       ;
               .endif ; (MON_ERROR_CHECK != MON_ERROR_CHECK_NONE)
;
;---------------------------------------------------------------------------
; Modify the INTM bit in ST0 to correctly show its re-enabling
; that can occur in MON_EINTR_VECS and MON_ETRAP_ENTRY.
;---------------------------------------------------------------------------
               .if (MON_CORE == MON_CORE_EXACT)
               LALK   #(~USR_INT_DSB)  ; Long immediate is required.
               OR     MON_ST0_SAVE     ;
               AND    MON_TEMP_ST0     ;
               SACL   MON_TEMP_ST0     ;
               .endif ; (MON_CORE == MON_CORE_EXACT)
;
;---------------------------------------------------------------------------
; Copy ACCL/ACCH from temporary storage in page zero to the register image.
; Copy PC & ST0-1 from temporary storage in page zero to the register image.
; This must be after MON_TEMP_ST0 is modified to show INTM correctly.
; This must be after MON_TEMP_ST1 is modified to disable single stepping.
;---------------------------------------------------------------------------
               .if (MON_CORE == MON_CORE_EXACT)
MON_SAVE_CORE  mon_save_core           ;
               .endif ; (MON_CORE == MON_CORE_EXACT)
;
;---------------------------------------------------------------------------
; Save the full register set to the register image.  Only if the
; monitor is entered via MON_TRAP_ENTRY is this interrupt protected.
;---------------------------------------------------------------------------
               .if (MON_FULL == MON_FULL_EXACT)
MON_SAVE_FULL  mon_save_full           ;
               .endif ; (MON_FULL == MON_FULL_EXACT)
;
;---------------------------------------------------------------------------
; We have now saved enough context to preserve the user state.
; Do an ERET to get out of the fully protected state.
;---------------------------------------------------------------------------
               mon_eret_null MON_ENTRY ;
;
               .page 
MON_ENTRY      .sect "mon_main"
;---------------------------------------------------------------------------
; Help the HLL Debugger to test for the occurence of a software breakpoint
; by passing a status value, masked by ATN_SWBP, to the debugger when it
; scans the analysis logic to check for hardware and software breakpoints.
; The test will check for the value '10' in bits 3-2 of Status Register #1.
;---------------------------------------------------------------------------
               LACK   #ATN_SWBP       ; Short immediate is ok.
               AND    MON_TEMP_ST1    ;
               OR     MON_STATUS      ;
               SACL   MON_STATUS      ;
;
;---------------------------------------------------------------------------
; Modify the INTM bit in ST0 to correctly show its re-enabling
; that can occur in MON_EINTR_VECS and MON_ETRAP_ENTRY.
;---------------------------------------------------------------------------
               .if (MON_CORE == MON_CORE_LAZY)
               LALK   #(~USR_INT_DSB) ; Long immediate is required.
               OR     MON_ST0_SAVE    ;
               AND    MON_TEMP_ST0    ;
               SACL   MON_TEMP_ST0    ;
               .endif ; (MON_CORE == MON_CORE_LAZY)
;
;---------------------------------------------------------------------------
; Copy ACCL/ACCH from temporary storage in page zero to the register image.
; Copy PC & ST0-1 from temporary storage in page zero to the register image.
; This must be after MON_TEMP_ST0 is modified to show INTM correctly.
; This must be after MON_TEMP_ST1 is modified to disable single stepping.
;---------------------------------------------------------------------------
               .if (MON_CORE == MON_CORE_LAZY)
MON_SAVE_CORE  mon_save_core          ;
               .endif ; (MON_CORE == MON_CORE_LAZY)
;
;---------------------------------------------------------------------------
; Save the full register set to the register image.  If this were
; ETRAP/ERET protected it would have a large latency and the
; software breakpoint code later on would have to be
; modified to use ST1 and not MON_TEMP_ST1.
;---------------------------------------------------------------------------
               .if (MON_FULL == MON_FULL_LAZY)
MON_SAVE_FULL  mon_save_full          ;
               .endif ; (MON_FULL == MON_FULL_LAZY)
;
;---------------------------------------------------------------------------
; If necessary disable sign extension operations in the accumalator.
;---------------------------------------------------------------------------
               .if (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
               RSXM                   ; Clear sign-extension mode.
               .endif ; (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
;
;---------------------------------------------------------------------------
; Whenever the monitor is entered through MON_ENTRY
; a status value is returned to the HLL Debugger.
;---------------------------------------------------------------------------
               SPLK   #AFTEREROR_S, MON_MACHINE
;
               .page 
;***************************************************************************
; Name: The MON_WAIT and MON_EVALUATE routines
;
; Description
;
; Input         MSG_STAT    = Message passing status.
;               MON_MACHINE = Monitor state machine value.
;
; Output        ACCL        = Internal monitor command.
;
; 1. Wait for the HLL Debugger to send a message by polling the active
;    high "MPSD FUNC" status. Then test for unexpected HLL Debugger
;    disconnects by testing the active low "MPSD STRAP" status.
;
; 2. When a message is found in the buffers, use the buffers status
;    in MSG_STAT to decide its type: empty; data to the monitor;
;    a command to the monitor; something from the monitor to the host.
;
; 3. The state of the monitor is represented by the value in MON_MACHINE,
;    whose four nibbles represent the internal commands to be used
;    based on the four possible buffer statuses.
;
; 4. The appropriate nibble is selected from MON_MACHINE and used as
;    an index into the table INTERNAL_TABLE that contains the entry-point
;    of the routine that implements the actions to be taken.
;
; 5. Its interesting to note that even with four possible buffer states
;    and four possible monitor states, there are actually only five
;    possible actions.
;
;***************************************************************************

MON_WAIT       .sect "mon_main"
;---------------------------------------------------------------------------
; Wait for the HLL Debugger to send a message by polling for FUNC
; then clear any outstanding emulation interrupts.
;
; We test for unexpected HLL Debugger disconnects here by testing the
; STAT_STRAP bit that is active-low. Zero indicates the strap state.
;
; If the the HLL Debugger isn't physically connected then it will see that
; the STAT_STRAP bit in MSG_STAT is zero and exit by branching to
; MON_EXECUTE to restore the registers and return to the application code.
;---------------------------------------------------------------------------
               BIT    MSG_STAT, 15-B_STATFUNC ; Check FUNC status,
               BBZ    MON_WAIT                ; if not FUNC then wait.
;
               SPLK   #MON_EMUMSK, IFR        ; Clear the monitor interrupt.
;
               BIT    MSG_STAT, 15-B_STATSTRP ; Check STRAP status,
               BBZ    MON_QUICKGO             ; if STRAP then exit monitor,
;
               .page 
;---------------------------------------------------------------------------
; This code checks the MSG_STAT register to learn the buffer's status from
; the WBFUL, CMD and RBFUL bits.  It then calculates which nibble of the
; state contains the appropriate action to be taken for the following cases:
;                                                 WBFUL   CMD    RBFUL
; Nibble 3 defines the action for buffer status:    1      d       d
; Nibble 2 defines the action for buffer status:    0      1       1
; Nibble 1 defines the action for buffer status:    0      0       1
; Nibble 0 defines the action for buffer status:    0      d       0
;
; Nibble 3 corresponds to commands/data from the monitor to the host.
; Nibble 2 corresponds to commands from the host to the monitor.
; Nibble 1 corresponds to data from the host to the monitor.
; Nibble 0 corresponds to the buffer being empty.
;
; If the buffer contains a command from the host and it is read by the
; monitor, its will become emty as the status changes from 011 to 010.
; If the buffer contains a datum from the host and it is read by the
; monitor, its will become empty as the status changes from 001 to 000.
;---------------------------------------------------------------------------
MON_EVALUATE   LACC   MON_MACHINE, 00h        ; Select nibble 3 if
               BIT    MSG_STAT, 15-B_STATWFUL ; WBFUL is set in status.
               BBNZ   DONE_EVALUATE           ;
;
               LACC   MON_MACHINE, 0Ch        ; Select nibble 0 if
               BIT    MSG_STAT, 15-B_STATRFUL ; RBFUL is clear in status.
               BBZ    DONE_EVALUATE           ;
;
               LACC   MON_MACHINE, 08h        ; Select nibble 1 if
               BIT    MSG_STAT, 15-B_STATCMND ; CMD is clear in status.
               BBZ    DONE_EVALUATE           ;
;
               LACC   MON_MACHINE, 04h        ; Select nibble 2 if
DONE_EVALUATE                                 ; none of above in status.
;
;---------------------------------------------------------------------------
; The desired nibble must be shifted into the low four bits of ACCL.
; Rotate everything left through both ACC registers and the C bit.
;---------------------------------------------------------------------------
               RPT    #((16 + 4 + 1) - 1) ;
               ROL                        ;
;
               .page 
;***************************************************************************
; Name: The MON_COMMAND routine
;
; Description
;
;    Read in a internal command in this format       :XXXX:XXXX:XXXX:X210:
;                                                                     |||
;                                                                     |||
;    Opcode Mask           0,1,2,3,4,5,6,7 values --------------------+++
;
;
;    Read in a external command in this format       :XXXX:XXXX:7654:3210:
;
;                                                               ||||  |||
;    Stop-Mode flag        0:no       1:stop      --------------+|||  |||
;    Single-Step flag      0:no       1:single    ---------------+||  |||
;    Forever-Run flag      0:no       1:stop      ----------------+|  |||
;    Auto-Run flag         0:no       1:single    -----------------+  |||
;                                                                     |||
;                                                                     |||
;    Opcode Mask           0,1,2,3,4,5,6,7 values --------------------+++
;
;                                                                    |  |
;    Auto-Increment Mask   0:no       1:increment -------------------+  |
;    Data Mask             0:read     1:write     ----------------------+
;
;    This routine executes internal and external monitor
;    commands that have been placed in ACCL.
;
;    The command's routine is accessed by using the low nibble of
;    the monitor command as an index to access an entry address and
;    a return address that is fetched from either INTERNAL_TABLE or
;    EXTERNAL_TABLE. They are put in ENTR_ADRS and EXIT_ADRS.
;
;    This routine ETRAP/ERET protects the code that implements
;    the command while maintaining a short interrupt latency.
;
; Input         ACCL      = Internal or external monitor command.
;
; Output        ENTR_ADRS = Selected entry address.
;               EXIT_ADRS = Selected exit address.
;
;***************************************************************************
MON_COMMAND    .sect "mon_main"
;---------------------------------------------------------------------------
; Mask off all but the opcode bits and use them to select the appropriate
; record in the action table and extract its entry and return address
; fields that are placed in ENTR_ADRS and EXIT_ADRS.
;---------------------------------------------------------------------------
INTERNAL_CMND  ANDK   #0Fh            ; Mask off all but the low nibble.
               ADLK   #INTERNAL_TABLE ; Add the tables base address.
               TBLR   EXIT_ADRS       ; Store action entry address field. 
               ADDK   #01h            ; Short immediate is obviously ok.
EXTERNAL_CMND  TBLR   ENTR_ADRS       ; Store action return address field.
;
               .page 
;---------------------------------------------------------------------------
; Initialise both the ARP in ST0 and ARB in ST1 to ARB.
;
; The ARB is included because ARP is loaded from it whenever the monitor
; does an LST #1 instruction in the IO space access routines when saving
; and restoring the CNF bit.
;
; This should be the one and only place we need to initialise
; ARP and ARB as the auxilliary register MON_AR should be
; used only in the protected code.
;---------------------------------------------------------------------------
               MAR    *, MON_AR      ; Set ARP to MON_AR.
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
               MAR    *, MON_AR      ; Set ARB to MON_AR.
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
;
;---------------------------------------------------------------------------
; Here is where we execute the ETRAP protected action routine whose
; entry address is ENTR_ADRS and whose exit address is EXIT_ADRS.
; The five temporary registers MON_TEMP0-4 are used for specific purposes:
;
;   MON_TEMP0/MON_TEMP_PC should hold the value ONLY_ETRAP.
;   MON_TEMP1/MON_TEMP_ST0 saves and restores the MON_AR auxilliary register
;   MON_TEMP2/MON_TEMP_ST1 saves and restores the ST1 status register.
;   MON_TEMP3/ENTR_ADRS holds the routine's entry address.
;   MON_TEMP4/EXIT_ADRS holds the routines return address.
;
; This should be the only place inside the monitor that we use the ETRAP,
; so the monitor can check for its incorrect usage by application code.
; This is done by checking that MON_TEMP_PC has the value ONLY_ETRAP.
;---------------------------------------------------------------------------
               .if (MON_ERROR_CHECK != MON_ERROR_CHECK_EXTRA)
               LACL   ENTR_ADRS          ; The action entry address.
               mon_etrap_load ONLY_ETRAP ; These lines are
               mon_etrap                 ; equivalent to a null etrap. 
ONLY_ETRAP     .endif ; (MON_ERROR_CHECK != MON_ERROR_CHECK_EXTRA)
;
               .if (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
               LALK   #ONLY_ETRAP        ;
               mon_etrap_load ONLY_ETRAP ; These lines are
               mon_etrap                 ; equivalent to a null etrap. 
ONLY_ETRAP     SUB    MON_TEMP_PC        ;
               BNZ    MON_TRAP_ENTRY     ;
               LACL   ENTR_ADRS          ; The action entry address.
               .endif ; (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
;         
;---------------------------------------------------------------------------
; Enter the action routine using the entry address in ENTR_ADRS.
; The action routine is ETRAP protected.
;---------------------------------------------------------------------------
               CALA                          ; Go to it.
;
;---------------------------------------------------------------------------
; Exit the action routine to here, the last ETRAP protected instruction.
;---------------------------------------------------------------------------
               mon_etrap_load MON_TRAP_ENTRY ; Null_etrap macro.
;
;---------------------------------------------------------------------------
; Branch using the exit address in EXIT_ADRS.
;---------------------------------------------------------------------------
               LACL   EXIT_ADRS              ; The action exit address.
               BACC                          ; Go to it.
;
               .page 
;***************************************************************************
; Name: The Internal Commands
;
;       SENDCMND, RECVCMND, DOAWRITE, SWCHOPRT, MAYBERUN
;
; Description
;    This is four pages of spaghetti code to implement the inner most
;    operations of the realtime monitor in a single section of the
;    smallest possible size. There has been no attempt to make it
;    easy to understand, so its for real geeks and nerds.
;
;    To comprehend what this code does you first need to look at the
;    five-item table of internal commands, INTERNAL_TABLE.  The second
;    value in each item is used as the entry point into ETRAP protected
;    code. The first value in each item is used as a branch address after
;    exiting from the protected code.
;
;    All of the five entry points and the corresponding protected code
;    are in the spaghetti code. All of the branch addresses that have
;    corresponding code are here also. When the branch address is
;    MON_WAIT then there is no code, as the branch is back to the routine
;    that polls for the next message from the HLL Debugger and decodes it.
;
;***************************************************************************
               .sect "mon_main"
;
               .page 
;---------------------------------------------------------------------------
; RECVCMND does the following:
; Read the message value from MSG_CMND.
; Write the message value to PRM_CMND.
; Exit fully protected mode and go to RECVCMND_EXIT.
;---------------------------------------------------------------------------
RECVCMND_ENTR  LACL    MSG_CMND ;
               SACL    PRM_CMND ;
               mon_eret         ; Exit fully protected mode.
;
;---------------------------------------------------------------------------
; SENDCMND does the following:
; Read the message value from MON_STATUS.
; Write the message value to MSG_CMND.
; Toggle message switch.
; Exit fully protected mode and go to MON_BUFFER.
;---------------------------------------------------------------------------
SENDCMND_ENTR  LACL    MON_STATUS                ;
               SACL    MSG_CMND                  ;
               SPLK    #AFTERSTAT_S, MON_MACHINE ;
;
;---------------------------------------------------------------------------
; SWCHOPRT does the following:
; Toggle message switch.
; Exit fully protected mode and go to MON_BUFFER.
;---------------------------------------------------------------------------
SWCHOPRT_ENTR  SPLK    #ANA_SWHI, MSG_CNTL    ; Set the message switch high.
               SPLK    #ANA_SWLO, MSG_CNTL    ; Set the message switch low.
NOACTION_ENTR  mon_eret                       ; Exit fully protected mode.
;
;---------------------------------------------------------------------------
; MAYBERUN does the following outside of protected mode:
; Test the buffer status to see if the monitor has received an empty
; datum or an empty command. If the latter then its a signal to exit
; the monitor and return to the application code.
;---------------------------------------------------------------------------
MAYBERUN_EXIT  BIT    MSG_STAT, 15-B_STATCMND ;
               BBZ    DOSWITCH                ;
               B      MON_EXECUTE             ;
;
               .page 
;---------------------------------------------------------------------------
; DOAWRITE is the exit-point/branch-address used for
; writing data values after receiving a data message.
; SETUP_IOWR is the entry point to prepare for IO writes in write commands.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
DOAWRITE_EXIT                                   ;
;
SETUP_IOWR     SPLK     #OPCODE_OUT, MON_TEMP0  ;
               B        SETUP_IOIO              ;
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
;
;---------------------------------------------------------------------------
; RECVCMND is the exit-point/branch-address used for
; reading data values after receiving a command message.
; Merge the extended address high and low bytes into PRM_XTND.
; Clear the monitor status in MON_STATUS.
; Test the external command in PRM_CMND to decide if a read or write command. 
; Set the next value of the monitor state in MON_MACHINE.
;---------------------------------------------------------------------------
RECVCMND_EXIT
               .if (MON_XTND_EITHER == MON_XTND_ON)
               LACL   PRM_CMND ; ACCH becomes zero.  
               RPT    #(8-1)   ; Shift zeros into ACCL high byte.
               SFR             ; Shift extended address into ACCL low byte.
               OR     PRM_XTND ; Low byte of PRM_XTND is zero or same value.
               SACL   PRM_XTND ; Merge high and low bytes of address.
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
;
               .if (MON_ERROR_CHECK == MON_ERROR_CHECK_NONE)
               SPLK   #ATN_OK, MON_STATUS       ; Clear the monitor status.
               .else ; (MON_ERROR_CHECK != MON_ERROR_CHECK_NONE)
               SPLK   #ATN_TRAP, MON_STATUS     ; Set the monitor status.
               .endif ; (MON_ERROR_CHECK)
;
WR_OR_RD       SPLK   #AFTERWRIT_S, MON_MACHINE ; Choose the next state,
               BIT    PRM_CMND, 15-B_RORWMASK   ; either AFTERWRIT_S
               BBNZ   DOSWITCH                  ; or AFTERREAD_S.
               SPLK   #AFTERREAD_S, MON_MACHINE ;
;
;---------------------------------------------------------------------------
; SETUP_IORD is the entry point to prepare for IO reads in read commands.
; SETUP_IOIO implements the option to load IO space access code in RAM.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
SETUP_IORD     SPLK    #OPCODE_IN, MON_TEMP0    ;
;
SETUP_IOIO     SPLK    #OPCODE_ERET, MON_TEMP1  ;
               LALK    #MON_IO_RUN              ;
               TBLW    MON_TEMP0                ;
               ADDK    #1                       ;
               TBLW    PRM_ADRS                 ;
               ADDK    #1                       ;
               TBLW    MON_TEMP1                ;
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_RAM)
;
               .if (MON_IO_MAP_USE != MON_IO_MAP_USE_RAM)
DOAWRITE_EXIT                                   ;
               .endif ; (MON_IO_MAP_USE != MON_IO_MAP_USE_RAM)
;
               .page 
;---------------------------------------------------------------------------
; Load both the program and data map registers with the new extended
; address value, that is provided by the debugger and held in PRM_XTND.
;---------------------------------------------------------------------------
               mon_load_new           ;
;
;---------------------------------------------------------------------------
; XTRNCMND sets up everything to decode the external command in
; PRM_CMND then execute it in ETRAP protected code.
; If a read command then read data and place it in MSG_DATA.
; If a write then take data from MSG_DATA and write it.
; Exit fully protected mode and go to XTRNCMND_EXIT.
;---------------------------------------------------------------------------
XTRNCMND       LACK   #CMNDMASK       ; Mask off all but the low nibble.
               AND    PRM_CMND        ; Execute an external command.
               ADLK   #EXTERNAL_TABLE ; Add the tables base address.
               SPLK   #XTRNCMND_EXIT, EXIT_ADRS ;
               B      EXTERNAL_CMND   ; May be either a read or a write.
XTRNCMND_EXIT
;
;---------------------------------------------------------------------------
; Load the program and data map registers with their old extended address
; values that were saved in MON_PMAP and MON_DMAP when entering the monitor.
;---------------------------------------------------------------------------
               mon_load_old           ;
;
;---------------------------------------------------------------------------
; AINC possibly increments the memory address in PRM_ADRS.
;---------------------------------------------------------------------------
               .if (MON_AUTO_INC == MON_AUTO_INC_EXTRA)
AINC_TEST      BIT    PRM_CMND, 15-B_AINCMASK ; Possibly increment
               BBZ    DOSWITCH                ; the memory address.
               .endif ; (MON_AUTO_INC == MON_AUTO_INC_EXTRA)
;
               .if (MON_AUTO_INC != MON_AUTO_INC_NONE)
DO_AINC        LACK   #1                      ; Short immediate is ok.
               ADD    PRM_ADRS                ; 
               SACL   PRM_ADRS                ;
               .endif ; (MON_AUTO_INC != MON_AUTO_INC_NONE)
;
;---------------------------------------------------------------------------
; DOSWITCH sets up everything up to do a SWCHOPRT internal command
; to toggle message switch in fully protected mode
; then return to MON_WAIT.
;---------------------------------------------------------------------------
DOSWITCH       LACK   #SWCHOPRT_C             ;
               B      INTERNAL_CMND           ;
;
               .page 
;***************************************************************************
; Name: The MON_UNUSED and MON_ADRSWR routines
;
; Description
;      Address value read and write function.
;
; Input     PRM_CMND = Command
;           MSG_DATA = Write Address
; Output    MSG_DATA = Read Address
;
;***************************************************************************
MON_UNUSED     .sect "mon_main"
;---------------------------------------------------------------------------
; Fetch the address value from PRM_ADRS.
; Exit fully protected mode and return to the calling routine.
;
; MON_UNUSED was previously defined as the "mon_eret" in the MON_ADRSWR.
; It just bounced an empty message.  It has been temporarily redefined
; to return data, in case it was causing message passing protocol failures.
;---------------------------------------------------------------------------
               LACL   PRM_ADRS ;
               SACL   MSG_DATA ;
;
               mon_eret        ;
;
MON_ADRSWR     .sect "mon_main"
;---------------------------------------------------------------------------
; Copy the command register high byte into the extended address register.
; The extended address high byte is initialised.
; The extended address low byte is erased (zero).
; Load the address value into PRM_ADRS.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               .if (MON_XTND_EITHER == MON_XTND_ON)
               LACL   PRM_CMND ;
               AND    #0FF00h  ; 
               SACL   PRM_XTND ;
               .endif ; (MON_XTND_EITHER == MON_XTND_ON)
;
               LACL   MSG_DATA ;
               SACL   PRM_ADRS ;
               mon_eret        ;
;
               .page 
;***************************************************************************
; Name: The MON_PMRD and MON_PMWR routines
;
; Description
;      Program memory space read and write function.
;
; Input     PRM_CMND = Command
;           PRM_ADRS = Address
;           MSG_DATA = Write Data
; Output    MSG_DATA = Read Data
;
;***************************************************************************
MON_PMRD       .sect "mon_main"
;---------------------------------------------------------------------------
; Load the address in PRM_ADRS into the accumalator.
; Write the program memory value from MSG_DATA.
; Do a ERET to disable access to any protected program memory.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               LACL    PRM_ADRS ;
               TBLR    MSG_DATA ;
               mon_eret         ;
;
MON_PMWR       .sect "mon_main"
;---------------------------------------------------------------------------
; Load the address in PRM_ADRS into the accumalator.
; Read the program memory value into MSG_DATA.
; Do a ERET to disable access to any protected program memory.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               LACL    PRM_ADRS ;
               TBLW    MSG_DATA ;
               mon_eret         ;
;
               .page 
;***************************************************************************
; Name: The MON_DMRD and MON_DMWR routines
;
; Description
;      Data memory space read and write function.
;
; Input     PRM_CMND = Command
;           PRM_ADRS = Address
;           MSG_DATA = Write Data
; Output    MSG_DATA = Read Data
;
;***************************************************************************
MON_DMRD       .sect "mon_main"
;---------------------------------------------------------------------------
; Load the address in PRM_ADRS into the auxilliary register.
; Read the value from data memory to MSG_DATA.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               SAR     MON_AR, MON_TEMP_ST0 ; Save auxilary register.
               LAR     MON_AR, PRM_ADRS ;
               LACL    *                ;
               SACL    MSG_DATA         ;
               LAR     MON_AR, MON_TEMP_ST0 ; Restore auxilary register.
               mon_eret                 ;
;
MON_DMWR       .sect "mon_main"
;---------------------------------------------------------------------------
; Load the address in PRM_ADRS into the auxilliary register.
; Write the value from MSG_DATA to data memory.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               SAR     MON_AR, MON_TEMP_ST0 ; Save auxilary register.
               LAR     MON_AR, PRM_ADRS ;
               LACL    MSG_DATA         ;
               SACL    *                ;
               LAR     MON_AR, MON_TEMP_ST0 ; Restore auxilary register.
               mon_eret                 ;
;
               .page 
;***************************************************************************
; This page is intentionally left blank.
;***************************************************************************
;
               .page 
;***************************************************************************
; Name: The MON_IORD and MON_IOWR routines
;
; Description
;      IO memory space read and write function.
;      This uses self modifying code which must be located in B0 RAM.
;
; Input     PRM_CMND = Command
;           PRM_ADRS = Address
;           MSG_DATA = Write Data
; Output    MSG_DATA = Read Data
;
;***************************************************************************
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
               .sect "mon_main"
MON_IORD
MON_IOWR
;---------------------------------------------------------------------------
; Create a pointer to either entry-point to the IO access routine,
; which differ in address by 1, the same as the difference in values
; of all read and write commands. 
;---------------------------------------------------------------------------
               LACK    #RORWMASK            ; Short immediate is ok.
               AND     PRM_CMND             ;
               ADLK    #IOIO_LOAD           ;
;
;---------------------------------------------------------------------------
; Clear the CNF bit to ensure B0 is located at 0x01xx@DATA.
; Copy the few words of message passing code from its load location
; in program memory to B0 in low data memory.
; Set the CNF bit to ensure B0 is located at 0xFFxx@PRGM.
;---------------------------------------------------------------------------
               SAR     MON_AR, MON_TEMP_ST0 ; Save auxilary register.
               CLRC    CNF                  ; Ensure B0 is at 0x01xx@DATA.
               LRLK    MON_AR, #MON_B0_LOAD ;
               RPTK    #(3-1)               ;
               TBLR    *+                   ;
               BLDD    PRM_ADRS, #(MON_B0_LOAD+1) ;
               SETC    CNF                  ; Ensure B0 is at 0xFFxx@PRGM.
;
;---------------------------------------------------------------------------
; Exectute the three words in B0 high program memory by this CALL.
;---------------------------------------------------------------------------
               POPD    ENTR_ADRS        ; Save stack to a free location
               CALL    MON_B0_RUN       ;
               PSHD    ENTR_ADRS        ; Restore stack.
;
;---------------------------------------------------------------------------
; Restore ST1 with it's CNF value.
; ST1 is saved in MON_TEMP_ST1 by the ETRAP macro.
; Exit fully protected mode and return to the calling routine.
;---------------------------------------------------------------------------
               LAR     MON_AR, MON_TEMP_ST0 ; Restore auxilary register.
               LST     #1, MON_TEMP_ST1 ; Restore ST1.
               mon_eret                 ;
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
;
               .page 
;---------------------------------------------------------------------------
; These are the three words that will be copied to B0 high program memory
; to be executed by the CALL MON_B0_RUN statement.
; Three words will be used for either IO access command: an IN/OUT opcode;
; a value to be over written with an address; and a RET opcode.
;---------------------------------------------------------------------------
               .if (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
               .label  IOIO_LOAD
               .word   OPCODE_IN   ; IO Read Opcode
               .word   OPCODE_OUT  ; IO Read Address, IO Write Opcode
               RET                 ; IO Read Return,  IO Write Address
               RET                 ;                  IO Write Return
               .endif ; (MON_IO_MAP_USE == MON_IO_MAP_USE_ROM)
;
               .page 
;***************************************************************************
; Name: The MON_EXECUTE and MON_EXIT routines
;
; Description
;    Execution function - single step and run
;    Context restore function for entering into application.
;
; 1. Restore the full register set lazily.
; 2. Do a null ETRAP to enter fully protected mode.
; 3. Remap the ETRAP vector for later monitor re-entry.
; 4. Restore the emulation and user interrupt masks.
; 5. Restore the full register set exactly.
; 6. Restore more of the context: PC, ACC & ST.
; 7. Do a ERET to enter application or exit real-time mode.
;
;***************************************************************************
MON_EXECUTE    .sect "mon_main"
;---------------------------------------------------------------------------
; The mon_wait macro may cause the monitor to wait for permission from
; interrupt driven code before performing certain HLL Debugger commands.
; The commands are STOPMODE (enter stopmode), STEP-n and RUN-n (single-step
; and single-run), RUN and RUNF (both run-forever).  The mon_wait macro
; is configured by the "MON_WAIT.." mmnemonics and mon_wait_test macro.
;---------------------------------------------------------------------------
               mon_wait                   ;
;
;---------------------------------------------------------------------------
; Restore the full register set lazily.
;---------------------------------------------------------------------------
               .if (MON_FULL == MON_FULL_LAZY)
               .if (MON_RGST == MON_RGST_ANY)
               LDP    #MON_RGST_DP ; Set DP to the register table.
               .endif ; (MON_RGST == MON_RGST_ANY)
               mon_rest_full       ;
               .if (MON_RGST == MON_RGST_ANY)
               LDP    #MON_PGE0_DP ; Set DP to page zero.
               .endif ; (MON_RGST == MON_RGST_ANY)
               .endif ; (MON_FULL == MON_FULL_LAZY)
;
               .page 
MON_EXIT       .sect "mon_main"
;---------------------------------------------------------------------------
; Do a null ETRAP to enter fully protected mode. Load the emulation
; trap vector so as to enter the monitor if an emulation trap occurs.
;---------------------------------------------------------------------------
               .if (MON_ERROR_CHECK != MON_ERROR_CHECK_EXTRA)
               mon_etrap_load EXIT_ETRAP     ; These lines are
               mon_etrap                     ; equivalent to a null etrap. 
EXIT_ETRAP     .endif ; (MON_ERROR_CHECK != MON_ERROR_CHECK_EXTRA)
;
               .if (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
               LALK   #EXIT_ETRAP            ;
               mon_etrap_load EXIT_ETRAP     ; These lines are
               mon_etrap                     ; equivalent to a null etrap. 
EXIT_ETRAP     SUB    MON_TEMP_PC            ;
               BNZ    MON_TRAP_ENTRY         ;
               SACL   MON_STATUS             ; Load zero into MON_STATUS.
               .endif ; (MON_ERROR_CHECK == MON_ERROR_CHECK_EXTRA)
;
               mon_etrap_load MON_TRAP_ENTRY ;
;
               .page 
MON_QUICKGO    .sect "mon_main"
;---------------------------------------------------------------------------
; Calculate which interrupts will be re-enabled as the monitor exits,
; in MON_EXECUTE, they are those that were disabled in MON_xxxx_ENTRY.
;---------------------------------------------------------------------------
               LACK  #MON_EMUMSK          ; Short immediate is ok.
               OR    MON_IMR_SAVE         ;
               ANDK  #(~MON_INT_MSK)      ; Long immediate sucks.
               OR    IMR                  ;
               SACL  IMR                  ;
;
;---------------------------------------------------------------------------
; Prepare to restore the entire context.
;---------------------------------------------------------------------------
               .if (MON_RGST == MON_RGST_ANY)
               LDP   #MON_RGST_DP   ; Set DP to the register table.
               .endif ; (MON_RGST == MON_RGST_ANY)
;
;---------------------------------------------------------------------------
; Restore the full register set exactly.
;---------------------------------------------------------------------------
               .if (MON_FULL == MON_FULL_EXACT)
               mon_rest_full        ;
               .endif ; (MON_FULL == MON_FULL_EXACT)
;
;---------------------------------------------------------------------------
; Push the program address that we wish to ERET to the top of stack.
;---------------------------------------------------------------------------
               PSHD  MON_PC         ; Restore PC via the top of the stack.
;
;---------------------------------------------------------------------------
; Restore ACCH, ACCL, ST1, ST0 from the register image.
;---------------------------------------------------------------------------
               ZALH  MON_ACCH       ; Restore ACCH.
               ADDS  MON_ACCL       ; Restore ACCL.
;
;---------------------------------------------------------------------------
; Restore ST1 and ST0 in this order so as to not overwrite ARP with ARB.
; Return to the application or stopmode with an ERET.
;---------------------------------------------------------------------------
               LST   #1, MON_ST1    ; Restore ST1.
               LST   #0, MON_ST0    ; Restore ST0.
               mon_eret             ; Return to application.
;
               .end
