;*********************************************************************
;
; Filename: EQ.ASM
;
;
; Author: Carl Nuzman, Hemanth Sampath
; Date: 21 December, 1995
; Last revision:21 december, 1995.
;
;*********************************************************************
;
; Description:  
;    This file has 2 routines: "equalize" and "eq_adapt".
; The routine "equalize", implements the convolution of data in input
; buffer with real and imaginary filter taps( phase-splitting fractionally
; spaced adaptive equaliser). The output of the adaptive equalizer taps,
; is demodulated by using an in-built carrier routine, that computes
; the updated cos(wc*T) and sin(wc*T). wc is the carrier frequency and T is
; baud period. The base_band output is quantised to the nearest constellation
; point, by calling the "slicer" routine. The base_band error is then 
; remodulated to form the pass-band error.
;       
; The routine "eq_adapt" updates 36 out of 144 real and imaginary filter taps
; every baud. It uses the pass_band error computed in the routine "equalize"
; to update the filter taps in double precision.
;
;*********************************************************************
;
; Usage:  Called by main "receiver routine" . The routine "equalize" calls
;         a routine "slicer", which is in file slicer.asm.
;
; Inputs: inpbuffer (Input buffer), hr1 ( higher 16 bits from real filter)
;         hr2 ( lower 16 bits from real filter), hi1 ( higher 16 bits from
;         imaginary filter),hi2 ( lower 16 bits from imaginary filter),angle
;         (measures carrier phase), cospre ( cos(angle)) , sinpre(sin(angle)),
;         adapt_shift( controls the rate at which taps are updated),
; Outputs: hr1,hr2,hi1,hi2 ( updated filter taps),angle( updated version),
;          cospre,sinpre(updated versions),viterbR1, viterbI1 ( outputs
;          to be used by Viterbi decoder.
;
;****************************************************************************
; Declare global variables.

	.global equalize,eq_adapt,slicer
	.global hr1,hr2,hr3,hrend,hi1,sintab,inpbuffer,endbuffer
	.global N1,eq_t0,eq_t1,eq_t2,eq_t3,sigR,sigI,gamma
	.global Mu,Munum,errorR,errorI,cospre,sinpre
	.global angle,angle_inc,modR,modI,sl_c_ims,cnR,cnI
	.global diffR,diffI,k1,k2,wcT,angle_acc,angle_accl
	.global twoPi,invPi,tapinit,SC_WAIT
	.global bit_rate,viterbR1,viterbR2,viterbI1,viterbI2
	.global adapt_shift,Chrisx,Chrisy

	.mmregs

	.text

;******************************************************************
;                  DESCRIPTION OF POINTERS USED
;
; AR0 -
; AR1 - used to update real adaptive filter coeffs.
; AR2 - used to update imaginary (Hilbert Transform)adaptive filter coeffs.
; AR3 - running count 3,2,1,0 , which keeps track of where we are in a baud.
; AR4 - used to update real adaptive filter coeffs ( double precision)
; AR5 - used by interrupt (xint) to write to inpbuffer.
; AR6 - pointer to the input circular buffer.
; AR7 - used to update imaginary ad. filter coeffs ( double precision).

;*******************************************************************
;		DATA FORMAT NOTATION
;
; The position of the binary point in our data is indicated with
; the following notation:
; x:y:z indicates that there are x sign bits, y integer bits, and
; z fractional bits.
;
; Examples:
;
; Form 1:1:15  means that the variable has 1 sign bit, 1 integer bit
; and 15 fractional bits.
;
; Form 1:5:10  means that the variable has 1 sign bit, 5 integer bits
; and 10 fractional bits.
;*********************************************************************



;******************************************************************
;               MAIN ROUTINE "EQUALIZE"
;*******************************************************************
; AR6 points to the input buffer (inpbuffer), which is a circular buffer.

equalize:

	ldp #0
	splk  #endbuffer, CBSR1
	splk  #inpbuffer, CBER1
	splk  #0eh, CBCR                ; AR6 is a pointer to circular buffer
	ldpk sigR

;-----------------------------------------------------------------------
; Use MACD operations for convolving the inpbuffer with real and imaginary
;   filter coefficients.
;-----------------------------------------------------------------------


	lamm AR5                        ; store current value from inpbuffer
	nop                             ; in accumulator.
	mar *, AR6
	samm AR6                        ; form = 1:4:11
	zap
	rpt   #143
	mac   hr1,*-
	apac                            ; accumulator has filter output
	ldpk sigR                       ; store real filter output in memory
					; form = 2:5:25

	sach sigR,2                     ; form = 1:4:11

	zap
	rpt   #143
	mac   hi1,*-
	apac                            ; acc has filter output
	sach sigI,2                     ; store output in memory.

					; form = 1:4:11

;-----------------------------------------------------------------------
; Save old Viterbi decoder inputs
;-----------------------------------------------------------------------


	lacc bit_rate                   ; load the correct shift into TREG1
	samm TREG1                      ;   depending on baud rate of the
					;   transmitted signal.

	lacc viterbR1                   ; save old viterbi inputs
	sacl viterbR2
	lacc viterbI1
	sacl viterbI2                   ; form = 1:5:10

;-----------------------------------------------------------------------
; Demodulate the filtered signal output by using the values of sin(wc*T)
;   and cos(wc*T) computed in previous baud, where
;     wc = carrier frequency
;     and T = baud period.
; sin(wc*T) is stored in "sinpre"
; cos(wc*T) is stored in "cospre"
; "modR" and "modI" are real and imaginary parts of the demodulated
;   filter outputs, used by equaliser.
; "viterbR1" and  "viterbI1" are demodulated filter outputs, used by
;   the Viterbi Decoder.
;-----------------------------------------------------------------------


	lt  sigR                        ; form = 1:4:11
	mpy cospre
	ltp sinpre
	mpy sigI
	lta sigI                        ; form = 2:6:26
	sach Chrisx, 2
	satl                            ; shift accumulator according to
					; speed of transmitted data

	sach modR, 2                    ; form = 1:4:11
	sach viterbR1                   ; save real part of demodulated
					; output for Viterbi Decoder.
	mpy  cospre
	ltp  sinpre
	mpy  sigR
	spac
	sach Chrisy, 2
	satl                            ; shift accumulator according to
					; baud rate of transmitted signal
	sach modI, 2                    ; form = 1:4:11
	sach viterbI1                   ; save imaginary part of demodulated
					; output for Viterbi Decoder

;-----------------------------------------------------------------------
; Call the slicer routine.  Slicer quantizes the demodulated output to the
;   nearest constellation point and outputs the real and imaginary part of
;   the corresponding constellation point in "cnR" and "cnI".  For the
;   details of slicer's operation, please see file:  "slicer.asm"
;-----------------------------------------------------------------------


	call slicer                     ; form= 1:4:11
;------------------------------------------------------------------------
; Compute the error between ideal and actual demodulated data and store
; imaginary part of error calculated in "diffI" and real part of error in
; "diffR". These values are to be used by the carrier tracker routine.
; They are also referred to as "mean squared error".
;------------------------------------------------------------------------
	zap
	ldpk cnR
	lacc cnR
	sub modR                        ; form = 1:4:11
	sacl diffR                      ; form = 1:4:11
	lacc cnI                        ; Form = 1:4:11
	sub modI
	sacl diffI
;-----------------------------------------------------------------------
; Remodulate the base-band error by using the previously computed values of
; cos(wc*T) and sin(wc*T) to get the pass-band error.
; complex multiplication: (errorR,errorI) = (diffR,diffI)*(cospre,sinpre)
;-----------------------------------------------------------------------

	lt  diffR                       ; form = 1:4:11
	mpy cospre                      ; form = 1:1:14
	ltp sinpre
	mpy diffI
	lts diffI
	sach errorR, 2                  ; form = 1:4:11
	mpy  cospre                      
	ltp  sinpre                   
	mpy  diffR
	apac
	sach errorI, 2                  ; form = 1:4:11

;************************************************************************
; Begin carrier tracking routine.
; For this section, consider demodulated data to be in the form (1:6:9).  
; This corresponds to the way we sliced the demodulated output to the  
; nearest constellation point and the way we calculate the
; square of the inverse magnitude, needed for carrier_tracking loop.
;***********************************************************************


carrier:
;--------------------------------------------------------------------
; First compute the derivative of the mean_squared error with respect 
; to phase_offset.
	
	zap
	lt modR                         ; form = 1:6:9
	mpy diffI                       ; form = 1:6:9
	lts modI
	mpy diffR                       
	apac                            ; form = 2:12:18

	bsar 7                          
					; lower 16 bits of acc has the
					; required value in the form:
					; 1:4:11

;-----------------------------------------------------------------------
; Normalise the derivative computed in the previous section:
	
	samm TREG0                      ; form = 1:4:11
	nop
	ldpk sl_c_ims                   ; form = 1:0:15
	mpy sl_c_ims
	pac                             ; form = 2:5:26



;-----------------------------------------------------------------------
; Implement 2nd order loop:  
; SKip 2nd order part of the loop, after symbol_clock recovery tracks 
; the symbol clock frequency of the transmitter. This is checked by testing
; the appropriate bit in "SC_wait".
;-----------------------------------------------------------------------


	sach angle_inc,1                ; for debugging purposes only


	bsar 3                          ; first order constant

	bit SC_WAIT, 15
	bcnd eq_skp, TC                 ; skip 2nd order part of loop.


	sacb
	bsar 15                         ; second order constant
	adds angle_accl
	add  angle_acc, 16
	sach angle_acc
	sacl angle_accl
	addb


eq_skp:                                 
;---------------------------------------------------------------------------
; Compute the new angle, by using the previous value of angle stored in
; "angle". 
	
	
	bsar 14                         ; form = 1:4:11
	add angle                       ; form = 1:4:11
	add wcT                         ; form = 1:4:11



;----------------------------------------------------------------------
; Check if the angle (in radians) is greater than 2*pi.  If yes, then
;   subtract 2*pi from the angle. Keep the angle within 0 and 2*Pi.
;-----------------------------------------------------------------------


eq_lt   bcnd eq_low, LT
	sub twoPi                       ; form = 1:4:11
eq_gt   bcnd eq_hi,GEQ

	add twoPi
	b sc_over

eq_low  add twoPi
	b eq_lt

eq_hi   sub twoPi
	b eq_gt

;-----------------------------------------------------------------------
; Convert the angle that has been computed in radians to a point in 
; the look up table:"sintab"
; The look up table is 64*5 words long (from 0 to 5*pi/2)  
; Corresponding point in the look up table  = angle * 128/Pi
; Memory "angle" has the new angle computed (in radians). Now use the
;   look-up table to read out the new values of sinpre and cospre. Store
;   them in the memory locations "sinpre" and "cospre"
;
;-----------------------------------------------------------------------


sc_over samm TREG0
	sacl angle
	nop
	mpy invPi                       ; form = 1:0:15
	pac                             ; form = 2:4:26
					; multiply by 128 (2:11:19)


	add #04000h, 4                  ; rounding bit

					
	bsar 16                         ; The integer is in the lower part
	bsar 3                          ;   of the ACC
	add #sintab
	samm AR1                        ; use a pointer to get the address
	mar *,AR1
	sacb
	lacc *                          ; load the sine (angle) to the ACC
	sacl sinpre                     ; make sure that the value is in
					;   1:1:14 form
					; store sin(angle) to memory
	lacb
	add #64
	samm AR1
	nop
	nop
	lacc *
	sacl cospre                     ; store the cos(angle) to memory
					; make sure that the value is of
					;   the form 1:1:14
	ret


;**************************************************************************
;                Routine  "eq_adapt"
;**************************************************************************
;This routine calculates the next set of real and imaginary filter
; coefficients based on the pass-band error we calculated in the previous
; routine.
; It updates 36 filter taps per baud in double precision.
;It is not possible to update all the taps within a baud.
;Note:
;1)
;You can implement 2777 instruction cycles between any 2 interrupts,
;because AIC sampling rate has been set to 7200 Hz ( interrrupts occur
;every sampling instant)
;Updating all the taps per baud needs more than 2777*3 instructions.
;One can check if all of the cycles are being used between 2 sampling
;instants by inspecting the "disaster" variable, which counts the number
;of times the interrupt gets called before the main control loop has
;finished.
;2)
; "tapinit" controls which part of the 144 taps are to be updated.
; Tapinit stores the values:  0, 108, 72, 36, 0, etc. If "tapinit" is
; 36, then we update all taps between 36 and 72 and so on.
;-----------------------------------------------------------------------


eq_adapt:
;       ret;;;;;;;;;;;;;;;;;;;;;;
;--------------------------------------------------------------------------
; Decrement "tapinit" by 36 every baud. Keep the value of "tapinit" between
; 0 and 108.

	ldpk tapinit
	lacc tapinit
	sub #36
	bcnd eq_36, GEQ
	lacc #108
eq_36   sacl tapinit

;----------------------------------------------------------------------------
; Initialise pointers :
; AR1 points to higher order real filter tap
; AR4 points to lower order real filter tap
; AR2 points to higher order imaginary filter tap.
; AR7 points to lower order imaginary filter tap.
; Make sure AR6, which is a pointer to the input buffer (inpbuffer), stays
; within bounds.

	lacc #hr1
	add tapinit
	samm AR1                        ; high order real
	add #144
	samm AR4                        ; low order real
	lacc #hi1
	add tapinit
	samm AR2                        ; high order imaginary
	add #144
	samm AR7                        ; low order imaginary

	lamm AR5
	nop
	nop
	sub tapinit                     ; make sure AR6 is within the
	sub # inpbuffer                 ; input buffer(circular buffer).
	bcnd Jonas,GEQ
	add #144

Jonas   add #inpbuffer
	samm AR6

	sacl eq_t3                      ; for debugging purposes only.
;---------------------------------------------------------------------------
; Implement the following equation to update real filter taps (in double
; precision):
; h_real(n+1) = h_real(n) + Mu * errorR * AR6
;               where , h_real(n) is the real filter tap at a given instant
;               ,n. AR6 points to the input circular buffer (inpbuffer).
;               Mu is a fraction, which is implemented by using "satl"
;               command, which shifts the accumulator right by appropriate
;               bits, specified by the memory location "adapt_shift".
;               "Adapt_shift" is controlled in the main receiver program
;               which calls eq_adapt.
; Note: H_real(n) is in double precision. Higher 16 bits are denoted by
;       hr1 and lower 16 bits are denoted by hr2.

	ldpk adapt_shift
	lacc adapt_shift                ; load the correct shift into
	samm TREG1                      ; TREG1 depending on the
					; baud rate of transmitted data.

	ldp #0                          ; Use a block repeat command
	splk #35, BRCR
	ldpk eq_t0
	mar *, AR6
	zap
	lt errorR
	rptb end_block -1


	mpy *,AR4                        ; Pr. register   = errorR * AR6
	ltp errorI                       ; acc = errorR * AR6
	sfl
	satl                             ; acc = Mu * errorR * AR6
	adds *, AR1                      ; acc = hr1 + hr2 - hr1*2^-15
	add *, 16

	sach *+,AR4                      ; Store updated filter taps.
	sacl *+,AR6                      ; store hr1 and hr2
;--------------------------------------------------------------------------
; Implement the following equation to update imaginary filter taps
; (in double precision):
; h_imaginary(n+1) = h_imaginary(n) + Mu * errorI * AR6
;               where , h_imaginary(n) is the imaginary filter tap at a
;               given instant,n. AR6 points to the input circular buffer
;               (inpbuffer).
;
; Note: H_imaginary(n) is in double precision. Higher 16 bits are denoted by
;       hi1 and lower 16 bits are denoted by hi2.


	mpy *-, AR7                             ; P   = errorI * AR6
	ltp errorR                              ; acc = errorI * AR6
	sfl
	satl
	adds *, AR2
	add *, 16

	sach *+,AR7
	sacl *+,AR6


end_block
	ldp #0                                  ; Turn off circular buffer.
	splk #0, CBCR

eq_ret  ret


;*********************************************************************







