;*********************************************************************
;
; Filename:    FAST.ASM
;
; Author: Carl Nuzman, Hemanth Sampath. 
; Date:
; Last revision: 25 December, 1995.
;
;*********************************************************************
;
; Description:

;               Routine: fast_equalise:
;  This routine computes the real and imaginary part of the adaptive  
; equaliser, when the transmitter sends a pp sequence. This routine           
; fast_trains the adaptive equaliser.
;
;               Routine: Inverse2
;    This routine computes the inverse of a given real number in a specified
; format. This gets called by fast_equalise.



; Algorithm:
;               Routine: fast_equalise:
; 1. copy data from input buffer (inpbuffer) to "fr(i)" and zero "fi(i)".     
; 2. Take 144 pt fft of data in "fr(i)" and "fi(i)", by calling the routine "fft"
; 3. This leaves the scrambled output in fr(i)/fi(i). Call them D(i). 
; 4. unscramble D(i),  doubling lower half and zeroing upper half. 
;    real part is storedin hr1(i); Imaginary part is stored in hi1(i).
; 5. Implement equation A8 ( see code below) to get C(i).
;      Real part is in fr(i) and imaginary part is in fi(i).
; 6. Compute inverse fourier transform of C(i) by calling fft routine. 
;    The output of fft is stored in fr(i) and fi(i).
; 7. Descramble fr(i) and then divide by 144.  Store output in gr(i).
;       Now, rotate gr(i) in a circular fashion in such a way that the 
;        maximum of gr(i) is in the center of the buffer. Copy the rotated 
;        version of gr(i) to hr1(i) , where i =0..143. This forms the 
;        real part of adaptive equalizer.
; 8. Repeat 7. for fi(i). Store output in hi(1) to get imaginary part of    
;    adaptive equaliser.
	

;*********************************************************************
;
; Usage: fast_equalise gets called by main receiver program.
;        inverse2 gets called by fast_equalise.
;        fast_equalise calls routines "fft" , "unscramble" and "unscramble2"
; Inputs: Inpbuffer( inputbuffer containing the pp sequence).
;
; Outputs: hr1 and hi1; the real and imaginary parts of the adaptive 
;          equaliser.
;
; Variables used:    see code below.
;
; Registers used:    see code below.
;
;*********************************************************************
;
; Comments:
;
;*********************************************************************
;       DECLARE GLOBAL VARIABLES
;*********************************************************************
	.mmregs
	.include macros.inc

	.global stop_baby
	.global unscramble
	.global unscramble2
	.global inpbuffer
	.global fast_equalise
	.global fr
	.global fi
	.global hr1
	.global hi1
	.global gr
	.global bitable
	.global ft_inv
	.global ft_temp
	.global ft_br
	.global ft_bi
	.global ft_maxindex
	.global three
	.global num
	.global den
	.global ans_sign
	.global fft
	.global div_disaster
	.global denom
	.global my_numerator
	.global inpdebug, inpdebug2
;*********************************************************************
;******************************************************************
;                  DESCRIPTION OF POINTERS USED
;
; AR3 - running count 3,2,1,0 , which keeps track of where we are in a baud.
; AR5 - used by interrupt (xint) to write to inpbuffer.
; AR6 - pointer to the input circular buffer.
;
;*******************************************************************
;               DESCRIPTION OF TYPE OF DATA USED
;
; The data in inputbuffer (inpbuffer) has both integer and fractional
; parts. The form of the data is indicated at appropriate places.
; Examples:
;
; Form 1:1:15  means that the variable has 1 sign bit, 1 integer bit
; and 15 fractional bits.
;
; Form 1:5:10  means that the variable has 1 sign bit, 5 integer bits
; and 10 fractional bits.
;*********************************************************************


;*********************************************************************
;                        ROUTINE: FAST_EQUALISE:
;*********************************************************************

fast_equalise:

;---------------------------------------------------------------------- 
;copy data from input buffer (inpbuffer) to "fr(i)" and zero "fi(i)".
;"fr(i)" and "fi(i)" are 144 words long each. "fr(i)" is the real part of
;data in inpbuffer and "fi(i)" is imaginary part of data in inpbuffer.

	ldpk fr
	lar AR0, #fr
	mar *, AR0

	rpt #143
	bldd #inpbuffer, *+

	lar AR0, #fi
	lacc #0
	rpt #143
	sacl *+

;-----------------------------------------------------------------------
; Take 144 pt fft of data in "fr(i)" and "fi(i)", by calling the routine "fft". 
; This leaves the scrambled output in fr(i)/fi(i). Call them D(i). 
; where i = 0..143

	lacc #15                        ; load constant in shift register 
					; for fft's.

	samm TREG1
	nop
	nop
	call fft                        ; call fft routine.

;-----------------------------------------------------------------------
; unscramble D(i),  doubling lower half and zeroing upper half. The real part
; of the output is stored in hr1(i) and the imaginary part of the output is
; stored in hi1(i), where i = 0.. 143.

	call unscramble                 ; output is stored in hr1,hi1

	lar AR7, #(hr1+72)
	mar *, AR7
	lacc #0
	rpt #71
	 sacl *+

	lar AR7, #(hi1+72)
	mar *, AR7
	lacc #0
	rpt #71
	 sacl *+





;************************************************************************
;------------------------------------------------------------------------

; Implement following equation to get C(i):   (EQUATION A8)
;                                       ____
;   C(i) =                B(i mod 48) * D(i)
;         ------------------------------------------------------------
;         | D(i mod 48)***2 + D(48+i mod 48)**2 + D(96+i mod 48)**2 |
;
; where i = 0..143.
; C(i) are to be stored in fr(i) and fi(i) .
; D(i) are stored in hr1(i) and hi1(i)
; The  computation time for computing the above eequaiton can be vastly
; reduced by noting that D(i) = 0, for i = 72 to 144.

	lacc #48
	ldp #0
	samm INDX
	lar AR0, #(hr1+48)             ; First initialise pointers.
	lar AR1, #(hi1+48)
	lar AR2, #(fr)
	lar AR4, #(fi)
	lar AR7, #bitable              ; stored in order: Real,Imaginary,
				       ; Real,Imaginary,and so on .....
	mar *, AR0

;-------------------------------------------------------------------------
; for i = 0 to 23 , implement the equation A8 through the following code:
;   1.    den = D(i)^2 + D(i+48)^2
;   2.    ft_inv =1/den
;   3.    ft_b = B(i)*ft_inv
;   4.    C(i) = ft_b * D(i)'
; end;

	ldp #0
	splk #23, BRCR                 ; Initialise repeat counter.
	ldpk den
	nop
	nop
	rptb (end_frst24 - 1)



	lt *                            ;form = 1:0:15
	mpy *0-, AR1
	ltp *
	mpy *0-, AR0
	lta *
	mpy *, AR1
	lta *
	mpy *, AR7                      ;form = 2:0:30
	apac

					;calculate inverse = 1.0 / denominator
	ldpk den
	sach den,5                      ;form = 1:0:15
	lacc den
	call inverse2                   ;quotient in accumulator
	ldpk ft_inv
	sacl ft_inv                     ;store inverse in ft_inv (Q-15 format)



	;calculate ft_b = B(i) * inverse

	lt ft_inv                       ; form = 1:0:15
	mpy *+                          ; form = 1:0:15
	ltp ft_inv
	mpy *+, AR0
	sach ft_br,1                    ; form = 1:0:15
	pac
	sach ft_bi,1                    ; form = 1:0:15

	;calculate C(i) = ft_b * D(i)'

	lt  ft_br
	mpy *, AR1
	ltp *, AR2
	mpy ft_bi
	lta ft_bi
	sach *0+, 7,AR0                 ; form = 1:0:15
	mpy  *0+, AR1
	ltp  *0+, AR4
	mpy  ft_br
	spac
	sach *0+, 7, AR0

	; calcute C(i+48) = ft_b * D(i+48)'

	lt  ft_br
	mpy *, AR1
	ltp *, AR2
	mpy ft_bi
	lta ft_bi
	sach *0-, 7,AR0
	mpy  *+, AR1
	ltp  *+, AR4
	mpy  ft_br
	spac
	sach *0-, 7                 ; form = 1:0:15
	mar *+, AR2
	mar *+, AR0


end_frst24

;--------------------------------------------------------------------------
; Initialise pointers to compute equation A8 for i = 24 to 47


	lar AR0, #(hr1+24)
	lar AR1, #(hi1+24)
	lar AR2, #(fr+24)
	lar AR4, #(fi+24)
	lar AR7, #(bitable+48)          
	mar *, AR0
				
;-------------------------------------------------------------------------
; for i = 24 to 47 , implement the equation A8 through the following code:
; 1.      den =  D(i)^2
; 2.      ft_inv =1/den       
; 3.      ft_b = B(i)*ft_inv
; 4.      C(i) = ft_b * D(i)'
; end;
; C(i) are to be stored in fr(i) and fi(i) , where fr(i) is the real part
; of C(i) and fi(i) is the imaginary part of C(i).

	
	ldp #0
	splk #23, BRCR
	nop
	nop
	rptb (end_scnd24 - 1)

	lt *                    ;form = 1:0:15
	mpy *, AR1
	ltp *
	mpy *, AR7
	apac                    ;form = 1:0:15


				;calculate inverse = 1.0/denominator
	ldpk den
	sach den,5              ;form = 1:0:15
	lacc den
	call inverse2
	sacl ft_inv             ;store inverse in ft_inv

				;calculate ft_b = B(i) * inverse
	lt ft_inv
	mpy *+
	ltp ft_inv
	mpy *+, AR0
	sach ft_br,1            ;form = 1:0:15
	pac
	sach ft_bi,1            ;form = 1:0:15

				;calculate C(i) = ft_b * D(i)'
	lt  ft_br
	mpy *, AR1
	ltp *, AR2
	mpy ft_bi
	lta ft_bi
	sach *+, 7,AR0
	mpy  *+, AR1
	ltp  *+, AR4
	mpy  ft_br
	spac
	sach *+, 7, AR0         ;form = 1:0:15

end_scnd24

;--------------------------------------------------------------------------
;zero out top half of fr(i) and fi(i), that have been computed in the previous 
;section.

	lar AR0, #(fr+72)
	lacc #0
	mar *, AR0
	rpt #73
	sacl *+

	lar AR0, #(fi+72)
	rpt #73
	sacl *+

;************************************************************************
;-----------------------------------------------------------------------
; At this point, we have computed C(i) for i = 0.. 143. Real part of
; C(i) is in fr(i) and imaginary part of C(i) is in fi(i).
; Compute inverse fourier transform of C(i) by calling fft routine. 
; The output of fft is stored in fr(i) and fi(i).




	call fft                        ; compute inverse fourier
					; transform of C(i).

;----------------------------------------------------------------------
; FIRST WORK WITH REAL PART OF IFFT OUTPUT: fr(i)
; Descramble fr(i) and then divide by 144.
; The output is stored in gr(i), where i = 0..143.


	lar AR0, #fr
	mar *, AR0
	call unscramble2                ; Unscramble fr(i).
					; Output is stored in gr(i).
;------------------------------------------------------------------------
; Search for maximum of gr(i). Then, find i corresponding to maximum gr(i).
;Now, rotate gr(i) in a circular fashion in such a way that the maximum of
;gr(i) is in the center of the buffer. Copy the rotated versionof gr(i) to
; hr1(i) , where i =0..143. This forms the real part of adaptive equalizer.


	lar AR0, #(gr)
	mar *, AR0

					;repeat 144 times
	ldp #0
	splk #143,BRCR
	ldpk ft_maxindex
	splk #3, three

	lacc #0
	sacb

	rptb end_b - 1
	lacc *
	abs
	crgt
	bcnd newmax, C
cont_b  mar *+
end_b
	b skip_max

newmax                                  ;update maximum index of gr(i)
	sar AR0, ft_maxindex
	b cont_b


skip_max:

	clrc SXM
	lacc #(gr+72)
	setc SXM
	neg
	adds ft_maxindex
	bcnd cont_c, GEQ
	lacl ft_maxindex
	add #72
	sacl ft_maxindex
	b cont_de
cont_c
	lacl ft_maxindex
	sub #72
	sacl ft_maxindex
cont_de

	;make sure that (ft_maxindex-gr) is divisible by three
	clrc SXM
	lacc  #gr
	setc SXM
	neg
	adds ft_maxindex
	rpt #15
	 subc three
	bsar 16
	neg
	adds ft_maxindex
	sacl ft_maxindex

					;set up circular buffer
	ldp #0                          ; and copy data into hr1(i)
	sacl AR6
	splk #(gr), CBSR1
	splk #(gr+143), CBER1
	splk #0eh, CBCR
	mar *, AR6

	rpt #143
	 bldd *+, #(hr1)



;-------------------------------------------------------------------------
; LET US WORK WITH THE IMAGINARY PART OF IFFT OUTPUT: fi(i).
; Now, descramble fi(i) and then divide by 144.
; The output is in stored in gr(i).

	lar AR0, #fi
	mar *, AR0
	call unscramble2                ; Unscramble ifft output.


;------------------------------------------------------------------------
; Search for maximum of gr(i). Then, find i corresponding to maximum gr(i).
;Now, rotate gr(i) in a circular fashion in such a way that the maximum of
;gr(i) is in the center of the buffer. Copy the rotated versionof gr(i) to
; hi1(i) , where i =0..143. This forms the imaginary part of adaptive
; equalizer.


				; Find maximum of gr(i)
				; and note the corresponding i.

	ldpk ft_maxindex
	lar AR6, ft_maxindex
	mar *, AR6

	rpt #143                           ; copy data to hi(i)
	 bldd *+, #(hi1)

	splk #0h, CBCR


	ret


;***************************************************************************
;               ROUTINE: INVERSE2
;***************************************************************************
; Description:
;Given divisor in low accumulator, calculate the inverse
;and store it back in the low accumulator

inverse2:
	ldpk ans_sign
	splk #0, ans_sign
	bcnd err_div2,EQ
	bcnd pos_div, GT
	neg
	splk #0ffffh, ans_sign
pos_div:
	sacl denom
	sub  my_numerator
	bcnd div_ovrflw, LT             ;max out if divisor is too small
	lacc my_numerator, 16
	rpt #15
	 subc denom

	sacl denom
	lacl denom
	bit ans_sign, 0
	bcnd pos_div2, NTC
	neg
pos_div2:
	ret

err_div2:                               ; set flag and max out if zero
	lacc div_disaster
	add #1
	sacl div_disaster
	lacc #7fffh
	ret

div_ovrflw:                             ; max out if overflowed
	lacc #7fffh
	bit ans_sign, 0
	bcnd pos_div3, NTC
	neg
pos_div3:
	ret

