;*********************************************************************
;
; Filename:  INVMAP.ASM
;
;
; Author:  Chris Buehler, Jonas Keating
; Date: 11/20/95
; Last revision:  12/23/95
;
;*********************************************************************
;
; Description:  This file contains the following functions for
;		implementing the inverse mapper:
;		1 unscramble16             - unscrambles a word of data
;		2 build_bitstream          - reconstructs output data
;					     from the ibits, the qbits,
;					     and the shell mapper index
;		  2a build_bitstream96       - subfunction for 9600
;		  2b build_bitstream144      -     "        "  14400
;		  2c build_bitstream192      -     "        "  19200
;		3 compute_ibits            - calculates the ibits by
;					     inverting the differential
;					     encoder
;		4 find_point_index         - finds the index of a 2D point
; 					     in the constellation table
; 		5 get_binary_subset_label  - determines the subset of a
;					     2D point from its coords
;					     (see Appendix D of paper)
;		6 rotate90_cw              - rotates a 2D point by a
;					     multiple of 90 degrees
;					     clockwise
;
;
; Algorithm:  1 unscramble16
;		undoes the scrambling of the linear shift
;		feedback register which uses generating polynomial:
;			   1 + x^-18 + x^-23
;		Past bit values are stored in randstat, and the original
;		data stream is recovered by doing three shifts and
;		two XORs of the past data bits and the current input.
;		16 bits are recovered in each iteration of the algorithm.
;	      2 build_bitstream
;		Through a series of loads,shifts, and stores the different
;		groups of bits (ibits, qbits, and unshell mapper output)
;		are regouped into blocks of words.
;	      3 compute_ibits
;		The ibits are computed by inverting the equations of
;		the differential encoder.  Specifically, the following
;		equations are used:
;
;		(I3,I2) = znew - zold mod 4
;		(I1,U0) = w - znew mod 4
;
;		w,znew, and zold are determined from the input points.
;	      4 find_point_index
;		Determines the table constellation table index (from
;		the transmitter) of a point given its coordinates.  This
;		is accomplished by using an inverse lookup table which
;		is indexed by the coords of the point and contains its
;		index.
;	      5 get_binary_subset_label
;		Determines the subset label of a 2D point.  This is
;		useful for determining the rotation factor that was used
;		(znew or w) to obtain the current point.
;		See Appendix D for a description of the algorithm in more
;		detail.
;	      6 rotate90_cw
;		Uses swaps an negations to implement 90 degree rotations
;		based on the value in the accumulator.
;
;
;*********************************************************************
;
; Usage:  1  unscramble16
;	     point AR0 at the word to unscramble, call unscramble16
;	  2  build_bitstream
;	     the arrays ibits and qbits
;	     the accumulator contains the shell mapper index
;	  3  compute_ibits
;	     upon calling computes the ibits from zold,znew,w
;	     zold is the previous value of z (rotation factor Z)
;	     znew the current value of z
;	     w is rotation factor W
;         4  find_point_index
;	     upon calling finds the index of (vx, vy)
;	     from the index the function also stores the qbits
;	     and the ring index
;	  5  get_binary_subset_label
;	     upon calling finds the label of (ux, uy)
;	  6  rotate90_cw
;	     upon calling rotates the point (ux, uy) into (vx, vy)
; Inputs: 1  unscramble16
;	     AR0
;	  2  build_bitstream
;	     ibits, qbits, ACC contains shell mapper index
;	  3  compute_ibits
;	     zold,znew,w
;         4  find_point_index
;	     vx, vy
;	  5  get_binary_subset_label
;	     ux, uy
;	  6  rotate90_cw
;	     ux, uy, ACC contains multiple of 90 degrees to rotate by
;
;
; Outputs:1  unscramble16
;	     whatever AR0 pointed to
;	  2  build_bitstream
;	     bitstream (4 words)
;	  3  compute_ibits
;	     ibits (4 words)
;         4  find_point_index
;	     qbits (8 words)
;	     mjk (8 words)
;	  5  get_binary_subset_label
;	     ACC
;	  6  rotate90_cw
;	     vx, vy
;
;
;*********************************************************************
;
; Comments:  Together with the unshell mapper, these functions constitute
;            the inverse mapper portion of the receiver.
;
;*********************************************************************
  .mmregs
  .text
  .include macros.inc

  .global rotate90_ccw,find_point_index, compute_ibits
  .global vx,vy,ux,uy
  .global m,ibits,z,w,u0
  .global get_binary_subset_label
  .global recvdata1,yx,yy,invtab,saveAR0,saveAR1,saveAR2
  .global bitstream,ibits,qbits
  .global mjk,q,znew,zold,qmask
  .global randstat,bitbuf,bit_addr
  .global build_bitstream,build_bitstream96
  .global build_bitstream144,build_bitstream192


unscramble16:
;Assumes that AR0 points to the word to unscramble and is current

;  mar 	*,AR0
  ldpk	randstat
  lacc	randstat+1,16
  or	randstat
  sacl	randstat+1
  bsar	2
  sacb
  bsar	5
  xorb
  xor	*
  sacb
  lacl	*
  sacl	randstat
  lacb
  sacl	*
  ret


build_bitstream:
  sacb
  ldpk	bit_addr
  lacc	bit_addr
  bacc

build_bitstream96:

  ldpk	recvdata1
  lacb
;//  bitstream[0] = ACC&0xFFFFL;
  sacl	bitstream
;//  bitstream[1] = (ACC>>16)&0xF;
  bsar	16
  and	#000Fh
  sacb
;//  bitstream[1] |= (ibits[0]<<4);
;//  bitstream[1] |= (ibits[1]<<7);
;//  bitstream[1] |= (ibits[2]<<10);
;//  bitstream[1] |= (ibits[3]<<13);
  lacc	ibits+3,3
  or	ibits+2
  sacl	bitstream+1
  lacc	bitstream+1,3
  or	ibits+1
  sacl	bitstream+1
  lacc	bitstream+1,3
  or	ibits
  sacl	bitstream+1
  lacc	bitstream+1,4
  orb
  sacl	bitstream+1

  lar	AR0,#(bitstream+1)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+0)
  call	unscramble16,*,AR0

  ; check for data
  lacc  bitstream
  add	#1
  bcnd	no_data,EQ
;  lacc	bitstream
;  ldpk	bitbuf
;  sacl	bitbuf
;  ldpk  bitstream
;  lacc  bitstream+1
;  ldpk  bitbuf
;  sacl  bitbuf+1
  ldpk	recvdata1
no_data:
  ret

build_bitstream144:

  ldpk	recvdata1
  lacb
  sacl	bitstream
  bsar	16
  and	#0FFFh
  sacb
  lacc  qbits+0,3
  or	ibits+0
  sacl	bitstream+1
  lacc	bitstream+1,12
  orb
  sacl	bitstream+1			; Done bitstream+1
  lacc	qbits+7
  sfl
  or	qbits+6
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	ibits+3
  sfl
  or	qbits+5
  sfl
  or	qbits+4
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	ibits+2
  sfl
  or	qbits+3
  sfl
  or	qbits+2
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	ibits+1
  sfl
  or	qbits+1
  sacl	bitstream+2			; Done bitstream+2

  lar	AR0,#(bitstream+2)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+1)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+0)
  call	unscramble16,*,AR0

  ret

build_bitstream192:

  ldpk	recvdata1
  lacb
  sacl	bitstream
  bsar	16
  and	#0FFFh
  sacb
  lacc  qbits+0,3
  or	ibits+0
  sacl	bitstream+1
  lacc	bitstream+1,12
  orb
  sacl	bitstream+1			; Done bitstream+1
  lacc	ibits+2,3
  or	qbits+3
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	qbits+2
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	ibits+1
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	qbits+1
  sacl	bitstream+2
  lacc	bitstream+2,3
  or	qbits+0
  bsar	1
  sacl	bitstream+2		    	; Done bitstream+2
  lacc	qbits+7,3
  or	qbits+6
  sacl	bitstream+3
  lacc	bitstream+3,3
  or	ibits+3
  sacl	bitstream+3
  lacc	bitstream+3,3
  or	qbits+5
  sacl	bitstream+3
  lacc	bitstream+3,3
  or	qbits+4
  sacl	bitstream+3
  lacc	bitstream+3,3
  or	ibits+2
  bsar	2
  sacl	bitstream+3			; Done bitstream+3

  lar	AR0,#(bitstream+3)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+2)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+1)
  call	unscramble16,*,AR0
  lar	AR0,#(bitstream+0)
  call	unscramble16,*,AR0

  ret

compute_ibits:

  ldpk	recvdata1
  sar	AR0,saveAR0
  lacl	m
  sfr
  add	#ibits
  samm	AR0
  mar	*,AR0
;//  ibits[(m>>1)] =  ((znew-zold)&3)<<1;
  lacc	znew,1
  sub	zold,1
  and	#6
  sacb
;//  ibits[(m>>1)] |= ((w-znew)>>1)&1;
  lacl	w
  sub	znew
  sfr
  and	#1
  orb
  sacl	*

;//  zold = znew;
  lacl	znew
  sacl	zold
  ret

find_point_index:
;// invtab is 18x18

  ldpk	recvdata1
  sar	AR0,saveAR0
  sar	AR1,saveAR1
  sar	AR2,saveAR2

  lacl	m
  add	#qbits
  samm	AR1
  sub	#qbits
  add	#mjk
  samm	AR2

  lacc	vx

;//  tempx = vx+35;
  add	#35

;//  tempx >>= 2;
  bsar	2
  sacl	vx
  lt	vx
  mpy	#18

;//  tempy = vy+35;
  lacc	vy
  add	#35
;//  tempy >>= 2;
  bsar	2
  apac

;//  ACC = invtab[tempx*18 + tempy];
  add	#invtab
  samm	AR0
  mar	*,AR0
  nop
  lacc	*,AR1				; make AR1 = qbits[2*m] current

  sacb
  and	qmask
  sacl	*,AR2
  lacc	q				; q = Q-1, Q = number of qbits
  bcndd	fpi_noshift,LT
    lacb
    nop

  rpt	q
    sfr

fpi_noshift:
  sacl	*

  lar	AR0,saveAR0
  lar	AR1,saveAR1
  lar	AR2,saveAR2
  ret

get_binary_subset_label:
;//  yx = yx - 1;
;//  yx >>= 1;
  ldpk	recvdata1
  lacc	uy
  sacl	yy
  lacc	ux
  sub	#1
  sfr
  sacl	yx

;//  j1 = yx&1;
  and	#1
  sacl	temp1

;//  yy = yy - 1;
;//  yy >>= 1;
  lacc	yy
  sub	#1
  sfr
  sacl	yy

;//  j0 = (yx + yy)&1;
  add	yx
  and	#1
  sacl	temp0
  lacc	temp1
;//  ACC = (j1<<1) + j0;
  sfl
  or	temp0
  ret

rotate90_ccw:
;//  switch(ACC)
;//  {
;//	case 0:  vx = ux;
;//		 vy = uy;
;//		 break;
;//	case 1:  vx = -uy;
;//		 vy = ux;
;//		 break;
;//	case 2:  vx = -ux;
;//		 vy = -uy;
;//		 break;
;//	case 3:  vx = uy;
;//		 vy = -ux;
;//		 break;
;//  }

  bcnd	check1,NEQ
  lacc	ux
  sacl	vx
  lacc	uy
  sacl	vy
  ret
check1:
  sub	#1
  bcnd	check2,NEQ
  lacc	ux
  sacl	vy
  lacc	uy
  neg
  sacl	vx
  ret
check2:
  sub	#1
  bcnd	check3,NEQ
  lacc	ux
  neg
  sacl	vx
  lacc	uy
  neg
  sacl	vy
  ret
check3:
  lacc	ux
  neg
  sacl	vy
  lacc	uy
  sacl	vx
  ret

  .end