;
; Filename: ALL96.ASM
;
; Author: Carl Nuzman, Hemanth Sampath, Hayden Metz, Chris Buehler,
;   Jonas Keating
; Date:
; Last revision:
;
;*********************************************************************
;
; Description:  Main program: data and control for the entire receiver
;
; Algorithm:  The receiver is considered a state machine. Each state is
; 	defined by the subroutines stored in a jump table. The program
;	cycles through the subroutines at the sampling rate, 7200 Hz.
;	The receiver changes from one state to the next by rewriting the
; 	jump table. The state changes occur when a timing counter gets
;	to zero or in response to signals from the transmitter.
;
;	This file also contains two interrupt routines.
;	INT1 is used in communicating between the PC and the
;	EVM. XINT, the transmit interrupt, is used to sample the incoming
;	analog signal at the sample rate and control sampling clock.
;
;*********************************************************************
;******************************************************************
;                  DESCRIPTION OF RESERVED AUXILARY REGISTERS
;
;	The following auxilary registers may not be modified by any
;	routine other than xint:
;
; AR3 - running count 2,1,0 , which keeps track of where we are in a baud.
; AR5 - used by interrupt (xint) to write to inpbuffer.
;
;	All other aux registers may be modified by any routine. Each
;	routine is responsable for saving and restoring auxilary register
;	contents between calls.
;
;*******************************************************************
;		DATA FORMAT NOTATION
;
; The position of the binary point in our data is indicated with
; the following notation:
; x:y:z indicates that there are x sign bits, y integer bits, and
; z fractional bits.
;
; Examples:
;
; Form 1:1:15  means that the variable has 1 sign bit, 1 integer bit
; and 15 fractional bits.
;
; Form 1:5:10  means that the variable has 1 sign bit, 5 integer bits
; and 10 fractional bits.
;*********************************************************************


	.include macros.inc

; subroutines
	.global sym_clock
	.global slicer
	.global decode
	.global equalize
	.global eq_adapt

; dual access intprog variables
	.global hr1
	.global hr2
	.global hr3
	.global hrend
	.global hi1
	.global sintab

; main variables
	.global inpbuffer
	.global endbuffer

; symclock variables
	.global  NL
	.global  NH
	.global  TEMP0
	.global  TEMP1
	.global  TEMP2
	.global  DELTAH
	.global  DELTAL
	.global  COUNTL
	.global  SC_WAIT
	.global  VN
	.global  final
	.global  dc_0
	.global  dc_1
	.global  dc_bit
	.global  ac_0
	.global  ac_1
	.global  ac_2
	.global  ac_bit
	.global  x
	.global  L
	.global  thresh
	.global  SC_FIRST
	.global  SC_AR1
	.global  SC_AR2

; equalizer variables
	.global N1
	.global eq_t0
	.global eq_t1
	.global eq_t2
	.global eq_t3
	.global sigR
	.global sigI
	.global gamma
	.global Mu
	.global Munum
	.global errorR
      .global errorI
      .global cospre
      .global sinpre
      .global modR
      .global modI
      .global cnR
      .global cnI
      .global diffR
      .global diffI
      .global k1
      .global k2
      .global wcT
      .global angle_acc
      .global angle_accl
      .global angle
      .global angle_inc
      .global twoPi
      .global invPi
      .global sl_rind
      .global sl_iind
      .global sl_ims
      .global sl_bits
      .global sl_c_ims
      .global tapinit

; Carrier detect variables
	.global cdetect
	.global powercalc
	.global power			; p(n-1)	form 1:15:0
	.global c          		; 1:15:0;  may be replaced by constant
	.global one_minus_c		; 1:15:0;  may be replaced by constant
	.global high_power		; turn-on power threshold
	.global low_power		; turn-off power threshold
	.global time_over               ; time power > high_thresh
	.global time_under		; time power < low_thresh
	.global high_time		; time above h.p.t. before turn-on
	.global low_time		; time below l.p.t. before turn-off
	.global signal			; set if signal is present

; extra stuff
	.global bit_rate
	.global viterbR1,viterbR2
	.global viterbI1, viterbI2
	.global adapt_shift
	.global quadrant
	.global descram_in
	.global Chrisx
	.global Chrisy
      .global pack
      .global pack_word

; fast training variables
	.global fft
	.global inverse
	.global unscramble
	.global fast_equalise
	.global fr
	.global fi
	.global gr
	.global bitable
	.global ft_inv
	.global ft_temp
	.global ft_br
	.global ft_bi
	.global ft_maxindex
	.global three
	.global num
	.global den
	.global ans_sign
	.global div_disaster
	.global denom
	.global my_numerator
	.global stop_baby

; fft data
	.global t0r
	.global t0i
	.global t1r
	.global t1i
	.global t2r
	.global t2i
	.global t3r
	.global t3i
	.global t4r
	.global t4i
	.global aar
	.global aai
	.global bbr
	.global bbi
	.global ccr
	.global cci
	.global ddr
	.global ddi
	.global w1r
	.global w1i
	.global w2r
	.global w2i
	.global count4b
	.global wtable3b
	.global wr
	.global inpdebug, inpdebug2

; decode data
  .global	recvdata1
;  .global	get_new_point
  .global	quantize4,calculate_errors,viterbi_fill
  .global	get_binary_subset_label,trace_back
  .global	precoder_filter_imag,round_precoder_output,precoder_filter_real
  .global	build_bitstream96,rotate90_ccw,compute_ibits
  .global	calc_branch_errors,unshell_map
  .global	update_path_metrics,compute_un,find_point_index,compute_xn
  .global	un_shell_map,quantize96

  .global   	parm0,parm1,parm2
  .global	m
  .global	x1,y1,x2,y2
  .global	error1,error2
  .global	bptr,bpts,bend,bitstream
  .global	znew,ci,cr,pi,pr,w,yx,yy,ytx,yty
  .global	pptr, stateptr, cmm, states, vt_cpm_curr, upm_finish
  .global	randstat,bitbuf
  .global	zold,vt_ppp0,bptr,oldpptr,vt_cpm1,vt_cpm2,vt_ppp15
  .global	vt_cpm_prev,bit_addr
  .global	build_bitstream96,build_bitstream144,build_bitstream192
  .global	g2_addr,g4_addr,z8_addr,build_bitstream
  .global	g2,g4,z8,g2_14_19,g4_14_19,z8_14_19,q,qmask,numshell

	.mmregs


; constants to initialize the Analog Interface Circuit (AIC)
;For 7200 Hz: A= 24 , B = 30 , upper cutoff frequency = 5400 Hz:

aiccommand0  .set        03c78h		; set A = 24;
aiccommand1  .set        00205h        ; set A' register to smallest level
aiccommand2  .set        03062h 	;  set B = 30;
aiccommand3  .set	 02c7h
sl_unit .set 512

;***************************************************************************
; DUAL-ACCESS MEMORY
;***************************************************************************
	.sect "intdata"
hr1: .space 72*16		; Real equalizer coefficients, upper word
hr1cen: .word 04000h            ; 1:1:14 format
hr1plus: .space 70*16
hrend: .word 0
hr2: .space 144*16              ; Real equalizer coefficients, lower word

; Imaginary equalizer coefficients, upper word. 1:1:14 format.
; Initialized with a Hilbert transform
hi1:
	.word	00000h, 0ffeah, 00000h, 0ffedh, 00000h, 0fff8h, 00000h, 00009h
	.word	00000h, 0001fh, 00000h, 00030h, 00000h, 00037h, 00000h, 0002dh
	.word	00000h, 00012h, 00000h, 0ffebh, 00000h, 0ffc0h, 00000h, 0ff9dh
	.word	00000h, 0ff92h, 00000h, 0ffa7h, 00000h, 0ffdch, 00000h, 00028h
	.word	00000h, 00079h, 00000h, 000b7h, 00000h, 000cah, 00000h, 000a3h
	.word	00000h, 00042h, 00000h, 0ffb7h, 00000h, 0ff22h, 00000h, 0feaeh
	.word	00000h, 0fe87h, 00000h, 0fecah, 00000h, 0ff80h, 00000h, 00092h
	.word	00000h, 001ceh, 00000h, 002e3h, 00000h, 00375h, 00000h, 00321h
	.word	00000h, 0017ch, 00000h, 0fde8h, 00000h, 0f66ch, 00000h, 0d8a8h
	.word	00000h, 02758h, 00000h, 00994h, 00000h, 00218h, 00000h, 0fe84h
	.word	00000h, 0fcdfh, 00000h, 0fc8bh, 00000h, 0fd1dh, 00000h, 0fe32h
	.word	00000h, 0ff6eh, 00000h, 00080h, 00000h, 00136h, 00000h, 00179h
	.word	00000h, 00152h, 00000h, 000deh, 00000h, 00049h, 00000h, 0ffbeh
	.word	00000h, 0ff5dh, 00000h, 0ff36h, 00000h, 0ff49h, 00000h, 0ff87h
	.word	00000h, 0ffd8h, 00000h, 00024h, 00000h, 00059h, 00000h, 0006eh
	.word	00000h, 00063h, 00000h, 00040h, 00000h, 00015h, 00000h, 0ffeeh
	.word	00000h, 0ffd3h, 00000h, 0ffc9h, 00000h, 0ffd0h, 00000h, 0ffe1h
	.word	00000h, 0fff7h, 00000h, 00008h, 00000h, 00013h, 00000h, 00016h

hi2: .space 144*16 		;Imaginary equalizer coefficients, lower word

; pad out to next data page
     .space 64*16

;***************************************************************************
; INTERNAL DATA MEMORY
;***************************************************************************
	.data
;-------------------------data page 2 and 3-----------------------;
;main routine variables
main_page0	.set	2
main_page1	.set    3
disaster:   .word 0
	   .space 20*16
testxf:	   .word 0
int_AR4:   .word 0
int_called: .word 0
; circular buffer for storing received samples. 144 samples long
inpbuffer: .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	   .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	   .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	   .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	   .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
	   .word 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
endbuffer: .word 0

	.space 88*16
;-----------------------------------------------------------------;
;-------------------------DATA PAGE 4-----------------------------;
;symbol clock  recovery variables
;-----------------------------------------------------------------;
sc_page0	.set 	4
eq_page0	.set    4
;q15 format, inverse magnitude squared of constellation points
sl_ims:
	.word	04000h, 00ccch, 004ech, 0028fh, 0018fh, 0010ch
	.word	00ccch, 0071ch, 003c3h, 00234h, 0016ch, 000fch
	.word	004ech, 003c3h, 0028fh, 001bah, 00135h, 000e0h
	.word	0028fh, 00234h, 001bah, 0014eh, 000fch, 000c0h
	.word	0018fh, 0016ch, 00135h, 000fch, 000cah, 000a2h
	.word	0010ch, 000fch,	000e0h, 000c0h, 000a2h, 00087h

NL:        .word 0,0,0
NH:        .word 0,0,0
TEMP0:     .word 0
TEMP1:     .word 0
TEMP2:     .word 0
DELTAH:     .word 0
DELTAL:     .word 0
COUNTL:		.word 012c0h   ; 2 second counter
SC_WAIT:    .word 1
VN:        .word 0
final:	    .word 0
dc_0:	.word 0
dc_1:	.word 0
dc_bit: .word 0
ac_0:	.word 0
ac_1:	.word 0
ac_2:	.word 0
ac_bit: .word 0
x:  .word 0
L:	.word 0018h ; 		; threshold value for random walk filter
thresh: .word 0
SC_FIRST: 	.word 1
SC_AR0:		.word 0
SC_AR1:		.word 0
SC_AR2:		.word 0
SC_AR4:		.word 0
SC_AR6:		.word 0
SC_AR7:		.word 0
SC_AR22:	.word 0

;-------------------------------------------------------------------
;equalizer variables
;-------------------------------------------------------------------
N1: .word 144
eq_t0: .word 0
eq_t1: .word 0
eq_t2: .word 0
eq_t3: .word 0
sigR: .word 0        		; form= 1:15:0
sigI: .word 0        		; form= 1:15:0
; use 1:1:14 format:
gamma: .word 02fffh
Mu:	.word 01000h     	; form = 1:1:14
Munum:  .word 040d0h    	; 7 second delay
errorR: .word 0         	; form: 1:4:11
errorI: .word 0         	; form : 1:4:11
cospre: .word 0         	; form : 1:1:14
sinpre: .word 0         	; form : 1:1:14
modR: .word 0          	 	; form : 1:4:11
modI: .word 0           	; form : 1:4:11
cnR:  .word 0
cnI:  .word 0
diffR: .word 0          	; form : 1:4:11
diffI: .word 0          	; form : 1:4:11
k1: .word 00400h
k2: .word 00040h         	; form : 1:1:14
wcT: .word 025b2h        	; form : 1:4:11   ; 3*Pi/2
angle_acc: .word 0
angle_accl: .word 0
angle: .word 0
angle_inc: .word 0
twoPi: .word 03243h      	; form 1:4:11      ; 2*Pi
invPi: .word 028beh             ; 1/Pi
sl_rind	.word	0		; real index
sl_iind .word	0		; imaginary index
sl_bits	 .word 0
sl_c_ims .word 0
tapinit	.word 0

;--------------------------------------------------------------------
;carrier detect variables
;--------------------------------------------------------------------
power		.word 0
c		.word 07a00h
one_minus_c	.word 00600h
high_power	.word 00040h  ; threshold to turn on reciever
low_power	.word 00020h  ; threshold to turn off receiver
time_over	.word 0
time_under	.word 0
;high_time	.word 360
high_time	.word 60
low_time	.word 96
signal		.word 0
xycount		.word 0
ysend		.word 0
;---------------------------------------------------------------------
;other variables
;---------------------------------------------------------------------
lvltst		.word 0
bit_rate_start  .set  3
bit_rate	.word bit_rate_start
viterbR2	.word 0         ; input to viterbi decoder
viterbI2        .word 0
viterbR1	.word 0
viterbI1	.word 0
adapt_shift	.word adapt_shift_start
adapt_shift_start  .set  1 ;0
quadrant	.word 0
descram_in	.word 0
Chrisx		.word 0
Chrisy		.word 0
pack_word	.word 0,0         ; latest 32 received bits
	.space 0*16


;----------------------------------------------------------------------
; -----------------------DATA PAGE 5--------------------------
;----------------------------------------------------------------------
; PC-EVM interface variables
;----------------------------------------------------------------------
debug_temp	.word 0
debug_buffer	.space 12*16
int1data:
; table of routines PC can call
int1jtable:	.word sendx, sendy, readhr, readhi, readp, scaleup, scaledn
		.word readdb, readb, data_to_PC,readbb,badcommand
i1comm	       	.word 0
i1commadd	.word badcommand
ysave:		.word 0
filtadd:	.word 0
filtr:		.word hr1
filti:		.word hi1
i1saveAR0:	.word 0
bitptr:		.word bitbuf
tcont:		.word 030h

MAXCOMMAND:	.set  11   ; number of routines in int1jtable

;-----------------------------------------------------------------------
; main control variables
;-----------------------------------------------------------------------

mainjtable:	.space 24*16
maincounter:	.word 0
timing_count:	.word 0
timing_count2:  .word 0
randstat	.word 0, 0
;bitbuf		.word 0ffffh,0ffffh,0ffffh,0ffffh
shoot	.word 0
;-----------------------------------------------------------------------
; fast training data
;-----------------------------------------------------------------------
ft_inv: 	.word 0
ft_temp: 	.word 0
ft_br: 		.word 0
ft_bi: 		.word 0
ft_maxindex: 	.word 0
three:    	.word 3
num: 		.word 0
den: 		.word 0
ans_sign: 	.word 0
div_disaster 	.word 0            ; divide by zero warning flag
denom:		.word 0
my_numerator	.word 00040h       ; Shift up to increase output from
				   ; equation A8. Shift down to decrease
				   ; it. (See fast.asm).
;------------------------------------------------------------------------
;fft variables
;------------------------------------------------------------------------
t0r	.word 0
t0i     .word 0
t1r	.word 0
t1i	.word 0
t2r	.word 0
t2i	.word 0
t3r	.word 0
t3i	.word 0
t4r	.word 0
t4i	.word 0
aar	.word 0
aai     .word 0
bbr	.word 0
bbi	.word 0
ccr	.word 0
cci	.word 0
ddr	.word 0
ddi	.word 0
w1r	.word 0
w1i	.word 0
w2r	.word 0
w2i	.word 0
count4b	.word 0
wtable3b .word 0,      	0,	48,	96,	96,	48
	 .word  16,	32,	64,	128,	112,	80
	 .word  32,	64,	80,	16,	128,	112

	.space 11*16

;***********************************************************************
; EXTERNAL DATA MEMORY
;***********************************************************************
	.sect "extdata"

pcum:	.space 205*16

;-----------------------------------------------------------------------
;144 point hilbert transform using kaiser window, 40db attenuation
; center at line 73
; 1:1:14 format
;-----------------------------------------------------------------------
hilbert:
	.word	00000h, 0ffeah, 00000h, 0ffedh, 00000h, 0fff8h, 00000h, 00009h
	.word	00000h, 0001fh, 00000h, 00030h, 00000h, 00037h, 00000h, 0002dh
	.word	00000h, 00012h, 00000h, 0ffebh, 00000h, 0ffc0h, 00000h, 0ff9dh
	.word	00000h, 0ff92h, 00000h, 0ffa7h, 00000h, 0ffdch, 00000h, 00028h
	.word	00000h, 00079h, 00000h, 000b7h, 00000h, 000cah, 00000h, 000a3h
	.word	00000h, 00042h, 00000h, 0ffb7h, 00000h, 0ff22h, 00000h, 0feaeh
	.word	00000h, 0fe87h, 00000h, 0fecah, 00000h, 0ff80h, 00000h, 00092h
	.word	00000h, 001ceh, 00000h, 002e3h, 00000h, 00375h, 00000h, 00321h
	.word	00000h, 0017ch, 00000h, 0fde8h, 00000h, 0f66ch, 00000h, 0d8a8h
	.word	00000h, 02758h, 00000h, 00994h, 00000h, 00218h, 00000h, 0fe84h
	.word	00000h, 0fcdfh, 00000h, 0fc8bh, 00000h, 0fd1dh, 00000h, 0fe32h
	.word	00000h, 0ff6eh, 00000h, 00080h, 00000h, 00136h, 00000h, 00179h
	.word	00000h, 00152h, 00000h, 000deh, 00000h, 00049h, 00000h, 0ffbeh
	.word	00000h, 0ff5dh, 00000h, 0ff36h, 00000h, 0ff49h, 00000h, 0ff87h
	.word	00000h, 0ffd8h, 00000h, 00024h, 00000h, 00059h, 00000h, 0006eh
	.word	00000h, 00063h, 00000h, 00040h, 00000h, 00015h, 00000h, 0ffeeh
	.word	00000h, 0ffd3h, 00000h, 0ffc9h, 00000h, 0ffd0h, 00000h, 0ffe1h
	.word	00000h, 0fff7h, 00000h, 00008h, 00000h, 00013h, 00000h, 00016h

; fast training data space
fr: .space 144*16
fi: .space 144*16
gr: .space 144*16
;--------------------------------------------------------------------------
; 48-point fft of V.34 standard PP sequence
;--------------------------------------------------------------------------
bitable:
	.word	01800h, 00ddbh, 0199ah, 00a9ah, 01bb6h, 00000h, 015fch, 0ef22h
	.word	017ffh, 00ddbh, 01b79h, 0fc62h, 00ddbh, 0e800h, 0ef22h, 0ea04h
	.word	00000h, 0e44ah, 0e666h, 0f566h, 0f225h, 017ffh, 0199ah, 00a9ah
	.word	01800h, 00ddbh, 00a9ah, 0e666h, 0e44ah, 00000h, 010deh, 015fch
	.word	017ffh, 00ddbh, 0fc62h, 0e487h, 0f225h, 01800h, 015fch, 0ef22h
	.word	00000h, 0e44ah, 0f566h, 0199ah, 00ddbh, 0e801h, 0f566h, 0199ah
	.word	01800h, 00ddbh, 0e666h, 0f566h, 01bb6h, 00000h, 0ea04h, 010deh
	.word	017ffh, 00ddbh, 0e487h, 0039eh, 00ddbh, 0e800h, 010deh, 015fch
	.word	00000h, 0e44ah, 0199ah, 00a9ah, 0f225h, 017ffh, 0e666h, 0f566h
	.word	01800h, 00ddbh, 0f566h, 0199ah, 0e44ah, 00000h, 0ef22h, 0ea04h
	.word	017ffh, 00ddbh, 0039eh, 01b79h, 0f225h, 01800h, 0ea04h, 010deh
	.word	00000h, 0e44ah, 00a9ah, 0e666h, 00ddbh, 0e801h, 00a9ah, 0e666h

;-------------------------------------------------------------------------
;1.25 cycle cosine table - 180 words, 1:1:14 format
;This table is required by the FFT routine in file fft3.asm
; This table computes the cosine of the angle between 0 and 5*Pi/2.
; Hence, every quadrant has been assigned 36 words.
;-------------------------------------------------------------------------
wr:
	.word	04000h, 03ff0h, 03fc1h, 03f73h, 03f07h, 03e7bh, 03dd1h, 03d09h
	.word	03c23h, 03b20h, 03a00h, 038c4h, 0376ch, 035fah, 0346ch, 032c6h
	.word	03106h, 02f2fh, 02d41h, 02b3ch, 02923h, 026f5h, 024b5h, 02263h
	.word	02000h, 01d8dh, 01b0ch, 0187dh, 015e3h, 0133eh, 01090h, 00ddah
	.word	00b1dh, 0085ah, 00593h, 002cah, 00000h, 0fd36h, 0fa6dh, 0f7a6h
	.word	0f4e3h, 0f226h, 0ef70h, 0ecc2h, 0ea1dh, 0e783h, 0e4f4h, 0e273h
	.word	0e001h, 0dd9dh, 0db4bh, 0d90bh, 0d6ddh, 0d4c4h, 0d2bfh, 0d0d1h
	.word	0cefah, 0cd3ah, 0cb94h, 0ca06h, 0c894h, 0c73ch, 0c600h, 0c4e0h
	.word	0c3ddh, 0c2f7h, 0c22fh, 0c185h, 0c0f9h, 0c08dh, 0c03fh, 0c010h
	.word	0c000h, 0c010h, 0c03fh, 0c08dh, 0c0f9h, 0c185h, 0c22fh, 0c2f7h
	.word	0c3ddh, 0c4e0h, 0c600h, 0c73ch, 0c894h, 0ca06h, 0cb94h, 0cd3ah
	.word	0cefah, 0d0d1h, 0d2bfh, 0d4c4h, 0d6ddh, 0d90bh, 0db4bh, 0dd9dh
	.word	0e000h, 0e273h, 0e4f4h, 0e783h, 0ea1dh, 0ecc2h, 0ef70h, 0f226h
	.word	0f4e3h, 0f7a6h, 0fa6dh, 0fd36h, 00000h, 002cah, 00593h, 0085ah
	.word	00b1dh, 00ddah, 01090h, 0133eh, 015e3h, 0187dh, 01b0ch, 01d8dh
	.word	02000h, 02263h, 024b5h, 026f5h, 02923h, 02b3ch, 02d41h, 02f2fh
	.word	03106h, 032c6h, 0346ch, 035fah, 0376ch, 038c4h, 03a00h, 03b20h
	.word	03c23h, 03d09h, 03dd1h, 03e7bh, 03f07h, 03f73h, 03fc1h, 03ff0h
	.word	04000h, 03ff0h, 03fc1h, 03f73h, 03f07h, 03e7bh, 03dd1h, 03d09h
	.word	03c23h, 03b20h, 03a00h, 038c4h, 0376ch, 035fah, 0346ch, 032c6h
	.word	03106h, 02f2fh, 02d41h, 02b3ch, 02923h, 026f5h, 024b5h, 02263h
	.word	02000h, 01d8dh, 01b0ch, 0187dh, 015e3h, 0133eh, 01090h, 00ddah
	.word	00b1dh, 0085ah, 00593h, 002cah, 00000h

;--------------------------------------------------------------------------
;321 word sine table.    1:1:14 format
;64 words per 90 degrees + 1 , from 0 to 5pi/2
;--------------------------------------------------------------------------
sintab
	.word	00000h, 00192h, 00323h, 004b5h, 00645h, 007d5h, 00964h, 00af1h
	.word	00c7ch, 00e05h, 00f8ch, 01111h, 01294h, 01413h, 0158fh, 01708h
	.word	0187dh, 019efh, 01b5dh, 01cc6h, 01e2bh, 01f8bh, 020e7h, 0223dh
	.word	0238eh, 024dah, 0261fh, 0275fh, 02899h, 029cdh, 02afah, 02c21h
	.word	02d41h, 02e5ah, 02f6bh, 03076h, 03179h, 03274h, 03367h, 03453h
	.word	03536h, 03612h, 036e5h, 037afh, 03871h, 0392ah, 039dah, 03a82h
	.word	03b20h, 03bb6h, 03c42h, 03cc5h, 03d3eh, 03daeh, 03e14h, 03e71h
	.word	03ec5h, 03f0eh, 03f4eh, 03f84h, 03fb1h, 03fd3h, 03fech, 03ffbh
	.word	04000h, 03ffbh, 03fech, 03fd3h, 03fb1h, 03f84h, 03f4eh, 03f0eh
	.word	03ec5h, 03e71h, 03e14h, 03daeh, 03d3eh, 03cc5h, 03c42h, 03bb6h
	.word	03b20h, 03a82h, 039dah, 0392ah, 03871h, 037afh, 036e5h, 03612h
	.word	03536h, 03453h, 03367h, 03274h, 03179h, 03076h, 02f6bh, 02e5ah
	.word	02d41h, 02c21h, 02afah, 029cdh, 02899h, 0275fh, 0261fh, 024dah
	.word	0238eh, 0223dh, 020e7h, 01f8bh, 01e2bh, 01cc6h, 01b5dh, 019efh
	.word	0187dh, 01708h, 0158fh, 01413h, 01294h, 01111h, 00f8ch, 00e05h
	.word	00c7ch, 00af1h, 00964h, 007d5h, 00645h, 004b5h, 00323h, 00192h
	.word	00000h, 0fe6eh, 0fcddh, 0fb4bh, 0f9bbh, 0f82bh, 0f69ch, 0f50fh
	.word	0f384h, 0f1fbh, 0f074h, 0eeefh, 0ed6ch, 0ebedh, 0ea71h, 0e8f8h
	.word	0e783h, 0e611h, 0e4a3h, 0e33ah, 0e1d5h, 0e075h, 0df19h, 0ddc3h
	.word	0dc72h, 0db26h, 0d9e1h, 0d8a1h, 0d767h, 0d633h, 0d506h, 0d3dfh
	.word	0d2bfh, 0d1a6h, 0d095h, 0cf8ah, 0ce87h, 0cd8ch, 0cc99h, 0cbadh
	.word	0cacah, 0c9eeh, 0c91bh, 0c851h, 0c78fh, 0c6d6h, 0c626h, 0c57eh
	.word	0c4e0h, 0c44ah, 0c3beh, 0c33bh, 0c2c2h, 0c252h, 0c1ech, 0c18fh
	.word	0c13bh, 0c0f2h, 0c0b2h, 0c07ch, 0c04fh, 0c02dh, 0c014h, 0c005h
	.word	0c000h, 0c005h, 0c014h, 0c02dh, 0c04fh, 0c07ch, 0c0b2h, 0c0f2h
	.word	0c13bh, 0c18fh, 0c1ech, 0c252h, 0c2c2h, 0c33bh, 0c3beh, 0c44ah
	.word	0c4e0h, 0c57eh, 0c626h, 0c6d6h, 0c78fh, 0c851h, 0c91bh, 0c9eeh
	.word	0cacah, 0cbadh, 0cc99h, 0cd8ch, 0ce87h, 0cf8ah, 0d095h, 0d1a6h
	.word	0d2bfh, 0d3dfh, 0d506h, 0d633h, 0d767h, 0d8a1h, 0d9e1h, 0db26h
	.word	0dc72h, 0ddc3h, 0df19h, 0e075h, 0e1d5h, 0e33ah, 0e4a3h, 0e611h
	.word	0e783h, 0e8f8h, 0ea71h, 0ebedh, 0ed6ch, 0eeefh, 0f074h, 0f1fbh
	.word	0f384h, 0f50fh, 0f69ch, 0f82bh, 0f9bbh, 0fb4bh, 0fcddh, 0fe6eh
	.word	00000h, 00192h, 00323h, 004b5h, 00645h, 007d5h, 00964h, 00af1h
	.word	00c7ch, 00e05h, 00f8ch, 01111h, 01294h, 01413h, 0158fh, 01708h
	.word	0187dh, 019efh, 01b5dh, 01cc6h, 01e2bh, 01f8bh, 020e7h, 0223dh
	.word	0238eh, 024dah, 0261fh, 0275fh, 02899h, 029cdh, 02afah, 02c21h
	.word	02d41h, 02e5ah, 02f6bh, 03076h, 03179h, 03274h, 03367h, 03453h
	.word	03536h, 03612h, 036e5h, 037afh, 03871h, 0392ah, 039dah, 03a82h
	.word	03b20h, 03bb6h, 03c42h, 03cc5h, 03d3eh, 03daeh, 03e14h, 03e71h
	.word	03ec5h, 03f0eh, 03f4eh, 03f84h, 03fb1h, 03fd3h, 03fech, 03ffbh
	.word	04000h

points: .space 128*16
pointadd: .word points
pointdead: .word -1
bitbuf:  .word 0ffffh, 0ffffh,0ffffh,0ffffh
errcount: .word 0
bcount:	  .word 0

inpdebug	.space 144*16
inpdebug2       .space 144*16

;**************************************************************************
;  MACROS
;**************************************************************************

; Initialize the dsp
init .macro  ; set up simple initialization
  ldp   #0
			; turn off interrupts
  setc  INTM
  clrc  CNF
  setc  SXM
  clrc  OVM
  splk  #0h,IMR
			; set up wait states
  splk  #0, CWSR
  splk  #0, PDWSR
  splk  #05555h, IOWSR
			; set up memory mode
  splk  #003eh, PMST

  splk  #08h,SPC        ; reset serial port
  opl   #0c0h,SPC       ; enable xmit and recv

  splk #0, PA2
  rpt   #40h            ; wait a bit
  nop
  splk  #0,DXR          ; initialize serial port .
  splk  #0, DRR
  splk  #030h, PA2
  spm 0			; make sure product mode is right

; send the AIC initialization commands
  lacc #3
  poll_xmit
  lacc #aiccommand0
  poll_xmit
  lacc #3
  poll_xmit
  lacc #aiccommand1
  poll_xmit
  lacc #3
  poll_xmit
  lacc #aiccommand2
  poll_xmit
  lacc #3
  poll_xmit
  lacc #aiccommand3
  poll_xmit

 .endm

;transmit macro

poll_xmit .macro
  ldp   #0
poll1?:
  bit   SPC, 4    	; test bit 11, the XRDY bit
  bcnd  poll1?,NTC      ; if 0, continue to loop
  samm  DXR             ; on transition to 1, write to DXR

  .endm

;**********************************************************
; Interrrupt locations
;**********************************************************

 .sect "vectors"
RS      B       _main
INT1    B       _int1          	;External interrupt 1
INT2    B       _int2          	;External interrupt 2
INT3    B       _int3          	;External interrupt 3
TINT    B       _int4          	;Timer interrupt
RINT    B       _rint          	;Serial port receive interrupt
XINT    B       _xint          	;Serial port transmit interrupt
TRNT    B       _trnt          	;TDM port receive interrupt
TXNT    B       _txnt          	;TDM port transmit interrupt
INT4    B       _int4          	;External interrupt 4
	.space  14*16 		;Reserves space -- 14 words
	.space  2*16            ;Software trap routine
NMI     B       _nonmask        ;Nonmaskable external interrupt
	.space  2*16            ;Reserved space
	.space  24*16           ;12 software interrupt vectors


;*******************************************************************
; PROGRAM MEMORY
;*******************************************************************
      .text


;-------------------------------------------------------------------
;MAIN LOOP
;-------------------------------------------------------------------
; Loop through the 24-element jump table, calling each subroutine in
; the table in order. Wait for one sampling interrupt to be called
; each time through the loop.
;---------------------- --------------------------------------------
_main:


	init
	call recv_start
	call power_start
	ldpk maincounter
	splk #(24), maincounter

mainloop:
	ldpk int_called
	apl #0000h, int_called
sym_sty bit int_called, 15
	bcnd sym_sty, NTC    	; wait for the sampling interrupt

	ldpk maincounter
	lacc maincounter
	sub #1
	bcnd mn_skip, GEQ
	lacc #23
mn_skip:
	sacl maincounter
	neg
	add #(mainjtable+23)
	samm AR0
	mar *, AR0
	nop
	lacc *
	cala
	b mainloop

;**************************************************************************
;  Routines handling the various states of the receiver
;**************************************************************************

detect_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnon, LT
	ret
turnon:
	ldpk disaster
	apl #0, disaster
	ldpk timing_count
	splk #(4824), timing_count  ;two seconds plus change
	;rewrite jump table
	ldpk	maincounter
	splk	#24,maincounter

	lar AR0, #mainjtable
	mar *, AR0
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	splk #dotting_0, *+
	splk #dotting_0, *+
	splk #dotting_2, *+
	ret


dotting_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	ret
dotting_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	ldpk timing_count
	lacc timing_count
	sub #1
	bcnd init_fasttrain,LT
	sacl timing_count
	ret

init_fasttrain:
	ldpk disaster
	apl #0, disaster
	call fast_equalise
	ldpk angle_acc
	splk #0, angle_acc
	splk #0, angle_accl
	splk #0, angle
	lacc #(864)        ; 3*288, length of PP sequence (could be less)
	ldpk disaster
	sub disaster          ; number of sampling instants used by
			      ; fast_equalise
	ldpk timing_count
	sacl timing_count

	;change jump table
	ldpk  maincounter
	splk  #24,maincounter
	lar AR0, #mainjtable
	mar *, AR0
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	splk #fasttrain_0, *+
	splk #fasttrain_0, *+
	splk #fasttrain_2, *+
	ret

fasttrain_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	ret
fasttrain_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	ldpk timing_count
	lacc timing_count
	sub #3
	bcnd init_training,LT
	sacl timing_count
	ret


init_training:
	ldpk SC_WAIT
	splk #0, SC_WAIT                ; symbol clock stable flag
	splk #0080h, L 			; down-shift symbol clock threshold

	;change jump table
	ldpk timing_count
	splk #(0), timing_count	;
	splk  #24,maincounter
	lar AR0, #mainjtable
	lar AR0, #mainjtable
	mar *, AR0
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #training_2, *+
	ret


training_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	ret
training_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	call eq_adapt
	call decode
	; search for magic word indicating end of training
	ldpk pack_word
	lacc pack_word+1,16
	or   pack_word
	add #1
	bcnd got_it, EQ
	ret

got_it:
	ldpk timing_count
	splk #(64), timing_count
	lar AR0, #mainjtable
	mar *, AR0
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	splk #gotit_0, *+
	splk #gotit_0, *+
	splk #gotit_2, *+
	ret

gotit_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	ret

gotit_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	call eq_adapt
	call decode
	ldpk timing_count
	lacc timing_count
	sub #1
	bcnd init_full, EQ
	sacl timing_count
	sacb
	and #00007h
	cc store_it, EQ
	ret

store_it:
	lacb
	bsar 3
	neg
	add #(debug_buffer+7)
	samm AR0
	ldpk pack_word
	lacc pack_word
	mar *, AR0
	sacl *
	ret

init_full:
	; adjust adapt_shift, which controls the rate of equalizer adaption
	; larger values slow it down and make it more stable.
	; Must be increased to handle bigger constellations.
	ldpk adapt_shift
	lacc adapt_shift
	add #4
	sacl adapt_shift

	; adjust the scaling of values going into the slicer
	; subtract 1 for 9600, 2 for 14400, 3 for 19200.
	lacc bit_rate
	ldpk debug_buffer
	sub debug_buffer	;secret code from transmitter
	ldpk bit_rate
	sacl bit_rate
	ldpk debug_buffer
	lacc debug_buffer
	ldpk bit_addr
	sub #1
	bcnd speed_96, EQ
	sub #1
	bcnd speed_144, EQ
	sub #1
	bcnd speed_192, EQ
	b stop_baby
speed_96:
	splk #g2, g2_addr
	splk #g4, g4_addr
	splk #z8, z8_addr
	splk #6, numshell
	splk #build_bitstream96,bit_addr
	ldpk recvdata1
	splk #(-1),q
	splk #0,qmask
	b keep_on_rollin
speed_144:
	splk #build_bitstream144,bit_addr
	splk #g2_14_19, g2_addr
	splk #g4_14_19, g4_addr
	splk #z8_14_19, z8_addr
	splk #12, numshell
	ldpk recvdata1
	splk #0,q
	splk #01h,qmask
	b keep_on_rollin
speed_192:
	splk #build_bitstream192,bit_addr
	splk #g2_14_19, g2_addr
	splk #g4_14_19, g4_addr
	splk #z8_14_19, z8_addr
	splk #12, numshell
	ldpk recvdata1
	splk #2,q
	splk #07h,qmask

	ldpk bit_rate
	lacc bit_rate
	add  #2
	sacl bit_rate			; set rate for 16 pt
	b init_train16				; training sequence

	;change jump table
keep_on_rollin:
	ldpk timing_count
	splk #(0), timing_count	;
	; init viterbi variables
	ldpk  maincounter
	splk  #24,maincounter		; reset main loop
	lar AR0, #mainjtable            ;   two decode 2 points
	mar *, AR0                      ;   to initialize viterbi variables
	splk #full_0, *+
	splk #full_0, *+
	splk #full_2, *+
	splk #full_0, *+
	splk #full_0, *+
	splk #full_1, *+

;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
;	splk #full_0, *+
;	splk #full_0, *+
;	splk #full_2, *+
	ret

init_train16:
	ldpk timing_count
	splk #(4800), timing_count	; train for 4800 bauds
	; init viterbi variables
	ldpk  maincounter
	splk  #24,maincounter
	lar AR0, #mainjtable
	mar *, AR0
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	splk #training_0, *+
	splk #training_0, *+
	splk #train16_2, *+
	ret

train16_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	call eq_adapt
	ldpk timing_count
	lacc timing_count
	sub #1
	bcnd go_for_192, EQ
	sacl timing_count
	ret

go_for_192:
	ldpk bit_rate
	lacc bit_rate
	sub  #2
	sacl bit_rate
	ldpk timing_count
	splk #(0), timing_count	;
	; init viterbi variables
	ldpk  maincounter
	splk  #24,maincounter		; reset main loop
	lar AR0, #mainjtable            ;   two decode 2 points
	mar *, AR0                      ;   to initialize viterbi variables
	splk #full_0, *+
	splk #full_0, *+
	splk #full_2, *+
	splk #full_0, *+
	splk #full_0, *+
	splk #full_1, *+
	ret

full_0:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	ret
full_1:
	ldpk	recvdata1
	splk	#0,m			; reset m counter
	lar AR0, #mainjtable
	mar *, AR0
	splk #full_calc_errors, *+
	splk #full_upm1, *+
	splk #full_equalizer, *+
	splk #full_upm2, *+
	splk #full_inverse_map, *+
	splk #full_equalizer, *+
	splk #full_calc_errors, *+
	splk #full_upm1, *+
	splk #full_equalizer, *+
	splk #full_upm2, *+
	splk #full_inverse_map, *+
	splk #full_equalizer, *+
	splk #full_calc_errors, *+
	splk #full_upm1, *+
	splk #full_equalizer, *+
	splk #full_upm2, *+
	splk #full_inverse_map, *+
	splk #full_equalizer, *+
	splk #full_calc_errors, *+
	splk #full_upm1, *+
	splk #full_equalizer, *+
	splk #full_upm2, *+
	splk #full_inverse_map, *+
	splk #full_equalizer, *+
full_2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock
	call equalize
	call eq_adapt
	ret

full_calc_errors:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock

	ldpk viterbR2
	lacc viterbR2
	ldpk recvdata1
	sacl ytx

	ldpk viterbI2
	lacc viterbI2
	ldpk recvdata1
	sacl yty

	splk #x1,parm0
	splk #y1,parm1
	call quantize4,*,AR0

	splk #error1,parm2
	call calculate_errors,*,AR0

;	call get_new_point
	ldpk viterbR1
	lacc viterbR1
	ldpk recvdata1
	sacl ytx
	ldpk viterbI1
	lacc viterbI1
	ldpk recvdata1
	sacl yty

	splk #x2,parm0
	splk #y2,parm1
	call quantize4,*,AR0

	splk #error2,parm2
	call calculate_errors,*,AR0
	call calc_branch_errors,*,AR0
	ret

full_upm1:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock

	ldpk recvdata1
	splk #03FFFh,cmm+1
	splk #0FFFFh,cmm
	splk #states,stateptr
	call update_path_metrics,*,AR0

	; update pointers for next time through
	ldpk stateptr
	lacl stateptr
	add  #32
	sacl stateptr
	lacl vt_cpm_curr
	add  #16
	sacl vt_cpm_curr
	lacl pptr
	add  #16
	sacl pptr
	ret

full_upm2:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock

	ldpk recvdata1
	call update_path_metrics,*,AR0
	call upm_finish
	call trace_back

	ldpk pointadd
	lar  AR0,pointadd
	mar  *,AR0
	ldpk yx
	lacl yx
	sacl *+
	lacl yy
	sacl *+
	lacl yx+1
	sacl *+
	lacl yy+1
	sacl *+
	ldpk pointadd
	lacc pointadd
	add  #4
	sacl pointadd
	sub  #pointadd
	nop
	nop
	xc   2, EQ
	  splk  #(pointadd-4), pointadd
	ret

full_inverse_map:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ
	call sym_clock

	ldpk recvdata1
	lacc yx
	bcnd inverse_map_end,EQ

	call compute_un
	call get_binary_subset_label
	sacl znew
	call rotate90_ccw

	call find_point_index

	lacc yx+1
	sacl yx
	lacc yy+1
	sacl yy
	lacl m
	add  #1
	sacl m

;	call compute_xn
	call compute_un
	call get_binary_subset_label
	sacl w
	call rotate90_ccw

	call find_point_index

	call compute_ibits
	lacl m
	add  #1
	and  #7
	sacl m
	bcnd inverse_map_end,NEQ
	call un_shell_map
	call build_bitstream

	ldpk bcount
	lacl bcount
	add  #1
	sacl bcount

	ldpk bitstream
	lacc bitstream
	add  #1
	bcnd inverse_map_end,EQ
	lacc bitstream+1,16
	or   bitstream
	ldpk bitbuf
	sach bitbuf+1
	sacl bitbuf
	ldpk bitstream
	lacc bitstream+3,16
	or   bitstream+2
	ldpk bitbuf
	sach bitbuf+3
	sacl bitbuf+2
	lacl errcount
	add  #1
	sacl errcount

inverse_map_end:
	ret

full_decode_data:
	call full_inverse_map
	call un_shell_map
	call build_bitstream
	ret

full_equalizer:
	call cdetect
	call pwrite
	lacc signal
	bcnd turnoff, EQ

	call sym_clock
	call equalize            ; call fast_eq at this point.
	call eq_adapt
	ret

;jump here for debugging purposes
stop_baby:
	ldp #0
	splk #00001h, IMR ;turn off everything but int1
halt_baby:
	b halt_baby


;---------------------------------------------------------------------
; reinitialize variables when carrier is lost
;---------------------------------------------------------------------
turnoff:
	mar *,AR2
	lar AR2, #hr1
	rptz #0023fh				; 575 to get hr1 thru hi2
	sacl *+					; 287 for only hr parts
	ldpk hr1cen
	opl #04000h, hr1cen
	ldpk hi1
	lar AR2, #hi1
	rpt #143
	bldd #hilbert, *+


;zero pages two and three

	lar AR2, #disaster
	rptz #255
	sacl *+

	ldpk sc_page0
	lar AR2, #NL
	rptz #91
	sacl *+
	opl #012c0h, COUNTL
	opl #00001h, SC_WAIT
	opl #008h, L
	opl #00001h, SC_FIRST
	opl #00090h, N1
	opl #02fffh, gamma
	opl #01000h, Mu
	opl #040d0h, Munum
	opl #00400h, k1
	opl #00040h, k2
	opl #025b2h, wcT
	opl #03243h, twoPi
	opl #028beh, invPi
	opl #07a00h, c
	opl #00600h, one_minus_c
	opl #00040h, high_power
	opl #00020h, low_power
	opl #00168h, high_time
	opl #00060h, low_time

	opl #(bit_rate_start), bit_rate
	opl #(adapt_shift_start), adapt_shift

	lar AR7,#pcum                  		; setup power array pointer and
	sar AR7, SC_AR7
	apl #00000h, SC_AR22
	ldpk	pack_word
	splk	#0,pack_word
	splk	#0,pack_word+1

; data for cvscop3 -- need to check on which of these need to be reset

	ldpk int1data
	apl #00000h, i1comm
	apl #00000h, ysave
	apl #00000h, filtadd
	apl #00000h, i1saveAR0
	splk #030h, tcont
	lar AR2, #pcum			; pcum
	mar *,AR2
	rptz #199
	sacl *+

; initialize the viterbi decoder variables

	ldpk bptr
	splk #bpts, bptr
	splk #vt_ppp0, pptr
	splk #states, stateptr
	splk #vt_ppp15, oldpptr
	splk #vt_cpm1, vt_cpm_prev
	splk #vt_cpm2, vt_cpm_curr
	splk #0, zold
	splk #0, znew
	splk #0, w
	splk #0, m
	lar  AR2, #bpts
	mar  *,AR2
	rptz #511
	 sacl *+
	ldpk bitbuf
	splk #0ffffh, bitbuf
	splk #0ffffh, bitbuf+1
	splk #0ffffh, bitbuf+2
	splk #0ffffh, bitbuf+3
	ldpk randstat
	splk #0,randstat
	splk #0,randstat+1


; initialize stuff from fast_training and the fft
; zero out data page 5

	ldpk ft_inv
	lar AR2, #ft_inv
	mar *,AR2
	rptz #50               			; 127-36 (skipping sl_c_ims table)
	sacl *+

	opl #00040h,my_numerator



	ldpk wtable3b
	lar AR2,#wtable3b
	mar *,AR2
	opl #0h,*+
	opl #0h,*+
	opl #48,*+
	opl #96,*+
	opl #96,*+
	opl #48,*+
	opl #16,*+
	opl #32,*+
	opl #64,*+
	opl #128,*+
	opl #112,*+
	opl #80,*+
	opl #32,*+
	opl #64,*+
	opl #80,*+
	opl #16,*+
	opl #128,*+
	opl #112,*

; intialize jump table

	lar AR0, #mainjtable
	mar *, AR0
	lacc #detect_0
	rpt #23
	sacl *+
	ldp #0


	splk #endbuffer,ARCR

	call power_start

	ret

;-------------------------------------------------------------------
;other subroutines
;-------------------------------------------------------------------
;write power values for use by PC-EVM interface
pwrite:
	ldpk SC_AR7
	lar AR7, SC_AR7
	lar AR2, SC_AR22

	mar *,AR2
	lamm AR2
	mar *+,AR7
	nop

	sub #199                     	; test if current power pointer
	bcnd calcdone,GT		; is beyond power array
	lacl power			; write power to power array
	sacl *+
	b nocalc

calcdone:
	lar AR2,#0		    	; reset power counter and pointer
	lar AR7,#pcum

nocalc:
	sar AR7, SC_AR7
	sar AR2, SC_AR22

	ret

; called before starting receiver
recv_start:
;intialize jump table
	lar AR0, #mainjtable
	mar *, AR0
	lacc #detect_0
	rpt #23
	sacl *+
	ldpk adapt_shift
	lacc #(adapt_shift_start)
	sacl adapt_shift
	zap
	lar AR5, #inpbuffer
	ldp #0
	splk #endbuffer,ARCR
	splk  #0021h, IMR     	; enable the transmit interrupt and int1;
	smmr  IFR,IFR
	lar AR3, #3
	clearmsg
	clrc  INTM            	; go interupts
	ret


power_start:
	ldpk SC_AR1
	lar AR7,#pcum                  	; setup power array pointer and
	mar *,AR2			;   counter
	lar AR2,#0
	sar AR7, SC_AR7
	sar AR2, SC_AR22
	ret



;************************************************************************
; VARIABLE DEFINITIONS FOR PC-EVM INTERFACE
;************************************************************************

my_output	.set modR 	; transmitted to AIC
cv_output_x	.set Chrisx     ; x-value for constellation display
cv_output_y	.set Chrisy     ; y-value for constellation display
my_scale	.set bit_rate   ; variable adjusted by "," and "." keys
				; and displayed by "b" key
my_coefr	.set hr1        ; array displayed by "r" key
my_coefi	.set hi1        ; array displayed by "i" key
my_coefp	.set pcum	; array displayed by "p" key

;************************************************************************
; INTERRUPT HANDLERS
;************************************************************************

;------------------------------------------------------------------------
; PC-EVM interface interrupt
;------------------------------------------------------------------------
_int1:

	ldpk	int1data
	sar	AR0,i1saveAR0
	lamm	PA2 			; check for command sent by host
	and	#1
	bcnd	i1_nocomm,EQ		; if = 0, then data accessed


	lamm	PA1			; read the sent command, clear flag
	sacl	i1comm
	bcnd	badcommand,LT		;   if command < 0 then BAD
	sub	#MAXCOMMAND
	bcnd	badcommand,GT		;   if command > MAX then BAD
					; form subroutine address
	add	#(int1jtable+MAXCOMMAND)
	samm	AR0
	mar	*,AR0
	nop
	lacc	*
	sacl	i1commadd		; save command address
i1_nocomm:
	clearmsg
	ldpk	int1data
	lacl	i1commadd		; load command address
	bacc				; branch to command address

;send contents of bitbuf to PC
readbb:
	ldpk	int1data
	splk	#readbb2, i1commadd
	splk	#(bitbuf), filtadd
readbb2:
	ldpk	int1data
	lacl	filtadd
	samm	AR0
	mar	*,AR0
	nop
	lacc	*+
	samm	PA0
	sar	AR0,filtadd
	lar	AR0,i1saveAR0
	rete
data_to_PC:
	ldpk	int1data
	splk	#data_to_PC2, i1commadd
	ldpk	bitptr
	splk	#(bitbuf), bitptr
	lacc	#bitbuf
	samm	PA0
	ldpk	int1data
	lar	AR0,i1saveAR0
	rete
data_to_PC2:
	ldpk	bitptr
	lacl	bitptr
	samm	AR0
	mar	*,AR0
	nop
	lacc	*
	samm	PA0
	splk	#0ffffh,*+
	sar	AR0,bitptr
	ldpk	int1data
	lar	AR0,i1saveAR0
	rete

; send x and y constellation points continuously to PC
sendx:
	ldpk	cv_output_x
	lacl	cv_output_x
	samm	PA0
	ldpk	cv_output_y
	lacl	cv_output_y
	ldpk	int1data
	sacl	ysave
	splk	#sendy, i1commadd
	lar 	AR0, i1saveAR0
	rete
sendy:
	ldpk	int1data
	lacl	ysave
	samm	PA0
	splk	#sendx, i1commadd
	lar 	AR0, i1saveAR0
	rete

; send value of my_scale variable to PC
readb:
	ldpk	my_scale
	lacl	my_scale
	samm	PA0
	ldpk	int1data
;	splk	#readb, i1commadd
	lar 	AR0, i1saveAR0
	rete

; send contents of debug_buffer to PC
readdb:
	ldpk	int1data
	splk	#readdb2, i1commadd
	splk	#(debug_buffer-1), filtadd
readdb2:
	ldpk	int1data
	lacl	filtadd
	add	#1
	samm	AR0
	mar	*,AR0
	sacl	filtadd
	lacc	*
	samm	PA0
	lar	AR0,i1saveAR0
	rete

; send contents of my_coefr array to PC
readhr:
	ldpk	int1data
	splk	#readhr2, i1commadd
	splk	#(my_coefr-1), filtadd
readhr2:
	ldpk	int1data
	lacl	filtadd
	add	#1
	samm	AR0
	mar	*,AR0
	sacl	filtadd
	lacc	*
	samm	PA0
	lar	AR0,i1saveAR0
	rete

; send contents of my_coefi array to PC
readhi:
	ldpk	int1data
	splk	#readhi2, i1commadd
	splk	#(my_coefi-1), filtadd
readhi2:
	ldpk	int1data
	lacl	filtadd
	add	#1
	samm	AR0
	mar	*,AR0
	sacl	filtadd
	lacc	*
	samm	PA0
	lar	AR0,i1saveAR0
	rete

; send contents of my_coefp array to PC
readp:
	ldpk	int1data
	splk	#readp2, i1commadd
	splk	#(my_coefp-1), filtadd
readp2:
	ldpk	int1data
	lacl	filtadd
	add	#1
	samm	AR0
	mar	*,AR0
	sacl	filtadd
	lacc	*
	samm	PA0
	lar	AR0,i1saveAR0
	rete

; increment my_scale variable
scaleup:
	ldpk	my_scale
	lacc	my_scale
	add 	#1
	sacl	my_scale
	samm 	PA0
	lar	AR0,i1saveAR0
	rete

; decrement my_scale variable
scaledn:
	ldpk	my_scale
	lacc	my_scale
	sub 	#1
	sacl	my_scale
	samm 	PA0
	lar	AR0,i1saveAR0
	rete

badcommand:
	ldpk	int1data
	lacc	#badcommand
	sacl	i1commadd
	lacc	#666
	samm	PA0
	lar 	AR0, i1saveAR0
	rete
	rete

_int2:  rete
_int3:  rete
_tint:  rete
_rint:  rete


;-------------------------------------------------------------------------
;Transmit interrupt:
; 	Used to read and write samples from the AIC at 7200Hz and control
;	the AIC clock
;-------------------------------------------------------------------------
_xint:


; AR5 keeps track of next input buffer location
; AR3 is running count 2,1,0,2,1,0,etc.

	ldpk int_called              	; main loop zeroes int_called flag
	lacc int_called
	bcnd int_ok, EQ
					; count number of times interrupt
	lacc disaster     		; is called before loop is done
	add #1
	sacl disaster
int_ok	opl #0ffffh, int_called

	mar *, AR3                   	; decrement AR3.
	banz decr
	lar AR3,#2
decr

					;increment AR5 and keep it in bounds
	mar *,AR5
	mar *+
	lamm AR5
	nop
	nop
	sub #(inpbuffer+143)
	bcnd cont, LEQ
	lar AR5, #inpbuffer
cont

; INPUT - copy data from DRR to inpbuffer

	lamm  DRR
	nop
	nop
	sacl *
	ldpk lvltst                    ; store current value in lvltst
	sacl lvltst                    ; for debugging.
	ldpk thresh

; OUTPUT - send aic commands, control xf bit

	lamm AR3
	nop
	nop
	bcnd  gofor,EQ

	sub #1
	bcnd debug, EQ

; AR3 = 1 or 2; no AIC clock adjustment
	ldpk my_output
	lacc my_output
	and #0fffch
	samm DXR
	rete

;AR3 =0 ---> beginning of a baud;
;Send aic commands and toggle XF bit.

gofor
	lacc thresh
	bcnd advance,LT
	bcnd retard,GT

; no advance or retard requested
	ldpk my_output
	lacc my_output
	and #0fffch
	samm DXR
	b toggle

retard
	ldpk my_output
	lacc my_output
	apl #0,thresh
	and #0fffch
	or #2
	samm DXR
	b toggle

advance
	ldpk my_output
	lacc my_output
	apl #0,thresh
	and #0fffch
	or #1
	samm DXR

; toggle XF bit for oscilloscope synchronization
toggle
	ldpk int_AR4
	sar AR4, int_AR4
	lar AR4, #testxf
	mar *, AR4
	sst #1, *
	xpl #0010h, *
	lst #1, *
	lar AR4, int_AR4
	rete


; for debugging purposes
debug
	ldpk my_output
	lacc my_output
	and #0fffch
	samm DXR
	nop
	nop
	rete

_trnt:  rete
_txnt:  rete
_int4:  rete
_nonmask:       rete

.end







