/*********************************************************************
;
; Filename: RECV96B.CPP
;
;
; Author: Chris Buehler, Jonas Keating
; Date: 10/13/95
; Last revision: 11/12/95
;
;*********************************************************************
;
; Description: A C++ simulation of the decoder.  Takes the output
;	       from TRANS96A.EXE and converts it back into the
;	       original data.
;
;*********************************************************************
;
; Usage: RECV96B < infile.dat > outfile.dat
;
;*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

// define coset labels  these are WEI's, will not need
#define A 0
#define B 2
#define C 1
#define D 3
#define M 6

typedef unsigned long int ulong;
typedef unsigned int uint;

void init_tables(void);
void init_shell_tables(void);
void show_tables(void);
void done(void);

int get_new_point(void);
int get_new_point2(void);

// viterbi.asm
void quantize4(void);
void calculate_errors(void);
void calc_branch_errs(void);
void update_path_metrics(void);
void trace_back(void);

// pre_filt.asm
void precoder_filter_real(void);
void precoder_filter_imag(void);
void round_precoder_output(void);
void quantize96(void);
void compute_xn(void);
void compute_un(void);

// invmap.asm
void compute_ibits(void);
void rotate90_ccw(void);
void find_point_index(void);
void get_binary_subset_label(void);
void build_bitstream96(void);
void outbits96(void);

// unshell.asm
void unshell_map(void);

void prediction_error_filter(void);

void check_2d_quantize(void);
void trace_last(void);
void viterbi_out(void);
void move_yt(void);
void move_y(void);
void write4dig(unsigned int);

/*
   conventions for v.34

   all constellation points lie on 2D lattice 2Z^2 + (1,1)

   the 2D lattice is partitioned into 4 subsets labeled 0,1,2,3

   0 -->  4Z^2 + (1,1)  (Wei's A)
   1 -->  4Z^2 + (1,3)  (Wei's C)
   2 -->  4Z^2 + (3,3)  (Wei's B)
   3 -->  4Z^2 + (3,1)  (Wei's D)

   the 4D lattice Z^4 is patitioned into 8 subsets which consist of
     pairs of pairs of 2D subsets.  these are labeled 0..7

   0 --> (0,0) + (2,2)
   1 --> (0,1) + (2,3)
   2 --> (0,2) + (2,0)
   3 --> (0,3) + (2,1)
   4 --> (1,1) + (3,3)
   5 --> (1,2) + (3,0)
   6 --> (1,3) + (3,1)
   7 --> (1,0) + (3,2)

   the 4D subset label consists of 3 bits, (Y2 Y1 Y0)
   (Y2 Y1), the inputs to the convolutional encoder, are found from a table
   Y0 is the previous output of the convolutional encoder
*/


// want history of 32 2D points, or 16 4D points for viterbi decoder

// yt's are input to viterbi decoder --> noise corrupted points
//  assumed to have format 9:7 (7 fractional bits)
int ytx, yty;
int xtx[4], xty[4];

// y's are outputs of viterbi decoder --> predicted lattice points
//   are inputs to unprecoder
int yx[2], yy[2];

// u's are outputs of unprecoder --> transmitted constellation points
//   are inputs to inverse MAP function
int xx[3], xy[3];
int ux, uy;
int vx, vy;

//  these are quantized points in the respective 2D coset (0,1,2,3)
int x1[4];
int y1[4];
int x2[4];
int y2[4];

// with associated 2D squared errors
// may use arrays as per above
int error1[4];
int error2[4];

// eight 4D squared branch errors for each of 8 4D subsets
// indexed array for indexing from viterbi lookup table
uint b_err[8];
// the x and y coords of the points in the eight 4D subsets to which
//   a sequence of 2D points has been sliced
//   indexed from 0 to 15 --> 8 points for 16 past 4D symbols
int bx1[16][8];
int by1[16][8];
int bx2[16][8];
int by2[16][8];

// viterbi trellis state table
//   16 4D symbols deep, with 16 states each
//   each state has 4 entries: cum. path metric, and prev. path pointer,x,y
//   circularly addressed

ulong vt_cpm[16][16];
uint vt_ppp[16][16];
uint vt_pts[16][16];

//int vt_x1[16][16];
//int vt_y1[16][16];
//int vt_x2[16][16];
//int vt_y2[16][16];

// 9600 baud constants
int q=0, qmask = 0;

// loop counter
int m;

int hr[3] = {0, 0, 0}; //-300 42 -1
int hi[3]= { 0, 0, 0};
int pr,pi;
int cr,ci;
int zold=0,znew=0,w=0;
ulong r0 = 0;
uint qbits[8];
uint ibits[4];
int mjk[8];
uint bitstream[4];

// vcurrent: newest viterbi table entry
//  vcurrent+1 mod 16 == oldest entry
int vcurrent = 0;
int minstate = 0;
// set to max
ulong curr_min_metric = 0xFFFFFFFFL;
// fake accumulator and buffer
long int ACC=0;
long int ACCB=0;

// auxillary register to play with
int *AR0, *AR1, *AR2;

// tables
int invtab[18][18];
int states[16][4];
ulong g2[4*(M-1) + 1];
ulong g4[8*(M-1) + 1];
ulong g8[8*(M-1) + 1];
ulong z8[8*(M-1) + 1];


// debug variables
int curr_err;
FILE *infile;
int *inptr;
int curr_2d_subset,curr_2d_subset1, curr_2d_subset2, curr_4d_subset;
char junk[256];
int label[4][4] = {{0, 1, 2, 3},{7, 4, 5, 6},{2, 3, 0, 1},{5, 6, 7, 4}};

void main(int argc, char *argv[])
{
  int n;

  init_tables();
  init_shell_tables();
//  show_tables();
//  exit(1);

  if(argc < 2)
    infile = fopen("big.out","r");
  else
    infile = fopen(argv[1],"r");

//  printf("<");
  // assumes enough points to initialize viterbi table
  //  the 16th time through, a call to trace_back() will yield the
  //  first transmitted point
  for(n=0; n < 15; n++)
  {
//    printf("%d\n",vcurrent);
    get_new_point();
    prediction_error_filter();
    //debug stuff
    AR0 = x1;
    AR1 = y1;
    quantize4();

    AR2 = error1;
    calculate_errors();

    // if fails, decode last point twice i guess
    get_new_point();
    prediction_error_filter();

    AR0 = x2;
    AR1 = y2;
    quantize4();

    AR2 = error2;
    calculate_errors();

    calc_branch_errs();
    curr_min_metric = 0xFFFFFFFFL;
    update_path_metrics();

    vcurrent = (vcurrent+1)&0xF;
  }

  while(1)
  {
    m = 0;
    while(m < 8)
    {
      if(get_new_point() == EOF)
	done();
      prediction_error_filter();	// this is in the adaptive eq.

      AR0 = x1;
      AR1 = y1;
      quantize4();

      AR2 = error1;
      calculate_errors();

      // if fails, decode last point twice i guess
      if(get_new_point() == EOF)
	done();
      prediction_error_filter();	// in the adaptive eq.

      AR0 = x2;
      AR1 = y2;
      quantize4();

      AR2 = error2;
      calculate_errors();

      calc_branch_errs();
      curr_min_metric = 0xFFFFFFFFL;
      update_path_metrics();

      trace_back();			// now have two points to play with
					// yx[0],yy[0] is first to decode
					// yx and yy are 9:7
      precoder_filter_real();
      round_precoder_output();
      pr = ACC;
      quantize96();
      cr = ACC;
      precoder_filter_imag();
      round_precoder_output();
      pi = ACC;
      quantize96();
      ci = ACC;
      compute_xn();
      compute_un();
      get_binary_subset_label();
      znew = ACC;
      rotate90_ccw();

      find_point_index(); 		// get qbits and shell from first pt.
      qbits[m] = ACC&qmask;
      mjk[m]   = ACC>>q;

      yx[0] = yx[1];			// do the second 2D point now
      yy[0] = yy[1];
      m++;

      precoder_filter_real();
      round_precoder_output();
      pr = ACC;
      quantize96();
      cr = ACC;
      precoder_filter_imag();
      round_precoder_output();
      pi = ACC;
      quantize96();
      ci = ACC;
      compute_xn();
      compute_un();
      get_binary_subset_label();
      w = ACC;
      rotate90_ccw();

      find_point_index(); 		// get qbits and shell from second pt.
      qbits[m] = ACC&qmask;
      mjk[m]   = ACC>>q;
      m++;

      compute_ibits();

      vcurrent = (vcurrent+1)&0xF;
    }
    unshell_map();
    build_bitstream96();
    outbits96();
  }
}

void outbits96(void)
{
 char c;
 c = char(bitstream[0]);
 if(isalnum(c) || isspace(c) || isprint(c))
   printf("%c",c);
 else
   printf("?");
 c = char(bitstream[0]>>8);
 if(isalnum(c) || isspace(c)|| isprint(c))
   printf("%c",c);
 else
   printf("?");
 c = char(bitstream[1]);
 if(isalnum(c) || isspace(c)|| isprint(c))
   printf("%c",c);
 else
   printf("?");
 c = char(bitstream[1]>>8);
 if(isalnum(c) || isspace(c)|| isprint(c))
   printf("%c",c);
 else
   printf("?");
// printf("%c%c%c%c",char(bitstream[0]),char(bitstream[0]>>8),
//		   char(bitstream[1]),char(bitstream[1]>>8));
}

// r0 is in ACC
// bitstream is laid out as follows
//
//   msb | bitstream[3] | bitstream[2] | bitstream[1] | bitstream[0] | lsb
//         msb      lsb   msb      lsb   msb      lsb   msb      lsb
void build_bitstream96(void)
{
  bitstream[0] = ACC&0xFFFFL;
  bitstream[1] = (ACC>>16)&0xF;
  bitstream[1] |= (ibits[0]<<4);
  bitstream[1] |= (ibits[1]<<7);
  bitstream[1] |= (ibits[2]<<10);
  bitstream[1] |= (ibits[3]<<13);
}

void done(void)
{
//  printf(">\nbye bye\n");
  exit(1);
}

void compute_ibits(void)
{
  ibits[(m>>1)] =  ((znew-zold)&3)<<1;
  ibits[(m>>1)] |= ((w-znew)>>1)&1;
  zold = znew;
}

// rotates a point pt by rot_factor*90 degrees (counter-clockwise)
//  90 degree rotations are performed by negating and swapping coords
void rotate90_ccw()
{
  switch(ACC)
  {
	case 0:  vx = ux;
		 vy = uy;
		 break;
	case 1:  vx = -uy;
		 vy = ux;
		 break;
	case 2:  vx = -ux;
		 vy = -uy;
		 break;
	case 3:  vx = uy;
		 vy = -ux;
		 break;
  }
}

void find_point_index(void)
{
  int tempx, tempy;

  tempx = vx+35;
  tempy = vy+35;
  tempx >>= 2;
  tempy >>= 2;
  ACC = invtab[tempx][tempy];
}

void compute_un(void)
{
  ux = (yx[0]>>7) - cr;
  uy = (yy[0]>>7) - ci;
}

void compute_xn(void)
{
  xx[0] = yx[0] - pr;
  xy[0] = yy[0] - pi;
}

void quantize96(void)
{
  // p stored in 9:7 format.  must quantize to multiple of 2w, w=1,2
  //     this routine is specific to 9600, it assumes 2w = 2!!!
  //this routine outputs integer values = 16:0
  // now it also works for negative numbers properly

  ACCB = ACC;
  ACC = (labs(ACC) + 0x07F)>>7;

  // we must mask out the 1st bit because we are rounding to the 2nd bit
  //  all numbers should be even coming out of quantize96
  ACC &= 0xFFFFFFFEL;
  if(ACCB < 0)
	ACC = -ACC;
}

void round_precoder_output(void)
{
  // rounding 11:21 number format to 9:7 = 1 sign,8 magn, 7 fractional

  ACCB = ACC;
  ACC = (labs(ACC) + 0x01FFFL)>>14;

  if(ACCB < 0)
	ACC = -ACC;
}

void prediction_error_filter(void)
{
  int i;

  ACC = (long(xtx[0])<<14) + long(xtx[1])*hr[0] + long(xtx[2])*hr[1] + long(xtx[3])*hr[2];
  ACC -= (long(xty[1])*hi[0] + long(xty[2])*hi[1] + long(xty[3])*hi[2]);
  round_precoder_output();
  ytx = ACC;

  ACC = (long(xty[0])<<14) + long(xty[1])*hr[0] + long(xty[2])*hr[1] + long(xty[3])*hr[2];
  ACC += long(xtx[1])*hi[0] + long(xtx[2])*hi[1] + long(xtx[3])*hi[2];
  round_precoder_output();
  yty = ACC;

  for(i=3; i>0; i--)
  {
    xtx[i] = xtx[i-1];
    xty[i] = xty[i-1];
  }
}

void precoder_filter_real(void)
{
  // h's are stored 2:14    = 1 sign bit, 1 magnitude bit, 14 fractional
  // x's are stored  9:7    = 1 sign bit, 8 magnitude bits, 7 fractional
  //    not sure about x's
  // so product is 11:21    - 2 sign bits (?) 9 magnitude, 21 fractional
  int p;
  ACC = 0;

  for(p=0; p < 3; p++)
  {
	ACC += xx[p]*hr[p] - xy[p]*hi[p];
  }
}

//simulates MACD
void precoder_filter_imag(void)
{
  // h's are stored 2:14    = 1 sign bit, 1 magnitude bit, 14 fractional
  // x's are stored  9:7    = 1 sign bit, 8 magnitude bits, 7 fractional
  //    not sure about x's
  // so product is 11:21    - 2 sign bits (?) 9 magnitude, 21 fractional
  int p;
  ACC = 0;

  for(p=0; p < 3; p++)
  {
	ACC += xy[p]*hr[p] + xx[p]*hi[p];
  }
  for(p=2; p > 0; p--)
  {
    xx[p] = xx[p-1];
    xy[p] = xy[p-1];
  }
}

// determines the 3 bits subset label for a particular constellation point
//   subset label is stored j2j1j0
//   point p is assumed to be in the format 15:1
//   the 1 fractional bit is the 1/2
//
// 11/24/95
// modified to only return j0 and j1. j2 is unneeded in the
//   16 state code
void get_binary_subset_label(void)
{
  uint j0,j1;
  int yx = ux, yy = uy;

  // Dr. Tretter's formula assumes that the lattice points
  // are shifted by .5,.5 ( and are (.5,.5) (1.5,1.5) etc...
  // ours are actually 2*his, so we'll divide by 2 to fix this
  // algorithm.. i hope

  // x0 = x-.5   y0 = y-.5
  yx = yx - 1;
  yy = yy - 1;
  // divide our coords by two, should give us Dr. Tretter's
  yx >>= 1;
  yy >>= 1;
  // j0 = mod(x0+y0,2)  that is, the remainder after dividing by 2
  //    or, the lsb

  j0 = (yx + yy)&1;

  // j1 = mod(x0,2) = lsb of x0

  j1 = yx&1;

  ACC = (j1<<1) + j0;
}


void viterbi_out(void)
{
//  printf("Cum. Path Error: %d\n",curr_min_metric);
  write4dig(yx[1]);
  write4dig(yy[1]);
  write4dig(yx[0]);
  write4dig(yy[0]);
}

void write4dig(unsigned int word)
{
  if(word < 0x1000L)
	printf("0");
  if(word < 0x0100L)
	printf("0");
  if(word < 0x0010L)
	printf("0");
  printf("%x\n",word);
}

void trace_back(void)
{
  int i, nextstate, branch,lastbaud;

  nextstate = minstate;
  for(i = 0; i < 15; i++)
    nextstate = vt_ppp[(vcurrent-i)&0xF][nextstate];
  branch = vt_pts[(vcurrent-15)&0xF][nextstate];
  lastbaud = (vcurrent-15)&0xF;
  yx[0] = bx1[lastbaud][branch];
  yy[0] = by1[lastbaud][branch];
  yx[1] = bx2[lastbaud][branch];
  yy[1] = by2[lastbaud][branch];

//  viterbi_out();
//  printf("[%d] state: %2d branch: %2d\n",vcurrent,nextstate,branch);
//  curr_err = vt_cpm[(vcurrent-15)&0xF][nextstate];
}

void update_path_metrics(void)
{
  int i,j;
  int prevstate,branch;
  ulong minmetric = 0xFFFFFFFFL, metric, newmetric;
  uint minst, minb;

  // loop through each state

  for(i=0; i < 16; i++)
  {
    minmetric = 0xFFFFFFFFL;
    // loop through each possible branch from prev state
    for(j=0; j < 4; j++)
    {
      prevstate = (states[i][j]>>3)&0xF;
      branch = states[i][j]&0x7;

      metric = vt_cpm[(vcurrent-1)&0xF][prevstate];
      metric = metric + b_err[branch];
//      if(newmetric < metric)
//	printf("we have overflow\n");
//      metric = newmetric;
      if(metric < minmetric)
      {
	minmetric = metric;
	minst = prevstate;
	minb = branch;
      }
    }
    vt_cpm[vcurrent][i] = minmetric;
    vt_ppp[vcurrent][i] = minst;
    vt_pts[vcurrent][i] = minb;
    if(minmetric < curr_min_metric)
    {
      curr_min_metric = minmetric;
      minstate = i;
    }
  }




  for(i=0; i < 16; i++)
  {
    vt_cpm[vcurrent][i] -= curr_min_metric;
  }
}

void calc_branch_errs(void)
{
/*
   0 --> (0,0) + (2,2)
   1 --> (0,1) + (2,3)
   2 --> (0,2) + (2,0)
   3 --> (0,3) + (2,1)
   4 --> (1,1) + (3,3)
   5 --> (1,2) + (3,0)
   6 --> (1,3) + (3,1)
   7 --> (1,0) + (3,2)
*/

  // subset 0  (0,0) vs. (2,2)
  if(error1[0]+error2[0]-(error1[2]+error2[2]) < 0)
  {
    b_err[0] = error1[0]+error2[0];
    bx1[vcurrent][0] = x1[0];
    by1[vcurrent][0] = y1[0];
    bx2[vcurrent][0] = x2[0];
    by2[vcurrent][0] = y2[0];
  }
  else
  {
    b_err[0] = error1[2]+error2[2];
    bx1[vcurrent][0] = x1[2];
    by1[vcurrent][0] = y1[2];
    bx2[vcurrent][0] = x2[2];
    by2[vcurrent][0] = y2[2];
  }
  // subset 1  (0,1) vs. (2,3)
  if(error1[0]+error2[1]-(error1[2]+error2[3]) < 0)
  {
    b_err[1] = error1[0]+error2[1];
    bx1[vcurrent][1] = x1[0];
    by1[vcurrent][1] = y1[0];
    bx2[vcurrent][1] = x2[1];
    by2[vcurrent][1] = y2[1];
  }
  else
  {
    b_err[1] = error1[2]+error2[3];
    bx1[vcurrent][1] = x1[2];
    by1[vcurrent][1] = y1[2];
    bx2[vcurrent][1] = x2[3];
    by2[vcurrent][1] = y2[3];
  }
  // subset 2  (0,2) vs. (2,0)
  if(error1[0]+error2[2]-(error1[2]+error2[0]) < 0)
  {
    b_err[2] = error1[0]+error2[2];
    bx1[vcurrent][2] = x1[0];
    by1[vcurrent][2] = y1[0];
    bx2[vcurrent][2] = x2[2];
    by2[vcurrent][2] = y2[2];
  }
  else
  {
    b_err[2] = error1[2]+error2[0];
    bx1[vcurrent][2] = x1[2];
    by1[vcurrent][2] = y1[2];
    bx2[vcurrent][2] = x2[0];
    by2[vcurrent][2] = y2[0];
  }
  // subset 3  (0,D) vs. (D,C)
  if(error1[0]+error2[3]-(error1[2]+error2[1]) < 0)
  {
    b_err[3] = error1[0]+error2[3];
    bx1[vcurrent][3] = x1[0];
    by1[vcurrent][3] = y1[0];
    bx2[vcurrent][3] = x2[3];
    by2[vcurrent][3] = y2[3];
  }
  else
  {
    b_err[3] = error1[2]+error2[1];
    bx1[vcurrent][3] = x1[2];
    by1[vcurrent][3] = y1[2];
    bx2[vcurrent][3] = x2[1];
    by2[vcurrent][3] = y2[1];
  }
  // subset 4  (A,C) vs. (B,D)
  if(error1[1]+error2[1]-(error1[3]+error2[3]) < 0)
  {
    b_err[4] = error1[1]+error2[1];
    bx1[vcurrent][4] = x1[1];
    by1[vcurrent][4] = y1[1];
    bx2[vcurrent][4] = x2[1];
    by2[vcurrent][4] = y2[1];
  }
  else
  {
    b_err[4] = error1[3]+error2[3];
    bx1[vcurrent][4] = x1[3];
    by1[vcurrent][4] = y1[3];
    bx2[vcurrent][4] = x2[3];
    by2[vcurrent][4] = y2[3];
  }
  // subset 5  (C,B) vs. (D,A)
  if(error1[1]+error2[2]-(error1[3]+error2[0]) < 0)
  {
    b_err[5] = error1[1]+error2[2];
    bx1[vcurrent][5] = x1[1];
    by1[vcurrent][5] = y1[1];
    bx2[vcurrent][5] = x2[2];
    by2[vcurrent][5] = y2[2];
  }
  else
  {
    b_err[5] = error1[3]+error2[0];
    bx1[vcurrent][5] = x1[3];
    by1[vcurrent][5] = y1[3];
    bx2[vcurrent][5] = x2[0];
    by2[vcurrent][5] = y2[0];
  }
  // subset 6  (A,D) vs. (B,C)
  if(error1[1]+error2[3]-(error1[3]+error2[1]) < 0)
  {
    b_err[6] = error1[1]+error2[3];
    bx1[vcurrent][6] = x1[1];
    by1[vcurrent][6] = y1[1];
    bx2[vcurrent][6] = x2[3];
    by2[vcurrent][6] = y2[3];
  }
  else
  {
    b_err[6] = error1[3]+error2[1];
    bx1[vcurrent][6] = x1[3];
    by1[vcurrent][6] = y1[3];
    bx2[vcurrent][6] = x2[1];
    by2[vcurrent][6] = y2[1];
  }
  // subset 7  (A,A) vs. (B,B)
  if(error1[1]+error2[0]-(error1[3]+error2[2]) < 0)
  {
    b_err[7] = error1[1]+error2[0];
    bx1[vcurrent][7] = x1[1];
    by1[vcurrent][7] = y1[1];
    bx2[vcurrent][7] = x2[0];
    by2[vcurrent][7] = y2[0];
  }
  else
  {
    b_err[7] = error1[3]+error2[2];
    bx1[vcurrent][7] = x1[3];
    by1[vcurrent][7] = y1[3];
    bx2[vcurrent][7] = x2[2];
    by2[vcurrent][7] = y2[2];
  }
}

// assumes AR0 points to x_, AR1 to y_, AR2 to error_
// calc error for current yt inputs
//  error is stored 6:10
void calculate_errors(void)
{
  int i;
  long temp;

  for(i = 0; i < 4; i++)
  {
    temp = long(AR0[i]-ytx)*long(AR0[i]-ytx)
	   + long(AR1[i]-yty)*long(AR1[i]-yty);
    AR2[i] = (temp>>4);
  }
}
/*
void move_y(void)
{
  int i;
  for(i = 31; i > 1; i-=2)
  {
    yx[i] = yx[i-2];
    yy[i] = yy[i-2];
    yx[i-1] = yx[i-3];
    yy[i-1] = yy[i-3];
  }
} */
/*
void move_yt(void)
{
  int i;
  for(i = 31; i > 0; i--)
  {
    ytx[i] = ytx[i-1];
    yty[i] = yty[i-1];
  }
}*/

// the ideal subset (exact) is in curr_2d_subset
// will check to make sure quantizer picked this as the best match
void check_2d_quantize(void)
{
  if(AR0[curr_2d_subset] != ytx)
    printf("quantizer error in x\n");
  if(AR1[curr_2d_subset] != yty)
    printf("quantizer error in y\n");
}

// quantizes the current x and y point to points in the 4 2D subsets
//  assumes AR0 is pointing to the array to store xA..xD in
//  for now AR1 is pointing to y array (but not in DSP probably)
void quantize4(void)
{
  int tx,ty, diffx,diffy;

  // subtract 1 for subset 0
  tx = ytx-128;
  // round to nearest multiple of 4 , still 9:7
  if(tx < 0)
  {
    tx = (labs(tx) + 256);
    tx &= 0xFFFFFE00;
    tx = -tx;
  }
  else
  {
    tx = (labs(tx) + 256);
    tx &= 0xFFFFFE00;
  }
  tx += 128;

  ty = yty-128;

  if(ty < 0)
  {
    ty = (labs(ty) + 256);
    ty &= 0xFFFFFE00;
    ty = -ty;
  }
  else
  {
    ty = (labs(ty) + 256);
    ty &= 0xFFFFFE00;
  }
  ty += 128;

  // calculating 0 subset done, figure out rest as offset from 0
  diffx = ytx-tx;
//  tx >>=7;
  diffy = yty-ty;
//  ty >>=7;
  // know these already
  AR0[0] = tx;
  AR1[0] = ty;
  AR1[3] = ty;
  AR0[1] = tx;
  if(diffx < 0)
  {
    AR0[3] = tx-256;
    AR0[2] = tx-256;
  }
  else
  {
    AR0[3] = tx+256;
    AR0[2] = tx+256;
  }
  if(diffy < 0)
  {
    AR1[1] = ty-256;
    AR1[2] = ty-256;
  }
  else
  {
    AR1[1] = ty+256;
    AR1[2] = ty+256;
  }
}

// m[]=
// m0 m1 m2 m3 m4 m5 m6 m7
//  n24   n23   n22   n21
//     n42,w42     n41,w41
//           n8,w8
void unshell_map(void)
{

  int n21, n22, n23, n24;
  int n41, n42;
  long int n8;
  int k,w41,w42,w2,w8;

  if(mjk[6] + mjk[7] < M)
    n21 = mjk[6];
  else
    n21 = M-1-mjk[7];

  if(mjk[4] + mjk[5] < M)
    n22 = mjk[4];
  else
    n22 = M-1-mjk[5];

  if(mjk[2] + mjk[3] < M)
    n23 = mjk[2];
  else
    n23 = M-1-mjk[3];

  if(mjk[0] + mjk[1] < M)
    n24 = mjk[0];
  else
    n24 = M-1-mjk[1];

  n41 = 0;
  w2 = mjk[4]+mjk[5];
  w41 = w2+mjk[6]+mjk[7];
  for(k = 0; k < w2; k++)
    n41 += g2[k]*g2[w41-k];
  n41+=n21*g2[w2];
  n41+=n22;

  n42 = 0;
  w2 = mjk[0]+mjk[1];
  w42 = w2+mjk[2]+mjk[3];
  for(k = 0; k < w2; k++)
    n42 += g2[k]*g2[w42-k];
  n42+=n23*g2[w2];
  n42+=n24;

  n8 = 0;
  w8 = w41+w42;
  for(k=0; k < w42; k++)
    n8 += g4[k]*g4[w8-k];
  n8+=n41*g4[w42];
  n8+=n42;

  ACC = z8[w8]+n8;
}

// read noisy point from "adaptive equalizer" (stdin)
int get_new_point(void)
{
  int x,y, result;

  result = scanf("%x %x",&x,&y);
  gets(junk);
  if(result == EOF) return EOF;
  if(result == 0) return EOF;
//  ytx = x;
//  yty = y;
  xtx[0] = x;
  xty[0] = y;
  return 1;
}

// read noisy point from "adaptive equalizer" (file "test.out")
int get_new_point2(void)
{
  int x,y, result;

  result = fscanf(infile,"%x %x",&x,&y);
  fgets(junk,256,infile);
  if(result == EOF)
  {
//    printf("end-of-file\n");
    return EOF;
  }
  if(result == 0)
  {
//    printf("short read\n");
    return EOF;
  }
//  ytx = x;
//  yty = y;
  xtx[0] = x;
  xty[0] = y;
  return 1;
}

void init_shell_tables(void)
{
  long int p,k,sum;
  // init g2
  for(p = 0; p <= 2*(M-1); p++)
    g2[p] = M - abs(p-M+1);
  for(p = 2*(M-1)+1; p <= 4*(M-1); p++)
    g2[p] = 0;

  //init g4
  for(p = 0; p <= 4*(M-1); p++)
  {
    sum = 0;
    for(k = 0; k<=p; k++)
      sum += g2[k]*g2[p-k];
    g4[p] = sum;
  }
  for(p = 4*(M-1)+1; p<=8*(M-1); p++)
    g4[p] = 0;

  // init g8
  for(p = 0; p <= 8*(M-1); p++)
  {
    sum = 0;
    for(k = 0; k <= p; k++)
      sum += g4[k]*g4[p-k];
    g8[p] = sum;
  }

  // init z8
  for(p = 0; p <= 8*(M-1); p++)
  {
    sum = 0;
    for(k = 0; k <= p-1; k++)
      sum += g8[k];
    z8[p] = sum;
  }
}


void init_tables(void)
{
  int i,j;
  #include "invtab.h"
  #include "states.h"

  ytx=0;
  yty=0;
  for(i=0; i < 32; i++)
  {
    yx[i] = 0;
    yy[i] = 0;
    xx[i] = 0;
    xy[i] = 0;
  }

  for(i=0; i < 4; i++)
  {
    x1[i] = 0;
    x2[i] = 0;
    y1[i] = 0;
    y2[i] = 0;
    error1[i] = 0;
    error2[i] = 0;
    b_err[i] = 0;
    b_err[i+4] = 0;
    xx[i] = 0;
    xy[i] = 0;
    xtx[i] = 0;
    xty[i] = 0;
  }
  for(i=0; i < 16; i++)
    for(j=0; j < 8; j++)
    {
      bx1[i][j] = 0;
      by1[i][j] = 0;
      bx2[i][j] = 0;
      by2[i][j] = 0;
    }

  for(i=0; i < 16; i++)
    for(j=0; j < 16; j++)
    {
      vt_cpm[i][j] = 0;
      vt_ppp[i][j] = 0;
      vt_pts[i][j] = 0;
//      vt_x1[i][j] = 0;
//      vt_y1[i][j] = 0;
//      vt_x2[i][j] = 0;
//      vt_y2[i][j] = 0;
    }


}

void show_tables(void)
{
  int i,j;
  printf("invtab:\n");
  for(i = 0; i < 18; i++)
  {
    printf("  .word ");
    for(j = 0; j < 17; j++)
      printf("%3d, ",invtab[i][j]);
    printf("%3d\n",invtab[i][17]);
  }
  printf("states:\n");
  for(i = 0; i < 16; i++)
  {
    printf("  .word ");
    for(j = 0; j < 3; j++)
      printf("%3d, ",states[i][j]);
    printf("%3d\n",states[i][3]);
  }





}