******************************************************************************
*             (C) COPYRIGHT TEXAS INSTRUMENTS, INC. 1998                     *
******************************************************************************
*  FILE NAME:  	demo.asm						     *
*									     *
*  VERSION:	1.0							     *
*									     *
*									     *
*  HARDWARE:	Pathway 2xx (TMS320F206) DSK				     *
*		Matrix Orbital LCD Display Module			     *
*		Keypad, Speaker, Microphone, RCA cables			     *
*									     *
*                                                                            *
*  HISTORY:  	This program is written in TMS320C2xx assembly. The reverb   *
*            	algorithm was adopted from code written by Wilfredo Almaguer *
*               and the tone generation algorithm from that provided in the  *
*		White Mountain 2xx DSK BSP.				     *
*									     *
*									     *
*  DESCRIPTION: This program incorporates two basic algorithms, reverb and   *
*               tone generation.  During execution, the user has the ability *
*               change the frequency of tones being played or to switch to   *  
*               the reverb demonstration.  The switching is done with a      *
*		keypad connected to the I/O pins on the DSK.  Messages are   *
*		also displayed on an LCD connected through the RS232C	     *
*		interface.						     *
*									     *
*									     *
*  Last Modified:  04-30-98  				            	     *
*		introduced mu-law compression to increase buffer size for    *
*		reverb delay.						     *
*									     *
*							Aaron K. Aboagye     *
******************************************************************************

	.title		"Demo"

	.include	"..\\include\\pathway.inc"
		

SSPCFG1	.set	0c002h			; settings for the SSPCR (in reset)
SSPCFG2	.set	0C032h			; settings for the SSPCR (out of reset)
 

* Constants
rxbuf	.set	0900h			; Recieve buffer starting address
buffend	.set	017ffh              	; Recieve buffer end address
size	.set	buffend-rxbuf		; Recieve buffer size
delay	.set    04ffh			; Length of reverb delay
size2	.set	01dfeh			; 2xsize = #mu-law compressed samples
page	.set    0300h			; data page of variables



* Variable locations - offsets
outval	.set	00h			; Storage space address for output value
buffpt	.set	01h			; Storage space address for current buffer ptr
cinfo   .set	02h			; address of cinfo variable 
flag	.set	03h			; address of flag to differentiate between tone generation and reverb
ton_cnt .set    04h			; address of tone frequency selector
retf	.set	05h			; address of ISR return flag
init	.set	06h			; address of init flag
ARG1	.set	09h         		; address of mu-law compress/expand data
hilo	.set	0ah			; address of hi or lo byte storage indicator (used in buffering for reverb)
MANT	.set	0bh			; addresss of mantissa of mu-law compressed sample
EXP	.set	0ch			; addresss of exponent of mu-law compressed sample
BIAS	.set	0dh			; address of bias used in mu-law compression


;	======== COEFFS ========
;   The table that follows supplies the coefficients and intial
;  	conditions for the difference equation of the digital sinusoidal
;	oscillator.
;
;	The basic formulas for a second-order digital sinusoidal oscillator
;	are:
;		Difference Equation:	y(n)  = coeff*y(n-1) - y(n-2)
;		Initial Conditions:		y(-1) = 0
;								y(-2) = -A*sin(w0)
;		where
;								coeff = 2*cos(w0)
;								w0    = 2pi*f/fs
;		and
;								A  = desired amplitude of sine wave
;								f  = desired frequency of sine wave
;								fs = sampling frequency
;
;		For more on Digital Sinusoidal Oscillators see:
;
;		Proakis, J., Manolakis, D. (1988) Introduction to Digital Signal
;		Processing, pp. 373-376, MacMillan, New York.
;
;
;	Due to the algorithm used to generate the sine wave data the formulas
;	for the coefficient and the intial conditions change slightly.  They
;	become:
;		coeff = cos(w0)*32768
;		y(-2) = -A*sin(w0)*32768
;
;	The format of the sections that follow are:
;		.word		coeff		;where coeff = cos(w0)*32768
;		.word		y(-1)		;where y(-1) = 0
;		.word		y(-2)		;where y(-2) = -A*sin(w0)*32768
;
;	Each section starts with a comment stating which values were used
;	for A, f, and fs, followed by the three calculated values of coeff
;	y(-1) and y(-2).  It is important to note that only one section can
;   be used (un-commented) at a time.
;
;	There are 2 limitations to this application:
;		
;		1)	Due to the fact that we are using only a second-order
;			oscillator the frequency observed on the output of the
;			AIC will not match exactly with the value of f that was
;			used to calculate the coefficients.  If more precision is 
;			required, a higher order oscillator may be used.
;
;		2)  The default state of the AIC at power-up is for a sampling
;			rate of 16kHz and a low-pass filter on the output with its
;			-3dB point located at 7.2kHz.  Therefore as f increases
;			towards the -3dB point of the filter the amplitude of the 
;			generated sine wave will decrease.  If increased bandwidth
;			is desired then the AIC can be setup to increase the sampling
;			rate and the frequency at which the -3dB point occurs in the
;			filter.  See the TLC320AC02C Data Manual (SLAS084B) for further
;			information.
;


* Coefficients - absolute addresses
COEFFS1	.set	311h			; address of 500Hz coefficients
COEFFS2	.set	314h			; address of 1kHz coefficients
COEFFS3	.set	317h			; address of 2kHz coefficients
COEFFS4	.set	31ah			; address of 3kHz coefficients
COEFFS	.set	31dh			; address of current coefficients

mess1	.set	320h			; address of 500Hz tone message characters
mess2	.set	338h			; address of 1KHz tone message characters
mess3	.set	350h			; address of 2KHz tone message characters
mess4	.set	368h			; address of 3KHz tone message characters
mess5	.set	380h			; address of reverb message characters
mess6	.set	399h			; address of initial message characters
mess7	.set	200h			; address of audio playback message
mess8	.set	218h			; start address of constant message section
mess9	.set	230h			; address of clear screen command

;	======== start ========

;	This is the main entry point for the program and is responsible 
;	for configuring the DSP before allowing the SSP interrupts 
;	to be serviced. 

	.ps	4000h
	.entry				; define the entry point of the program

start:	ldp	#0h
	setc	intm			; INTM = 1, disable global interrupts
	clrc	cnf			; CNF = 0,  map DARAM to data space (0x200-0x2ff)
	setc	sxm			; SXM = 1, use sign extension mode
	clrc	ovm			; OVM = 0, results overflow normally in accumulator
	spm	1			; set ALU to handle fractional multiplication

	ldp	#page	         	; use the DARAM block with Direct Addressing

	lacc	#coefs			; move coefficients and message characters
	lar	ar0, #COEFFS1		; from program memory to data memory
	mar	*, ar0
	rpt	#250
	tblr	*+

	lacc	#mesg7
	lar	ar0, #0200h
	rpt	#85
	tblr	*+

	lacc	#vectors		; load vectors to data memory location 800h which
	lar	ar0, #0800h		; is mapped to program memory address 8000h.
	rpt	#49
	tblr	*+

	splk	#0000h, outval		; outval = 0

	; set all wait-states to zero
	splk	#0000h, cinfo		; cinfo = 0
	out	cinfo, WSGR		; Set all wait states to zero

	; setup the Sync Serial Port
	splk	#SSPCFG1, cinfo		; place the ssp into reset and set it
	out	cinfo, SSPCR		; up for continuous mode operation.
	splk	#SSPCFG2, cinfo
	out	cinfo, SSPCR	 	; take the ssp out of reset

	; reset the AIC so any old settings get wiped out
	in      cinfo, IOSR    		; read IOSR
	lacl    cinfo          		; acc = cinfo
	and     #0fff7h        		; make IO3 = 0
	sacl    cinfo          		; store acc to cinfo
	out     cinfo, IOSR    		; write to IOSR, put AIC in reset

	in      cinfo, IOSR         	; read IOSR
	lacl    cinfo               	; acc = cinfo
	or      #40f8h			; make IO3 = 1
	sacl    cinfo			; store acc to cinfo
	out     cinfo, IOSR		; write to IOSR, take AIC out of reset


        splk 	#0c188h, cinfo        	; reset the UART by writing 0
	out 	cinfo, ASPCR		; 1 stop bit, tx interrupt, input i/o
        splk 	#0e188h, cinfo        	; Enable the serial port
	out 	cinfo, ASPCR
        splk 	#0041h, cinfo		; set baud rate =1200 @ 20MHz CLKOUT1
	out 	cinfo, BRD


	; unmask required interrupts
	ldp	#0h			; set data page pointer
	splk	#EN_RINT, IMR		; enable the XINT & RINT interrupts
					; also enable the TXRXINT (Async Serial Port)
					; interrupt (required by the monitor)

	ldp	#page
	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0502h, outval
	out	outval, SDTR		; configure register 5 for aux input

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0124h, outval
	out	outval, SDTR		; register A = 36, sampling rate ~= 4k

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0212h, outval
	out	outval, SDTR		; register B = 36, sampling rate ~= 4k


	splk	#0000h, flag		; initialize flags and counters
	splk	#0000h, retf	
	splk	#0002h, ton_cnt	
	splk	#0001h, init
	splk	#0000h, hilo

	lacl	#7f7fh			; initialize all bytes of reverb receive
	lar	ar2, #rxbuf		; buffer to mu-law 7f (linear 0)
	lar	ar4, #size
	mar	*, ar2

init_buf:
	sacl	*+, 0, ar4
	banz	init_buf, *-, ar2	

	lar	ar1, #rxbuf		; buffer ptr -> top of buffer
	lar	ar3, #size2		; ar2=size of buffer

	lar	ar2, #mess6		; set up pointers for startup message
	lar	ar0, #86		; number of characters in message
	call	display			; go diplay message

	ldp	#0h
	splk	#0ffffh, IFR		; Reset IFR flags
	clrc	intm			; enable global interrupts
	

loop:	
	ldp	#page 			; enter loop and allow DSP to process
	splk	#0000h, retf		; RINT interrupts and inputs from IO0 and IO1
	in	cinfo, IOSR		
	lacl	cinfo			
	and	#0007h
	sub	#0007h
	bit	retf, 15
	bcnd	loop, tc
	bcnd	chg_freq, neq
	b	loop				


rcvIsr:
	ldp	#page
	bit	flag, 15		; reverb flag set?
	bcnd	rxrev, tc
	bit	flag, 14		; no! echo flag set?
	bcnd	echo, tc

	
;
;	======== txtone ========
;
;	This is the interrupt service routine called when the transmit
;	FIFO of the F206 is empty (something needs to be sent).  It
;	will write data to the transmit FIFO which will send it back out to the
;	TLC320AC02C audio codec.  Also this is the routine that will generate
;	the actual sine wave data.
;

txtone:  
	ldp	#0
	splk	#CLR_RINT, IFR		; Clear pending interrupt 
	in	cinfo, SDTR


; start calculating sine wave data

	lar	ar0, #(COEFFS+2)	; ar0 = COEFFS+2
	lar	ar2, #0000h		; ar2 = 0
	mar	*, ar2			; ARP = ar2
	lacc	#0			; acc = 0
	mar	*0+			; ar2 += ar0
	sub	*-, 15			; ah = -(1/2)*y(n-2)
	dmov	*			; y(n-2) = y(n-1)
	lt	*-			; TREG = y(n-1)
	mpy	*+			; PREG = coeff*y(n-1)
	apac				; ah = coeff*y(n-1) - (1/2)*y(n-2)
	sfl				; ah = 2*coeff*y(n-1) - (1/2)*y(n-2)
	sach	*			; y(n-1) = ah
	sach	outval			; outval = y(n)

; sine wave calculation complete

       	lacl	outval			; acc = outval
	and	#0fffch			; need to keep control bits set to 0
	sacl	outval			; store acc to outval
	out	outval, SDTR		; send the sine wave data to the AIC

	ldp	#0h			; set data page pointer
	splk	#CLR_XINT, IFR		; mark the interrupt as serviced 
	ldp	#page
	splk	#0001, retf
	clrc	intm			; re-enable global interrupts
	ret				; return to caller


chg_freq:				; this routine processes frequency change and
	setc	intm			; switch to reverb requests
	ldp	#page			; set data page pointer		
	in	cinfo, IOSR
	lacl	cinfo
	lar	ar4, #2
	lar	ar5, #COEFFS


IO0:	bit	cinfo, 15		; change on IO0?
	bcnd	IO1, tc			; no! check IO1
	or	#0010h
	splk	#0000h, flag		; clear reverb flag
	
rel0:	in	cinfo, IOSR		; loop to detect switch released during a
	bit	cinfo, 15		; frequency change request
	bcnd	rel0, ntc

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0124h, outval
	out	outval, SDTR		; register A = 36, sampling rate ~= 4k

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0212h, outval
	out	outval, SDTR		; register B = 36, sampling rate ~= 4k

first:
	bit	ton_cnt, 15		; change to 500Hz?
	bcnd	second, ntc		
	lar	ar6, #COEFFS1		; yes! point ar6 to COEFFS1
	splk	#0002, ton_cnt		; set the flag for 1kHz on next request
	lar	ar2, #mess1		; set up pointer for message to be displayed
	lar	ar0, #23		; number of characters in message
	call 	display			; go write message to display
	b	DET

second:
	bit	ton_cnt, 14		; change to 1kHz?
	bcnd	third, ntc
	lar	ar6, #COEFFS2		; yes! point ar6 to COEFFS2
	splk	#0004, ton_cnt		; set the flag for 2kHz on next request
	bit	init, 15
	bcnd	not_init, ntc
	splk	#0h, init
	lar	ar2, #mess9		; set up pointer for message to be displayed
	lar	ar0, #1 		; number of characters in message
	call 	display			; go write message to display

	lar	ar2, #mess8
	lar	ar0, #23
	call	display

not_init:
	lar	ar2, #mess2		; set up pointer for message to be displayed
	lar	ar0, #23		; number of characters in message
	call 	display			; go write message to display
	b	DET

third:					
	bit	ton_cnt, 13		; change to 2kHz?
	bcnd	fourth, ntc
	lar	ar6, #COEFFS3		; yes! point ar6 to COEFFS3
	splk	#0008, ton_cnt		; set the flag for 3kHz on next request
	lar	ar2, #mess3		; set up pointer for message to be displayed
	lar	ar0, #23		; number of characters in message
	call 	display			; go write message to display
	b	DET

fourth:					
	bit	ton_cnt, 12		; change to 3kHz?
	bcnd	DONE, ntc
	lar	ar6, #COEFFS4		; yes! point ar6 to COEFFS4
	splk	#0001, ton_cnt		; set the flag for 500Hz on next request
	lar	ar2, #mess4		; set up pointer for message to be displayed
	lar	ar0, #23		; number of characters in message
	call 	display			; go write message to display
	b	DET
		
IO1:	bit	cinfo, 14		; change on IO1?
	bcnd	IO2, tc			; no! check IO2!
	or	#0020h
	sacl	cinfo
	out	cinfo, IOSR
	splk	#0001h, flag		; time for reverb !

	in	cinfo, IOSR
	lacl	cinfo
	and	#0fff7h 		; make IO3 = 0
	sacl	cinfo
	out     cinfo, IOSR    		; write to IOSR, put AIC in reset

	in      cinfo, IOSR         	; read IOSR
	lacl    cinfo               	; acc = cinfo
	or      #40f8h			; make IO3 = 1
	sacl    cinfo			; store acc to cinfo
	out     cinfo, IOSR		; write to IOSR, take AIC out of reset

buf_emp:
	in	cinfo, SSPCR
	lacl	cinfo
	and	#02000h
	bcnd	buf_emp, neq

	splk	#0003h, outval
	out	outval, SDTR
	splk	#0502h, outval
	out	outval, SDTR

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0124h, outval
	out	outval, SDTR		; register A = 36, sampling rate ~= 4k

	splk	#0003h, outval
	out	outval, SDTR		; request secondary communications
	splk	#0212h, outval
	out	outval, SDTR		; register B = 36, sampling rate ~= 4k

rel1:
	in	cinfo, IOSR		; loop to detect switch released during a
	bit	cinfo, 14		; switch to reverb request
	bcnd	rel1, ntc

	bit	init, 15
	bcnd	not_init1, ntc
	splk	#0h, init
	lar	ar2, #mess9		; set up pointer for message to be displayed
	lar	ar0, #1 		; number of characters in message
	call 	display			; go write message to display

	lar	ar2, #mess8
	lar	ar0, #23
	call	display

not_init1:
	lar	ar2, #mess5		; set up pointer for message to be displayed
	lar	ar0, #23		; number of characters in messages
	call	display			; go write message to display
	b	DONE

IO2:	bit	cinfo, 13		; change on IO2?
	bcnd	DONE, tc		; no! DONE!
	or	#0040h
	sacl	cinfo
	out	cinfo, IOSR
	splk	#0002h, flag		; time for echo !

	in	cinfo, IOSR
	lacl	cinfo
	and	#0fff7h 		; make IO3 = 0
	sacl	cinfo
	out	cinfo, IOSR    		; write to IOSR, put AIC in reset

	in	cinfo, IOSR         	; read IOSR
	lacl	cinfo               	; acc = cinfo
	or	#40f8h			; make IO3 = 1
	sacl	cinfo			; store acc to cinfo
	out	cinfo, IOSR		; write to IOSR, take AIC out of reset

rel2:
	in	cinfo, IOSR		; loop to detect switch released during a
	bit	cinfo, 13		; switch to reverb request
	bcnd	rel2, ntc

	bit	init, 15
	bcnd	not_init2, ntc
	splk	#0h, init
	lar	ar2, #mess9		; set up pointer for message to be displayed
	lar	ar0, #1 		; number of characters in message
	call 	display			; go write message to display

	lar	ar2, #mess8
	lar	ar0, #23
	call	display

not_init2:
	lar	ar2, #mess7
	lar	ar0, #23
	call	display
	b	DONE

DET:
	mar	*, ar6
	sacl	cinfo
	out	cinfo, IOSR

SWAP:					; replace current COEFFS with new ones
	lacl	*+, ar5
	sacl	*+, 0, ar4
	banz	SWAP, *-, ar6

DONE:
	mar	*, ar6
	lar	ar6, #38

bounce:					; delay loop to ignore effects of switch bounce
	rpt	#65535
	nop
	banz	bounce, *-, ar6
	
	clrc	intm			; re-enable global interrupts
	b	loop

***********************************************************************************
*	REVERB SECTION
***********************************************************************************

rxrev:					; RINT interrupt service routine
	in	ARG1, SDTR          	; Read in current sample
	lacl	ARG1
	sacl	outval
	call	mucomp			; compress into 8 bit mu-law

uplim:					; uplim and lowlim perform silence
	lacl	ARG1			; detection by means of thresholding
	sub	#080h			; between 7ch and 80h (note: 7fh in
	bcnd	valid, gt		; mu law is linear 0).

lowlim:
	lacl	ARG1
	add	#07ch
	bcnd	valid, gt
	b	skip			; received sample is silence, exit!
	
valid:					; received sample is not silence!
	mar	*, ar1
	sar	ar1, buffpt		; buffptr=current buffer location
	lacl	hilo
	bcnd	low, eq

hi:					; deal with high byte
	lacl	ARG1
	rpt	#7
	sfl
	or	*
	sacl	*+, 0, ar7
	b	go

low:					; deal with low byte
	lacl	ARG1
	sacl	*, 0, ar7

go:
	lacc	buffpt			; Load accum with current buffer location
	sub	#delay			; Subtract the delay
	sacl	buffpt			; buffpt-> delayed entry
	sub	#rxbuf			; Subtract top of buffer to see if
	bcnd	OK1, geq		; buffpt lies beyond its limits
	add	#buffend		; if it does add the end address to it
	sacl	buffpt			; in order to circle back through the end

OK1:	
        lar	ar7, buffpt		; Load ar7 with delayed buffer pointer
	lacl	hilo
	bcnd	lo1, eq

hi1:
	lacl	*			; high byte - mu-law
	rpt	#7
	sfr
	and	#00ffh
	sacl	ARG1
	b	go1

lo1:					; low byte - mu-law
	lacl	*			; load acc with delayed sample - mu-law
	and	#00ffh
	sacl	ARG1

go1:
	call	muexp			; convert delayed sample to linear form
	lacc	outval, 16	;15	; Load accum w/outval decimated to prevent overflow
	add	ARG1, 13		; Add decimated delay value to current value
	sach	outval			; Store sum in output value
	lacc	buffpt              	; Same comments for next four blocks of code
	sub	#delay
	sacl	buffpt
	sub	#rxbuf
	bcnd	OK2, geq
	add	#buffend   
	sacl	buffpt

OK2:
	lar	ar7, buffpt
	lacl	hilo
	bcnd	lo2, eq
hi2:	
	lacl	*
	rpt	#7
	sfr
	and	#00ffh
	sacl	ARG1
	b	go2
lo2:
	lacl	*
	and	#00ffh
	sacl	ARG1
go2:
	call	muexp
	lacc	outval, 16
        add	ARG1, 11
        sach	outval
 		
   	lacc	buffpt
	sub	#delay
	sacl	buffpt
	sub	#rxbuf
	bcnd	OK3, geq
	add	#buffend   
	sacl	buffpt

OK3:
        lar	ar7, buffpt
	lacl	hilo
	bcnd	lo3, eq
hi3:
	lacl	*
	rpt	#7
	sfr
	and	#00ffh
	sacl	ARG1
	b	go3

lo3:
	lacl	*
	and	#00ffh
	sacl	ARG1

go3:
	call	muexp
	lacc	outval, 16
        lar	ar7, buffpt
        add	ARG1, 9
        sach	outval
 		           
 	lacc	buffpt
	sub	#delay
	sacl	buffpt
	sub	#rxbuf
	bcnd	OK4, geq
	add	#buffend   
	sacl	buffpt
OK4:
        lar	ar7, buffpt
	lacl	hilo
	bcnd	lo4, eq
hi4:	
	splk	#0000h, hilo
	lacl	*, ar3
	rpt	#7
	sfr
	and	#00ffh
	sacl	ARG1
	b	go4
lo4:
	splk	#0001h, hilo
	lacl	*, ar3
	and	#00ffh
	sacl	ARG1
go4:
	call	muexp
	lacc	outval, 16
        add	ARG1, 7
        sach	outval

	lacc	outval			; Load accumulator with output value
	and	#0fffch			; 2 LSB's=0 to avoid ac02 secondary cmmd mode 
	sacl	outval, 1
	out	outval, SDTR		; Write out through sync serial port to ac02

	banz	skip, *-, ar1        	; If at the end of the buffer
	lar	ar1, #rxbuf       	; circle back to the beginning
	lar	ar3, #size2       	; to store samples.  		

skip:	ldp	#0
	splk	#CLR_RINT, IFR		; Clear pending interrupt request 			
	ldp	#page
	splk	#0001, retf
	clrc	intm			; Enable maskable interrupts  
	ret				; Return

******************************************************************************
* Routine to echo data received from codec right back out
******************************************************************************

echo:
	ldp	#page
	in	outval, SDTR
	out	outval, SDTR
	ldp	#0
	splk	#CLR_RINT, IFR		; Clear pending interrupt request 			
	ldp	#page
	clrc	intm
	ret


******************************************************************************
* Routine to write to display through UART
******************************************************************************

display:   
	mar	*, ar2
uart:
	in	cinfo, IOSR		; poll TEMT bit
	bit	cinfo, 3
	bcnd	uart, ntc
	setc 	xf			; toggle xf bit
	out 	*+, ADTR		; transmit character from data buffer@300
	mar 	*, ar0
	banz 	uart, *-, ar2		; check if size=0, and return
	ret


******************************************************************************
* Mu-law compression and expansion
******************************************************************************

mucomp:
	LDP	#page
	LACL	#33			; One-time initialization
	SACL	BIAS
	LACC	ARG1, 14		; Load data into high accumulator
					; Right shift by two because mu-law only deals with
					; 14-bit data, and we have 16-bit data as input.
  

; Begin mu-law compress (ACC HI contains 14-bit sign-magnitude)
; EXP corresponds to S bits, MANT corresponds to Q bits
; Final number is in form (ACC HI)=(P S2 S1 S0 Q3 Q2 Q1 Q0)


	MAR	*, AR5
	BCND	NEGAT, LT
	ADD	BIAS, 16		; Add bias  
	LAR	AR5, #-9		; Initialize exponent to -9 because this
					; will yield correct results when later
					; taking ABS to get mu-law exponent.
	RPT	#8			; 7 is the maximum sized exponent that
					; can be held in S bits.  Execute 7+2=9
					; times because first two leftmost bits
					; are zero since it's a 14-bit input.
	NORM	*+			; Find exponent
	AND	#0F000h, 14		; Mask four bits following leftmost 1.
					; These will become Q (representing the mantissa).
	SACH	MANT			; Mantissa is in bits 2-5 of high accumulator.
	SAR	AR5, EXP		; Exponent (S bits) is currently negated.
	LACC	EXP, 16
	ABS				; Correct exponent by negating.
	ADD	MANT, 2			; Merge exponent with mantissa, 
					; shifting left by two bits to get it in bits 4-6.
	XOR     #0FF00h, 4		; Invert all bits (keeping sign positive).
	B	FINISH


NEGAT:
	ABS				; Make positive
	ADD     BIAS, 16		; Add bias
	LAR     AR5, #-9		; Initialize exponent to -9 because this
					; will yield correct results when later
					; taking ABS to get mu-law exponent.
	RPT	#8			; 7 is the maximum sized exponent that
					; can be held in S bits.  Execute 7+2=9
					; times because first two leftmost bits
					; are zero since it's a 14-bit input.                              
	NORM	*+			; Find exponent
	AND	#0F000h, 14		; Mask four bits following leftmost 1.
					; These will become Q (representing the mantissa).
        SACH    MANT			; Mantissa is in bits 2-5 of high accumulator.
	SAR     AR5, EXP		; Exponent (S bits) is currently negated.
	LACC    EXP, 16
	ABS				; Correct exponent by negating.
	ADD     MANT, 2			; Merge exponent with mantissa, 
					; shifting left by two bits to get it in bits 4-6.
	XOR	#07F00h, 4		; Invert all 7 low bits and make polarity 
					; bit=0 (negative)


FINISH:					; Done mu-law compress (result in low byte of high accumulator)
	SACH	ARG1, 4			; Write over orignal C value
	RET


******************************************************************************
* Mu-law expander with decoding table
* input 8-bit mu-law encoded value in accumulator
******************************************************************************

muexp:
	LACL	ARG1			; ACC = data
	ADD	#CODADD			; Add accumulator to table base address
	TBLR	ARG1			; Read table entry and store back in data
	RET				; End mu-law decode


* CODEC-TO-LINEAR CONVERSION TABLE

CODADD:
	.word	-32124
	.word	-31100
	.word	-30076
	.word	-29052
	.word	-28028
	.word	-27004
	.word	-25980
	.word	-24956
	.word	-23932
	.word	-22908
	.word	-21884
	.word	-20860
	.word	-19836
	.word	-18812
	.word	-17788
	.word	-16764
	.word	-15996
	.word	-15484
	.word	-14972
	.word	-14460
	.word	-13948
	.word	-13436
	.word	-12924
	.word	-12412
	.word	-11900
	.word	-11388
	.word	-10876
	.word	-10364
	.word	-9852
	.word	-9340
	.word	-8828
	.word	-8316
	.word	-7932
	.word	-7676
	.word	-7420
	.word	-7164
	.word	-6908
	.word	-6652
	.word	-6396
	.word	-6140
	.word	-5884
	.word	-5628
	.word	-5372
	.word	-5116
	.word	-4860
	.word	-4604
	.word	-4348
	.word	-4092
	.word	-3900
	.word	-3772
	.word	-3644
	.word	-3516
	.word	-3388
	.word	-3260
	.word	-3132
	.word	-3004
	.word	-2876
	.word	-2748
	.word	-2620
	.word	-2492
	.word	-2364
	.word	-2236
	.word	-2108
	.word	-1980
	.word	-1884
	.word	-1820
	.word	-1756
	.word	-1692
	.word	-1628
	.word	-1564
	.word	-1500
	.word	-1436
	.word	-1372
	.word	-1308
	.word	-1244
	.word	-1180
	.word	-1116
	.word	-1052
	.word	-988
	.word	-924
	.word	-876
	.word	-844
	.word	-812
	.word	-780
	.word	-748
	.word	-716
	.word	-684
	.word	-652
	.word	-620
	.word	-588
	.word	-556
	.word	-524
	.word	-492
	.word	-460
	.word	-428
	.word	-396
	.word	-372
	.word	-356
	.word	-340
	.word	-324
	.word	-308
	.word	-292
	.word	-276
	.word	-260
	.word	-244
	.word	-228
	.word	-212
	.word	-196
	.word	-180
	.word	-164
	.word	-148
	.word	-132
	.word	-120
	.word	-112
	.word	-104
	.word	-96
	.word	-88
	.word	-80
	.word	-72
	.word	-64
	.word	-56
	.word	-48
	.word	-40
	.word	-32
	.word	-24
	.word	-16
	.word	-8
	.word	0
	.word	32124
	.word	31100
	.word	30076
	.word	29052
	.word	28028
	.word	27004
	.word	25980
	.word	24956
	.word	23932
	.word	22908
	.word	21884
	.word	20860
	.word	19836
	.word	18812
	.word	17788
	.word	16764
	.word	15996
	.word	15484
	.word	14972
	.word	14460
	.word	13948
	.word	13436
	.word	12924
	.word	12412
	.word	11900
	.word	11388
	.word	10876
	.word	10364
	.word	9852
	.word	9340
	.word	8828
	.word	8316
	.word	7932
	.word	7676
	.word	7420
	.word	7164
	.word	6908
	.word	6652
	.word	6396
	.word	6140
	.word	5884
	.word	5628
	.word	5372
	.word	5116
	.word	4860
	.word	4604
	.word	4348
	.word	4092
	.word	3900
	.word	3772
	.word	3644
	.word	3516
	.word	3388
	.word	3260
	.word	3132
	.word	3004
	.word	2876
	.word	2748
	.word	2620
	.word	2492
	.word	2364
	.word	2236
	.word	2108
	.word	1980
	.word	1884
	.word	1820
	.word	1756
	.word	1692
	.word	1628
	.word	1564
	.word	1500
	.word	1436
	.word	1372
	.word	1308
	.word	1244
	.word	1180
	.word	1116
	.word	1052
	.word	988
	.word	924
	.word	876
	.word	844
	.word	812
	.word	780
	.word	748
	.word	716
	.word	684
	.word	652
	.word	620
	.word	588
	.word	556
	.word	524
	.word	492
	.word	460
	.word	428
	.word	396
	.word	372
	.word	356
	.word	340
	.word	324
	.word	308
	.word	292
	.word	276
	.word	260
	.word	244
	.word	228
	.word	212
	.word	196
	.word	180
	.word	164
	.word	148
	.word	132
	.word	120
	.word	112
	.word	104
	.word	96
	.word	88
	.word	80
	.word	72
	.word	64	
	.word	56
	.word	48
	.word	40
	.word	32
	.word	24
	.word	16
	.word	8
	.word	0


******************************************************************************
* Coefficients and messages
******************************************************************************

coefs:
	.word	32138
	.word	0
	.word	-1598

	.word	30274
	.word	0
	.word	-3135

	.word	23170
	.word	0
	.word	-5793

	.word	12540
	.word	0
	.word	-7568

	.word	32138
	.word	0
	.word	-1598



mesg1:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	'5'			
	.word	'0'
	.word	'0'
	.word	'H'
	.word	'z'

	.word	' '
	.word	'T'
	.word	'o'
	.word	'n'
	.word	'e'

	.word	' '
	.word	'P'			
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'i'
	.word	'n'
	.word	'g'
	.word	'.'
	.word	'.'


mesg2:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	'1'			
	.word	'K'
	.word	'H'
	.word	'z'

	.word	' '
	.word	'T'
	.word	'o'
	.word	'n'
	.word	'e'

	.word	' '
	.word	'P'			
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'i'
	.word	'n'
	.word	'g'
	.word	'.'
	.word	'.'
	.word	'.'


mesg3:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	'2'			
	.word	'K'
	.word	'H'
	.word	'z'

	.word	' '
	.word	'T'
	.word	'o'
	.word	'n'
	.word	'e'

	.word	' '
	.word	'P'			
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'i'
	.word	'n'
	.word	'g'
	.word	'.'
	.word	'.'
	.word	'.'


mesg4:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	'3'			
	.word	'K'
	.word	'H'
	.word	'z'

	.word	' '
	.word	'T'
	.word	'o'
	.word	'n'
	.word	'e'

	.word	' '
	.word	'P'			
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'i'
	.word	'n'
	.word	'g'
	.word	'.'
	.word	'.'
	.word	'.'


mesg5:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	'R'			
	.word	'e'
	.word	'v'
	.word	'e'
	.word	'r'
	.word	'b'

	.word	' '
	.word	'D'			
	.word	'e'
	.word	'm'
	.word	'o'
	.word	'n'
	.word	's'
	.word	't'
	.word	'r'
	.word	'a'
	.word	't'
	.word	'i'
	.word	'o'
	.word	'n'


mesg6:
	.word	00feh			; LCD command prefix
	.word	'I'			; Initialize the display
	.word	0014h			; Number of columns
	.word	0004h			; Number of rows
	.word	00feh			; LCD command prefix
	.word	'C'			; turn automatic line wrapping on
	.word	00feh			; LCD command prefix
	.word	'X'			; clear display

	.word	' '
	.word	'T'
	.word	'E'
	.word	'X'
	.word	'A'
	.word	'S'
	.word	' '
	.word	' '
	.word	'I'
	.word	'N'
	.word	'S'
	.word	'T'
	.word	'R'
	.word	'U'
	.word	'M'
	.word	'E'
	.word	'N'
	.word	'T'
	.word	'S'
	.word	' '

	.word	' '
	.word	' '
	.word	'M'			
	.word	'C'
	.word	'U'
	.word	'/'
	.word	'D'
	.word	'S'
	.word	'P'
	.word	' '
	.word	' '
	.word	'S'
	.word	'E'
	.word	'M'
	.word	'I'
	.word	'N'
	.word	'A'
	.word	'R'
	.word	' '
	.word	' '


	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'
	.word	'*'

	.word	'5'			
	.word	'0'
	.word	'0'
	.word	'H'
	.word	'z'

	.word	' '
	.word	'T'
	.word	'o'
	.word	'n'
	.word	'e'
	
	.word	' '
	.word	'P'			
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'i'
	.word	'n'
	.word	'g'
	.word	'.'
	.word	'.'


mesg7:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	1			; row 1

	.word	' '
	.word	' '
	.word	'A'
	.word	'u'
	.word	'd'
	.word	'i'
	.word	'o'
	.word	' '
	.word	'P'
	.word	'l'
	.word	'a'
	.word	'y'
	.word	'b'
	.word	'a'
	.word	'c'
	.word	'k'
	.word	' '
	.word	'!'
	.word	' '
	.word	' '


mesg8:
	.word	00feh			; LCD command prefix
	.word	'G'			; outputs 0x47
	.word	1			; column 1
	.word	4			; row 4

	.word	'T'
	.word	'M'
	.word	'S'
	.word	'3'
	.word	'2'
	.word	'0'
	.word	'F'
	.word	'2'
	.word	'0'
	.word	'6'
	.word	' '
	.word	'D'
	.word	'S'
	.word	'K'
	.word	' '
	.word	'D'
	.word	'e'
	.word	'm'
	.word	'o'
	.word	'!'


mesg9:
	.word	00feh			; LCD command prefix
	.word	'X'			; clear display

;
;	======== vectors ========
;
;	This is the interrupt vector table that tells the DSP where
;	the various ISRs are located that will be serviced during the
;	programs execution.  Since we will be servicing the 
;	synchronous serial port RECEIVE interrupt (RINT) in this program, 
;	we place a branch instruction at location 08h that points to the 
;	xmitIsr function.  All other interrupt vectors branch to the 
;   	the reset vector at location 0 which will effectively restart
;   	the monitor.
;

vectors:
	b	0000h                   ;00; restart the monitor
	b	0000h                   ;02; restart the monitor
	b	0000h                   ;04; restart the monitor
	b   	0000h                 	;06; restart the monitor
	b	rcvIsr                  ;08; Receive ISR
	b	0000h                   ;0a; restart the monitor
	b	0000h                	;0c; restart the monitor
	.space 2*16                 	;0e; Reserved Space
	b	0000h                   ;10; restart the monitor
	b	0000h                  	;12; restart the monitor
	b	0000h                   ;14; restart the monitor
	b	0000h                   ;16; restart the monitor
	b	0000h                   ;18; restart the monitor
	b	0000h                   ;1a; restart the monitor
	b	0000h                   ;1c; restart the monitor
	b	0000h                   ;1e; restart the monitor
	b	0000h                   ;20; restart the monitor
	b	0000h                   ;22; restart the monitor
	b	0000h                   ;24; restart the monitor
	.space 2*16                 	;26; Reserved Space
	b	0000h                   ;28; restart the monitor
	b	0000h                   ;2a; restart the monitor
	b	0000h                   ;2c; restart the monitor
	b	0000h                   ;2e; restart the monitor
	b	0000h                   ;30; restart the monitor
	b	0000h                   ;32; restart the monitor
	b	0000h                   ;34; restart the monitor
	b	0000h                   ;36; restart the monitor
	b	0000h                   ;38; restart the monitor
	b	0000h                   ;3a; restart the monitor
	b	0000h                   ;3c; restart the monitor
	b	0000h                   ;3e; restart the monitor
        .end                            ; signify end of program
