**********************************************************************
* C32BOOT - TMS320C32 BOOT LOADER PROGRAM     (142 words)      7-7-94
*	    (C) COPYRIGHT TEXAS INSTRUMENTS INC., 1994	      	v.26
*====================================================================*
*
* NOTE:
*
*    1. Following device reset, the program waits for an external 
*	interrupt. The interrupt type determines the initial address 
*	from which the bootloader will start loading the boot table
*	to the destination memory :
*        _________________________________________________________
*	|               |                           |             |
*	| INTERRUPT PIN	| BOOT TABLE START ADDRESS  | BOOT SOURCE | 
*	|_______________|___________________________|_____________|
*	|               |                           |             |
*       |    INTR0	|	1000h  (STRB0)	    |	EPROM	  |
*	|---------------|---------------------------|-------------|	
*       |    INTR1	|     810000h  (IOSTRB)	    |	EPROM	  |
*	|---------------|---------------------------|-------------|
*       |    INTR2	|     900000h  (STRB1)	    |	EPROM	  |
*	|---------------|---------------------------|-------------|
*       |    INTR3	|     80804Ch  (sport0 Rx)  |	SERIAL	  |
*	|---------------|---------------------------|-------------|
*       | INT0 and INT3	|   1000h  (STRB0)   ASYNC  |EPROM,XF0/XF1|
*	|---------------|---------------------------|-------------|	
*       | INT1 and INT3	| 810000h  (IOSTRB)  ASYNC  |EPROM,XF0/XF1|
*	|---------------|---------------------------|-------------|
*       | INT2 and INT3	| 900000h  (STRB1)   ASYNC  |EPROM,XF0/XF1|
*	|_______________|___________________________|_____________|
*
*	If INT3 is asserted together with (INT2 or INT1 or INT0) 
*	following reset, that indicates that the boot table is to 
*	be read asynchronously from EPROM using pins XF0 and XF1 for
*	handshaking. The handshaking protocol assumes that the data
*	ready signal generated by the host arrives through pin XF1.
*	The data acknowledge signal is output from the C32 on pin XF0.
*	Both signals are active low. The C32 will continuously toggle
*	the IACK signal while waiting for the host to assert data ready
*	signal (pin XF1).
*
*    2. The boot operation involves transfer of one or more source 
*	blocks from the boot media to the destination memory. The block
*	structure of the boot table serves the purpose of distributing
*	the source data/program among different memory spaces. Each
*	block is preceded by several 32 bit control words describing
*	the block contents to the bootloader program.
*
*    3. When loading from serial port the bootloader reads the source 
*	data/program and writes it to the destination memory. There is
*	only one way to read the serial port. When loading from EPROM,
*	however, there are 4 different ways to read and assemble the
*	source contents, depending on the width of boot memory and the
*	size of the program/data being transfered. Because there is a 
*	possibility that reads and writes can span the same STRB space,
*	the bootloader loads the appropriate STRB control registers 
*	before each read and write.
*
*    4. If the boot source is EPROM whose physical width is less then 
*	32 bits, the physical interface of the EPROM device(s) to the
*	processor should be the same as that of the 32 bit interface.
*	(This involves a specific connection to C32's strobe and address 
*	signals). The reason for such arrangement is that in order to 
*	function properly the bootloader program always expects 32 bit 
*	data from 32 bit wide memory during the boot load operation.
*	Valid boot EPROM widths are : 1, 2, 4, 8, 16 and 32 bits.
*
*    5. A single source block cannot cross STRB boundaries. For example,
*	it's destination cannot overlap STRB0 space and IOSTRB space.
*	Additionaly, all of the destination addresses of a single source
*	block should reside in physical memory of the same width. It is
*	also not permitted to mix prg and data in the same source block.
*
*    6. The bootloader stops boot operation when it finds 0 in the block
*	size control word. Therefore each boot table should always end
*	with a 0 prompting the bootloader to branch to the first address
*	of the first block and start program execution from that location.
*
*====================================================================*
* C32 bootloader program register assignments, and altered mem locations
*====================================================================*
* AR7 - peripheral memory map			IOF - XF0 (handshake O)
* AR0 - read cntrl data subr pointer		IOF - XF1 (handshake I)
* AR1 - read block data/prg subr pointer
*
*  R2 - read STRB value			 R4 - write STRB value
* AR2 - read STRB pointer		AR4 - write STRB pointer
* AR3 - read data/prg pointer		AR5 - write data/prg pointer
*
*		 	read --> R1 --> write
*
* IR0 - EXEC start flag 		stack - 808024h - TIM0 cnt reg
* IR1 - EXEC start address	      		808028h - TIM0 per reg
*					IOSTRB-	808004h - DMA0 dst reg
*  R3 - data SIZE			STRB0 -	808006h - DMA0 dst reg
*  R5 - mem WIDTH			STRB1 -	808008h - DMA0 cnt reg
*	
*  R6 - memory read value		AR6,R7,R0,BK - scratch registers
*====================================================================*

reset		.word	start		; reset vector
		.space	44h		; program starts @45h

*====================================================================*
* Init registers : 808000h --> AR7,  808023h --> SP,  -1 --> IR0
*====================================================================*

start		LDI	4040h,AR7	; load peripheral memory map
		LSH	9,AR7		; base address = 808000h
		LDI	23h,SP		; initialize stack pointer to
		OR	AR7,SP		; 808023h (timer counter - 1)
		LDI	-1,IR0		; reset exec start addr flag

*====================================================================*
* Test for INT3 and, if set exclusively, procede with serial bootload.
* Else, load AR3 with 1000h if INT0, 810000h if INT1, 900000h if INT2.
* Also load appropriate boot strobe pointer  --> AR2 and
* force the boot strobe value to reflect 32bit memory width.
* If (INT0 or INT1 or INT2) and INT3 then turn on the handshake mode.
*====================================================================*

wait1		LDI	IF,R0
		AND	0Fh,R0		; clean
		CMPI	8,R0		; test for INT3
		BEQ	serial	;*******; serial bootload mode
		LDI	AR7,AR2

		ADDI	60h,AR2		; 808060h (IOSTRB)   -->  AR2
		TSTB	2,R0		; test for  INT1
		LDINZ	4080h,AR3	; 810000h / 2**9
		BNZ	exit3	;*******;

		ADDI	4,AR2		; 808064h  (STRB0)   -->  AR2
		TSTB	1,R0		; test for  INT0
		LDINZ	8,AR3		; 001000h / 2**9
		BNZ	exit3	;*******;

		ADDI	4,AR2		; 808068h  (STRB1)   -->  AR2
		TSTB	4,R0		; test for  INT2
		LDINZ	4800h,AR3	; 900000h / 2**9
		BZ	wait1	;*******;

exit3		TSTB	8,R0	      ;*; test#1 - INT3 asserted
		BZ	exit2	      ;*; test#2 - INXF1 low (not used)
		TSTB	80h,IOF	      ;*; enable handshake mode if
		LDI	6,IOF	      ;*; test#1 passed

exit2		LDI	0Fh,R2
		LSH	16,R2		; force boot data size to 32
		OR	*AR2,R2		; force boot mem width to 32
		STI	R2,*AR2
		LSH	9,AR3		; boot mem start addr --> AR3


*						      xx000001 -  1 bit
*==================================================== xx000010 -  2 bit
* Process MEMORY WIDTH control word (32 bits long)    xx000100 -  4 bit
*==================================================== xx001000 -  8 bit
*						      xx010000 - 16 bit
*						      xx100000 - 32 bit


		LDI	read_mc,AR0	; use memory to read cntrl words
					;             read_mc -->  AR0
				
		LDI	1,R5		; mem width = 1	   (init)
		LDI	32,AR6		; mem reads = 32   (init)
	       CALLU	read_m		; read memory once (1st read)

loop2		TSTB	1,R6
		BNZ	label4
		LSH	-1,R6		; look at next bit
		LSH	-1,AR6		; decr mem reads
		LSH	 1,R5		; incr mem width       -->  R5	
		BU	loop2	;*******;

label4		SUBI	2,AR6
		CMPI	0,AR6		; set flags
		BN	strobes	;*******; total # of mem reads = 32/R5
label5	       CALLU	read_m		; read memory once
		DBU	AR6,label5 ;****;

*====================================================================*
* Read and save IOSTRB, STRB0 & STRB1 (to be loaded at end of bootload)
*====================================================================*

strobes	       CALLU	AR0
		STI	R1,*+AR7(4)	; IOSTRB    -->     (DMA src)
	       CALLU	AR0
		STI	R1,*+AR7(6)	; STRB0     -->     (DMA dst)
	       CALLU	AR0
		STI	R1,*+AR7(8)	; STRB1     -->     (DMA cnt)
				
*====================================================================*
* Process block size (# of bytes, half-words or words after STRB cntrl)
*====================================================================*

block	       CALLU	AR0		; read boot memory cntrl word
		LDI 	R1,R1		; is this the last block ?
		BNZ	label2	;*******; no, go around

		LDI	*+AR7(4),R0	; 		    (DMA src)
		STI 	R0,*+AR7(60h)	; restore IOSTRB
		LDI	*+AR7(6),R0	;		    (DMA dst)
		STI	R0,*+AR7(64h)	; restore  STRB0
		LDI	*+AR7(8),R0	; 		    (DMA cnt)
		STI	R0,*+AR7(68h)	; restore  STRB1
		BU	IR1	;*******; branch to start of program

label2		LDI	R1,RC		; setup transfer loop
		SUBI	1,RC		; RC - 1 --> RC

*====================================================================*
* Process block destination address, save start address of first block
*====================================================================*
		
	       CALLU	AR0		; read boot memory cntrl word
		LDI	R1,AR5		; set dest addr 	 --> AR5
		CMPI	0,IR0		; look at EXEC start addr flag
		LDINZ	AR5,IR1		; if -1, EXEC start addr --> IR1
		LDINZ	0,IR0		; set EXEC start addr flag

*====================================================================*
* (For internal destination this word should be 0 or 60h. The first case
* will result in 0 --> DMA cntrl reg, in second case 0 --> IOSTRB reg.)
* Process block destination strobe control (sss...sss 0110 xx00)
*========================================= strb value ==== 00 - IOSTRB
*					     		   01 -  STRB0
	       CALLU	AR0		;		   10 -  STRB1	
		LDI	R1,R4				
		AND	6Ch,R1		; dest mem strb pntr --> AR4
		OR3	AR7,R1,AR4

		LSH	-8,R4		; dest memory strobe --> R4

		LDI	R4,R3
		LSH	-16,R3
		AND	3,R3		; dest data size     --> R3
		TSTB	0Ch,R1		; (IOSTRB case)
		LDIZ	3,R3

*====================================================================*
* Look at R5 and choose serial or memory read for block data/program
*====================================================================*

		CMPI	0,R5
		LDIEQ	read_s0,AR1	; read serial port0
		LDINE	read_mb,AR1	; read memory
 
*====================================================================*
* Transfer one block of data or program
*====================================================================*

		RPTB	loop4
	       CALLU	AR1		; read data/prg
		STI	R4,*AR4		; set write strobe
		NOP			; pipeline
loop4		STI	R1,*AR5++	; write data/prg
	    ||  STI	R2,*AR2		; set read strobe
		BU	block	;*******; process next block

*====================================================================*
* Load R5 with 0, load read_s0 to AR0 and initialize serial port_0
*====================================================================*

serial		LDI	read_s0,AR0	; use serial to read cntrl words
		LDI	0,R5		; memory WIDTH = serial
		LDI	0,R2		; dummy
		LDI	AR7,AR2		; dummy

		LDI	111h,R0		; 0000111h --> R0
		STI	R0,*+AR7(43h)	; set CLKR,DR,FSR as serial
		LDI	0A30h,R7	;                    port pins
		LSH	16,R7		; A300000h --> R7
		STI	R7,*+AR7(40h)	; set serial global cntrl reg
		BU	strobes	;*******; process first block

*====================================================================*
* Read a single value from serial or boot memory. The number of memory
* reads depends on mem WIDTH and data SIZE. R1 returns the read value.
* (Serial sim: NOP --> BZ read_s0 & LDI @4000H,R1 --> LDI *+AR7(4Ch),R1)  
*====================================================================*
 
read_s0		TSTB	20h,IF		; look at RINT0 flag
	   	BZ	read_s0		; wait for receive buffer full
		AND	0FDFh,IF	; reset interrupt flag
		LDI	*+AR7(4Ch),R1	; read data	       -->  R1	
		RETSU
*---------------------------------------------------------------------
read_mc		LDI	 3,R3		; data size = 32,    3 -->  R3

read_mb		LDI	1,BK		; 00000001  (ex: mem width=8)
		LSH	R5,BK		; 00000100 
		SUBI	1,BK		; 000000FF  =   mask   -->  BK

		LDI	R3,AR6		;  0 -   1 000  EXPAND
		ADDI	1,AR6		;  1 -  10 000   DATA  --> AR6
		LSH	3,AR6		; 11 - 100 000   SIZE	

		LDI	R5,R0
loop3		CMPI	 1,R0
		BEQ	exit1		; DATA SIZE
		LSH	-1,R0		; --------- - 1        --> AR6
		LSH	-1,AR6		; MEM WIDTH
		BU	loop3	;*******;
exit1		SUBI	1,AR6
		
		LDI	0,R0		; init shift value
		LDI	0,R1		; init accumulator
loop1		ADDI	3,SP		; 808027h --> SP
	       CALLU	read_m		; read memory once     -->  R6
		SUBI	3,SP		; 808024h --> SP
		AND3	R6,BK,R7	; apply mask
		LSH	R0,R7		; shift
		OR	R7,R1		; accumulate	       -->  R1
		ADDI	R5,R0		; increment shift value
		DBU	AR6,loop1 ;*****; decrement #of chunks --> AR6
		RETSU

*====================================================================*
* Perform a single memory read from the source boot table.
* Handshake enabled if IOXF0 bit of IOF reg is set, disabled when reset.
* IACK will pulse continously if handshake enabled and data not ready
* (to achieve zero-glue interface when connecting to a C40 comport)
*====================================================================*
 
read_m		TSTB	2,IOF		; handshake mode enabled ?
		BNZ	loop5		; yes, jump over
		LDI	*AR3++,R6	; no,  just read memory & return
		RETSU
*------------------------------------------------------------------ (C40)
loop5		IACK	*AR7		;*; intrnl dummy read pulses IACK
		TSTB	80h,IOF		;*; wait for data ready     
		BNZ	loop5		;*; (XF1 low from host)

		LDI	*AR3++,R6	;*; read memory once   -->  R6

		LDI	2,IOF		;*; assert data acknowledge
					;*; (XF0 low to host)

loop6		TSTB	80h,IOF		;*; wait for data not ready
		BZ	loop6		;*; (XF1 high from host)

		LDI	6,IOF		;*; deassert data acknowledge
					;*; (XF0 high to host)
		RETSU
*====================================================================*