/*----------------------------------------------------------------------
 *			 TMS340 Graphics Library
 *       Copyright (c) 1988-1990 Texas Instruments Incorporated.
 *                         All Rights Reserved
 *----------------------------------------------------------------------
 * Demonstration Program
 *
 *   Use bitblt function to animate running horses.
 *----------------------------------------------------------------------
 * Revision History:
 *   08/31/88...Original version written...................J.G.Littleton
 *   01/02/90...Modified for general portability...........J.R. Van Aken
 *----------------------------------------------------------------------
 */
#include <gsptypes.h>
#include "colors.h"

/* If MODE not defined at preprocessor command line, set to default. */
#ifndef  MODE
#define  MODE	  0	  /* default graphics mode */
#endif

#define NHORSES 4     /* number of horses in herd */
#define PITCH	144   /* pitch of running horse bitmap */
#define W	138   /* width of running horse bitmap */
#define H	73    /* height of running horse bitmap */

/* A horse is specified by its xy coordinates and animation frame #. */
typedef struct { long x, y; short frame; } HORSE;

extern short horse0[], horse1[], horse2[], horse3[],
	     horse4[], horse5[], horse6[], horse7[];

static CONFIG cfg;

static short *horseframe[] = {
    horse0, horse1, horse2, horse3, horse4, horse5, horse6, horse7
};

static HORSE herd[NHORSES];


/*
 *  This function draws horse bitmap number n at coordinates (x,y).
 */
draw_horse(horse)
HORSE *horse;
{
    set_fcolor(BLACK);
    fill_rect(W, H, (horse->x>>16)-W/2, (horse->y>>17)-H/2);
    horse->x += horse->y >> 5;
    horse->y -= horse->x >> 5;
    set_fcolor(WHITE);
    set_srcbm(horseframe[(horse->frame++)&7], PITCH, W, H, 1);
    bitblt(W, H, 0, 0, (horse->x>>16)-W/2, (horse->y>>17)-H/2);
}


/*
 *  Main demo program.
 */
main()
{
    int hres, vres, n;

    /* Initialize graphics environment. */
    if (!set_config(MODE, 1))
	exit(1);		 /* invalid graphics mode number */
    clear_screen(0);
    get_config(&cfg);
    hres = cfg.mode.disp_hres;	 /* horizontal resolution */
    vres = cfg.mode.disp_vres;	 /* vertical resolution */
    set_draw_origin(hres/2, vres/2);

    herd[0].x = 0;
    herd[0].y = -3*vres/8 << 16;
    herd[0].frame = 0;
    herd[1].x = 3*hres/8 << 16;
    herd[1].y = 0;
    herd[1].frame = 4;
    herd[2].x = 0;
    herd[2].y = -herd[0].y;
    herd[2].frame = 2;
    herd[3].x = -herd[1].x;
    herd[3].y = 0;
    herd[3].frame = 7;

    for (;;)
	 for (n = 0; n < NHORSES; n++) {
	     draw_horse(&herd[n]);
	     draw_rect(hres-1, vres-1, -hres/2, -vres/2);
	     wait_scan(vres);
	 }
}


/*-----------------------------------------------------------------------
 *		       8 bit-mapped images of horse
 *-----------------------------------------------------------------------
 */

short horse0[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0030,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0028,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0034,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0012,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0x0039,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x6000,0x007f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xda80,0x00dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x7f20,0x0177,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xddc8,0x01dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x7754,0x0377,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xddfb,0x03dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x6000,0x777f,0x0777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xcc00,0xdddf,0x0ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf300,0x7777,0x1777,0x0000,
     0x0000,0x0000,0xff80,0x003f,0x0000,0xfc00,0xfddd,0x3ddd,0x0000,
     0x0000,0x0000,0x0070,0x01c0,0x0000,0x7fe0,0x7777,0x7777,0x0000,
     0x0000,0x0000,0x6d8c,0x0e15,0x0000,0xdfd8,0xfddd,0xdddd,0x0000,
     0x0000,0x0000,0x7b63,0x70f7,0x0000,0x76fc,0xff77,0x7777,0x0001,
     0x0000,0xf800,0xddd8,0xcddd,0xe007,0xdd87,0xfddd,0xddff,0x0002,
     0x0000,0x57e0,0x7776,0x3777,0x1ffc,0x7770,0xff77,0x776f,0x0007,
     0x0000,0x7818,0xdddd,0xdddd,0xc001,0xdddf,0xbfdd,0xddfa,0x0005,
     0x0000,0xb686,0x7776,0x7777,0x7777,0x7777,0x1f77,0x77e0,0x0007,
     0x0000,0x3ddb,0xdddd,0xdddd,0xdddd,0xdddd,0x0fdd,0xd780,0x0007,
     0xe000,0xa7ee,0x7777,0x7777,0x7777,0x7777,0x07f7,0xfe00,0x0006,
     0xb800,0x6367,0xdddd,0xdddd,0xdddd,0xdddd,0x03dd,0xb800,0x0001,
     0x4c00,0xa1e3,0x7777,0x7777,0x7777,0x7777,0x01f7,0xe000,0x0000,
     0xc200,0x60d9,0xdddd,0xdddd,0xdddd,0xdddd,0x00fd,0x0000,0x0000,
     0xd900,0xa074,0x7777,0x7777,0x7777,0x7777,0x0077,0x0000,0x0000,
     0x7480,0x603f,0xdddd,0xdddd,0xdddd,0xdddd,0x003d,0x0000,0x0000,
     0x3ec0,0xc00f,0x7777,0x777f,0x7777,0x7777,0x003f,0x0000,0x0000,
     0xdfc0,0xc003,0xdddd,0xdddf,0xdddd,0xdddd,0x001d,0x0000,0x0000,
     0x85b0,0x4001,0x7777,0x7777,0x7777,0x7777,0x000b,0x0000,0x0000,
     0x72c8,0xc000,0xdddd,0xdddf,0xdddd,0xdddd,0x000f,0x0000,0x0000,
     0xfde0,0x4000,0x7777,0x7777,0x7777,0x7777,0x000f,0x0000,0x0000,
     0x0fe0,0x4000,0xdddf,0xdddf,0xdddd,0xdddd,0x0007,0x0000,0x0000,
     0x01f0,0xc000,0xf774,0x777f,0x7777,0x7777,0x0007,0x0000,0x0000,
     0x00fe,0xc000,0xdddd,0xddff,0xdddd,0xdddd,0x0003,0x0000,0x0000,
     0x003c,0xf800,0x7775,0x77ff,0x77f7,0xf777,0x0001,0x0000,0x0000,
     0x0000,0xfe00,0xdddc,0xdff1,0xdddd,0xdddd,0x0001,0x0000,0x0000,
     0x0000,0x7f00,0xf776,0xffc0,0x77f7,0xf777,0x0000,0x0000,0x0000,
     0x0000,0x1f00,0x7ddd,0xfe00,0xdfff,0x7ddd,0x0000,0x0000,0x0000,
     0x0000,0x43e0,0x3777,0xf000,0x7fff,0x3f77,0x0000,0x0000,0x0000,
     0x0000,0xe818,0x1ddd,0x8000,0xdfff,0x1ddd,0x0000,0x0000,0x0000,
     0x0000,0x7744,0x0e77,0x0000,0x7ffc,0x0f77,0x0000,0x0000,0x0000,
     0x0000,0xddc4,0x07dd,0x0000,0xffc0,0x07dd,0x0000,0x0000,0x0000,
     0x0000,0xf774,0x01ff,0x0000,0xfe00,0x0376,0x0000,0x0000,0x0000,
     0x0000,0xffda,0x0000,0x0000,0xfe00,0x01df,0x0000,0x0000,0x0000,
     0x0000,0x7ff6,0x0000,0x0000,0xfe00,0x0175,0x0000,0x0000,0x0000,
     0x0000,0xfc7d,0x0000,0x0000,0xff00,0x00df,0x0000,0x0000,0x0000,
     0x8000,0xf817,0x0000,0x0000,0x7f00,0x00f7,0x0000,0x0000,0x0000,
     0x8000,0xf80d,0x0000,0x0000,0x7f80,0x00de,0x0000,0x0000,0x0000,
     0x4000,0xf007,0x0000,0x0000,0x3f80,0x00f6,0x0000,0x0000,0x0000,
     0xe000,0xf805,0x0000,0x0000,0x3f80,0x00dc,0x0000,0x0000,0x0000,
     0x6000,0xf803,0x0000,0x0000,0x1fc0,0x00f4,0x0000,0x0000,0x0000,
     0xd000,0x7c01,0x0000,0x0000,0x0fc0,0x00dc,0x0000,0x0000,0x0000,
     0xf800,0x7e00,0x0000,0x0000,0x07e0,0x00f4,0x0000,0x0000,0x0000,
     0xdc00,0x3f00,0x0000,0x0000,0x03e0,0x00d8,0x0000,0x0000,0x0000,
     0x77c0,0x1f80,0x0000,0x0000,0x01f0,0x0178,0x0000,0x0000,0x0000,
     0x5cc0,0x0f80,0x0000,0xf1c0,0x01ff,0x01d8,0x0000,0x0000,0x0000,
     0x3760,0x0780,0x0000,0xffe0,0x00ff,0x0178,0x0000,0x0000,0x0000,
     0x1eb0,0x0380,0x0000,0xfff0,0x007f,0x01d0,0x0000,0x0000,0x0000,
     0x0750,0x0180,0x0000,0xfff0,0x001f,0x0170,0x0000,0x0000,0x0000,
     0x01f0,0x0000,0x0000,0xfff8,0x0001,0x01d0,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x39f8,0x0000,0x0170,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x03d0,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0360,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x03e0,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0f60,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x1de0,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x37c0,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x5e00,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf400,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xdc00,0x0000,0x0000,0x0000,
};

short horse1[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x2000,0x0002,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x9cc0,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xcf20,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x77c8,0x0007,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xddf0,0x000c,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x777f,0x0017,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0xddfc,0x003d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x777e,0x0077,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xe000,0xdddf,0x00dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0x7777,0x01f7,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xfc00,0xdddf,0x03dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf900,0x7777,0x0777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xfe00,0xfddd,0x0ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7fc0,0x7777,0x1777,0x0000,
     0x0000,0x0000,0xff00,0x0003,0x0000,0xdff8,0xfddd,0x3ddd,0x0000,
     0x0000,0x0000,0x00c0,0x01fc,0x0000,0x77fe,0x7f77,0x7777,0x0000,
     0x0000,0x0000,0xdc30,0x3e01,0x8000,0xddff,0xffdd,0xdddd,0x0000,
     0x0000,0x0000,0x7718,0xc077,0xffff,0x7777,0x1f77,0xf776,0x0000,
     0x0000,0x7f80,0xddc4,0x0ddd,0x8000,0xdddd,0x0fdd,0xddf8,0x0000,
     0x0000,0xa8c0,0x7777,0x7777,0x7777,0x7777,0x07f7,0x77c0,0x0000,
     0x0000,0xbe30,0xddda,0xdddd,0xdddd,0xdddd,0x03dd,0x3f00,0x0000,
     0x0000,0xf7b8,0x7771,0x7777,0x7777,0x7777,0x01f7,0x0000,0x0000,
     0x0000,0xf596,0xdddd,0xdddd,0xdddd,0xdddd,0x00fd,0x0000,0x0000,
     0x0000,0x9bd9,0x7774,0x7777,0x7777,0x7777,0x00f7,0x0000,0x0000,
     0xc000,0x8f7e,0xdddd,0xdddd,0xdddd,0xdddd,0x007d,0x0000,0x0000,
     0xa000,0x41b5,0x7777,0x7777,0x7777,0x7777,0x0077,0x0000,0x0000,
     0x5000,0xc0fa,0xdddd,0xdddd,0xdddd,0xdddd,0x003d,0x0000,0x0000,
     0x2800,0x40ef,0x7777,0x7777,0x7777,0x7777,0x003f,0x0000,0x0000,
     0xb400,0xc07e,0xdddd,0xdfdd,0xdddd,0xdddd,0x001d,0x0000,0x0000,
     0xaa00,0x403b,0x7777,0x7f77,0x7777,0x7777,0x001f,0x0000,0x0000,
     0xcd00,0x401d,0xdddd,0xdddd,0xdddd,0xdddd,0x000d,0x0000,0x0000,
     0x7ec0,0x801d,0x7777,0x7f77,0x7777,0x7777,0x000f,0x0000,0x0000,
     0xfd80,0x800e,0xdddd,0xdddd,0xdddd,0xdddd,0x0007,0x0000,0x0000,
     0xbec0,0x0006,0x7777,0x7f77,0x7777,0x7777,0x0007,0x0000,0x0000,
     0xaa60,0x0007,0xddde,0xfddd,0xdddd,0xdddd,0x0003,0x0000,0x0000,
     0xd510,0x0001,0x7778,0xf777,0x7777,0xf777,0x0001,0x0000,0x0000,
     0xffcc,0x0000,0xdde0,0xfddd,0xdddd,0xdddd,0x0000,0x0000,0x0000,
     0x307e,0x0000,0x7780,0xf777,0x77f7,0xf777,0x0000,0x0000,0x0000,
     0x0ff8,0x0000,0xde00,0xfddd,0xdfdf,0xfddd,0x0001,0x0000,0x0000,
     0x0000,0x0000,0x7c00,0xdf77,0x777f,0xff77,0x0003,0x0000,0x0000,
     0x0000,0x0000,0xd800,0x8ddd,0xdfff,0xfddd,0x000f,0x0000,0x0000,
     0x0000,0x0000,0x7000,0x0777,0x77fe,0xfff7,0x001f,0x0000,0x0000,
     0x0000,0x0000,0xf000,0x03dd,0xdff8,0xfe3d,0x003f,0x0000,0x0000,
     0x0000,0x0000,0x6000,0x0377,0x7780,0xf007,0x007f,0x0000,0x0000,
     0x0000,0x0000,0xe000,0x01dd,0xdc00,0x800d,0x00ff,0x0000,0x0000,
     0x0000,0x0000,0x6000,0x01f7,0x7800,0x000f,0x00fe,0x0000,0x0000,
     0x0000,0x0000,0xf000,0x00dd,0xd000,0x001d,0x00fc,0x0000,0x0000,
     0x0000,0x0000,0x7000,0x0077,0x6000,0x0017,0x0078,0x0000,0x0000,
     0x0000,0x0000,0xf800,0x005d,0xc000,0x003d,0x007c,0x0000,0x0000,
     0x0000,0x0000,0x7800,0x7ff7,0x0000,0x0077,0x003c,0x0000,0x0000,
     0x0000,0x0000,0xf800,0xffdd,0x0003,0x005e,0x003e,0x0000,0x0000,
     0x0000,0x0000,0xf800,0xff77,0x0007,0x00f6,0x001e,0x0000,0x0000,
     0x0000,0x0000,0xf000,0xfddd,0x000f,0x78dc,0x001f,0x0000,0x0000,
     0x0000,0x0000,0xc000,0xf777,0x0f9f,0xfcf8,0x000f,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x1ddc,0x1f9f,0xfcd8,0x0007,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x37e0,0xf73e,0xfeff,0x0003,0x0000,0x0000,
     0x0000,0x0000,0x0000,0xdf00,0xdf7e,0xfedd,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0xf400,0x76fd,0x3e77,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x5800,0xfcff,0x003f,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0xe000,0x0037,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0xc000,0x003d,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0077,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x005e,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0074,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fc,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fc,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fc,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse2[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xb000,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xce18,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xc7a4,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0xcde9,0x0002,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x77fe,0x0007,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xa800,0xddff,0x000d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf600,0x7777,0x000f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf000,0xdddd,0x001d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf980,0x7777,0x0037,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xbe40,0xdddd,0x007d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7f00,0x7777,0x00f7,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xd7a8,0xdddd,0x01dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x77d0,0x7777,0x0377,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x8000,0xdff1,0xdddd,0x07dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x75fe,0x7f77,0x0f77,0x0000,
     0x0000,0x0000,0x0000,0x0000,0xc000,0xdd5b,0xdfdd,0x1ddd,0x0000,
     0x0000,0x0000,0xfe00,0x0007,0xe000,0x775f,0x7ff7,0x3f77,0x0000,
     0x0000,0x0000,0x01c0,0x007c,0xbc00,0xddd7,0xe7dd,0x3ddf,0x0000,
     0x0000,0x0000,0x7070,0xff81,0xfbff,0x7775,0x03f7,0x377f,0x0000,
     0x0000,0x0000,0xdd18,0x003d,0xf800,0xdddd,0x01dd,0x3df8,0x0000,
     0x0000,0x0000,0x7744,0x7777,0x7777,0x7777,0x01f7,0x1fe0,0x0000,
     0x0000,0x0000,0xddd7,0xdddd,0xdddd,0xdddd,0x00fd,0x0f80,0x0000,
     0x0000,0x8000,0x7772,0x7777,0x7777,0x7777,0x00f7,0x0000,0x0000,
     0x0000,0x7fc0,0xddd9,0xdddd,0xdddd,0xdddd,0x007d,0x0000,0x0000,
     0x0000,0xfc78,0x7775,0x7777,0x7777,0x7777,0x007f,0x0000,0x0000,
     0x0000,0xaf07,0xdddc,0xdddd,0xdddd,0xdddd,0x0035,0x0000,0x0000,
     0x8000,0xfac1,0x7775,0x7777,0x7777,0x7777,0x003f,0x0000,0x0000,
     0x6000,0xf6e1,0xdddd,0xdddd,0xdddd,0xdddd,0x001d,0x0000,0x0000,
     0x4800,0xf9ac,0x7777,0x7777,0x7777,0x7777,0x001f,0x0000,0x0000,
     0x3400,0xdbda,0xdddd,0xdddd,0xdddd,0xdddd,0x000d,0x0000,0x0000,
     0x7000,0x8ff7,0x7777,0x7777,0x7777,0x7777,0x000f,0x0000,0x0000,
     0x1d80,0x83b4,0xdddf,0xdddd,0xdddd,0xdddd,0x000f,0x0000,0x0000,
     0x9a40,0x03fd,0x7777,0x7777,0x7777,0x7777,0x0007,0x0000,0x0000,
     0xb700,0x01cf,0xdddf,0xdddd,0xdddd,0xdddd,0x0007,0x0000,0x0000,
     0xf5c0,0x007f,0x777e,0x77f7,0x7777,0x7777,0x0003,0x0000,0x0000,
     0x1d60,0x000f,0xddde,0xdddd,0xdddd,0xdddd,0x000f,0x0000,0x0000,
     0xe7a0,0x0003,0x775c,0x77f7,0x7777,0xf777,0x001f,0x0000,0x0000,
     0xf8d0,0x0000,0xdd78,0xdfdd,0xdddd,0xdddd,0x007f,0x0000,0x0000,
     0x1ec8,0x0000,0x77f0,0x7f77,0x7777,0xf777,0x01ff,0x0000,0x0000,
     0x07e8,0x0000,0xdbe0,0xffdd,0xfddd,0xfddd,0x07ff,0x0000,0x0000,
     0x01f4,0x0000,0x7fc0,0xfff7,0x7ff7,0xff77,0x0fff,0x0000,0x0000,
     0x003c,0x0000,0xff80,0xe1fd,0xdfff,0xcddd,0x0fff,0x0000,0x0000,
     0x001a,0x0000,0xff00,0x00d7,0x77ff,0x0777,0x0ffc,0x0000,0x0000,
     0x000a,0x0000,0xfe00,0x00dd,0xe000,0x03dd,0x0f80,0x0000,0x0000,
     0x0004,0x0000,0xfc00,0x0077,0x8000,0x0777,0x0f80,0x0000,0x0000,
     0x0002,0x0000,0xfc00,0x00df,0x0000,0x0ddf,0x0f00,0x0000,0x0000,
     0x0002,0x0000,0xf800,0x00f7,0x0000,0x1778,0x0f00,0x0000,0x0000,
     0x0004,0x0000,0xf800,0x01df,0x0000,0x1df0,0x0f00,0x0000,0x0000,
     0x0000,0x0000,0xf800,0x0177,0x0000,0x3740,0x0f00,0x0000,0x0000,
     0x0000,0x0000,0xfc00,0x03df,0x0000,0x5d80,0x0f00,0x0000,0x0000,
     0x0000,0x0000,0xfc00,0x0377,0xc000,0xf707,0x0f00,0x0000,0x0000,
     0x0000,0x0000,0xfc00,0x0ddc,0xf000,0xdc07,0x1f00,0x0000,0x0000,
     0x0000,0x0000,0xfe00,0x377c,0xf000,0xf707,0x1f00,0x0000,0x0000,
     0x0000,0x0000,0xfe00,0xddf8,0xe000,0x5dfd,0x1f80,0x0000,0x0000,
     0x0000,0x0000,0xfe00,0x77f0,0xc003,0x3777,0x1f80,0x0000,0x0000,
     0x0000,0x0000,0xfe00,0xdfc0,0x800d,0x0ddd,0x0fc0,0x0000,0x0000,
     0x0000,0x0000,0xfc00,0x7c01,0x0077,0x03f7,0x07f0,0x0000,0x0000,
     0x0000,0x0000,0xfc00,0xe001,0x01dd,0x007c,0x03f8,0x0000,0x0000,
     0x0000,0x0000,0xf800,0x0001,0x0f77,0x0000,0x03f8,0x0000,0x0000,
     0x0000,0x0000,0xf000,0x0003,0x3ddc,0x0000,0x01f0,0x0000,0x0000,
     0x0000,0x0000,0xe000,0x0003,0x7ff0,0x0000,0x01e0,0x0000,0x0000,
     0x0000,0x0000,0xe000,0x0007,0xff00,0x0000,0x00c0,0x0000,0x0000,
     0x0000,0x0000,0xc000,0x0007,0xfe00,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0xc000,0x000f,0x7c00,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x8000,0x000f,0x1c00,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x8000,0x003f,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x03ff,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x07ff,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0ffe,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x1fe0,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x1fc0,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse3[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0600,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x3000,0x0580,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x4000,0x067f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xfb00,0x030f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xfc80,0x034d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x7f30,0x0777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x77f3,0x1f77,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xdfc0,0x0ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xddf4,0x35dd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0x777d,0x3777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x3800,0xddf7,0x7ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0x777f,0x7777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf300,0xdddf,0xdddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf400,0x7777,0xf77f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xffe0,0xdddd,0xdddd,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7f80,0x7777,0x777f,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xdfc0,0xdddd,0xdddd,0x0003,
     0x0000,0x0000,0xfc00,0x03ff,0x0000,0x77e0,0x7777,0x777f,0x0003,
     0x0000,0x0000,0x0380,0x1c00,0x0000,0xddde,0xdddd,0xdde7,0x0007,
     0x0000,0x0000,0x7060,0xe177,0xf001,0x7771,0xf777,0x7781,0x0007,
     0x0000,0x0000,0xdd18,0x0ddd,0x0ffe,0xdddc,0x7ddd,0xdd00,0x000d,
     0x0000,0x0000,0x7744,0x7777,0x7000,0x7777,0x1f77,0x7e00,0x000f,
     0x0000,0x0000,0xddd4,0xdddd,0xdddd,0xdddd,0x0fdd,0xdc00,0x000d,
     0x0000,0x3ffc,0x7772,0x7777,0x7777,0x7777,0x0777,0x7800,0x000f,
     0x8000,0xfc07,0xddda,0xdddd,0xdddd,0xdddd,0x035d,0xb000,0x000b,
     0xc000,0xf1d2,0x7773,0x7777,0x7777,0x7777,0x03f7,0xe000,0x0006,
     0xe000,0xe357,0xdddd,0xdddd,0xdddd,0xdddd,0x01dd,0xc000,0x0001,
     0x5800,0xfaf7,0x7776,0x7777,0x7777,0x7777,0x01f7,0x0000,0x0000,
     0xfc00,0xfead,0xdddd,0xdddd,0xdddd,0xdddd,0x00dd,0x0000,0x0000,
     0xaa00,0xdf9a,0x7777,0x7777,0x7777,0x7777,0x00f7,0x0000,0x0000,
     0x7780,0xc7c3,0xdddd,0xdddd,0xdddd,0xdddd,0x00dd,0x0000,0x0000,
     0x5940,0x83f8,0x7777,0x7777,0x7777,0x7777,0x00f7,0x0000,0x0000,
     0x04a0,0x003f,0xdddd,0xdddd,0xdddd,0xdddd,0x00dd,0x0000,0x0000,
     0xfed0,0x8003,0x7777,0x7777,0x7777,0x777f,0x00f7,0x0000,0x0000,
     0x3f68,0xc000,0xdddd,0xdddd,0xdddd,0xdddd,0x00dd,0x0000,0x0000,
     0x0734,0xc000,0x7777,0x7777,0x7777,0x777f,0x00f7,0x0000,0x0000,
     0x03b4,0xc000,0xdddb,0xdddd,0xdddd,0xddfd,0x3ddd,0x0000,0x0000,
     0x00f4,0xc000,0x7773,0x7777,0x7777,0x7777,0xd777,0x0000,0x0000,
     0x00ba,0xc000,0xddd7,0xdffd,0xdddf,0xddfd,0xdddd,0x0001,0x0000,
     0x0018,0xc000,0x776f,0x7eff,0x7777,0x77f7,0x7777,0x0003,0x0000,
     0x0018,0xc000,0xddcf,0xf83d,0xbddf,0xdfdd,0xdddd,0x0005,0x0000,
     0x0008,0xe000,0x779f,0xc00f,0xf77f,0x7fff,0x7f77,0x0007,0x0000,
     0x0008,0xf000,0xdf7f,0x000f,0xfffe,0xfe0f,0xfffd,0x0005,0x0000,
     0x0004,0xf800,0x77ff,0x0007,0x3ff0,0xf800,0x7fff,0x0007,0x0000,
     0x0000,0xfc00,0xdeff,0x0007,0x0000,0x0000,0xffc0,0x0005,0x0000,
     0x0000,0xff00,0x753f,0x0003,0x0000,0x0000,0x7f00,0x0007,0x0000,
     0x0000,0xffc0,0xdf1f,0x0003,0x0000,0x0000,0xfe00,0x0005,0x0000,
     0x0000,0xffe0,0xf70f,0x0001,0x0000,0x0000,0x7c00,0x0007,0x0000,
     0x0000,0xffe0,0xdf03,0x0001,0x0000,0x0000,0xf800,0x0005,0x0000,
     0x0000,0xffe0,0xf700,0x0000,0x0000,0x0000,0x7000,0x000f,0x0000,
     0x0000,0x1ff0,0x5f00,0x0000,0x0000,0x0000,0xc000,0x001d,0x0000,
     0x0000,0x07f0,0x7780,0x0000,0x0000,0x0000,0x6000,0x007f,0x0000,
     0x0000,0x03f0,0x3d80,0x0000,0x0000,0x0000,0xd000,0x01ff,0x0000,
     0x0000,0x01f8,0x37c0,0x0000,0x0000,0x0000,0x7800,0x0ffd,0x0000,
     0x0000,0x01f8,0x1dc0,0x0000,0x0000,0x0000,0xf800,0x3ff1,0x0000,
     0x0000,0x00f8,0x1f60,0x0000,0x0000,0x0000,0xf800,0x7fe0,0x0000,
     0x0000,0x00fc,0x0de0,0x0000,0x0000,0x0000,0xf800,0xff80,0x0000,
     0x0000,0x007c,0x17e0,0x0000,0x0000,0x0000,0x7800,0xfc00,0x0001,
     0x0000,0x007e,0x3dc0,0x0000,0x0000,0x0000,0x7800,0xf800,0x0003,
     0x0000,0x003e,0x77c0,0x0000,0x0000,0x0000,0x1800,0xf000,0x0003,
     0x0000,0x003f,0xdf00,0x0000,0x0000,0x0000,0x0000,0xf000,0x0003,
     0x0000,0x001f,0xf400,0x0001,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x001f,0xd800,0x0003,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x001f,0x7000,0x0007,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x001f,0xe000,0x001d,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x003e,0x4000,0x00f7,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x007c,0x8000,0x07dd,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x01f8,0x0000,0x0ff7,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x03f0,0x0000,0x1ffe,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x07f0,0x0000,0x3fe0,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse4[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x3800,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x1400,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x13e9,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xa000,0x19fe,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xc800,0x1f7f,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xfe40,0x3f5d,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xdf82,0x37f7,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xffdc,0x3ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x77b0,0x7777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0xdffc,0xdddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x1000,0x77ff,0x7777,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xe000,0xddff,0xdddd,0x0003,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf880,0x7777,0x7777,0x0003,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xff00,0xdddf,0xdddd,0x0005,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7d10,0x7777,0x7777,0x000f,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xffe0,0xdddd,0xdddd,0x000d,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7fd4,0x7777,0x77f7,0x001f,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xdff8,0xdddd,0xddfd,0x001d,
     0x0000,0x0000,0xf000,0x00ff,0x0000,0x77fe,0x7777,0x77ff,0x0037,
     0x0000,0x0000,0x0f00,0x0f00,0x8000,0xddff,0xdddd,0xdfff,0x007d,
     0x0000,0x0000,0xa0e0,0xf077,0xffff,0x777f,0xf777,0x7e07,0x0077,
     0x0000,0x0000,0xdc18,0xc2dd,0xf801,0xdddd,0xdddd,0xf803,0x00dd,
     0x0000,0x0000,0x7744,0x7777,0x7777,0x7777,0xf777,0x7001,0x01f7,
     0x0000,0x0060,0xddd2,0xdddd,0xdddd,0xdddd,0xfddd,0xe000,0x01df,
     0x0000,0x1f90,0x7773,0x7777,0x7777,0x7777,0x7777,0xc000,0x0177,
     0x8000,0xfafd,0xdddd,0xdddd,0xdddd,0xdddd,0x3ddd,0x8000,0x01df,
     0x4000,0xe0e3,0x7777,0x7777,0x7777,0x7777,0x3777,0x0000,0x00ff,
     0xf000,0xde30,0xdddd,0xdddd,0xdddd,0xdddd,0x3ddd,0x0000,0x007c,
     0x6800,0xfb16,0x7775,0x7777,0x7777,0x7777,0x1f77,0x0000,0x0000,
     0x1400,0xdff7,0xdddd,0xdddd,0xdddd,0xdddd,0x1ddd,0x0000,0x0000,
     0xca00,0x87ed,0x7775,0x7777,0x7777,0x7777,0x1f77,0x0000,0x0000,
     0x6500,0x817c,0xdddd,0xdddd,0xdddd,0xdddd,0x1ddd,0x0000,0x0000,
     0x7280,0xc0f6,0x7775,0x7777,0x7777,0x7777,0x1f77,0x0000,0x0000,
     0xb900,0xc01f,0xdddb,0xdddd,0xdddd,0xdddd,0x0ddd,0x0000,0x0000,
     0xdd80,0xc007,0x7773,0x77f7,0x7777,0x7777,0x0f77,0x0000,0x0000,
     0xc7c0,0xc001,0xdddb,0xdddd,0xdddd,0xdddd,0x0ddd,0x0000,0x0000,
     0x3362,0xc000,0x777b,0x77f7,0x7777,0x7777,0xff77,0x001f,0x0000,
     0x3edc,0xe000,0xddd7,0xdffd,0xdddd,0xdddd,0xdddf,0x007d,0x0000,
     0x03f0,0xe000,0x7777,0x77ff,0x7777,0xf777,0x7777,0x00f7,0x0000,
     0x0000,0xe000,0xddd7,0xff3d,0xdddd,0xfddd,0xdddf,0x01dd,0x0000,
     0x0000,0xf000,0x776f,0xfc1f,0x777f,0xfff7,0x7777,0x0377,0x0000,
     0x0000,0xf800,0xddcf,0xe00f,0xdddf,0xffff,0xdddf,0x03dd,0x0000,
     0x0000,0xfe00,0x776f,0x000f,0xffff,0x01ff,0x777e,0x077f,0x0000,
     0x0000,0xff00,0xddcf,0x0007,0xfffc,0x001f,0x7ff0,0x05c0,0x0000,
     0x0000,0xff80,0xf75f,0x0003,0x0000,0x0000,0x7fc0,0x0f40,0x0000,
     0x0000,0xffc0,0xfddf,0x0003,0x0000,0x0000,0x7fc0,0x0d80,0x0000,
     0x0000,0xffe0,0xf77d,0x0001,0x0000,0x0000,0xff80,0x1780,0x0000,
     0x0000,0x7ff0,0xfddc,0x0000,0x0000,0x0000,0xff80,0x3d00,0x0000,
     0x0000,0x1ff8,0x7f76,0x0000,0x0000,0x0000,0xff00,0x3701,0x0000,
     0x0000,0x0ff8,0x1fdf,0x0000,0x0000,0x0000,0xff00,0x7e01,0x0000,
     0x0000,0xc7fc,0x07f7,0x0000,0x0000,0x0000,0xfe00,0xf803,0x0000,
     0x0000,0xe1fc,0x01fd,0x0000,0x0000,0x0000,0xfc00,0xd003,0x0000,
     0x0000,0x60fe,0x0077,0x0000,0x0000,0x0000,0xf800,0x7807,0x0001,
     0x0000,0xf03f,0x001e,0x0000,0x0000,0x0000,0xf000,0xf807,0x0001,
     0x8000,0x701f,0x0007,0x0000,0x0000,0x0000,0xe000,0xf80f,0x0001,
     0xe000,0xd00f,0x0003,0x0000,0x0000,0x0000,0xc000,0xf81f,0x0001,
     0xf000,0x7007,0x0003,0x0000,0x0000,0x0000,0xc000,0xc03f,0x0001,
     0xf000,0xe007,0x0003,0x0000,0x0000,0x0000,0x8000,0x007f,0x0000,
     0xf000,0x6003,0x0007,0x0000,0x0000,0x0000,0x8000,0x00ff,0x0000,
     0xffc0,0xc001,0x0005,0x0000,0x0000,0x0000,0x0000,0x01ff,0x0000,
     0xffe0,0xc000,0x0007,0x0000,0x0000,0x0000,0x0000,0x03f8,0x0000,
     0x7fe0,0xc000,0x0005,0x0000,0x0000,0x0000,0x0000,0x0ff0,0x0000,
     0x1ff0,0x8000,0x000f,0x0000,0x0000,0x0000,0x0000,0x3fe0,0x0000,
     0x07f0,0x8000,0x000d,0x0000,0x0000,0x0000,0x0000,0x7fe0,0x0000,
     0x03f8,0x8000,0x000f,0x0000,0x0000,0x0000,0x0000,0xffc0,0x0000,
     0x00f8,0x8000,0x001d,0x0000,0x0000,0x0000,0x0000,0xfc00,0x0001,
     0x0000,0x0000,0x0037,0x0000,0x0000,0x0000,0x0000,0xfc00,0x0001,
     0x0000,0x0000,0x00ff,0x0000,0x0000,0x0000,0x0000,0xf800,0x0001,
     0x0000,0x0000,0x01f6,0x0000,0x0000,0x0000,0x0000,0x7800,0x0000,
     0x0000,0x0000,0x03de,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x077c,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0de0,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0fe0,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse5[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x4000,0x1000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x2400,0x3ee2,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xc900,0x0bff,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xfe20,0x09bf,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xffc0,0x0edf,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x77f8,0x1f77,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xddf2,0x1ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x777c,0x3777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0xdddf,0x3ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x9000,0x7777,0x7777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xe800,0xdddf,0x5ddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xfb00,0x7775,0xf777,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x7c40,0xdddd,0xdddd,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x5f90,0x7777,0xf77f,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xd7e0,0xdddd,0xdddd,0x0001,
     0x0000,0x0000,0xf800,0x0007,0x0000,0x75fe,0x7777,0x777f,0x0003,
     0x0000,0x0000,0x0780,0x007c,0x8000,0xdd3f,0xdddd,0xdddf,0x0003,
     0x0000,0x0000,0x7060,0xff83,0xffff,0x7750,0xf777,0x777f,0x0003,
     0x0000,0x0000,0xdd98,0x005d,0x0000,0xdddd,0xdddd,0xddff,0x0005,
     0x0000,0xbf00,0x7764,0x7777,0x7777,0x7777,0xf777,0x77e0,0x0007,
     0x0000,0xc0c4,0xddd3,0xdddd,0xdddd,0xdddd,0x7ddd,0xdbc0,0x0009,
     0x0000,0xfe38,0x7769,0x7777,0x7777,0x7777,0x3f77,0x7780,0x000f,
     0x4000,0xfb9e,0xdddd,0xdddd,0xdddd,0xdddd,0x1ddd,0xde00,0x001d,
     0x8000,0xf7ff,0x7775,0x7777,0x7777,0x7777,0x0f77,0x7c00,0x0017,
     0xc000,0xbeea,0xdddc,0xdddd,0xdddd,0xdddd,0x07dd,0xf800,0x001d,
     0xe000,0x9df4,0x7776,0x7777,0x7777,0x7777,0x0777,0x7000,0x0017,
     0x7c00,0x875d,0xddde,0xdddd,0xdddd,0xdddd,0x03dd,0xe000,0x000a,
     0x1800,0x81ef,0x7774,0x7777,0x7777,0x7777,0x0377,0x8000,0x0007,
     0xcc00,0x807f,0xddde,0xdddd,0xdddd,0xdddd,0x03dd,0x0000,0x0000,
     0xe600,0x801e,0x7774,0x7777,0x7777,0x7777,0x03f7,0x0000,0x0000,
     0x7940,0x8007,0xfddd,0xdddd,0xdddd,0xdddd,0x01dd,0x0000,0x0000,
     0x9da0,0x8001,0x7771,0x7777,0x7777,0x7777,0x01f7,0x0000,0x0000,
     0x7d80,0x8000,0xddd9,0xfddd,0xdddd,0xdddd,0x01dd,0x0000,0x0000,
     0x3e60,0x8000,0xf773,0xf777,0x7777,0x7777,0x01f7,0x0000,0x0000,
     0x0ff8,0xc000,0xdddb,0xdddf,0xdddd,0xdddd,0x01dd,0x0000,0x0000,
     0x0060,0xfe00,0x777b,0xf777,0x7777,0x7777,0x01f7,0x0000,0x0000,
     0x0000,0xff80,0xdddb,0xdddf,0xdddf,0xdddd,0x03dd,0x0000,0x0000,
     0x0000,0xffe0,0x7773,0x7fff,0x7777,0x7777,0x0777,0x0000,0x0000,
     0x0000,0xfff0,0xdddb,0xffff,0xdddf,0xdddd,0x0ddd,0x0000,0x0000,
     0x0000,0xfff8,0xf771,0xe001,0x777f,0x7777,0x7f77,0x0000,0x0000,
     0x0000,0xfffc,0x7ddd,0x0000,0xddff,0xdddd,0xaddd,0x0003,0x0000,
     0x0000,0x7ffe,0x1f75,0x0000,0xfffc,0xffff,0x7777,0x000f,0x0000,
     0x0000,0x87ff,0x0fe8,0x0000,0xffc0,0xffff,0xdddf,0x001d,0x0000,
     0x8000,0x707f,0x0377,0x0000,0x0000,0xff00,0x77f9,0x0017,0x0000,
     0xc000,0x8c1f,0x01dd,0x0000,0x0000,0xff00,0xdfc1,0x003d,0x0000,
     0xe000,0x63cf,0x00f7,0x0000,0x0000,0xfe00,0x7e03,0x00f7,0x0000,
     0xf000,0xd823,0x003d,0x0000,0x0000,0xfe00,0xf003,0x01dd,0x0000,
     0xfbe0,0x77a0,0x000f,0x0000,0x0000,0xfc00,0x0003,0x0377,0x0000,
     0xfff0,0xfd90,0x0001,0x0000,0x0000,0xfc00,0x0003,0x0ddc,0x0000,
     0x7ff8,0x1f50,0x0000,0x0000,0x0000,0xfc00,0x0003,0x3778,0x0000,
     0x1ff8,0x03c8,0x0000,0x0000,0x0000,0xf800,0x0003,0x5de0,0x0000,
     0x0ffc,0x0168,0x0000,0x0000,0x0000,0xf800,0x0003,0xf780,0x0000,
     0x03fe,0x01dc,0x0000,0x0000,0x0000,0xf000,0x0003,0xde00,0x0007,
     0x0000,0x00f4,0x0000,0x0000,0x0000,0xf000,0x0003,0x7c00,0x001f,
     0x0000,0x005c,0x0000,0x0000,0x0000,0xe000,0x0003,0xf000,0x003d,
     0x0000,0x0076,0x0000,0x0000,0x0000,0xe000,0x0003,0x8000,0x007f,
     0x0000,0x005e,0x0000,0x0000,0x0000,0xe000,0x0003,0x0000,0x007f,
     0x0000,0x0037,0x0000,0x0000,0x0000,0xe000,0x0003,0x0000,0x003f,
     0x0000,0x003d,0x0000,0x0000,0x0000,0xe000,0x0003,0x0000,0x001f,
     0x8000,0x0016,0x0000,0x0000,0x0000,0xe000,0x0003,0x0000,0x0000,
     0x8000,0x001d,0x0000,0x0000,0x0000,0xc000,0x0007,0x0000,0x0000,
     0x8000,0x0017,0x0000,0x0000,0x0000,0xc000,0x0007,0x0000,0x0000,
     0x8000,0x003d,0x0000,0x0000,0x0000,0xc000,0x000f,0x0000,0x0000,
     0x8000,0x0037,0x0000,0x0000,0x0000,0xc000,0x003f,0x0000,0x0000,
     0x0000,0x00df,0x0000,0x0000,0x0000,0xc000,0x007f,0x0000,0x0000,
     0x0000,0x01ee,0x0000,0x0000,0x0000,0x8000,0x00ff,0x0000,0x0000,
     0x0000,0x01dc,0x0000,0x0000,0x0000,0x0000,0x00ff,0x0000,0x0000,
     0x0000,0x03f8,0x0000,0x0000,0x0000,0x0000,0x01fc,0x0000,0x0000,
     0x0000,0x03f8,0x0000,0x0000,0x0000,0x0000,0x01fc,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse6[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x1800,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x1e00,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0b80,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x16e9,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xd800,0x3bbf,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xe380,0x2eef,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xff30,0x7bbb,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xffc0,0xeeee,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xbffe,0xbbbb,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xc000,0xebd8,0xeeee,0x0002,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0c00,0xbaff,0xbbbb,0x0003,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf100,0xeebf,0xeeee,0x0006,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xfe60,0xbbaf,0xbbbb,0x000f,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xff88,0xeeeb,0xeeee,0x000a,
     0x0000,0x0000,0xfe00,0x0007,0x0000,0xfff0,0xbbbb,0xbbbb,0x001b,
     0x0000,0x0000,0x0180,0x0078,0x8000,0xcffd,0xeeee,0xeeee,0x001e,
     0x0000,0x0000,0xba60,0x0f83,0x0000,0xbbfe,0xbbbb,0xbbbb,0x003b,
     0x0000,0xe400,0xee97,0x702e,0x0000,0xeebf,0xeeee,0xeeee,0x006e,
     0x0000,0x7840,0xbbab,0x83bb,0xffff,0xbbbb,0xbbbb,0xbffb,0x00fb,
     0x0000,0x7f80,0xeeef,0x0eee,0xea00,0xeeee,0xeeee,0xfffe,0x01ee,
     0x0000,0xedf8,0xbbbf,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0xb83f,0x01bb,
     0x0000,0x7f67,0xeeee,0xeeee,0xeeee,0xeeee,0xeeee,0xe007,0x03ee,
     0xc000,0x1b75,0xbbbe,0xbbbb,0xbbbb,0xbbbb,0xfbbb,0xc001,0x02bb,
     0x7000,0x0edb,0xeeee,0xeeee,0xeeee,0xeeee,0xeeee,0x8000,0x03ef,
     0xac00,0x07ae,0xbbb9,0xbbbb,0xbbbb,0xbbbb,0x7bbb,0x0000,0x017b,
     0xab00,0x03aa,0xeeed,0xeeee,0xeeee,0xeeee,0x3eee,0x0000,0x01fe,
     0xfa80,0x01ff,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0x1bbb,0x0000,0x00f8,
     0x4ac0,0x807d,0xeeee,0xeeee,0xeeee,0xeeee,0x0eee,0x0000,0x0000,
     0xffb0,0x4017,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0x0fbb,0x0000,0x0000,
     0x7f7a,0x4001,0xeeef,0xeeee,0xeeee,0xeeee,0x06ee,0x0000,0x0000,
     0x4ffc,0x4000,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0x07bb,0x0000,0x0000,
     0x2bf8,0x4000,0xeeef,0xeefe,0xeeee,0xeeee,0x06ee,0x0000,0x0000,
     0x0000,0xc000,0xbbba,0xbbbb,0xbbbb,0xbbbb,0x07bb,0x0000,0x0000,
     0x0000,0xf800,0xeeee,0xeefe,0xeeee,0xeeee,0x06ee,0x0000,0x0000,
     0x0000,0x7fc0,0xbbbb,0xbbbf,0xbbbb,0xbbbb,0x07bb,0x0000,0x0000,
     0x0000,0xffe0,0xeeee,0xeeee,0xeeee,0xeeee,0x06ee,0x0000,0x0000,
     0x0000,0xfff0,0xbbba,0xbbff,0xbbbb,0xbbbb,0x03bb,0x0000,0x0000,
     0x0000,0x7ff0,0xeeee,0xfff3,0xeeee,0xeeee,0x02ee,0x0000,0x0000,
     0x0000,0x7ff8,0xfbbb,0xff81,0xbbbb,0xbbbb,0x03bb,0x0000,0x0000,
     0x0000,0xbffc,0xffee,0xf800,0xeeff,0xeeee,0x03ee,0x0000,0x0000,
     0x0000,0xbffe,0x7fbb,0x8000,0xbfff,0xbbbb,0x06bb,0x0000,0x0000,
     0x0000,0xf8ff,0x3fee,0x0000,0xfffc,0xeedf,0x0eee,0x0000,0x0000,
     0x8000,0xb83f,0x0ffb,0x0000,0xffc0,0xffff,0x1bbb,0x0000,0x0000,
     0xc000,0xee0f,0x03ee,0x0000,0xe000,0x83ff,0x3eef,0x0000,0x0000,
     0xf0e0,0xbbc7,0x007f,0x0000,0x0000,0x01fc,0x7bbf,0x0000,0x0000,
     0xfb10,0xeeb3,0x000f,0x0000,0x0000,0x01fe,0xeef8,0x0000,0x0000,
     0xffe8,0xfb89,0x0001,0x0000,0x0000,0x01fe,0xbbc0,0x0001,0x0000,
     0x7ff8,0x3ae4,0x0000,0x0000,0x0000,0x00fe,0xef80,0x0003,0x0000,
     0x3ffc,0x0fb4,0x0000,0x0000,0x0000,0x00fe,0xbe00,0x0007,0x0000,
     0x0ffc,0x03ea,0x0000,0x0000,0x0000,0x007e,0xec00,0x000e,0x0000,
     0x0000,0x005b,0x0000,0x0000,0x0000,0x003e,0xf800,0x003b,0x0000,
     0x0000,0x003d,0x0000,0x0000,0x0000,0x003e,0xf000,0x006e,0x0000,
     0x8000,0x001f,0x0000,0x0000,0x0000,0x003e,0xc000,0x00fb,0x0000,
     0xc000,0x000e,0x0000,0x0000,0x0000,0x003f,0x8000,0x01ef,0x0000,
     0xa000,0x0007,0x0000,0x0000,0x0000,0x001f,0x0000,0x03bc,0x0000,
     0xfc00,0x0007,0x0000,0x0000,0x0000,0x001f,0x0000,0x0ef0,0x0000,
     0xbaf8,0x0003,0x0000,0x0000,0x0000,0x001f,0x0000,0xfbe0,0x0000,
     0xef88,0x0001,0x0000,0x0000,0x0000,0x000f,0x0000,0xefc0,0x0003,
     0x7bf4,0x0000,0x0000,0x0000,0x8000,0x000f,0x0000,0xbf80,0x000f,
     0x1ef4,0x0000,0x0000,0x0000,0x8000,0x0007,0x0000,0xf000,0x001f,
     0x07fa,0x0000,0x0000,0x0000,0x8000,0x0007,0x0000,0xf000,0x001f,
     0x01fe,0x0000,0x0000,0x0000,0x8000,0x0007,0x0000,0xe000,0x000f,
     0x003c,0x0000,0x0000,0x0000,0xc000,0x0007,0x0000,0xe000,0x0003,
     0x0000,0x0000,0x0000,0x0000,0xc000,0x000f,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0xc000,0x003f,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0xc000,0x00ff,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x8000,0x00ff,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x01fc,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x01fc,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

short horse7[] =
{
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x7000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x9000,0x6e0c,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xe400,0x13ff,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xf880,0x1957,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xff20,0x18ed,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xbfd0,0x1bbb,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0xebe2,0x6eee,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x4000,0xbafc,0x7bbb,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8000,0xee7f,0xeeee,0x0001,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x8400,0xbbbf,0xbbbb,0x0003,
     0x0000,0x0000,0x0000,0x0000,0x0000,0xf880,0xeeef,0xeeee,0x0006,
     0x0000,0x0000,0xfc00,0x001f,0x0000,0xff10,0xbbb9,0xbbbb,0x0007,
     0x0000,0x0000,0x0380,0x07e0,0x0000,0x5fe0,0xeeee,0xeeee,0x000e,
     0x0000,0x0000,0xb86a,0x381b,0x0000,0x97c0,0xbbbb,0xbbbb,0x0013,
     0x0000,0xf000,0xee9f,0xc7ee,0x002f,0xf5ff,0xeeee,0xeeee,0x001e,
     0x0000,0x7c00,0xbbaa,0x1bbb,0xfff8,0xbab5,0xbbbb,0xbbbb,0x002b,
     0x0000,0x3f50,0xeee6,0xfeee,0x0007,0xeea8,0xeeee,0xeeee,0x006e,
     0x0000,0x8ff8,0xbbb5,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0xbbab,0x007b,
     0x4000,0xe35f,0xeee4,0xeeee,0xeeee,0xeeee,0xeeee,0xeffe,0x00ee,
     0xc000,0x33a9,0xbbba,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0xbc07,0x01bb,
     0x7000,0x38e8,0xeeea,0xeeee,0xeeee,0xeeee,0xeeee,0xf001,0x01ee,
     0x1a00,0x1cbb,0xbbba,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0xe000,0x01bb,
     0x8600,0x071e,0xeeea,0xeeee,0xeeee,0xeeee,0x6eee,0x8000,0x01af,
     0x6b00,0x01d5,0xbbbb,0xbbbb,0xbbbb,0xbbbb,0x3bbb,0x0000,0x00ff,
     0x5180,0x00ff,0xeeeb,0xeeee,0xeeee,0xeeee,0x3eee,0x0000,0x00fe,
     0xfc80,0x800b,0xbbba,0xbbbb,0xbbbb,0xbbbb,0x1bbb,0x0000,0x0038,
     0x2ec0,0x0000,0xeeea,0xeeee,0xeeee,0xeeee,0x0eee,0x0000,0x0000,
     0x03c0,0x8000,0xbbba,0xbbbb,0xbbbb,0xbbbb,0x0fbb,0x0000,0x0000,
     0x00e4,0x8000,0xeee6,0xeeee,0xeeee,0xeeee,0x06ee,0x0000,0x0000,
     0x0078,0xf000,0xbbb5,0xbbbb,0xbbbb,0xbbbb,0x07bb,0x0000,0x0000,
     0x0000,0xfc00,0xeeef,0xeefe,0xeeee,0xeeee,0x03ee,0x0000,0x0000,
     0x0000,0xfe00,0xbbab,0xbbbb,0xbbbb,0xbbbb,0x03bb,0x0000,0x0000,
     0x0000,0xfe00,0xeeeb,0xeefe,0xeeee,0xeeef,0x03ee,0x0000,0x0000,
     0x0000,0xfe00,0xbbab,0xbbff,0xbbbb,0xbbbb,0x01bb,0x0000,0x0000,
     0x0000,0xfe00,0xeeef,0xff87,0xeeee,0xeeef,0x01ee,0x0000,0x0000,
     0x0000,0xff00,0xfbb1,0xfc00,0xffbf,0xbbbb,0x01bb,0x0000,0x0000,
     0x0000,0x0fe0,0x6eee,0x0000,0xffff,0xeeee,0x00ee,0x0000,0x0000,
     0x0000,0xb010,0x3bbb,0x0000,0xffe0,0xbbbb,0x00bb,0x0000,0x0000,
     0x0000,0xeee8,0x1eae,0x0000,0xfe00,0xefee,0x01ee,0x0000,0x0000,
     0x0000,0xbbb8,0x07db,0x0000,0xc000,0xbfff,0x01bb,0x0000,0x0000,
     0x0000,0xeeec,0x00fe,0x0000,0xc000,0xf3ff,0x03ee,0x0000,0x0000,
     0x0000,0xffba,0x0003,0x0000,0xe000,0xe3ff,0x03bb,0x0000,0x0000,
     0x0000,0x07ee,0x0000,0x0000,0xe000,0xc1ff,0x02ef,0x0000,0x0000,
     0x8000,0x07bb,0x0000,0x0000,0xf000,0xc0ff,0x07bf,0x0000,0x0000,
     0x8000,0x03ee,0x0000,0x0000,0xf000,0x807f,0x06ef,0x0000,0x0000,
     0xc040,0x01fb,0x0000,0x0000,0xf800,0x003f,0x07be,0x0000,0x0000,
     0xe0a0,0x007e,0x0000,0x0000,0xf800,0x001f,0x0efc,0x0000,0x0000,
     0xb91c,0x0007,0x0000,0x0000,0xfc00,0x000f,0x1bb0,0x0000,0x0000,
     0xeec6,0x0003,0x0000,0x0000,0xfe00,0x0003,0x1ee0,0x0000,0x0000,
     0xbbbc,0x0000,0x0000,0x0000,0xfe00,0x0000,0x1bc0,0x0000,0x0000,
     0x6ef8,0x0000,0x0000,0x0000,0x7f00,0x0000,0x2ec0,0x0000,0x0000,
     0x3be0,0x0000,0x0000,0x0000,0x3f80,0x0000,0x7b80,0x0000,0x0000,
     0x0f80,0x0000,0x0000,0x0000,0x1fc0,0x0000,0x6f00,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0fe0,0x0000,0xba00,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x03f0,0x0000,0xec00,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x01f8,0x0000,0xb800,0x0001,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fc,0x0000,0xf000,0x0002,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x007e,0x0000,0xa000,0x0007,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x007f,0x0000,0xe000,0x0006,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x003f,0x0000,0xc000,0x000b,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x003f,0x0000,0xc000,0x001e,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x003e,0x0000,0x8000,0x00fb,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x007c,0x0000,0x8000,0x00ef,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fe,0x0000,0x0000,0x03bf,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x00fe,0x0000,0x0000,0x06f8,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x01fe,0x0000,0x0000,0x07b8,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x01fe,0x0000,0x0000,0x07f8,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
     0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
};

