;-------------------------------------------------------------------------;
;                                  TIGA                                   ;
;        Copyright (c) 1987-1990  Texas Instruments Incorporated.         ;
;			   All Rights Reserved				  ;
;-------------------------------------------------------------------------;
;   TIGA - Graphics Manager                                               ;
;-------------------------------------------------------------------------;
;                                                                         ;
;   void set_palet(size, offset, palet_ary)                               ;
;                                                                         ;
;   This function loads the global "palet" array with the palette values  ;
;   in the parameters and then goes on to load the palette itself.        ;
;   This routine is intended for display boards using the TMS34070        ;
;   palette in line-load mode.                                            ;
;                                                                         ;
;-------------------------------------------------------------------------;
;                                                                         ;
;   int  set_palet_entry(index, r, g, b, i)                               ;
;                                                                         ;
;   This function loads one entry of the global "palet" array,            ;
;   and then goes on to load the palette itself.                          ;
;                                                                         ;
;   Notes: - A8 is False (0) if an invalid index is requested             ;
;-------------------------------------------------------------------------;
;  01/01/87  Original Version Written                      J.G.Littleton  ;
;  07/12/88  Modified for 8-bit RGB, global palet, TIGA dm Graham Short   ;
;  08/26/88  Modified to support multiple display pages    J.G.Littleton  ;
;  09/12/88  Access globals thru environment struct        W.S.Egr        ;
;  11/21/88  Added return value to _entry function         Graham Short   ;
;  01/10/89  Modified to store 4 MSBs in _palet only       Graham Short   ;
;  01/25/89  Modified clear CONTROL to not clear CD bit    Graham Short   ;
;  01/26/89  Cleared PMASK and wrote to all display pages  Graham Short   ;
;  07/16/89  Added monochrome monitor detection            Graham Short   ;
;  10/20/89  Modified for the 34020 SDB board              Graham Short   ;
;-------------------------------------------------------------------------;
        .include   oem.inc
        .include   gspreg.inc
        .include   gsptypes.inc
        .include   gspglobs.inc
;
;   Define Functions        
        .globl     _set_palet
        .globl     _set_palet_entry
        .globl     _dm_set_palet_entry

Rcount  .set    A0
Rarg    .set    A1
Rcol    .set    A2
Rpal    .set    A3
Rbrook  .set    A4
Roffset .set    A5

Rindex  .set    A0
Rred    .set    A1
Rgrn    .set    A2
Rblu    .set    A6
Rint    .set    A7

COLOR_MONOCHROME_FLAG   .set    1

;-------------------------------------------------------------------------;
;  set_palet                                                              ;
;-------------------------------------------------------------------------;
_set_palet:
        mmtm    SP,A0,A1,A2,A3,A4,A5
        move    *-STK,Rcount,1              ; number  of palette vals
        move    *-STK,Roffset,1             ; offset into the palet  
        move    *-STK,Rarg,1                ; address of palette vals
;-------------------------------------------------------------------------;
;  if count is zero, abort                                                ;
;-------------------------------------------------------------------------;
check_count: 
        or      Rcount,Rcount               
        jrz     exit 
;-------------------------------------------------------------------------;
;  check that the number of entries being set does not overflow the palet ;
;-------------------------------------------------------------------------;
set_Rcount:
        move    @_config+CONFIG_PALET_SIZE,Rpal,1 ; get size of palet in temp           
        move    Roffset,Rbrook
        add     Rcount,Rbrook           ;get Offset + Count in temporary
        cmp     Rbrook,Rpal             ;compare against size of palette
        jrgt    init_brook
        move    Rpal,Rcount             ;if larger, adjust count 
        sub     Roffset,Rcount              
init_brook:
        movi    0FFh,Rbrook                     ; Set up read mask
        move    Rbrook,@PALET_READ_MASK
        move    Roffset,Rpal                    ;Address of palet entry
        sla     5,Rpal                          ;32 bits per palet entry
        addi    _palet,Rpal                     ;Address of global palet array
        move    Roffset,@PALET_WRITE_ADDR       ;Set palette address register
        movi    PALET_RAM,Rbrook                ;Setup address of palette data register
        setf    8,0,0
        move    @_monitorinfo,A8,1   
        .if     GSP_34010
        move    *A8(MONITOR_FLAGS10),A8,1 ; Get color/mono flag
        .endif
        .if     GSP_34020
        move    *A8(MONITOR_FLAGS20),A8,1 ; Get color/mono flag
        .endif
        andi    COLOR_MONOCHROME_FLAG,A8
        jrz     color_monitor
monochrome_monitor:
        move    *Rarg+,Rcol,0           ; read "r" component
        move    Rcol,*Rpal+,0           ; store "r" in global palet array
        move    *Rarg+,Rcol,0           ; read "g" component
        move    Rcol,*Rpal+,0           ; store "g" in global palet array
        move    *Rarg+,Rcol,0           ; read "b" component
        move    Rcol,*Rpal+,0           ; store "b" in global palet array
        move    *Rarg+,Rcol,0           ; read intensity component
        move    Rcol,*Rpal+,0           ; store in global palet array
        move    Rcol,*Rbrook,1          ; store in red component of real palet
        move    Rcol,*Rbrook,1          ; store in green component of real palet
        move    Rcol,*Rbrook,1          ; store in blue component of real palet
        dsjs    Rcount,monochrome_monitor
        jruc    exit
color_monitor:
        move    *Rarg+,Rcol,0           ; read red component
        move    Rcol,*Rpal+,0           ; store in global palet array
        move    Rcol,*Rbrook,1          ; store in real palet
        move    *Rarg+,Rcol,0           ; read green component
        move    Rcol,*Rpal+,0           ; store in global palet array
        move    Rcol,*Rbrook,1          ; store in real palet
        move    *Rarg+,Rcol,0           ; read blue component
        move    Rcol,*Rpal+,0           ; store in global palet array
        move    Rcol,*Rbrook,1          ; store in real palet
        move    *Rarg+,Rcol,0           ; read intensity component
        move    Rcol,*Rpal+,0           ; store in global palet array
        dsjs    Rcount,color_monitor
;
exit:  
        mmfm    SP,A0,A1,A2,A3,A4,A5
        rets    2


;-------------------------------------------------------------------------;
;  set_palet_entry                                                        ;
;-------------------------------------------------------------------------;
;                                                                         ;
; DIRECT-MODE ENTRY POINT                                                 ;
;                                                                         ;
;-------------------------------------------------------------------------;
_dm_set_palet_entry:
        mmtm    SP,A0,A1,A2,A3,A4,A5,A6,A7
;-------------------------------------------------------------------------;
;  get arguments                                                          ;
;-------------------------------------------------------------------------;
        move    *-STK,A8,1                  ;address of data area
        setf    16,0,0
        move    *A8+,Rindex,1               ;get index
        move    *A8+,Rred,0                 ;get red color
        move    *A8+,Rgrn,0                 ;get green color
        move    *A8+,Rblu,0                 ;get blue color
        move    *A8,Rint,0                  ;get intensity color
        jruc    common_ep2
;-------------------------------------------------------------------------;
;                                                                         ;
; C-PACKET ENTRY POINT                                                    ;
;                                                                         ;
;-------------------------------------------------------------------------;
_set_palet_entry
        mmtm    SP,A0,A1,A2,A3,A4,A5,A6,A7
;-------------------------------------------------------------------------;
;  get arguments                                                          ;
;-------------------------------------------------------------------------;
        move    *-STK,Rindex,1              ;pop index
        move    *-STK,Rred,1                ;pop red color
        move    *-STK,Rgrn,1                ;pop green color
        move    *-STK,Rblu,1                ;pop blue color
        move    *-STK,Rint,1                ;pop intensity color
        setf    16,0,0
;-------------------------------------------------------------------------;
;                                                                         ;
; 2-ENTRY POINTS JOIN UP HERE                                             ;
;                                                                         ;
;-------------------------------------------------------------------------;
;  check index, if too big abort and return FALSE                         ;
;-------------------------------------------------------------------------;
common_ep2:
        clr     A8
        move    @_config+CONFIG_PALET_SIZE,Rbrook,1 
        cmp     Rindex,Rbrook                      
        jrlo    exit2                              
;
set_Rpal2:  
        movi    0FFh,Rbrook
        move    Rbrook,@PALET_READ_MASK
        move    Rindex,Rpal                     ;Address of palet entry
        sla     5,Rpal                          ;32 bits per palet entry
        addi    _palet,Rpal                     ;Address of global palet array
        move    Rindex,@PALET_WRITE_ADDR        ;Set palette address register
        movi    PALET_RAM,Rbrook                ;Setup address of palette data register
        setf    8,0,0
        move    @_monitorinfo,A8,1   
        .if     GSP_34010
        move    *A8(MONITOR_FLAGS10),A8,1 ; Get color/mono flag
        .endif
        .if     GSP_34020
        move    *A8(MONITOR_FLAGS20),A8,1 ; Get color/mono flag
        .endif
        andi    COLOR_MONOCHROME_FLAG,A8
        jrz     color_monitor2
monochrome_monitor2:
        move    Rred,*Rpal+,0           ; store "r" in global palet array
        move    Rgrn,*Rpal+,0           ; store "g" in global palet array
        move    Rblu,*Rpal+,0           ; store "b" in global palet array
        move    Rint,*Rpal+,0           ; store "i" in global palet array
        move    Rint,*Rbrook,1          ; store in red component of real palet
        move    Rint,*Rbrook,1          ; store in green component of real palet
        move    Rint,*Rbrook,1          ; store in blue component of real palet
        movk    1,A8                    ; normal return (TRUE, A8 <> 0)
        jruc    exit2
color_monitor2:
        move    Rred,*Rpal+,0           ; store in global palet array
        move    Rred,*Rbrook,1          ; store in real palet
        move    Rgrn,*Rpal+,0           ; store in global palet array
        move    Rgrn,*Rbrook,1          ; store in real palet
        move    Rblu,*Rpal+,0           ; store in global palet array
        move    Rblu,*Rbrook,1          ; store in real palet
        move    Rint,*Rpal+,0           ; store intensity in palet array
        movk    1,A8                    ; normal return (TRUE, A8 <> 0)
exit2:
        mmfm    SP,A0,A1,A2,A3,A4,A5,A6,A7
        rets    2
