
*----------------------------------------------------------------------
*                                    TIGA
*          Copyright (C) 1989-1990  Texas Instruments Incorporated.
*                            All Rights Reserved
*----------------------------------------------------------------------
*                       TIGA 3D Flight Simulator
*----------------------------------------------------------------------
*
* Define data structures for modules written in GSP Assembly Language:
*     CLIP3D       parameters used in clipping to 3D viewport
*     ZCHECK       parameters for comparing object radius to distance
*     VIEWPOINT    viewing orientation unit vectors, position and speed
*     VIEWINFO     sine and cosine of yaw, pitch and roll angles
*     NODE         node in 3D database organized as binary tree
*     LEAF         leaf or terminal node in 3D database
*     NONLEAF      nonleaf or intermediate node in 3D database
*     DBASE3D      header information for 3D database
*
*                                                  J. Van Aken 03/23/89
*----------------------------------------------------------------------

*----------------------------------------------------------------------
* typedef struct {
*    short xleft, ytop;  /* top left corner of viewport on screen     */
*    short sx, sy;       /* viewport half width and half height       */
*    short cx, cy;       /* xy offsets to viewport center from corner */
*    /* Set rdiag = sqrt(sx*sx+sy*sy) */
*    short rdiag;        /* viewport half diagonal length             */
*    short d;            /* viewer's assumed distance from screen     */
*    short znear;        /* z value at "hither" clipping plane        */
*    short bufsize;      /* size of each buffer in bits               */
*    long *buf[2];       /* pointers to buffers for intermediate vals */
*    /* Set kx = (d<<18)/sx and ky = (d<<18)/sy. */
*    long kx, ky;        /* clipping volume normalization factors     */
* } CLIP3D;
*----------------------------------------------------------------------
; Displacements into CLIP3D data structure, global variable _vpclip:
XLEFT    .set     0
YTOP     .set     16
SX       .set     2*16
SY       .set     3*16
CX       .set     4*16
CY       .set     5*16
RDIAG    .set     6*16
D        .set     7*16
ZNEAR    .set     8*16
BUFSIZE  .set     9*16
BUF0     .set     10*16
BUF1     .set     12*16
KX       .set     14*16
KY       .set     16*16


*----------------------------------------------------------------------
* typedef struct {
*    /* If z/r > kvis, object is too distant to be visible. */
*    short kvis;         /* d/Rvis-1, where Rvis = threshhold radius  */
*    /* If z/r > kdot, object appears as a single dot on the screen. */
*    short kdot;         /* d/Rdot-1, where Rvis = threshhold radius  */
*    /* If z/r > knpc, object is too small to require preclipping. */
*    short knpc;         /* d/Rnpc-1, where Rvis = threshhold radius  */
* } ZCHECK;
*----------------------------------------------------------------------
; Displacements into ZCHECK data structure, global variable _ztest:
KVIS     .set     0
KDOT     .set     16
KNPC     .set     2*16


*----------------------------------------------------------------------
* typedef struct {
*    long u[3];      /* view "right" vector         (30-bit fraction) */
*    long v[3];      /* view "down" vector          (30-bit fraction) */
*    long w[3];      /* viewplane normal vector     (30-bit fraction) */
*    long xyz[3];    /* view reference point         (8-bit fraction) */
*    long speed;     /* translation per frame        (8-bit fraction) */
* } VIEWPOINT;
*----------------------------------------------------------------------
; Displacements into VIEWPOINT data structure, global variable _view:
UX       .set     0
UY       .set     32
UZ       .set     2*32
VX       .set     3*32
VY       .set     4*32
VZ       .set     5*32
WX       .set     6*32
WY       .set     7*32
WZ       .set     8*32
XV       .set     9*32
YV       .set     10*32
ZV       .set     11*32
SPEED    .set     12*32


*----------------------------------------------------------------------
* typedef struct {
*    long csyaw, snyaw;  /* cos, sin of yaw angle   (30-bit fraction) */
*    long cspch, snpch;  /* cos, sin of pitch angle (30-bit fraction) */
*    long csrol, snrol;  /* cos, sin of roll angle  (30-bit fraction) */
* } VIEWINFO;
*----------------------------------------------------------------------
; Displacements into VIEWPOINT data structure, global variable _vuparams:
CSYAW    .set     0
SNYAW    .set     32
CSPCH    .set     2*32
SNPCH    .set     3*32
CSROL    .set     4*32
SNROL    .set     5*32


*----------------------------------------------------------------------
* typedef struct {
*     unsigned bsrad : 8;     /* radius of node's bounding sphere     */
*     unsigned log2f : 5;     /* log2 of scaling factor for bsrad     */
*     unsigned ntyp  : 3;     /* node type = 7 (leaf), 0 (nonleaf)    */
*     unsigned dotcolor : 16; /* color of distant object seen as dot  */
* } NODE;
*----------------------------------------------------------------------
* typedef struct {
*     NODE n;                 /* node type = 7 (leaf node)            */
*     unsigned short bbdisp;  /* offset in words to bounding box data */
* } LEAF;
*----------------------------------------------------------------------
* typedef struct {
*     NODE n;                 /* node type = 0 (nonleaf node)         */
*     NODE *lson;             /* pointer to node's left son           */
*     ldisp[4] : 8;           /* xyz displacements to left son        */
*     NODE *rson;             /* pointer to node's right son          */
*     rdisp[4] : 8;           /* xyz displacements to right son       */
*     bpp[4] : 4;             /* binary partitioning plane coeff's    */
* } NONLEAF;
*----------------------------------------------------------------------
; Offsets common to both leaf and nonleaf NODE structures
BSRAD   .set      0                   ;radius of bounding sphere
LOG2F   .set      8                   ;log2(scale factor) for BSRAD
NTYP    .set      13                  ;node type (7=leaf, 0=nonleaf)
DOTCOLOR .set     16                  ;node's color at great distance
; Offsets specific to LEAF node structure
BBDISP  .set      32                  ;word displacement to bounding box
; Offsets specific to NONLEAF node structure
LSON    .set      1*32                ;pointer to node's left son
LDISP   .set      2*32                ;left son's xyz disp. from node
RSON    .set      3*32                ;pointer to node's right son
RDISP   .set      4*32                ;right son's xyz disp. from node
BPP     .set      5*32                ;binary partitioning plane coeff's


*----------------------------------------------------------------------
* typedef struct {    /* header information for 3D object database       */
*     long dbsize;       /* length of database in bytes                  */
*     long rootnode;     /* bit offset to root node of binary tree       */
*     POINT3D start;     /* starting xyz coordinates in world space      */
*     PALET palet[16];   /* palette of 16 colors (see typedefs.h)        */
*     PALET bwpatn[16];  /* 16 4x8 patterns for monochrome displays      */
*     long bcdata;       /* bit offset to object data for bouncing cube  */
*     long bcbbox;       /* bit offset to bounding box for bouncing cube */
* } DBASE3D;
*----------------------------------------------------------------------
; Displacements into DBASE3D data structure, global variable _db3d:
DBSIZE   .set     0                   ;database size in bytes
ROOTNODE .set     DBSIZE+32           ;bit offset to root node of tree
START    .set     ROOTNODE+32         ;starting xyz coordinates in FIX8
PALET    .set     START+3*32          ;16-color palette
BWPATN   .set     PALET+8*4*16        ;16 patterns for 1-bit-per-pixel
BCDATA   .set     BWPATN+8*4*16       ;offset to masher 3D model data
BCBBOX   .set     BCDATA+32           ;offset to masher bounding box data

