/*======================================================================*/
/*  isrdemo.c   TIGA Interrupt Service Routine demonstration            */
/*                                                                      */
/*  Copyright (C) 1990, Texas Instruments, All Rights Reserved          */
/*======================================================================*/
#include <tiga.h>
#include <extend.h>
#include <sys\types.h>
#include <sys\timeb.h>

/*----------------------------------------------------------------------*/
/*  Miscellaneous defs                                                  */
/*----------------------------------------------------------------------*/
#define NUM_ISRS    4               /* Total # of ISRs we will install  */
#define DI_TRAP     10              /* Display Interrupt trap #         */
#define WV_TRAP     11              /* Windows Violation trap #         */
#define HI_TRAP     9               /* Host Interrupt trap #            */
#define SW_TRAP     16              /* Example Software trap #          */
#define ENABLE      1               /* ISR enable flag                  */
#define DISABLE     0               /* ISR disable flag                 */
#define LOOP_COUNT  10000           /* Number of test loops             */    
#define HINT        0x0020          /* Hint flag for set_xstate()       */

/*----------------------------------------------------------------------*/
/*  Define entry point calls to functions in 'isrmodul.rlm'             */
/*----------------------------------------------------------------------*/
#define get_isr_service_count(trap_index)  \
        cp_ret(USER_CP(isr_module | 0),1,_WORD(trap_index))
#define invoke_sw_trap(trapnum)         \
        cp_cmd(USER_CP(isr_module | 1),1,_WORD(trapnum))
#define get_last_scanline()             \
        cp_ret(USER_CP(isr_module | 2),0)

/*----------------------------------------------------------------------*/
/*  Global variable definitions                                         */
/*----------------------------------------------------------------------*/
         short  isr_module;         /* RLM module number                */
unsigned short  priority[NUM_ISRS]; /* Array to contain ISR priorities  */


main()
{
    short           i, count, error;
    struct timeb    start_time, stop_time;   

    /*------------------------------------------------------------------*/
    /*  Initialize the TIGA environment and load extended graphics lib  */
    /*  The init_tiga() function is defined in the file tiga.c          */
    /*------------------------------------------------------------------*/
    printf( "Initializing TIGA........ " );
    init_tiga(1);
    printf( "OK.\n" );

    /*------------------------------------------------------------------*/
    /*  Load TIGA extension required for this demonstration             */
    /*------------------------------------------------------------------*/
    printf( "Loading [isrmodul.rlm]... " );
    if( (isr_module = install_rlm("isrmodul")) < 0 )
    {
        printf( "\n  Error [%d] loading 'isrmodul.rlm'\n", isr_module );
        term_tiga();
    }
    else
        printf( "OK.\n" );

    /*------------------------------------------------------------------*/
    /*  ISRs have been loaded into TIGA.  We now need to get the        */
    /*  priority of each ISR to be used for ISR identification.         */
    /*------------------------------------------------------------------*/
    get_isr_priorities( NUM_ISRS, priority );

    /*------------------------------------------------------------------*/
    /*  Start timer.  This is used to calculate elapsed time required   */
    /*  for screen refresh value.                                       */
    /*------------------------------------------------------------------*/
    ftime( &start_time );

    /*------------------------------------------------------------------*/
    /*  Enable all ISRs using the set_interrupt() function              */
    /*------------------------------------------------------------------*/
    set_interrupt( DI_TRAP, priority[0], ENABLE, get_last_scanline() );
    set_interrupt( WV_TRAP, priority[1], ENABLE, -1 );
    set_interrupt( HI_TRAP, priority[2], ENABLE, -1 );
    set_interrupt( SW_TRAP, priority[3], ENABLE, -1 );

    /*------------------------------------------------------------------*/
    /*  Cause each ISR to be serviced a given number of times.          */
    /*  Display Interrupt will be serviced each time the screen is      */
    /*  refreshed.                                                      */
    /*------------------------------------------------------------------*/
    printf( "Testing ISRs............. " );
    set_clip_rect(1,1,0,0);         /* Make clip rect small to gen WV   */
    set_windowing(2);               /* WV interrupt on write outside    */
    for( i=0; i<LOOP_COUNT; i++ )
    {
        draw_point(10,10);          /* Generate WV interrupt            */
        set_xstate(HINT);           /* Generate Host Interrupt          */
        while( get_xstate() & HINT )/* Wait until TMS340 has processed  */
            ;                       /*  prior Host Interrupt            */
        invoke_sw_trap(SW_TRAP);    /* Generate software trap 16 int    */
    }
    printf( "OK.\n" );

    /*------------------------------------------------------------------*/
    /*  Stop the timer.                                                 */
    /*------------------------------------------------------------------*/
    ftime( &stop_time );

    /*------------------------------------------------------------------*/
    /*  Disable all ISR's                                               */
    /*------------------------------------------------------------------*/
    set_interrupt( DI_TRAP, priority[0], DISABLE, -1 );
    set_interrupt( WV_TRAP, priority[1], DISABLE, -1 );
    set_interrupt( HI_TRAP, priority[2], DISABLE, -1 );
    set_interrupt( SW_TRAP, priority[3], DISABLE, -1 );

    /*------------------------------------------------------------------*/
    /*  Calculate the number of milliseconds elapsed while the display  */
    /*  interrupt was enabled.  Then, calculate the screen refresh rate */
    /*  based on this elapsed time and the number of times the DI was   */
    /*  serviced.                                                       */
    /*------------------------------------------------------------------*/
    if( stop_time.millitm < start_time.millitm )
    {
        stop_time.time--;
        stop_time.millitm += 1000;
    }
    count = ((stop_time.time - start_time.time) * 1000) +
                (stop_time.millitm - start_time.millitm);
    printf( "Screen refresh........... %d frames/sec.\n",
        (get_isr_service_count(0) * 1000L) / count );

    /*------------------------------------------------------------------*/
    /*  Now, read back the number of times each ISR was serviced.       */
    /*  These counts should equal LOOP_COUNT.                           */
    /*------------------------------------------------------------------*/
    printf( "Checking ISR status...... " );
    for( error=0, i=1; i<NUM_ISRS; i++ )
        if( (count = get_isr_service_count(i)) != LOOP_COUNT )
        {
            printf( "\n  ISR #%d: Expected: %d, Read: %d", i, LOOP_COUNT, count );    
            error = 1;
        }
    printf( "%s", error ? "\n" : "All ISRs OK.\n" );

    /*------------------------------------------------------------------*/
    /*  Terminate the TIGA environment and return to DOS                */
    /*  The term_tiga() function is defined in the file tiga.c          */
    /*------------------------------------------------------------------*/
    term_tiga();
}
