/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                      TIGA DEMO FRAME                                     */
/*                                                                          */
/* Draw TI logo and a series of outlines emanating from the centre of the   */
/* screen.                                                                  */
/*                                                                          */
/*    9/14/86...Original version written..................C. Killebrew      */
/*    7/16/87...Modified for Siggraph demo................J. Van Aken       */
/*   12/29/88...Converted to TIGA, rewrote Math funcs.....Graham Short      */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/* Microsoft-c insert file */
#include <malloc.h>

#include <typedefs.h>
#include <tiga.h>
#include <extend.h>

#include "tigademo.h"
#include "colors.h"

typedef long  FIX;
typedef short SCREEN;

#define Z1 0

/*-------------------------------- data ------------------------------------*/

/* List of x, y, and z coordinates for vertices in TI logo. */
static FIX tibug[]={
   37,-45,Z1,  41,-46,Z1,  56,-46,Z1,  68,-43,Z1,
   72,-40,Z1,  71,-30,Z1,  73,-22,Z1,  74,-18,Z1,
   77,-14,Z1,  79,-9,Z1,   79,5,Z1,    76,19,Z1,
   71,23,Z1,   63,26,Z1,   46,36,Z1,   38,45,Z1,

   35,52,Z1,   35,60,Z1,   37,66,Z1,   40,71,Z1,
   40,74,Z1,   34,76,Z1,   24,76,Z1,   21,75,Z1,
   9,69,Z1,    -1,59,Z1,   -8,47,Z1,   -14,35,Z1,
   -17,26,Z1,  -22,24,Z1,  -26,24,Z1,  -31,26,Z1,

   -38,33,Z1,  -43,33,Z1,  -51,29,Z1,  -58,20,Z1,
   -59,16,Z1,  -62,10,Z1,  -69,3,Z1,   -73,1,Z1,
   -76,-3,Z1,  -80,-9,Z1,  -80,-17,Z1, -41,-17,Z1,
   -41,-77,Z1, -41,-77,Z1, 0,-77,Z1,   0,-51,Z1,
   10,-46,Z1,
      
   7,-25,Z1,   -8,-25,Z1,  -11,-10,Z1, 3,-10,Z1,   /* hollow "t" */
   -3,18,Z1,   -2,25,Z1,   0,29,Z1,    4,33,Z1,
   11,36,Z1,   19,36,Z1,   23,35,Z1,   28,33,Z1,
   34,27,Z1,   34,22,Z1,   31,24,Z1,   28,24,Z1,
   25,21,Z1,   26,18,Z1,   31,-10,Z1,  42,-10,Z1,
   45,-25,Z1,  33,-25,Z1,

   13,-38,Z1,  31,-38,Z1,  21,18,Z1,   3,18,Z1,    /* letter "i" */

   27,-59,Z1,  29,-58,Z1,  32,-55,Z1,  33,-53,Z1,  /* the dot */
   33,-49,Z1,  32,-47,Z1,  29,-44,Z1,  27,-43,Z1,
   22,-43,Z1,  20,-44,Z1,  17,-47,Z1,  16,-49,Z1,
   16,-53,Z1,  17,-55,Z1,  20,-58,Z1,  22,-59,Z1
};
#define NUM_TIBUG sizeof(tibug)/(3 * sizeof(FIX))

/* Connectivity information for Texas map */
static short texas_topo[]={
    0, 1, 2, 3, 4, 5, 6, 7, 8,     /* map of Texas */
    9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22,
    23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35, 36,
    37, 38, 39, 40, 41, 42, 43,
    44, 45, 46, 47, 48, 0
};
#define NUM_TEXAS sizeof(texas_topo)/(sizeof(short))

/* Connectivity information for TI logo */
/* map of Texas */
static short ti_topo[]={
    0, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22,
    23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35, 36,
    37, 38, 39, 40, 41, 42, 43,
    44, 45, 46, 47, 48, 49, 50,
    51, 52, 53, 54, 55, 56, 57,
    58, 59, 60, 61, 62, 63, 64,
    65, 66, 67, 68, 69, 70, 0
};
#define NUM_TI sizeof(ti_topo)/(sizeof(short))

/* letter "i" */
static short i_topo[]={
    71, 72, 73, 74, 71
};
#define NUM_I sizeof(i_topo)/(sizeof(short))

/* dot the "i" */
static short dot_topo[]={
    75, 76, 77, 78, 79, 80,
    81, 82, 83, 84, 85, 86,
    87, 88, 89, 90, 75
};
#define NUM_DOT sizeof(dot_topo)/(sizeof(short))

static int first_time = TRUE;

/*-------------------------------- draw functions --------------------------*/

/*--------------------------------------------------------------------------*/
/* copy_vertex                                                              */
/* Function to copy a list of n vertices from an input array to an output   */
/*--------------------------------------------------------------------------*/
void copy_vertex(n, vertexin, vertexout)
short    n;
FIX      *vertexin;
FIX      *vertexout;
{
   register int i;

   for (i = 0; i < n; i++)
   {
      *vertexout++ = *vertexin++;
      *vertexout++ = *vertexin++;
      *vertexout++ = *vertexin++;
   }
}

/*--------------------------------------------------------------------------*/
/* draw_polyline_v                                                          */
/* draws a polyline of n points, the point identifies are in ptlist which   */
/* indicate the subset of vertices in vtlist to draw.                       */
/*--------------------------------------------------------------------------*/
void draw_polyline_v(n, vtlist, ptlist)
short       n;
SCREEN      *vtlist;
SCREEN      *ptlist;
{
   register int i, vertex;
   SCREEN       *vlist, *vlist_save;

   vlist_save = vlist = (SCREEN *) malloc (2 * n * sizeof(SCREEN));
   for (i = 0; i < n; i++)
   {
      vertex = vtlist[i];
      vertex <<= 1;
      *vlist++ = ptlist[vertex++];
      *vlist++ = ptlist[vertex];
   }
   draw_polyline(n, vlist_save);
   free(vlist_save);
}


/*--------------------------------------------------------------------------*/
/* fill_polygon_v                                                           */
/* fills a polygon of n points, the point identifies are in ptlist which    */
/* indicate the subset of vertices in vtlist to draw.                       */
/*--------------------------------------------------------------------------*/
void fill_polygon_v(n, vtlist, ptlist)
short       n;
SCREEN      *vtlist;
SCREEN      *ptlist;
{
   register int i, vertex;
   SCREEN       *vlist, *vlist_save;

   vlist_save = vlist = (SCREEN *) malloc (2 * n * sizeof(SCREEN));
   for (i = 0; i < n; i++)
   {
      vertex = vtlist[i];
      vertex <<= 1;
      *vlist++ = ptlist[vertex++];
      *vlist++ = ptlist[vertex];
   }
   c_fill_polygon(n, vlist_save);
   free(vlist_save);
}


/*-------------------------- fixed point matrix functions ------------------*/

/*--------------------------------------------------------------------------*/
/* fix_mult                                                                 */
/* Return the result of the fixed point multiply of two numbers a and b.    */
/*--------------------------------------------------------------------------*/
FIX fix_mult(a, b)
FIX a;
FIX b;
{
   long  al, bl, am, bm;
   FIX   result;

   /* MS 16 bits */
   am = a>>16;
   bm = b>>16;
   /* LS 16 bits */
   al = a & 0x0000FFFFL;
   bl = b & 0x0000FFFFL;
   /* 4 multiplies */
   result = ((al * bl)>>16) + (al * bm) + (am * bl) + ((am * bm)<<16);
   return(result);
}


/*--------------------------------------------------------------------------*/
/* fl2fix                                                                   */
/* Converts a floating point number to fixed point.                         */
/*--------------------------------------------------------------------------*/
FIX fl2fix(fl)
float fl;
{
   return ( (FIX) (fl * (float) 0x00010000L));
}


/*--------------------------------------------------------------------------*/
/* init_matrix                                                              */
/* initializes the matrix passed to the 4x4 identity matrix                 */
/*--------------------------------------------------------------------------*/
void init_matrix(matrix)
FIX      *matrix;
{
   register int i, j;

   for (i = 0; i < 3; i++)
   {
      *matrix++ = 0x00010000L;
      for (j = 0; j < 4; j++)
         *matrix++ = 0L;
   }
   *matrix = 0x00010000L;
}


/*--------------------------------------------------------------------------*/
/* scale_matrix                                                             */
/* scales the 4x4 matrix by the 3 fixed-point values for x, y and z         */
/*--------------------------------------------------------------------------*/
scale(matrix, factor)
FIX     *matrix;
FIX     *factor;
{
   FIX      *factor_copy;
   register int i, j;

   factor_copy = factor;

   for (i = 0; i < 4; i++)
   {
      factor = factor_copy;
      for (j = 0; j < 3; j++)
         if (*factor == 0L)
         {
            *matrix++ = 0L;
            factor++;
         }
         else
         {
            *matrix = fix_mult(*factor, *matrix);
            matrix++;
            factor++;
         }
      /* skip last column of matrix */
      matrix++;
   }
}


/*--------------------------------------------------------------------------*/
/* transform                                                                */
/* Apply the 4x4 matrix to the list of n vertices passed                    */
/*--------------------------------------------------------------------------*/
transform(matrix, n, vertices)
FIX     *matrix;
short n;
FIX     *vertices;
{
   register int   i, j;
   FIX            tempx, tempy, tempz;
   FIX            *matrix_copy;

   matrix_copy = matrix;

   for (i = 0; i < n; i++)
   {
      matrix = matrix_copy;
      tempx = tempy = tempz = 0L;
      for (j = 0; j < 3; j++)
      {
         if (*vertices)
         {
            if (*matrix != 0L)
               tempx += fix_mult(*vertices, *matrix);
            matrix++;
            if (*matrix != 0L)
               tempy += fix_mult(*vertices, *matrix);
            matrix++;
            if (*matrix != 0L)
               tempz += fix_mult(*vertices, *matrix);
            matrix++;
         }
         else
            matrix += 3;
         *vertices++;
         /* skip last column of matrix*/
         *matrix++;
      }
      /* store back result adding in translation part of the matrix */
      vertices -= 3;
      *vertices++ = tempx + *matrix++;
      *vertices++ = tempy + *matrix++;
      *vertices++ = tempz + *matrix;
   }
}


/*--------------------------------------------------------------------------*/
/* vertex_to_point                                                          */
/* Convert the list of n vertices (x,y,z) values to a list of points (x,y)  */
/*--------------------------------------------------------------------------*/
vertex_to_point(n, vlist, plist)
short       n;
FIX         *vlist;
SCREEN      *plist;
{
   register int i;

   for (i = 0; i < n; i++)
   {
      *plist++ = (*vlist++ + 0x00008000L)>>16;
      *plist++ = (*vlist++ + 0x00008000L)>>16;
      vlist++;
   }
}


/*--------------------------------------------------------------------------*/
/* outline                                                                  */
/* Draw a series of outlines of TI logo scaled from fstart to fend by an    */
/* increment of finc. Starting color is color, maximum colors is max_colors.*/
/*--------------------------------------------------------------------------*/
outline(color, max_colors, fstart, fend, finc, nverts, verts)
short    color;
short    max_colors;
FIX      fstart;
FIX      fend;
FIX      finc;
short    nverts;
FIX      *verts;
{
   FIX         factor;
   FIX         scale_vals[3], matrix[16];
   FIX         *verts2;
   SCREEN      *vertsp, term;

   verts2 = (FIX        *) malloc (nverts * 3 * sizeof(FIX));
   vertsp = (SCREEN     *) malloc (nverts * 2 * sizeof(SCREEN));
   if ((!verts2) || (!vertsp))
      return(0);

   /* Z scaling of 1 */
   scale_vals[2] = 0x00010000L;
   /* set up "term" to decide between fstart<fend and fstart>fend */
   term = fstart <= fend;
   for (factor = fstart;
        ((factor <= fend) && term) || ((factor > fend) && !term);
        factor += finc)
   {
      init_matrix(matrix);
      /* copy only the vertices of the texas map */
      copy_vertex(nverts, verts, verts2);
      scale_vals[0] = factor;
      scale_vals[1] = factor;
      scale(matrix, scale_vals);
      transform(matrix, nverts, verts2);
      vertex_to_point(nverts, verts2, vertsp);
      if( iscolor )
      {
         set_fcolor(phys_color[color]);
         if (++color == max_colors)
            color = 1;
      }
      else
        set_fcolor( color++ % 2 );
      draw_polyline_v(nverts, texas_topo, vertsp);
   }
   free(verts2);
   free(vertsp);
   if( !iscolor )
      set_fcolor(0xFFFFFFFF);
   return(1);
}


/*--------------------------------------------------------------------------*/
/* frame2 - main subroutine                                                 */
/*--------------------------------------------------------------------------*/
void frame2()
{
   short       i;
   long        pitch;
   SCREEN      *tibugp;

   tibugp = (SCREEN     *) malloc (NUM_TIBUG * 2 * sizeof(SCREEN));
   if (!tibugp)
   {
      printf("Not enough DOS memory to store points\n");
      return;
   }

   /* move drawing origin to centre of the screen */
   set_draw_origin(config.mode.disp_hres>>1, config.mode.disp_vres>>1);

   /* convert TIBUG to fixed point */
   if (first_time)
   {
      for (i = 0; i < (sizeof(tibug)/sizeof(FIX)); i++)
         tibug[i] <<= 16;
      first_time = FALSE;
   }

   /* Draw converging outlines. */
   if (!outline(1, max_colors, fl2fix(10.0), fl2fix(1.05), fl2fix(-0.1), NUM_TEXAS, tibug))
   {
      printf("Not enough DOS memory to store points\n");
      return;
   }

   /* Draw filled TI logo in center of picture. */
   set_phys_fcolor(TI_SYMBOL);
   vertex_to_point(NUM_TIBUG, tibug, tibugp);
   fill_polygon_v(NUM_TI, ti_topo, tibugp);
   fill_polygon_v(NUM_I, i_topo, tibugp);
   fill_polygon_v(NUM_DOT, dot_topo, tibugp);

   /* draw diverging outlines in different colors to provide some
      pseudo animation */
   for (i = 0; i < 5; i++)
      if (!outline(i+2, max_colors, fl2fix(1.55), fl2fix(10.0), fl2fix(0.1), NUM_TEXAS, tibug))
      {
         printf("Not enough DOS memory to store points\n");
         return;
      }

   /* tidy up */
   set_draw_origin(0,0);
   free(tibugp);           
}



