/******************************************************************
 * NOTICE: The information contained in this file is proprietary  *
 * to Cadence Design Systems and is being made available          *
 * to Cadence customers under strict non-disclosure agreements.   *
 * Use or disclosure of this information is permissible only      *
 * under the terms of the existing non-disclosure agreement.      *
 ******************************************************************/

/* Waveform for the rcount module described at several levels */
module wrc;
    reg clock, inhibit, clear;
    wire outbe, outf1, outf2, outf3, outgs;
    tri out;
    assign
        out = outbe,
        out = outf1,
        out = outf2,
        out = outf3,
        out = outgs;

    rcount_beha cbe(outbe, clock, inhibit, clear);
    rcount_fun1 cf1(outf1, clock, inhibit, clear);
    rcount_fun2 cf2(outf2, clock, inhibit, clear);
    rcount_fun3 cf3(outf3, clock, inhibit, clear);
    rcount_gate cgs(outgs, clock, inhibit, clear);

    parameter d = 1000;
    initial
        begin
            #d
               //cgs.identify;
               cgs.disp;
               inhibit = 0;
            #d
               clock = 0;
               clear = 0;
            #d cgs.disp;
            #d clear = 1;
            #d repeat(171)
                begin
                    #d cgs.disp;
                       clock = 1;
                    #d clock = 0;
                end
            $finish;
        end

    initial
        fork
            begin
                #26010 inhibit = 1;
                #980 inhibit = 0;
            end
            begin
                #28010 inhibit = 1;
                #989 inhibit = 0;
            end
        join

    always @out $display($time,,"out=", out);

    always @outbe $display($time,, "outbe=", outbe);
    always @outf1 $display($time,, "outf1=", outf1);
    always @outf2 $display($time,, "outf2=", outf2);
    always @outf3 $display($time,, "outf3=", outf3);
    always @outgs $display($time,, "outgs=", outgs);

endmodule // wrc




/****************************************************************************/
/*                         BEHAVIORAL LEVEL                                 */

module rcount_beha(out, clock, inhibit, clear);
    output out;
    input clock, inhibit, clear;
    reg out;

    always begin :seq
        repeat(169) @(negedge(clock | inhibit));
        out = 0;
        @(negedge(clock | inhibit));
        out = 1;
    end

    always @clear begin
        disable seq;
        if(clear)
            deassign out;
        else
            assign out = 1;
    end

    checktwo checkci(clock, inhibit);

    task identify;
        $display("Simulating at the behavioral level");
    endtask

    task disp;
    //    $display("%d\tout=%b clock=%b inhibit=%b clear=%b", 
    //        $time, out, clock, inhibit, clear);
    ;
    endtask
endmodule // rcount_beha
    

module checktwo(i1, i2);
    parameter margin = 20;
    input i1, i2;
    time lasttime;
    initial lasttime = 0;
    always @i1 begin
        if($time <= lasttime + margin)
            $display("**Warning! at time %d checktwo @i1 failed lasttime=%d",
                $time, lasttime);
        lasttime = $time;
    end
    always @i2 begin
        if($time <= lasttime + margin)
            $display("**Warning! at time %d checktwo @i2 failed lasttime=%d",
                $time, lasttime);
        lasttime = $time;
    end
endmodule




/****************************************************************************/
/*                        FUNCTIONAL LEVEL 1                                */

module rcount_fun1(out, clock, inhibit, clear);
    output out;
    input clock, inhibit, clear;
    reg[7:0] q;
    wire
        out = q != 169,
        fedbk = q != 170;

    always @(negedge(clock | inhibit)) q = q + 1;

    always @(clear or fedbk) begin
        if(clear & fedbk)
            deassign q;
        else
            assign q = 0;
    end

    task identify;
        $display("Simulating at the functional 1 level");
    endtask

    task disp;
        $display("%d\tout=%b clock=%b inhibit=%b clear=%b q=%b", 
            $time, out, clock, inhibit, clear, q);
    endtask
endmodule // rcount_fun1





/****************************************************************************/
/*                        FUNCTIONAL LEVEL 2                                */

module rcount_fun2(out, clock, inhibit, clear);
    output out;
    input clock, inhibit, clear;
    reg[7:0] q;
    wire #15
        out = q != 169,
        fedbk = q != 170;
    wire #5
        c4 = clock | inhibit,
        clr = c4 | (clear & fedbk);

    always @(negedge c4) q = q + 1;

    always @(clr) begin
        if(clr)
            deassign q;
        else
            assign q = 0;
    end

    task identify;
        $display("Simulating at the functional 2 level");
    endtask

    task disp;
        $display("%d\tout=%b clock=%b inhibit=%b clear=%b q=%b", 
            $time, out, clock, inhibit, clear, q);
    endtask
endmodule // rcount_fun2





/****************************************************************************/
/*                        FUNCTIONAL LEVEL 3                                */

module rcount_fun3(out, clock, inhibit, clear);
    output out;
    input clock, inhibit, clear;
    wire  qa,  qb,  qc,  qd,  qe,  qf,  qg,  qh;
    wire nqa, nqb, nqc, nqd, nqe, nqf, nqg, nqh;
    wire
        a1 = qa & nqb & nqc & qd,
        a2 = nqa & qb & nqc & qd,
        a3 = nqe & qf & nqg & qh;

    wire #15
        out = ~(a1 & a3),
        fedbk = ~(a2 & a3);
    wire #5
        c4 = clock | inhibit,
        clr = c4 | (clear & fedbk);

    fnegdms
        toga(qa, nqa, c4, nqa, clr),
        togb(qb, nqb, qa, nqb, clr),
        togc(qc, nqc, qb, nqc, clr),
        togd(qd, nqd, qc, nqd, clr),

        toge(qe, nqe, qd, nqe, clr),
        togf(qf, nqf, qe, nqf, clr),
        togg(qg, nqg, qf, nqg, clr),
        togh(qh, nqh, qg, nqh, clr);

    task identify;
        $display("Simulating at the functional 3 level");
    endtask
    task disp;
        $display("%d\tout=%b clock=%b clear=%b a1-3=%b%b%b qa-h=%b", 
            $time, out, clock, clear,
            a1, a2, a3, {qh,qg,qf,qe,qd,qc,qb,qa});
    endtask
endmodule // rcount_fun3


/* functional negative d-type master-slave flip-flop */
module fnegdms(q, q_, clk, d, clear);
    output q, q_;
    input clk, d, clear;
    reg s;
    wire #5 q = s, q_ = !s;
    always @(negedge clk) s = d;
    always @clear if(clear) deassign s; else assign s = 0;
endmodule // fnegdms




/****************************************************************************/
/*                      GATE AND SWITCH LEVEL                               */

module rcount_gate(out, clock, inhibit, clear);
    output out;
    input clock, inhibit, clear;
    wire i1, c1, c2, c3, c4;
    wire clr, fedclr, a1, a2, a3, fedbk;
    wire  qa,  qb,  qc,  qd,  qe,  qf,  qg,  qh;
    wire nqa, nqb, nqc, nqd, nqe, nqf, nqg, nqh;

    not gi1(i1, inhibit), gc1(c1, clock);
    nand gc2(c2, i1, c1);
    not gc3(c3, c2), gc4(c4, c3);
    or gclr(clr, c4, fedclr);
    and gfedclr(fedclr, clear, fedbk);
    and ga1(a1, qa, nqb, nqc, qd),
        ga2(a2, nqa, qb, nqc, qd),
        ga3(a3, nqe, qf, nqg, qh);
    nand #50 gout(out, a1, a3), gfedbk(fedbk, a2, a3);

    dnegdms
        toga(qa, nqa, c4, nqa, 1, clr),
        togb(qb, nqb, qa, nqb, 1, clr),
        togc(qc, nqc, qb, nqc, 1, clr),
        togd(qd, nqd, qc, nqd, 1, clr),

        toge(qe, nqe, qd, nqe, 1, clr),
        togf(qf, nqf, qe, nqf, 1, clr),
        togg(qg, nqg, qf, nqg, 1, clr),
        togh(qh, nqh, qg, nqh, 1, clr);

    task identify;
        $display("Simulating at the gate and switch level");
    endtask

    task disp;
        $display("%d\tout=%b c4=%b clr=%b a1=%b a2=%b a3=%b qh-a=%b", 
            $time, out, c4, clr,
            a1, a2, a3, {qh,qg,qf,qe,qd,qc,qb,qa});
    endtask
endmodule // rcount_gate



/* dynamic negative d-type master-slave flip-flop */
module dnegdms(q, nq, clock, d, preset, clear);
    output q, nq;
    input clock, d, preset, clear;
    wire q, nq, nd, nclock;

    not gd(nd, d), gc(nclock, clock);

    trireg master, wm1, wm2;
    nmos emast(wm1, nd, clock), lmast(wm1, wm2, nclock);
    nand gm1(master, wm1, preset), gm2(wm2, master, clear);

    trireg slave, ws1, ws2;
    nmos eslav(ws1, master, nclock), lslav(ws1, ws2, clock);
    nand gs1(slave, ws1, clear), gs2(ws2, slave, preset);

    not #1 gq(q, slave), gnq(nq, ws1);
endmodule // dnegdms
