/******************************************************************
 * NOTICE: The information contained in this file is proprietary  *
 * to Cadence Design Systems and is being made available          *
 * to Cadence customers under strict non-disclosure agreements.   *
 * Use or disclosure of this information is permissible only      *
 * under the terms of the existing non-disclosure agreement.      *
 ******************************************************************/

//Dynamic MOS serial shift register circuit description module
module shreg (out, in, phase1, phase2);
    /*IO port declarations, where 'out' is the inverse
      shift of 'in' controlled by the dual-phased clock*/
    output out; //shift register output
    input in; //shift register input
    input phase1, phase2; //clocks

    trireg
        wa1, wa2, wa3; //charge storage nodes

    tri #1
        wb1, wb2, out; //wires pulled up to VDD

    supply0
        gnd; //ground supply

    pullup //depletion mode pullup devices
        (wb1), (wb2), (out);

    nmos #(3,4) //pass devices and their interconnections
        a1(wa1,in,phase1), b1(wb1,gnd,wa1),
        a2(wa2,wb1,phase2), b2(wb2,gnd,wa2),
        a3(wa3,wb2,phase1), gout(out,gnd,wa3);
endmodule

// Waveform description module to drive the shift register
module wave_shreg;
    wire shiftout; //net to receive circuit output value
    reg shiftin; //register to drive value into circuit
    reg phase1, phase2; //clock driving values

    //make the circuit a sub-module and define the
    //waveform-circuit interconnections
    shreg
        cct (shiftout, shiftin, phase1, phase2);

    //define the waveform time step
    parameter d = 100;

    //main part of waveform description
    initial
        begin :main
            //initialize waveform input stimulus
            shiftin = 0;
            phase1 = 0;
            phase2 = 0;

            // setup the monitoring information
            setmon;

            //shift data in
            repeat(2) clockcct;
            if (shiftout == shiftin) error; //and check the output

            //set new value, shift and check
            $display("\nShift in 1...");
            shiftin = 1;
            repeat(2) clockcct;
            if (shiftout == shiftin) error;
        end

    //display header and setup monitoring
    task setmon;
        begin
            $display("      time  clocks  in out  wa1-3  wb1-2");
            $monitor($time,,,,
                phase1, phase2,,,,,,
                shiftin,,,, shiftout,,,,,
                cct.wa1, cct.wa2, cct.wa3,,,,,
                cct.wb1, cct.wb2);
        end
    endtask

    //produce dual-phased clock pulse
    task clockcct;
        begin
            #d phase1 = 1; //time step defined by parameter d
            #d phase1 = 0;
            #d phase2 = 1;
            #d phase2 = 0;
        end
    endtask

    //error routine
    task error;
        begin
            $display("Error at time %d", $time,
                "shiftin=%b shiftout=%b", shiftin, shiftout);
            $stop;
        end
    endtask
endmodule
