/* ba_int_nowired_path.c 
 *
 * This file contains the routines that make up a simple delay
 * back annotator.  This back annotator reads a file that contains
 * cell interconnect net names and their associated incremental rise and
 * fall delays.  For each net/delay set, the back annotator scans
 * all cells with output or inout ports that are connected to the
 * interconnect net and adds the delays to the paths within each cell
 * that are connected to the interconnect net.  This back annotator is
 * meant to be used with designs containing cells described with path
 * delay timing.  The following shows example entries in a delay file:
 *
 *      top.m1.dm1 10 12
 *      top.m1.dm2 10 12
 *      top.m1.q[0] 25 27
 *      top.m1.qb[0] 30 35
 *      top.m1.q[1] 25 27
 *      top.m1.qb[1] 30 35
 *
 * To associate these routines with a Verilog-XL system task,
 * place the following entry into the 'veriusertfs' data
 * structure in the 'veriuser.c' file:
 *
 *      s_tfcell veriusertfs[] = {
 *                                  {usertask,0,
 *                                   ba_inp_nowired_check,0,
 *                                   ba_inp_nowired_call,0,
 *                                   "$ba_delays",0}, 
 * 
 *                                  {0} 
 *                               };
 *
 * Once this file has been compiled and linked with the Verilog-XL
 * object files provided with the release, the delay back annotator
 * may be called from the Verilog-XL source description to perform
 * delay back annotation.  The system task must be invoked with one
 * argument - the name of the delay back annotation file.  The
 * following shows an example partial source description that invokes
 * the delay back annotator:
 *
 *      module top;
 *         .
 *         .
 *         .
 *         initial
 *            $ba_delays("test_inp.dat");
 *         .
 *         .
 *         .
 *      endmodule
 *
 */

#include <stdio.h>
#include "acc_user.h"
#include "ba.h"
#include "veriuser.h"

/* checktf routine for $ba_delays() */
ba_inp_nowired_check()
{ /* ba_inp_nowired_check() */

   /* check for one parameter of type string - delay file name */
   if (tf_nump() != 1)
   {
      strcpy(message,
                  "incorrect number of parameters passed to $ba_delays()");
      tf_error(message);
      return;
   }

   if (tf_typep(1) != tf_string)
   {
      strcpy(message,"illegal parameter passed to $ba_delays()\n");
      strcat(message,"parameter must be string with delay file name");
      tf_error(message);
      return;
   }
} /* ba_inp_nowired_check() */


/* calltf routine for $ba_delays() - contains
   the actual back-annotation functionality */
ba_inp_nowired_call()
{ /* ba_inp_nowired_call() */
FILE *ba_fp;                       /* pointer to back-annotation file */
char ext_net_name[LARGE_STRING];   /* external hierarchical net name */
handle ext_net_handle;             /* handle to the external net */
handle int_net_handle;             /* handle to the internal net */
handle mod_handle;                 /* handle to the parent module instance */
handle curr_driver;                /* handle to the current driver */
handle curr_path;                  /* handle to current path delay */
double rise, fall;                 /* incremental rise and fall delays */

int i;                             /* loop variable */
char temp[LARGE_STRING];           /* temporary string */

   /* get back-annotation file name and open */
   if ((ba_fp = fopen(tf_getcstringp(1),"r")) == null)
   {
      sprintf(message,"unable to open back-annotation file %s",
                                                  tf_getcstringp(1));
      tf_error(message);
      return;
   }   

   /* initialize access routines */
   acc_initialize();
#if 0
   acc_configure(accDevelopmentVersion,"1.5a");
#endif

   /* configure path delay access routines to accept a rise and
      a fall delay */
   acc_configure(accPathDelayCount,"2");

   /* while there are more lines in the file, read them and
      back-annotate the delays */
   while (fscanf(ba_fp,"%s %lf %lf",ext_net_name,&rise,&fall) != EOF)
   {
      /* get handle to the external net */
      ext_net_handle = acc_handle_object(ext_net_name);
      if (ext_net_handle == null)
      {
         sprintf(message,"illegal net name %s found in file",ext_net_name);
         tf_error(message);
         return;
      }

      /* scan all modules with output ports that drive this net and
         annotate the delays */
      curr_driver = null;
      while ((curr_driver = acc_next_driver(ext_net_handle,curr_driver))
                                                                    != null)
      {
         /* get the connecting cell output net and the parent module */
         int_net_handle = acc_handle_conn(curr_driver);
         mod_handle = acc_handle_parent(int_net_handle);

         /* find correct path delays and annotate delays */
         curr_path = null;
         while (curr_path = acc_next_modpath(mod_handle,curr_path))
         {
            /* check for correct path and annotate delays */
            if (acc_handle_pathout(curr_path) == int_net_handle)
            {
               acc_append_delays(curr_path,rise,fall);

               /* set pulse control back to 100% x-generation region
                  for new delays */
               acc_set_pulserx(curr_path,0,100);
            }
         }
      }
   }

   /* clean up access routines */
   acc_close();
} /* ba_inp_nowired_call() */
