/* ba_path_path.c 
 *
 * This file contains the routines that make up a simple delay
 * back annotator.  This back annotator reads a file that contains
 * cell instance names, nets that identify path delays within the
 * cells and the incremental rise and fall delays associated with
 * each path, and nets and constants that identify timing checks within
 * the cells and the incremental limit associated with each timing check.
 * For each path/delay or timing check/limit pair, the back annotator
 * adds the given delay or limit to the appropriate path or timing check.
 * This back annotator is meant to be used with designs containing cells
 * described with path delay timing.
 *
 * The following gives the syntax for the delay file:
 *
 *      <cell instance name> <number of path delays>
 *         <path input net> <path output net> <rise delay> <fall delay>
 *                  .
 *                  .
 *                  .
 *         <number of timing checks>
 *         <timing check type> <lower bound net name> <lower bound edge type> 
 *                                                                            ^
 *                                                                            |
 *                                                                           /
 *                                                                          /
 *               /---------------------------------------------------------/
 *              /
 *             /
 *         <upper bound net name> <upper bound edge type> <limit>
 *                  .
 *                  .
 *                  .
 *
 * The timing check types and edge types are given as constants that
 * are defined in the access routine include file 'acc_user.h'.  Please
 * refer to the "PLI Reference Manual" for detailed information on these
 * constants.  The following shows example entries in a delay file:
 *
 *      top.m1.m1 2
 *         sel y 10 12
 *         i1 y 10 12
 *         0
 *
 *      top.m1.m2 2
 *         sel y 10 12
 *         i1 y 10 12
 *         0
 *
 *      top.m1.m3 2
 *         clk q 25 27
 *         clk qb 30 35
 *         2
 *         366 d 0 clk 9 20
 *         367 clk 13 d 0 17
 *
 *      top.m1.m4 2
 *         clk q 25 27
 *         clk qb 30 35
 *         2
 *         366 d 0 clk 9 20
 *         367 clk 13 d 0 17
 *
 * To associate these routines with a Verilog-XL system task,
 * place the following entry into the 'veriusertfs' data
 * structure in the 'veriuser.c' file:
 *
 *      s_tfcell veriusertfs[] = {
 *                                  {usertask,0,
 *                                   ba_ppt_check,0,
 *                                   ba_ppt_call,0,
 *                                   "$ba_delays",0},
 *
 *                                  {0}
 *                               };
 *
 * Once this file has been compiled and linked with the Verilog-XL
 * object files provided with the release, the delay back annotator
 * may be called from the Verilog-XL source description to perform
 * delay back annotation.  The system task must be invoked with one
 * argument - the name of the delay back annotation file.  The
 * following shows an example partial source description that invokes
 * the delay back annotator:
 *
 *      module top;
 *         .
 *         .
 *         .
 *         initial
 *            $ba_delays("test_ppt.dat");
 *         .
 *         .
 *         .
 *      endmodule
 *
 */

#include <stdio.h>
#include "acc_user.h"
#include "ba.h"
#include "veriuser.h"
/* checktf routine for $ba_delays() */
ba_ppt_check()
{ /* ba_ppt_check() */

   /* check for one parameter of type string - delay file name */
   if (tf_nump() != 1)
   {
      strcpy(message,
                  "incorrect number of parameters passed to $ba_delays()");
      tf_error(message);
      return;
   }

   if (tf_typep(1) != tf_string)
   {
      strcpy(message,"illegal parameter passed to $ba_delays()\n");
      strcat(message,"parameter must be string with delay file name");
      tf_error(message);
      return;
   }
} /* ba_ppt_check() */


/* calltf routine for $ba_delays() - contains
   the actual back-annotation functionality */
ba_ppt_call()
{ /* ba_ppt_call() */
FILE *ba_fp;                       /* pointer to back-annotation file */
char hier_name[LARGE_STRING];      /* hierarchical module instance name */
int num_paths;                     /* number of path delays for a module
                                      instance */
handle mod_handle;                 /* handle to module instance to back-annotate
                                      to */
handle curr_path;                  /* handle to current path delay */
char pathin_name[SMALL_STRING];    /* name of current path input */
char pathout_name[SMALL_STRING];   /* name of current path output */
double rise, fall;                 /* incremental delays */

int num_tchecks;                   /* number of timing checks for a module 
                                      instance */ 
char sig1_str[SMALL_STRING];       /* timing check signal one string */
char sig2_str[SMALL_STRING];       /* timing check signal two string */  
handle tcheck_handle;              /* handle to timing check */
int tcheck_id;                     /* integer identifying timing check */
int edge1_id;                      /* integer identifying signal one edge */
int edge2_id;                      /* integer identifying signal two edge */
double tcheck_incdel;              /* incremental delay to add to timing
                                      check */

int i;                             /* loop variable */
char temp[LARGE_STRING];           /* temporary string */

   /* get back-annotation file name and open */
   if ((ba_fp = fopen(tf_getcstringp(1),"r")) == null)
   {
      sprintf(message,"unable to open back-annotation file %s",
                                                  tf_getcstringp(1));
      tf_error(message);
      return;
   }   

   /* initialize access routines */
   acc_initialize();
#if 0
   acc_configure(accDevelopmentVersion,"1.5a");
#endif

   /* configure path delay access routines to accept a single delay
      for all transitions */
   acc_configure(accPathDelayCount,"2");

   /* while there are more lines in the file, read them and
      back-annotate the delays */
   while (fscanf(ba_fp,"%s %d",hier_name,&num_paths) != EOF)
   {
      /* get handle to module instance */ 
      mod_handle = acc_handle_object(hier_name);
      if (mod_handle == null)
      {
         sprintf(message,"illegal instance name %s found in file",hier_name);
         tf_error(message);
         return;
      }

      /* read path IDs and delays and annotate the delays */
      for (i = 0; i < num_paths; i++)
      {
         fscanf(ba_fp,"%s %s %lf %lf",pathin_name,pathout_name,&rise,&fall);

         /* get a handle to the path */
         curr_path = acc_handle_modpath(mod_handle,pathin_name,pathout_name);
         if (curr_path == null)
         {  
            sprintf(message,"illegal path found in file - input %s, output %s",
                                                      pathin_name,pathout_name);
            tf_error(message);
            return;
         }

         /* add incremental delays to path */
         acc_append_delays(curr_path,rise,fall);

         /* set pulse control back to 100% x-generation region for
            new path delays */
         acc_set_pulserx(curr_path,0.0,1.0);
      }

      /* process timing checks */
      
      /* next entry in file is number of timing checks */ 
      fscanf(ba_fp,"%d",&num_tchecks);

      /* only process if there are timing checks for this module instance */
      if (num_tchecks > 0)
      {
         /* read in and process each timing check */
         for (i = 0; i < num_tchecks; i++)
         {
            /* read in data for one timing check */
            fscanf(ba_fp,"%d %s %d %s %d %lf",&tcheck_id,sig1_str,&edge1_id,
                                             sig2_str,&edge2_id,&tcheck_incdel);

            /* get handle to timing check and add extra delay */
            tcheck_handle = acc_handle_tchk(mod_handle,tcheck_id,sig1_str,
                                                    edge1_id,sig2_str,edge2_id);
            if (tcheck_handle == null) 
            {
               strcpy(message,"timing check found in file does not exist");
               tf_error(message);
               return;
            }

            acc_append_delays(tcheck_handle,tcheck_incdel);
         }
      }
   }

   /* clean up access routines */
   acc_close();
} /* ba_ppt_call() */
