/* File dl_calc_ba_icn.c */

/*******************************************************************************
*
*  This file contains routines which demonstrate how to create an access 
*  routine based delay calculator for a primitive delay or path delay
*  based library.
*
*  In order to understand this code, it will be helpful to have reviewed
*  the access routine documentation, and to have it handy for reference.
*  It should not have been necessary to have studied the access routines in
*  detail.  Indeed, studying this code will likely be very helpful in 
*  understanding the details within the access routines document.
*
*  A detailed application note that describes how to use the access routines
*  to write back annotators and delay calculators is provided with the PLI
*  documentation.  It may be necessary to review this applications note
*  before delving into this code.
*
*  Note that the term "cell" and "cell instance" are used here, and seen
*  in routines such as "acc_next_cell".  A "cell" is a module which has
*  been defined as a cell by having its definition follow the
*  compiler directive `celldefine (`endcelldefine is also available).
*  Also, all modules retrieved from libraries will automatically
*  be defined as cells, unless the argument '+nolibcell' has been included on
*  the command line.
*
*  This file contains routines for two distinct delay calculators : one
*  that may be applied to primitive output (distributed) delay cells, and
*  one that may be applied to path delay cells.  It is convenient to place the
*  two calculators in the same file since they share many auxiliary routines
*  (for example, for calculating load due to fanout on a given port).  Both
*  delay calculators retrieve interconnect wire capacitance from an optional
*  capacitance back annotation file.  The delay calculators assume that the
*  file contains cell interconnect net names and their associated capacitance
*  values.  The delay calculators use the routines in the file 'dl_backan.c' to
*  manipulate the capacitance back annotation file.  If no capacitance
*  back annotation file is provided, the calculators estimate the capacitance
*  due to interconnect wire.
*
*  The delay calculators make the following assumptions about the libraries
*  they serve:
*          > cell ports are not explicitly named (their names are the
*            same as the net to which they connect within the cell)
*            (Do not confuse this with connection by name, which is fine.)
*          > each cell contains specparams which indicate the rise and
*            fall drive strength of each output, and the capacitive
*            loading factor of each input.  These specparams will be
*            detected by access routine acc_fetch_attribute, and
*            must follow the naming conventions, which are described
*            as follows :
*              To represent the drive of a port output "Q" :
*                    specparam RiseStrength$Q = 1;
*                    specparam FallStrength$Q = 2;
*              To represent the drive of a path from "A" to "Q" :
*                    specparam RiseStrength$A$Q = 3;
*                    specparam FallStrength$A$Q = 4;
*
*  There is nothing hard-coded into Verilog-XL or the access routines
*  that requires these naming conventions.  These conventions are decided
*  on by the developer of the library and delay calculator.
*
*  A call to the function :
*
*      acc_fetch_attribute(net,"RiseStrength$");
*
*  for the net A in the example above will return 1.0.
*
*  The distributed delay based calculator scans the output port of each cell
*  instance, determines the drive strengths of the port, determines the
*  load on the port due to both driven cell inputs and interconnect wire,
*  calculates an additional delay for the port, and adds this delay to
*  that of all primitives in the cell that drive the port.
*
*  The path delay based calculator scans each path of each cell instance,
*  determines the drive strengths of the path, determines the load on the 
*  path due to both driven cell inputs and interconnect, and calculates
*  and places an additional delay on the path.
*
*  To associate these routines with Verilog-XL system tasks,
*  place the following entries into the 'veriusertfs' data
*  structure in the 'veriuser.c' file:

     {usertask,0, 0,0, dl_prim_invoke,0, "$dcalc_prim",1},
     {usertask,0, 0,0, dl_path_invoke,0, "$dcalc_path",1},

*  Also, the functions dl_prim_invoke() and dl_path_invoke() need to
*  be delcared in veriuser.c.   Add the following line above the
*  definition of veriusertfs in the veriuser.c file:
 
   int dl_prim_invoke(), dl_path_invoke();

*  Invoking Delay Calculation :
*    Delay calculation is invoked by the system tasks '$dcalc_prim'
*    and '$dcalc_path' which are defined in the 'veriusertfs' data
*    structure.  Two arguments are passed to the system tasks :
*        System task argument 1 :
*            The scope of the delay calculation.  This argument must be
*            a valid path name of a module or null.  If the value is null,
*            the scope will be set to that of the module which contains
*            $dcalc_prim or $dcalc_path.
*        System task argument 2 :
*            This argument indicates whether interconnect delays will be
*            estimated or back annotated.  If this argument is null, or
*            is "estimate", interconnect capacitance will be derived from
*            the array "wirecap_est_table". Otherwise, the argument will be
*            assumed to be a back annotation filename, where the file
*            contains one line entries composed of the hierarchical name
*            of a cell interconnect net, followed by a float value
*            which will indicate the capacitance.
*
*    An example invokation within a Verilog description would be :
*
*        $dcalc_prim(top.chip1,"backan.dat");
*                     // scope is mod instance top.chip1,
*                     // interconnect capacitance read from file
*
*        $dcalc_prim; // scope is invoking module, 
*                     // interconnect capacitance is estimated 
*
*******************************************************************************/

#include "acc_user.h"
#include "veriuser.h"
#include "dl.h"

static void 	dl_calc_prim_icn_delays PROTO_PARAMS((handle start_mod, int estimate_flag));
static void 	dl_calc_path_icn_delays PROTO_PARAMS((handle start_mod, int estimate_flag));
static double 	dl_load_factor PROTO_PARAMS((handle net));
static handle 	dl_handle_icn PROTO_PARAMS((handle mod, handle cell_output_net));
static double 	dl_estimate_capacitance PROTO_PARAMS((int fanout_count));
static handle 	dl_process_scope_arg PROTO_PARAMS((void));
static int 	dl_process_estimate_arg PROTO_PARAMS((void));

/*** System task invocation routine ***/
global void dl_prim_invoke();
global void dl_path_invoke();
global void dl_invoke();

/*** Main delay calculation routines ***/
local void dl_calc_prim_icn_delays();
local void dl_calc_path_icn_delays();

/*** Auxiliary routines ***/
local double dl_load_factor();
local double dl_estimate_capacitance();
local handle dl_handle_icn();

/*** Argument processing routines ***/
local handle dl_process_scope_arg();
local bool dl_process_estimate_arg();

/*** Global variables ***/
bool dl_verbose_flag = false;

/*** Wire capacitance estimation table ***/
int wirecap_est_table_size = 10;
double wirecap_est_table[] =
        {1.2, 2.2, 3.4, 5.0, 6.7, 8.8, 10.2, 12.7, 15.3, 18.0};

#define dlPrim 1
#define dlPath 2

double dl_backan_fetch();

/******************************************************************************/
/* System task invocation routine */

/*******************************************************************************
*   DL_PRIM_INVOKE
*   Begin delay calculation for primitive delay based models
*******************************************************************************/
exfunc void dl_prim_invoke()
{
    dl_invoke(dlPrim);
}

/*******************************************************************************
*   DL_PATH_INVOKE
*   Begin delay calculation for path delay based models
*******************************************************************************/
exfunc void dl_path_invoke()
{
    dl_invoke(dlPath);
}

/*******************************************************************************
*   DL_INVOKE
*   This routine invokes delay calculation processing. It accepts an
*   argument which indicates whether to begin path delay ot primitive
*   delay based calculations
*   This routine performs the following functions :
*       > Initialize and configure the access routines.
*       > Process command line plusargs
*       > Call routines to process system task arguments
*       > Call routine to calculate interconnect delays
*
*******************************************************************************/
exfunc void dl_invoke(prim_or_path)
    int prim_or_path;
{
    handle  dl_scope_g;
    int     mtmparam;
    handle  scope;
    bool    estimate_flag;

    acc_initialize();
#if 0
    acc_configure(accDevelopmentVersion,"1.5a");
#endif
    acc_configure(accToHiZDelay,"average");
    acc_configure(accPathDelayCount,"2");
    acc_configure(accDefaultAttr0,"true");

    /****************************************************************/
    /*** If "+dlverbose" appears on command line, set global flag ***/
    /****************************************************************/
    if (mc_scan_plusargs("dlverbose"))
    {
        dl_verbose_flag = true;
        if (prim_or_path == dlPrim)
            io_printf("Delay calculation invoked for primitive delay cells : \n");
        else
            io_printf("Delay calculation invoked for path delay cells : \n");
    }

    /****************************************************************/
    /*** Process argument to determine scope of delay calculation ***/
    /****************************************************************/
    scope = dl_process_scope_arg();

    /******************************************************/
    /*** Process argument to determine wire capacitance ***/
    /*** estimation or back annotation                  ***/
    /******************************************************/
    estimate_flag = dl_process_estimate_arg();

    /*************************************/
    /*** Calculate interconnect delays ***/
    /*************************************/
    if (prim_or_path == dlPath)
        dl_calc_path_icn_delays(scope,estimate_flag);
    else
        dl_calc_prim_icn_delays(scope,estimate_flag);

    /*******************************************************/
    /*** If necessary, free back-annotation table memory ***/
    /*******************************************************/
    if (!estimate_flag)
        dl_free_backan_table();

    /******************/
    /*** Aesthetics ***/
    /******************/
    if (dl_verbose_flag)
        io_printf("\n");

    /***********************************/
    /*** Access routine use complete ***/
    /***********************************/
    acc_close();
}


/******************************************************************************/
/* Main delay calculation processing loops */

/*******************************************************************************
*   DL_CALC_PRIM_ICN_DELAYS
*   This routine accepts a handle to a module, scans each offspring cell,
*   and performs delay calculation and annotation for each output port.
*******************************************************************************/
static void dl_calc_prim_icn_delays(start_mod,estimate_flag)
    handle  start_mod;
    bool    estimate_flag;
{
    handle  mod, term, prim, driver;
    handle  portout, portout_net, portout_term, interconnect_net;

    int     fanout_count;
    double  fanout_load, wire_load;
    double  rise_strength, fall_strength, rise, fall;

    /**********************************************************/
    /*** Process each cell module instance within the scope ***/
    /**********************************************************/
    mod = null;
    while (mod = acc_next_cell(start_mod, mod))
    {
        /****************************************************/
        /*** Process each output port in this cell module ***/
        /****************************************************/
        portout = null;
        while (portout = acc_next_portout(mod,portout))
        {
            /***********************************************/
            /*** Process each net connected to this port ***/
            /***********************************************/
            portout_net = null;
            interconnect_net = null;
            while (portout_net = acc_next_loconn(portout,portout_net))
            {
                /**********************************************************/
                /*** If capacitance is back annotated, get interconnect ***/
                /*** net for this output net - break out at first       ***/
                /*** unconnected bit                                    ***/
                /**********************************************************/
                if (!estimate_flag)
                { 
                    interconnect_net = acc_next_hiconn(portout,interconnect_net);
                    if (interconnect_net == null)
                        break;
                }
            
                /*********************/
                /*** Count fanouts ***/
                /*********************/
                fanout_count = acc_count(acc_next_cell_load,portout_net);
                if (estimate_flag && (fanout_count == 0))
                    continue;
    
                /***********************************************/ 
                /*** Determine loading on net due to fanouts ***/
                /***********************************************/ 
                fanout_load = dl_load_factor(portout_net);
    
                /**************************************************************/
                /*** Determine estimated or back-annotated wire capacitance ***/
                /**************************************************************/
                if (estimate_flag)
                    wire_load = dl_estimate_capacitance(fanout_count);
                else
                    wire_load = dl_backan_fetch(interconnect_net);
    
                /********************************************************/
                /*** Determine attributes associated with portout_net ***/
                /********************************************************/
                rise_strength = acc_fetch_attribute(portout_net, "RiseStrength$");
                fall_strength = acc_fetch_attribute(portout_net, "FallStrength$");
    
                /***************************************/ 
                /*** Calculate load dependant delays ***/
                /***************************************/ 
                rise = rise_strength * (fanout_load + wire_load);
                fall = fall_strength * (fanout_load + wire_load);
    
                /*************************************************/ 
                /*** Scan the primitives which drive this port ***/
                /*************************************************/ 
                driver = null;
                while (driver = acc_next_driver(portout_net,driver))
                {
                    prim = acc_handle_parent(driver);
                    if (acc_handle_parent(prim) != mod)
                        continue;

                    /*******************************************/ 
                    /*** Add calculated delays to the delays ***/
                    /*** on the driving primitive            ***/
                    /*******************************************/ 
                    acc_append_delays(prim, rise, fall);
    
                    /************************************************************/ 
                    /*** If requested, print detailed calculation information ***/ 
                    /************************************************************/
                    if (dl_verbose_flag) {
                        double newrise, newfall, newz;
                        acc_fetch_delays(prim,&newrise,&newfall,&newz);
                        io_printf("\nNet %s delays : rise = %6.2f  fall = %6.2f\n",
                            acc_fetch_fullname(portout_net), newrise, newfall);
                        io_printf("  rise_strength  =%6.2f  fall_strength =%6.2f\n",
                             rise_strength, fall_strength);
                        io_printf("  fanout_load    =%6.2f  wire_load     =%6.2f\n",
                            fanout_load, wire_load);
                        io_printf("  added rise     =%6.2f  added fall    =%6.2f\n",
                            rise, fall);
                    }
                }
            }
        }
    }
}

/*******************************************************************************
*   DL_CALC_PATH_ICN_DELAYS
*   This routine accepts a handle to a module, scans each offspring cell,
*   and performs delay calculation and annotation for each path in the cell.
*******************************************************************************/
static void dl_calc_path_icn_delays(start_mod,estimate_flag)
    handle  start_mod;
    bool    estimate_flag;
{
    handle  mod, path;
    handle  pathout_net, interconnect_net;
    int     fanout_count;
    double  fanout_load, wire_load;
    double  rise_strength, fall_strength, rise, fall;

    /**********************************************************/
    /*** Process each cell module instance within the scope ***/
    /**********************************************************/
    mod = null;
    while (mod = acc_next_cell(start_mod, mod))
    {
        /*********************************************/
        /*** Process each path in this cell module ***/
        /*********************************************/
        path = null;
        while (path = acc_next_modpath(mod,path))
        {
            /*****************************/
            /*** Get number of fanouts ***/
            /*****************************/
            pathout_net = acc_handle_pathout(path);
            fanout_count = acc_count(acc_next_cell_load,pathout_net);
            if (estimate_flag && (fanout_count == 0))
                continue;
            if (!estimate_flag)
                interconnect_net = dl_handle_icn(mod,pathout_net);

            /***********************************************/
            /*** Determine loading on net due to fanouts ***/
            /***********************************************/
            fanout_load = dl_load_factor(pathout_net);

            /**************************************************************/
            /*** Determine estimated or back-annotated wire capacitance ***/
            /**************************************************************/
            if (estimate_flag)
                wire_load = dl_estimate_capacitance(fanout_count);
            else
                wire_load = dl_backan_fetch(interconnect_net);

            /*************************************************/
            /*** Determine attributes associated with path ***/
            /*************************************************/
            rise_strength = acc_fetch_attribute(path, "RiseStrength$");
            fall_strength = acc_fetch_attribute(path, "FallStrength$");

            /***************************************/
            /*** Calculate load dependant delays ***/
            /***************************************/
            rise = rise_strength * (fanout_load + wire_load);
            fall = fall_strength * (fanout_load + wire_load);

            /*****************************************/
            /*** Append delays to the driving path ***/
            /*****************************************/
            acc_append_delays(path, rise, fall);

            /**********************************************/
            /*** Set pulse control to 100% x-generation ***/
            /*** for new delay                          ***/
            /**********************************************/
            acc_set_pulserx(path,0.0,1.0);

            /************************************************************/
            /*** If requested, print detailed calculation information ***/
            /************************************************************/
            if (dl_verbose_flag) {
                double newrise, newfall, newz;
                acc_fetch_delays(path,&newrise,&newfall,&newz);
                io_printf("\nPath %s delays : rise = %6.2f  fall = %6.2f\n",
                    acc_fetch_fullname(path), newrise, newfall);
                io_printf("  rise_strength  =%6.2f  fall_strength =%6.2f\n",
                     rise_strength, fall_strength);
                io_printf("  fanout_load    =%6.2f  wire_load     =%6.2f\n",
                    fanout_load, wire_load);
                io_printf("  added rise     =%6.2f  added fall    =%6.2f\n",
                    rise, fall);
            }
        }
    }
}
 


/******************************************************************************/
/* Auxiliary routines */

/*******************************************************************************
*   DL_LOAD_FACTOR
*   This routine accepts a pointer to a net, scans its connected terminals,
*   and returns the total load factor on the net.  This is determined
*   by finding specify parameters called FanoutLoad$netname associated
*   with each load and driver and summing their values.
*******************************************************************************/
static double dl_load_factor(net)
    handle  net;
{
    double  total_lf = 0;
    handle  load, load_net, driver_prim, driver_mod, driver_net, driver;

    /************************************/
    /*** Process each cell input load ***/
    /************************************/
    load = null;
    while (load = acc_next_cell_load(net,load))
    {
        /**************************************/
        /*** Get net connected to this load ***/
        /**************************************/
        load_net = acc_handle_conn(load);

        total_lf += acc_fetch_attribute(load_net, "FanoutLoad$");
    }

    /*************************************************/
    /*** If load has multiple drivers, scan fanin, ***/
    /*** adding each lf to total_lf                ***/
    /*************************************************/
    if (acc_count(acc_next_driver,net) > 1)
    {
        handle net_mod = acc_handle_parent(net);

        /*******************************/
        /*** Process each net driver ***/
        /*******************************/
        driver = null;
        while (driver = acc_next_driver(net, driver))
        {
            /*****************************************************/
            /*** Don't include load factor of driving terminal ***/
            /*****************************************************/
            driver_prim = acc_handle_parent(driver);
            driver_mod  = acc_handle_parent(driver_prim);
            if (driver_mod == net_mod)
                continue;

            /****************************************/
            /*** Get net connected to this driver ***/
            /****************************************/
            driver_net = acc_handle_conn(driver);

            total_lf += acc_fetch_attribute(driver_net, "FanoutLoad$");
        }
    }

    return (total_lf);
}


/*******************************************************************************
*   DL_HANDLE_ICN
*   This routine accepts a handle to a scope and a net,
*   and returns the interconnect net (hiconn) that the 
*   net is connected to.
*******************************************************************************/
static handle dl_handle_icn(mod,cell_output_net)
    handle  mod, cell_output_net;
{
    handle portout;

    /*********************************************************/
    /*** Scan output ports until the one fed by the net is ***/
    /*** found, then return the hiconn net                 ***/                 
    /*********************************************************/
    portout = null;
    while (portout = acc_next_portout(mod,portout))
    {
        if (acc_next_loconn(portout,null) == cell_output_net)
            return(acc_next_hiconn(portout,null));
    }

    /******************************************************/
    /*** Print warning and return null if net not found ***/
    /******************************************************/
    io_printf("Warning from dl_handle_icn : output net \"%s\" not found\n",
                                           acc_fetch_fullname(cell_output_net));
    return(null);
}


/*******************************************************************************
*   DL_ESTIMATE_CAPACITANCE
*   This routine accepts an integer indicating the fanout count on a net,
*   and returns the estimated wire capacitance from table wirecap_est_table
*   (defined eralier in this file).  If the fanout count is larger than the
*   table allows, a warning is printed, and the largest value in the table
*   is returned.
*******************************************************************************/
static double dl_estimate_capacitance(fanout_count)
int fanout_count;
{
    if (fanout_count > wirecap_est_table_size) {
        io_printf("Warning from dl_estimate_capacitance : excessive fanout\n");
        fanout_count = wirecap_est_table_size;
    }

    return (wirecap_est_table[fanout_count-1]);
}


/******************************************************************************/
/* Argument processing routines */

/*******************************************************************************
*   DL_PROCESS_SCOPE_ARG
*   Process first argument : Full hierarchical name of scope to
*   which delay calculation will be performed.
*   This routine returns the scope indicated by the first argument.
*   If the first argument is null, the invoking module is returned as the scope.*   If the first argument does not represent a valid scope,
*   the invoking module is returned as the scope and an error is reported.
*   This routine uses some PLI "tf_" routines and defined values
*******************************************************************************/
static handle dl_process_scope_arg()
{
    handle  scope = acc_handle_tfarg(1);
    int     arg_type;

    if (acc_fetch_type(scope) != accModule)
    {
        /**************************************************************/
        /*** if task argument is null, set scope to invoking module ***/
        /**************************************************************/
        arg_type = tf_typep(1);
        if (arg_type == tf_nullparam)
            scope = acc_handle_parent((handle)tf_getinstance());
        /***********************************************************/
        /*** if task argument not null and is not a valid scope, ***/
        /*** set scope to invoking module display error          ***/
        /***********************************************************/
        else {   
            scope = acc_handle_parent((handle)tf_getinstance());
            io_printf("%s%s","Warning from delay calculation :",
                " invalid scope argument, setting scope to invoking module\n");
        }
    }    
 
    if (dl_verbose_flag)
        io_printf("  Scope : %s\n", acc_fetch_fullname(scope));
 
    return (scope);
}


/*******************************************************************************
*   DL_PROCESS_ESTIMATE_ARG
*   Process second argument : estimate wire capacitance or read from file.
*   Return true to indicate estimated capacitance, false to indicate
*   back-annotated capacitance.
*   This routine uses various PLI "tf_" routines and defined values
*******************************************************************************/
static bool dl_process_estimate_arg()
{
    int     arg_type = tf_typep(2);
    char   *arg = (char *)tf_getp(2);
    char    net_filename[256];

    if (arg_type != tf_string)
    {
        io_printf("%s%s","Warning from delay calculation :",
            " invalid second argument, will estimate wire capacitance\n");
        /*** return true to indicate estimated capacitance ***/
        return (true);
    }

    if ((arg == null) || (strcmp(arg, "estimate") == 0))
    {
        if (dl_verbose_flag)
            io_printf("  Wire capacitance will be estimated\n");

        /*** return true to indicate estimated capacitance ***/
        return (true);
    }
    else
    {
        if (dl_verbose_flag)
            io_printf("  Wire capacitance file : \"%s\"\n", arg);

        strcpy(net_filename, arg);

        if (!dl_make_backan_table(net_filename))
        {    
            io_printf("  Wire capacitance will be estimated\n"); 
            return(true); 
        }

        /*** return false to indicate back annotated capacitance ***/
        return (false);
    }
}
