/* File dl_calc_scan_icn.c */

/*******************************************************************************
*
*  This file contains routines which demonstrate how to create an access 
*  routine based delay calculator for a primitive delay or path delay
*  based library.
*
*  In order to understand this code, it will be helpful to have reviewed
*  the access routine documentation, and to have it handy for reference.
*  It should not have been necessary to have studied the access routines in
*  detail.  Indeed, studying this code will likely be very helpful in 
*  understanding the details within the access routines document.
*
*  A detailed application note that describes how to use the access routines
*  to write back annotators and delay calculators is provided with the PLI
*  documentation.  It may be necessary to review this applications note
*  before delving into this code.
*
*  Note that the term "cell" and "cell instance" are used here, and seen
*  in routines such as "acc_next_cell".  A "cell" is a module which has
*  been defined as a cell by having its definition follow the
*  compiler directive `celldefine (`endcelldefine is also available).
*  Also, all modules retrieved from libraries will automatically
*  be defined as cells, unless the argument '+nolibcell' has been included on
*  the command line.
*
*  This file contains routines for two distinct delay calculators : one
*  that may be applied to primitive output (distributed) delay cells, and
*  one that may be applied to path delay cells.  It is convenient to place the
*  two calculators in the same file since they share many auxiliary routines
*  (for example, for calculating load due to fanout on a given port).  Both
*  delay calculators use the entries in a capacitance back annotation file
*  to 'drive' the delay calculation.  The delay calculators assume that the
*  file contains interconnect net names and their associated capacitance values.
*  The following shows entries in an example capacitance back annotation
*  file:
*      
*        top.m1.dm1 3.6
*        top.m1.dm2 4.2
*        top.m1.q[0] 6.2
*        top.m1.qb[0] 6.2
*        top.m1.q[1] 6.2
*        top.m1.qb[1] 6.2
*
*  The delay calculators make the following assumptions about the libraries
*  they serve:
*          > cell ports are not explicitly named (their names are the
*            same as the net to which they connect within the cell)
*            (Do not confuse this with connection by name, which is fine.)
*          > each cell contains specparams which indicate the rise and
*            fall drive strength of each output, and the capacitive
*            loading factor of each input.  These specparams will be
*            detected by access routine acc_fetch_attribute, and
*            must follow the naming conventions, which are described
*            as follows :
*              To represent the drive of a port output "Q" :
*                    specparam RiseStrength$Q = 1;
*                    specparam FallStrength$Q = 2;
*              To represent the drive of a path from "A" to "Q" :
*                    specparam RiseStrength$A$Q = 3;
*                    specparam FallStrength$A$Q = 4;
*
*  There is nothing hard-coded into Verilog-XL or the access routines
*  that requires these naming conventions.  These conventions are decided
*  on by the developer of the library and delay calculator.
*
*  A call to the function :
*
*      acc_fetch_attribute(net,"RiseStrength$");
*
*  for the net A in the example above will return 1.0.
*
*  For each interconnect net in the capacitance back annotation file, 
*  the distributed delay based calculator scans all cell output nets connected
*  to the interconnect net.  For each cell output net, the calculator
*  determines the drive strengths of the port associated with the cell output
*  net and the load on the net due to driven cell inputs, calculates an
*  additional delay for the net, and adds this delay to that of all primitives
*  in the cell that drive the net.
*
*  For each interconnect net in the capacitance back annotation file, 
*  the path delay based calculator scans all cell output nets connected
*  to the interconnect net.  For each cell output net, the calculator
*  scans all path delays within the appropriate cell that have the cell output
*  net as their path output net.  For each path delay, the calculator
*  determines the drive strengths of the path and the load on the cell output
*  net due to driven cell inputs, calculates an additional delay for the
*  path, and adds this delay to the path.
*
*  To associate these routines with Verilog-XL system tasks,
*  place the following entries into the 'veriusertfs' data
*  structure in the 'veriuser.c' file:
*
      {usertask,0, 0,0, dl_prim_invoke,0, "$dcalc_prim",0},
      {usertask,0, 0,0, dl_path_invoke,0, "$dcalc_path",0},

*  Also, the functions dl_prim_invoke() and dl_path_invoke() need to
*  be delcared in veriuser.c.   Add the following line above the
*  definition of veriusertfs in the veriuser.c file:
 
   int dl_prim_invoke(), dl_path_invoke();


*  Invoking Delay Calculation :
*    Delay calculation is invoked by the system tasks '$dcalc_prim'
*    and '$dcalc_path' which are defined in the 'veriusertfs' data
*    structure.  One argument must be passed to the system task - the
*    name of the capacitance back annotation file.
*
*    An example invokation within a Verilog description would be :
*
*        $dcalc_prim("test_inp.dat");
*
*******************************************************************************/

#include <stdio.h>
#include "acc_user.h"
#include "veriuser.h"
#include "dl.h"

static void 	dl_calc_delays PROTO_PARAMS((int prim_or_path, char *net_filename));
static void 	dl_ann_prim_delays PROTO_PARAMS((handle net, double fanout_load, double wire_load));
static void 	dl_ann_path_delays PROTO_PARAMS((handle net, double fanout_load, double wire_load));
static double 	dl_load_factor PROTO_PARAMS((handle net));
static void 	dl_net_init PROTO_PARAMS((void));
static void 	dl_net_insert PROTO_PARAMS((handle net));
static int 	dl_net_done PROTO_PARAMS((handle net));

/*** System task invocation routine ***/
global void dl_prim_invoke();
global void dl_path_invoke();
global void dl_invoke();

/*** Main delay calculation routines ***/
local void dl_calc_delays();
local void dl_ann_prim_delays();
local void dl_ann_path_delays();

/*** Auxiliary routines ***/
local double dl_load_factor();
local void dl_net_init();
local void dl_net_insert();
local bool dl_net_done();

/*** Global variables ***/
bool dl_verbose_flag = false;
int cellout_count;


#define dlPrim 1
#define dlPath 2
#define MAX_CELLOUT_HANDLES 256

/*** More global variables ***/
handle cellout_array[MAX_CELLOUT_HANDLES];


/******************************************************************************/
/* System task invocation routine */

/*******************************************************************************
*   DL_PRIM_INVOKE
*   Begin delay calculation for primitive delay based models
*******************************************************************************/
exfunc void dl_prim_invoke()
{
    dl_invoke(dlPrim);
}

/*******************************************************************************
*   DL_PATH_INVOKE
*   Begin delay calculation for path delay based models
*******************************************************************************/
exfunc void dl_path_invoke()
{
    dl_invoke(dlPath);
}

/*******************************************************************************
*   DL_INVOKE
*   This routine invokes delay calculation processing. It accepts an
*   argument which indicates whether to begin path delay ot primitive
*   delay based calculations
*   This routine performs the following functions :
*       > Initialize and configure the access routines.
*       > Process command line plusargs
*       > Call routines to process system task arguments
*       > Call routine to calculate interconnect delays
*
*******************************************************************************/
exfunc void dl_invoke(prim_or_path)
    int prim_or_path;
{
    handle  dl_scope_g;
    int     mtmparam;
    int     arg_type = tf_typep(1);
    char   *arg = (char *)tf_getp(1);
    char    net_filename[256];

    acc_initialize();
#if 0
    acc_configure(accDevelopmentVersion,"1.5a");
#endif
    acc_configure(accToHiZDelay,"average");
    acc_configure(accPathDelayCount,"2");
    acc_configure(accDefaultAttr0,"true");

    /****************************************************************/
    /*** If "+dlverbose" appears on command line, set global flag ***/
    /****************************************************************/
    if (mc_scan_plusargs("dlverbose"))
    {
        dl_verbose_flag = true;
        if (prim_or_path == dlPrim)
            io_printf("Delay calculation invoked for primitive delay cells : \n");
        else
            io_printf("Delay calculation invoked for path delay cells : \n");
    }

    /********************************************************/
    /*** Process argument to get name of capacitance file ***/
    /********************************************************/
    if (arg_type != tf_string)
    {
        io_printf("%s%s\n","Warning from delay calculation :",
            " invalid argument, no delay calculation will be performed");
        return;
    }

    if (dl_verbose_flag)
        io_printf("  Wire capacitance file : \"%s\"\n", arg);
 
    strcpy(net_filename, arg);

    /*************************************/
    /*** Calculate interconnect delays ***/
    /*************************************/
    dl_calc_delays(prim_or_path,net_filename);

    /******************/
    /*** Aesthetics ***/
    /******************/
    if (dl_verbose_flag)
        io_printf("\n");

    /***********************************/
    /*** Access routine use complete ***/
    /***********************************/
    acc_close();
}


/*******************************************************************************
*   DL_CALC_DELAYS
*   This routine reads a back annotation file with capacitance values
*   associated with interconnect nets and performs delay calculation,
*   placing the delays on the primitives driving the associated cell
*   output net or the paths feeding the associated cell output net in
*   all cells with output or inout ports that feed this net. 
*******************************************************************************/
static void dl_calc_delays(prim_or_path,net_filename)
    int prim_or_path;
    char *net_filename;
{
    handle  interconnect_net, cellout_net;
    handle  driver;
    FILE   *net_file; 
    int     fanout_count;
    double  fanout_load, wire_load;
    char    interconnect_name[256];

    /***************************************************************/
    /*** Read each net/capacitance pair and calculate the delays ***/
    /*** for the net                                             ***/
    /***************************************************************/
    if ((net_file = fopen(net_filename,"r")) == null)
    {
        io_printf("Warning: Back-annotate file \"%s\" not found.  %s\n",
                                net_filename,"No delays will be calculated.");
        return;
    }

    while (fscanf(net_file,"%s%lf\n",interconnect_name,&wire_load) != EOF)
    {
        if ((interconnect_net = acc_handle_object(interconnect_name)) == null)
        {
            io_printf("Warning: Interconnect net \"%s\" not found.\n",
                                                            interconnect_name);
            continue;
        }

        /****************************************************/
        /*** Scan all cells with ports that feed this net ***/ 
        /****************************************************/
        dl_net_init();
        driver = null;
        while (driver = acc_next_driver(interconnect_net,driver))
        {
            /**************************************/
            /*** Skip if this net has been done ***/
            /**************************************/
            cellout_net = acc_handle_conn(driver);
            if (dl_net_done(cellout_net))
                continue;

            /*********************/
            /*** Count fanouts ***/
            /*********************/
            fanout_count = acc_count(acc_next_cell_load,cellout_net);
 
            /***********************************************/
            /*** Determine loading on net due to fanouts ***/
            /***********************************************/
            fanout_load = dl_load_factor(cellout_net);
 
            /*****************************************/
            /*** Calculate and annotate the delays ***/
            /*****************************************/
            if (prim_or_path == dlPrim)
                dl_ann_prim_delays(cellout_net,fanout_load,wire_load);
            else
                dl_ann_path_delays(cellout_net,fanout_load,wire_load);

            dl_net_insert(cellout_net);
        }
    }
}


/*******************************************************************************
*   DL_ANN_PRIM_DELAYS
*   This routine accepts a handle to a cell output net and the total
*   fanout loading on the net, retrieves the rise and fall strength
*   attributes associated with the net, calculates the delays for the net
*   and annotates them to all drivers on the net within the current
*   scope.
*******************************************************************************/
static void dl_ann_prim_delays(net,fanout_load,wire_load)
    handle  net;
    double  fanout_load, wire_load;
{
    double rise_strength, fall_strength;
    double rise, fall;
    handle driver, prim, parent;

    /****************************************************/
    /*** Determine attributes associated with the net ***/
    /****************************************************/
    rise_strength = acc_fetch_attribute(net, "RiseStrength$");
    fall_strength = acc_fetch_attribute(net, "FallStrength$");

    /***************************************/ 
    /*** Calculate load dependent delays ***/
    /***************************************/ 
    rise = rise_strength * (fanout_load + wire_load);
    fall = fall_strength * (fanout_load + wire_load);
    
    /*******************************************************/ 
    /*** Add the delays to all primitives in the current ***/
    /*** module that drive this net                      ***/
    /*******************************************************/ 
    parent = acc_handle_parent(net);
    driver = null;
    while (driver = acc_next_driver(net,driver))
    {
        prim = acc_handle_parent(driver);
        if (acc_handle_parent(prim) != parent)
            continue;

        /*******************************************/ 
        /*** Add calculated delays to the delays ***/
        /*** on the driving primitive            ***/
        /*******************************************/ 
        acc_append_delays(prim, rise, fall);
    
        /************************************************************/ 
        /*** If requested, print detailed calculation information ***/ 
        /************************************************************/
        if (dl_verbose_flag)
        {
            double newrise, newfall, newz;
            acc_fetch_delays(prim,&newrise,&newfall,&newz);
            io_printf("\nNet %s delays : rise = %6.2f  fall = %6.2f\n",
                                  acc_fetch_fullname(net), newrise, newfall);
            io_printf("  rise_strength  =%6.2f  fall_strength =%6.2f\n",
                                  rise_strength, fall_strength);
            io_printf("  fanout_load    =%6.2f  wire_load     =%6.2f\n",
                                  fanout_load, wire_load);
            io_printf("  added rise     =%6.2f  added fall    =%6.2f\n",
                                  rise, fall);
        }
    }
}


/*******************************************************************************
*   DL_ANN_PATH_DELAYS
*   This routine accepts a handle to a cell output net and the total
*   fanout loading on the net, retrieves the rise and fall strength
*   attributes associated with the path that feeds the net, calculates
*   the delays for the net and annotates them to all paths that feed the
*   net.
*******************************************************************************/
static void dl_ann_path_delays(net,fanout_load,wire_load)
    handle  net;
    double  fanout_load, wire_load;
{
    handle  parent, path;
    double rise_strength, fall_strength, rise, fall;

    /**************************************************/
    /*** For each path that feeds the net, retrieve ***/
    /*** the strength attributes, calculate and     ***/
    /*** annotate the delays                        ***/
    /**************************************************/
    parent = acc_handle_parent(net);
    path = null;
    while (path = acc_next_modpath(parent,path))
    {
        if (acc_handle_pathout(path) != net)
            continue;

        /*************************************************/
        /*** Determine attributes associated with path ***/
        /*************************************************/
        rise_strength = acc_fetch_attribute(path, "RiseStrength$");
        fall_strength = acc_fetch_attribute(path, "FallStrength$");

        /***************************************/
        /*** Calculate load dependent delays ***/
        /***************************************/
        rise = rise_strength * (fanout_load + wire_load);
        fall = fall_strength * (fanout_load + wire_load);

        /*****************************************/
        /*** Append delays to the driving path ***/
        /*****************************************/
        acc_append_delays(path, rise, fall);

        /************************************************************/
        /*** If requested, print detailed calculation information ***/
        /************************************************************/
        if (dl_verbose_flag)
        {
            double newrise, newfall, newz;
            acc_fetch_delays(path,&newrise,&newfall,&newz);
            io_printf("\nPath %s delays : rise = %6.2f  fall = %6.2f\n",
                                  acc_fetch_fullname(path), newrise, newfall);
            io_printf("  rise_strength  =%6.2f  fall_strength =%6.2f\n",
                                  rise_strength, fall_strength);
            io_printf("  fanout_load    =%6.2f  wire_load     =%6.2f\n",
                                  fanout_load, wire_load);
            io_printf("  added rise     =%6.2f  added fall    =%6.2f\n",
                                  rise, fall);
        }
    }
}
 


/******************************************************************************/
/* Auxiliary routines */

/*******************************************************************************
*   DL_LOAD_FACTOR
*   This routine accepts a pointer to a net, scans its connected terminals,
*   and returns the total load factor on the net.  This is determined
*   by finding specify parameters called FanoutLoad$netname associated
*   with each load and driver and summing their values.
*******************************************************************************/
static double dl_load_factor(net)
    handle  net;
{
    double  total_lf = 0;
    handle  load, load_net, driver_prim, driver_mod, driver_net, driver;

    /************************************/
    /*** Process each cell input load ***/
    /************************************/
    load = null;
    while (load = acc_next_cell_load(net,load))
    {
        /**************************************/
        /*** Get net connected to this load ***/
        /**************************************/
        load_net = acc_handle_conn(load);

        total_lf += acc_fetch_attribute(load_net, "FanoutLoad$");
    }

    /*************************************************/
    /*** If load has multiple drivers, scan fanin, ***/
    /*** adding each lf to total_lf                ***/
    /*************************************************/
    if (acc_count(acc_next_driver,net) > 1)
    {
        handle net_mod = acc_handle_parent(net);

        /*******************************/
        /*** Process each net driver ***/
        /*******************************/
        driver = null;
        while (driver = acc_next_driver(net, driver))
        {
            /*****************************************************/
            /*** Don't include load factor of driving terminal ***/
            /*****************************************************/
            driver_prim = acc_handle_parent(driver);
            driver_mod  = acc_handle_parent(driver_prim);
            if (driver_mod == net_mod)
                continue;

            /****************************************/
            /*** Get net connected to this driver ***/
            /****************************************/
            driver_net = acc_handle_conn(driver);

            total_lf += acc_fetch_attribute(driver_net, "FanoutLoad$");
        }
    }

    return (total_lf);
}


/*******************************************************************************
*   DL_NET_INIT
*   This routine initializes the global net storage variables and structures.
*******************************************************************************/
static void dl_net_init()
{
    cellout_count = 0;
}


/*******************************************************************************
*   DL_NET_INSERT
*   This routine accepts a handle to a net and stores the handle in the net
*   storage structure.
*******************************************************************************/
static void dl_net_insert(net)
    handle net;
{
    if (cellout_count >= MAX_CELLOUT_HANDLES - 1)
    {
        io_printf("Error in dl_net_insert : cell handle overflow.  %s\n",
            "Delay calculation may contain errors");
        return;
    }

    cellout_array[cellout_count++] = net;
}


/*******************************************************************************
*   DL_NET_DONE
*   This routine accepts a handle to a net and checks to see if it exists in
*   the net storage structure.  If it exists, this routine returns TRUE.
*   Otherwise, this routine returns FALSE.
*******************************************************************************/
static bool dl_net_done(net)
    handle net; 
{ 
    int i;

    if (cellout_count > 0)
    {
        for (i = 0; i < cellout_count; i++)
        {
            if (cellout_array[i] == net)
                return(TRUE);
        }
    }

    return(FALSE);
}
