/* Last Edit: Mon Mar 25 11:05:51 1991 by rays */
/* Last Edit: Fri Oct 5 1991 by manojb which ensures that the memories with */ 
/* width of more than 8 bits get set properly */

/*                          stuff_memory.c

     This file contains routines that place new values into memories.  It
     also contains a long explanation of how one accesses a memory in the
     PLI.  We expect to implement acc_ routines that will make this process
     easier in the future.

This file contains the following functions:

   void stuff_memory();
   handle my_handle_mem_tfarg();
   int memory_depth();
   int memory_width();
   void set_memory_bit();
   void set_memory();


		 HOW TO ACCESS MEMORIES WITH THE PLI

Values of verilog memories are stored in a string that is structured
like this:

struct
{
  char avalbits[number_of_groups]; S* one group for 8 bits of width *S
  char bvalbits[number_of_groups]; S* one group for 8 bits of width *S
} memval[mem_size]

The two character arrays store the memory information.  Each group is
one character and contains the information for 8 bits of the verilog
memory.  You derive the information from avalbits and bvalbits by
matching up the bits in each character and looking at the
combinations:

		A	B	Value
		0	0	  0
		1 	0  	  1
		0   1	  z
		1	1	  x



The structure described above doesn't exist in C because C doesn't
allow structures of variable widths.  So PLI gives you a string that
contains the information, and you have to index into that string
yourself.

The string looks like this for a 16 bit memory that's two words deep:

Verilog Declaration:
reg [15:0] mem [0:1] 

results in a string that is organized like this:

struct
{
  char avalbits[2];
  char bvalbits[2];
} memval[2];

We can look at this string graphically.

	 ADDRESS 0				  ADDRESS 1
+-----------------------------------|----------------------------------+
| GRP0-A | GRP1-A | GRP0-B | GRP1-B | GRP0-A | GRP1-A | GRP0-B |GRP1-B |
+-----------------------------------|----------------------------------+
   0         1        2       3         4        5        6        7

As you can see, this string contains 8 bytes.  You take the address,
and the number of groups and figure out which byte in the character
string to use.  Then you have to look at a bit within that byte to find
out the data for a specific memory bit.

The above string is stored in a structure that is returned from
tf_nodeinfo.  The code to get this structure might look like this:

  s_tfnodeinfo mem;
  tf_nodeinfo(tfarg_numb,&mem);

We now have the memory information in structure mem.  The above string
is stored in mem.node_value.memoryval_p.  This is a pointer to the above
string.  So:
  
  char *memvalue;
  memvalue = mem.node_value.memoryval_p

If you examine the above string, you'll see that we need to know the
number of groups in each word to access the proper bytes.  We get the
number of groups from the same structure:

  int ngroups;
  ngroups = mem.node_ngroups;


Now, to access a specific verilog memory word, we have to get the avalbits
and bvalbits that store the value for that word.  We can use the following
equation for this:

  a_index = ngroups * 2 * address; 
  b_index = a_pointer + ngroups;

Let's get the indices for the above example if we want address 1:

  a_index = 2 * 2 * 1 = 4 
  b_index = 4 + 2     = 6

We can use the indices to generate pointers to the aval string and bval string:

  char *aptr, *bptr;

  aptr = memvalue + aindex;
  bptr = memvalue + bindex;


Now you can use the fact that there are 8 bits per group to access the
specific memory bit that you want.  For example if you want to get a certain
bit number from the memory:

  int group_number, bit_number, 
      get_mem_value(); S* returns the scalar value of the memory location *S
  char agroup,bgroup;
  group_number = (mem_bit_number / 8);
  bit_number   = (mem_bit_number % 8);

  agroup = aptr[group_number];
  bgroup = pbtr[group_number];

  get_mem_value (agroup, bgroup, bit_number);

*/


#include "set_values.h"

/***********************************************************************
			     STUFF MEMORY

This routine will store the value string into a memory in the tfarg list.  It
uses routines that I've written to simplify the task.  Evetually we will have
acc routines that handle the same funtions.

This program gets a handle to the memory, then it loops throuh all the memory
locations and stuffs in the value from the value string.  However there are
two possible loops.

If the value string contains a value, then I get the new value and loop
through the memory storing it.  Since the new value doesn't change I 
get it outside of the loop.

If the value string calls for a random value, then I get a new value through
each pass in the loop.  This is much slower, so I only do it when the
user calls for a random value.

***********************************************************************/

void stuff_memory(tfarg_numb)
int tfarg_numb;
{
	handle memory; /* The memory on the tfarg list */
	int addr, /* A loop counter that will address the memory */
	   depth, /* The total depth of the memory */
	   width; /* The width of the memory */
	char* new_value; /* A string representing the new value of the memory */
    char* register_value(); /* Returns a value based on the value string and
                               the memory width */
    char* random_value();   /* Returns a random value string based on width */
  
    int memory_depth(),  /* A routine that gets the memory depth */
        memory_width();  /* A routine that gets the memory width */
    
    void set_memory(); /* Routine to set the value of a memory index */
    handle my_handle_mem_tfarg(); /* My routine that gets a handle to a memory */

  

    memory = my_handle_mem_tfarg(tfarg_numb);  /* Get handle to memory */
	depth = memory_depth (memory);             
	width = memory_width (memory);

	if (!strcmp(strtoupper(acc_fetch_tfarg_str(1)), "RANDOM"))
	{
 	    for (addr = 0; addr < depth; addr++)
	    {
			new_value = random_value(width);
			set_memory(memory, addr, new_value);
	    }
	}
	else
	{
		new_value = register_value(width);
    	for (addr = 0; addr <= depth; addr++)
			set_memory(memory, addr, new_value);
	}
}
	


/*R  void set_memory();
 *
 *	Inputs: 
 *			handle memory  -- A handle to the memory.
 *			int    address -- The address for which the value is desired.
 *			char*  valueString -- The string containing the new value.
 *
 *	Returns:
 *			A string that contains the value at the address input
 *
 *	Synopsis:
 *			The routine takes the handle to the memory whose data is to be  
 *			accessed and the address of the location whose value is to be set
 *			and a string that contains the value. 
 */
void set_memory(memory, address, valueString)
handle memory;
int address;
char* valueString;
{
    void set_memory_bit();
	p_tfnodeinfo nodeInfo = (p_tfnodeinfo)memory;
	int i;

	int memWidth = nodeInfo->node_vec_size;
	int ngroups = nodeInfo->node_ngroups;

	char* memVal = nodeInfo->node_value.memoryval_p;

	char* aPtr = memVal + ngroups*2*address;
	char* bPtr = memVal + (ngroups*2*address + ngroups);

	for (i = 0; i < memWidth; i++)
		set_memory_bit(aPtr, bPtr, i, ngroups, valueString[i]);

}

/*R  void set_memory_bit();
 *
 *	Inputs: 
 *			char* aVal     : Pointer to the "a" value string
 *			char* bVal     : Pointer to the "b" value string
 *			int   bitPos   : The bit that needs to have its value set
 *			int   ngroups  : Number of groups in the memory
 *			char  bitValue : The value that is to be put in the bitPos
 *
 *	Synopsis:
 *		The routine takes the pointers to the "a" and "b" value strings and the 
 * 		bit that needs to be set along with the number of groups of the memory 
 *		and the value that needs to be put into the specified bit and puts the 
 *		value in the memory bit.
 */
void set_memory_bit(aVal, bVal, bitPos, ngroups, bitValue)
char *aVal, *bVal;
int bitPos, ngroups;
char bitValue;
{
	int bitNum = bitPos % MEMGROUPLENGTH;
	int groupVal = ngroups -1 - bitPos/MEMGROUPLENGTH;

	switch (bitValue)
	{
		case '0' :
		{
			clear_bit(&aVal[groupVal], bitNum);
			clear_bit(&bVal[groupVal], bitNum);
			break;
		}
		case '1' :
		{
			set_bit(&aVal[groupVal], bitNum);
			clear_bit(&bVal[groupVal], bitNum);
			break;
		}
		case 'z':
		case 'Z' :
		{
			clear_bit(&aVal[groupVal], bitNum);
			set_bit(&bVal[groupVal], bitNum);
			break;
		}
		case 'x':
		case 'X' :
		{
			set_bit(&aVal[groupVal], bitNum);
			set_bit(&bVal[groupVal], bitNum);
			break;
		}
	}
}


/***********************************************************************
			 MY HANDLE MEM TFARG

This routine gets a pointer to the memories tfnodeinfo structure and
returns it as a handle.

***********************************************************************/


handle my_handle_mem_tfarg (tfarg)
     int tfarg;
{

/* Set aside memory space for the structure */
  p_tfnodeinfo mem = (p_tfnodeinfo) malloc (sizeof(s_tfnodeinfo));

  if (!tf_nodeinfo(tfarg, mem)) /* Get the nodeinfo structure */
    tf_message(ERR_ERROR, "Verilog", "MEMERR", 
	       "Error reading memory info from command line");
  
  return (handle) mem; /* Return the pointer to the structure as a handle */
}


/***********************************************************************
			     MEMORY DEPTH

This routine returns the number of words in a Verilog memory.  For example:

If we had 

reg [15:0] mem[0:1023] 

memory_depth would return 1024.

I have to cast the handle back to being a pointer to a tfnodeinfo structure
then I can use the -> operator to get the memory depth.

***********************************************************************/

int memory_depth(mem)
     handle mem;
{
  return (((p_tfnodeinfo) mem) -> node_mem_size);
}



/***********************************************************************
			     MEMORY WIDTH

This routine returns the width of a Verilog memory.  For example:

If we had 

reg [15:0] mem[0:1023] 

memory_width would return 16.

I have to cast the handle back to being a pointer to a tfnodeinfo structure
then I can use the -> operator to get the memory width.
***********************************************************************/

int memory_width(mem)
     handle mem;
{
  return (((p_tfnodeinfo) mem) -> node_vec_size);
}

