/*******************************************************/
/* file: lenval.c                                      */
/* abstract:  This file contains routines for using    */
/*            the lenVal data structure.               */
/*******************************************************/
#include "lenval.h"
#include "ports.h" /* for DEBUG_MODE define */

/* return the value represented by this lenval */
long value(lenVal *x)
{
    int i;
    long result=0;          /* result to hold the accumulated result */
    for (i=0;i<x->len;i++)
    {
	result=result<<8;   /* shift the accumulated result */
	result+=x->val[i];  /* get the last byte first */
    }
    return result;
}

/* set x to value; assumes value<512 */
void initLenVal(lenVal *x, long value)
{
    x->len=1;  
    x->val[0]=(unsigned char) value;
}

/* return TRUE iff actual=expected (after masking out some bits using mask */
short EqualLenVal(lenVal *expected, lenVal *actual, lenVal *mask)
{
    int i;
#ifdef DEBUG_MODE
    /* since we can't read TDO, just assume TDO matches whatever is expected */
    return 1;
#endif
    for (i=0;i<expected->len;i++)
    {
	unsigned char byteVal1=expected->val[i];   /* i'th byte of expected */
	unsigned char byteVal2=actual->val[i];     /* i'th byte of actual   */
	byteVal1 &= mask->val[i];                  /* mask out expected     */
	byteVal2 &=mask->val[i];                   /* mask out actual       */
	if (byteVal1!=byteVal2)
	    return 0;                              /* values are not equal  */
    }
    return 1;                                      /* values are equal      */
}



/* return the (byte, bit) of lv (reading from left to right) */
short RetBit(lenVal *lv, int byte, int bit)
{
    int i;
    unsigned char ch=lv->val[byte];   /* get the correct byte of data */
    unsigned char mask=128; /* 10000000 */
    for (i=0;i<bit;i++)
	mask=mask>>1;       /* mask the correct bit of the byte       */
    return ((mask & ch) !=0);  /* return 1 if the bit is 1, 0 otherwise */
}

/* set the (byte, bit) of lv equal to val (e.g. SetBit("00000000",byte, 1)
   equals "01000000" */
void SetBit(lenVal *lv, int byte, int bit, short val)
{
    int i;
    unsigned char *ch=&(lv->val[byte]);
    unsigned char OrMask=1, AndMask=255; 
    for (i=0;i<7-bit;i++)
	OrMask=OrMask<<1;
    AndMask-=OrMask;
    *ch = *ch & AndMask;  /* 0 out the bit */
    if (val)
	*ch = *ch | OrMask;   /* fill in the bit with the correct value */
}

/* add val1 to val2 and store in resVal;      */
/* assumes val1 and val2  are of equal length */
void addVal(lenVal *resVal, lenVal *val1, lenVal *val2)
{
    unsigned char carry=0;
    short i;
    resVal->len=val1->len;  /* set up length of result */
    /* start at least significant bit and add bytes    */
    for (i=val1->len-1;i>=0;i--)
    {
	unsigned char v1=val1->val[i];  /* i'th byte of val1 */
	unsigned char v2=val2->val[i];  /* i'th byte of val2 */
	/* add the two bytes plus carry from previous addition */
	unsigned char res=v1+v2+carry;  
	/* set up carry for next byte */
	if (v1+v2+carry>255)            
	    carry=1;                    /* carry into next byte */
	else
	    carry=0;
	resVal->val[i]=res;             /* set the i'th byte of the result */
    }
}

/* read from XSVF numBytes bytes of data into x */
void readVal(lenVal *x, short numBytes)
{
    int i;
    for (i=0;i<numBytes;i++)
        readByte(&(x->val[i]));  /* read a byte of data into the lenVal */
    x->len=numBytes;             /* set the length of the lenVal        */
}
