
;	Routines to do clipping with the Sutherland/Hodge
;	algorithm.  See Foley/VanDam pg146.
;	This implementation uses the following assignment of bits
;	in the 'outcode' that records a points position relative
;	to the clipping window.
; 	Bit 3 - point is left of window.	TOLEFT
;	Bit 2 - point is right of window.	TORIGHT
;	Bit 1 - point is below window.		TOBOTTOM
;	Bit 0 - point is above window.		TOTOP
;
;	The routines assume the following data structures  :
;	
;	typedef POINT struct{int x, y} 
;	typedef WINDOW struct {int
;		xleft, ybottom,		lower left corner
;		xright, ytop,		upper right corner
;		win_dx, win_dy		(xleft-xright), (ytop-ybottom)	
;		flags1, flags2		no current use
;		win_ident		every window needs a name
;
;	Entries
;		vcode(p,winp)
;		POINT *p;
;		WINDOW *winp
;
;		usage :
;			push winp, p
;			outcode is returned in acc.
;
;		clipline(p1,p2,winp)
;		POINT *p1, *p2;
;		WINDOW *winp
;
;		usage :
;			push winp,p2,p1
;			Returns with zero flag set iff some or all
;			of the line is visible.  In this case the
;			coordinates pointed to by p1,p2 are the
;			endpoints of the displayable portion.
;			If invisible, the coordinates are junk.


TOLEFT	equ	8		; Outcode bit assignments.
TORIGHT equ	4
TOBOTTOM equ	2
TOTOP	equ	1

WINXL	equ	0		; Window structure field offsets.
WINXR	equ	4
WINYB	equ	2
WINYT	equ	6
WINDX	equ	8
WINDY	equ	10

POINTX	equ	0		; Point structure field offsets.
POINTY	equ	2



;	Clip line to window.
;	Call seq:
;		Push wp, p2, p1.
;	On return zero flag clear iff the line is not visible.
;	In this case the values of the points p1,p2 are junk.
;	If visible, p1,p2 are the endpoints of the visible portion.
;	All registers used.
;
;	Stack offsets for parameters and local vars :
;
CLWP	equ 11		; Parameters.
CLP2	equ 9
CLP1	equ 7

; return address equ 5

CLCOD1	equ 3		; Local vars.
CLCOD2	equ 1



clipl
	rep	#0x30		; M = X  = 1

	lda	##0		; reserve space on stack for 
	pha			; and init local vars.
	pha
	tsx			; Get pointer to parameters etc.
				; The stack pointer is a little too
				; fluctuatious for use with constant
				; offsets.

;	Get outcodes of the points p1, p2.
;	Save the outcodes at CLCOD1, CLCOD2.

	lda	CLWP,x		; push wp for vcode.
	pha
	lda	CLP1,x		; ditto p1		
	pha
	jsr	vcode		; get code for p1 in acc
	sta	CLCOD1,x	; save code
	pla			; pop p1
	lda	CLP2,x		; push p2 for vcode (wp still stacked)
	pha
	jsr	vcode		; get code for p2
	sta	CLCOD2,x	; save code
	ply			; clean up stack
	ply

;	test for trivial accept - both codes 0

	lda	CLCOD1,x
	ora	CLCOD2,x	; see if both codes == 0
	bne	cl1		; branch if no, clip may be needed.
	brl	result		; trivial accept, clean up and return.

;	test for trivial reject - codes AND <> 0
cl1
	lda	CLCOD1,x
	and	CLCOD2,x
	beq	cl2		; branch if clip needed.
	brl	result		; trivial reject, cleanup and return.
cl2	

;	We're here because the line couldn't be trivially accepted 
;	or rejected.  Call the bisect routine for each point out
;	of the window.  Bisect adjusts the point and its visibility
;	code.

	lda	CLWP,x		; push window pointer
	pha

	lda	CLCOD1,x	; need to clip p1 ?
	beq	cli1		; branch if no.

; set for call to bisect - p1 needs clipping

	pha			; push p1 code - bisect modifies.
	lda	CLP1,x		; push p1 - bisect modifies coords.
	pha				
	lda	CLCOD2,x	; push p2 code.
	pha
	lda	CLP2,x		; push p2
	pha	
	jsr	bisect		; adjust p1 and its code.
	pla			; clean up stack.
	pla
	pla
	pla			; this is the new code for p1.  
	beq	cli1		; if <> 0, line is invisible.
	ply			; so clean stack and return
	brl	result		; with failure indication (acc <> 0).
cli1
	lda	CLCOD2,x	; need to clip p2 ?
	bne	cli2		; branch if yes.
	ply			; no, it's already inside.  clean up
	brl	result		; and return success (acc = 0).
cli2

; set up for call to bisect - p2 needs clipping.
; remember wp is already stacked

	pha			; push p2 code - bisect modifies.
	lda	CLP2,x		; push p2 - bisect modifies coords.
	pha				
	lda	##0		; push p1 code.
	pha
	lda	CLP1,x		; push p1
	pha	
	jsr	bisect		; adjust p2 and its code.
	pla			; clean up stack.
	pla
	pla
	pla			; this is the new code for p2 and also
				; the result of the clip - success.
cli3
	ply			; clean up stack.

result
	tay			; save result
	pla			; clean up stack
	pla
	tya			; restore result, set/clear zero flag
	rts

;	Parameter stack offsets

BIWP	equ 23
BICOD2	equ 21		; visibility code of clipped point.
BIP2	equ 19		; point to clip.
BICOD1	equ 17		; visibility code of fixed point.
BIP1	equ 15		; Fixed point.

; ret add equ 13

;	Local storage stack offsets

BIPY	equ 11		; One end of line to be bisected.
BIPX	equ 9
BIMY	equ 7		; Midpoint of line bisected.
BIMX	equ 5
BICODM	equ 3		; Visibility code of midpoint.
BIADDM	equ 1		; Address of BIMX, for vcode.

bisect
	tsc		; reserve 5 words on stack for local vars.
	clc
	adc	##10
	tcs				

;	copy point parameter p1 to bip	

	ldy	##POINTX
	lda	(BIP1,s),y
	sta	BIPX,s
	ldy	##POINTY
	lda	(BIP1,s),y
	sta	BIPY,s

;	find mid(bip,p2), save in bim

cutit

	ldy	##POINTX
	lda	(BIP2,s),y
	clc
	adc	BIPX,s
	asl	a
	sta	BIMX,s

	ldy	##POINTY
	lda	(BIP2,s),y
	clc
	adc	BIPY,s
	asl	a
	sta	BIMY,s

;	see if midpoint equals 1 of the endpoints.
;	in that case, we return successfully - the midpoint
;	is on the edge of the window.

	lda	BIMX,s
	cmp	BIPX,s
	bne	bi1
	lda	BIMY,s
	cmp	BIPY,s
	beq	success
bi1
	ldy	##POINTX
	lda	BIMX,s
	cmp	(BIP2,s),y
	bne	bi2
	ldy	##POINTY
	lda	BIMY,s
	cmp	(BIP2,s),y
	bne	bi2

success

	ldy	##POINTX	; return point of intersection.
	lda	BIPX,s	
	sta	(BIP2,s),y
	ldy	##POINTY
	lda	BIPY,s
	sta	(BIP2,s),y

	lda	##0		; return visibility code = 0
	sta	BICOD2		; indicating success.
				
	tsc			; clean up stack.
	clc
	adc	##24
	tcs
	rts	

;	Haven't found intersection yet.  Get visibility code
;	of midpoint.
bi2
	tsx
	lda	BIWP,x
	pha
	lda	BIADDM,x
	pha
	jsr	vcode		; returns with code in acc.

	plx			; clean up stack
	plx		
	
;	see if segment from midpoint to clip point
;	intersects the window.

	sta	BICODM,s	; and codes, result is 0 if 
	and	BICOD2,s	; not outside.
	beq	bi4		; branch if intersection.

;	segment from mid to clip is outside, see if same 
;	for fixed point to mid.

	lda	BICODM,s	; and codes, result 0 if
	and	BICOD1,s	; not outside.
	beq	bi3		; branch if intersection.

;	Line can be rejected.  Return same outcode as fixed
;	point.  

	lda	BICOD1,s
	sta	BICOD2,s

	tsc			; clean up stack and return.
	clc
	adc	##24
	tcs
	rts			

bi3

;	Here because segment from mid to clip was outside.
;	Move clip to mid and go bisect again.

	lda	BIMX,s
	ldy	##POINTX
	sta	(BIP2,s),y

	lda	BIMY,s
	ldy	##POINTY
	sta	(BIP2,s),y

	lda	BICODM,s
	sta	BICOD2,s
	brl	cutit

bi4

;	Here because segment from fixed to clip was outside.
;	Move fixed to mid and go bisect again.

	lda	BIMX,s
	sta	BIPX,s
	lda	BIMY,s
	sta	BIPY,s

	lda	BICODM,s
	sta	BICOD1,s
	brl	cutit


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Routine to compute outcode of point with respect to a	;
;	window.  Call seq : push wp, p.				;
;	Return outcode in acc. 					;
;	Also, zero flag is set iff point in window.		;
;	Uses X, Y registers.					;
;								;
;	Stack offsets of parameters and local variables.	;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

VCPOFF	equ	1
VCWOFF	equ	3	

vcode
	lda	##0		; Init outcode.
	pha

	tsx			; For addressing outcode (RMW).
	
	ldy	##POINTX
	sec			; Compute *p.x - *winp.xleft.
	lda	(VCPOFF,s),y	; Carry will be set if point is left 
	ldy	##WINXL		; of window.
	sbc	(VCWOFF,s),y
	rol	0,x		; Save carry in outcode.


	ldy	##WINDY		; Compare result(-1) with window  
	dec	a		; width.  Carry will be set if point
	cmp	(VCWOFF,s),y	; is INSIDE.  We'll toggle it on the
	rol	0,x		; way out.

; 	As above, for Y.


	ldy	##POINTY
	sec			; Compute *p.y - *winp.ybottom.
	lda	(VCPOFF,s),y	; Carry will be set if point is below 
	ldy	##WINYB		; window.
	sbc	(VCWOFF,s),y
	rol	0,x		; Save carry in outcode.

	ldy	##WINDY		; Compare result(-1) with window  
	dec	a		; height.  Carry will be set if point
	cmp	(VCWOFF,s),y	; is <= top.  We'll toggle it on the
	rol	0,x		; way out.

	pla			; Get outcode.
	eor	##TOLEFT+TOTOP ; Take care of reversed bits.
	rts			; Return with outcode in acc.

