/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

#include "lib.h"
#include "uilib.h"
#include "info.h"
#include "osf.h"
#include "mcheck.h"
#include "cpu.h"		/* interface this file implements */
#include "cpu/ev6.h"		/* private data and definitions */


/* EV6 revision codes, as given in the CHIP_ID field in I_CTL */
/* These strings should be no longer than 16 characters */

const String EV6_rev[] = {

    /* EV6 Revisions */
    "ALERT: EV6 P1!",				/* 000000b */
    "ALERT: EV6 P2!",				/* 000001b */
    "ALERT: EV6 P2.2!",				/* 000010b */
    "EV6 Rev 2.3",				/* 000011b */
    "EV6 Pass 3",				/* 000100b */
    "EV6 Rev 2.4",				/* 000101b */
    "EV6 Pass 2.5",				/* 000110b */
    "EV6 (unknown P7)",				/* 000111b */

    /* EV67 Revisions - pointed at by EV6__EV67_REVBASE */
    "EV67 Rev 2.1",				/* 001000b */
    "EV67 Rev 2.2",				/* 001001b */
    "EV67 Rev 2.1.1",				/* 001010b */
    "EV67 Rev 2.2.1",				/* 001011b */
    "EV67 Rev 2.3",				/* 001100b */
    "EV67 Rev 2.1.2",				/* 001101b */
    "EV67 Rev 2.2.2",				/* 001110b */
    "EV67 Rev 2.2.3",				/* 001111b */

    /* EV68 Revisions - pointed at by EV6__EV68_REVBASE */
    "EV68(Al) Pass 1",				/* 10000 = EV68AL Pass 1 */
    "EV68(Al) Pass 2",				/* 10001 = EV68AL Pass 2 */
    "EV68(Al) P2.1A",				/* 10010 = Pass 2.1/2.1A */
    "Reserved [0x13]",				/* 10011 = Reserved EV6x */
    "EV68CB Pass 1",				/* 10100 = EV68CB Pass 1 */
    "Reserved [0x15]",				/* 10101 = Reserved EV6x */
    "Reserved [0x16]",				/* 10110 = Reserved EV6x */
    "Reserved [0x17]",				/* 10111 = Reserved EV6x */
    "Reserved [0x18]",				/* 11000 = Reserved EV6x */
    "Reserved [0x19]",				/* 11001 = Reserved EV6x */
    "Reserved [0x1A]",				/* 11010 = Reserved EV6x */
    "Reserved [0x1B]",				/* 11011 = Reserved EV6x */
    "Reserved [0x1C]",				/* 11100 = Reserved EV6x */
    "Reserved [0x1D]",				/* 11101 = Reserved EV6x */
    "Reserved [0x1E]",				/* 11110 = Reserved EV6x */
    "Reserved [0x1F]",				/* 11111 = Reserved EV6x */

    "Unrecognised rev!"				/* = CPU_NREVS */
};

#define CPU_NREVS (sizeof( EV6_rev ) / sizeof( String ) - 1)


String ev6_cpurev( unsigned long I_CTL )
{
    unsigned long revno;
    revno = (I_CTL >> 24) & 0x3F;			/* extract CHIP_ID */
    if ( revno > CPU_NREVS )    revno = CPU_NREVS;	/* error case */
    return EV6_rev[revno];
}

int ev6_cpurevno( unsigned long I_CTL )
{
    unsigned long revno;
    revno = (I_CTL >> 24) & 0x3F;			/* extract CHIP_ID */
    return revno;
}


/*----------------------------------------------------------------------*/
/* Machine-check interpretation, according to the 21264 reference manual */

void cpu_mchkinterp( String interp, int scb, LogoutFrame_t * L)
{
    /* just in case we fall through */
    sprintf_dbm(interp, "No match: EXC_ADDR=0x%X CADDR=0x%X\n"
		"ISTAT=0x%X CSTAT=0x%X DCSTAT=0x%X, MMSTAT=0x%X",
		L->EXC_ADDR, L->C_ADDR,
		L->I_STAT, L->C_STAT, L->DC_STAT, L->MM_STAT );	

    switch (scb) {

    case SCB_Q_PROCERR:

	/* Icache data or tag parity error */
	if (L->I_STAT & EV6__I_STAT__PAR) 
	{
	    sprintf_dbm(interp,
			"CPU Internal Error:\n"
			"Icache parity error");
	    break;
	}

	/* Dcache tag parity error (on issue) (DFAULT) */
	if (L->MM_STAT & EV6__MM_STAT__DC_TAG_PERR) 
	{
	    sprintf_dbm(interp,
			"CPU Internal Error:\n"
			"Dcache tag parity (issue)" );
	    break;
	}

	/* Errors relating to D-stream set non-zero DC_STAT - mask CRD bits */
	switch (L->DC_STAT &
		(EV6__DC_STAT__ECC_ERR_ST | EV6__DC_STAT__ECC_ERR_LD))
	{

	case EV6__DC_STAT__ECC_ERR_ST:
	    /* Dcache single-bit ECC error on small store */
	    sprintf_dbm(interp,
			"CPU Internal Error:\n"
			"Dcache ECC on read-mod-write");
	    break;

	case EV6__DC_STAT__ECC_ERR_LD:
	    switch (L->C_STAT) 
	    {
	    case 0:
		/* Dcache single-bit error on speculative load */
		/* Bcache victim read on Dcache/Bcache miss */
		sprintf_dbm(interp,
			"Slot-B Error:\n"
			"Dcache speculative or Bcache victim");
		break;

	    case EV6__C_STAT__DSTREAM_DC_ERR:
		/* Dcache single bit error on load */
		sprintf_dbm(interp,
			"CPU Internal Error:\n"
			"Dcache ECC error, bit %d",
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
		break;


	    case EV6__C_STAT__DSTREAM_BC_ERR:
		/* Bcache single-bit error on Dcache fill */
		sprintf_dbm(interp,
			"Slot-B Error:\n"
			"Bcache/Dcache - ECC bit %d",
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
		break;

	    case EV6__C_STAT__DSTREAM_MEM_ERR:
		/* Memory single-bit error on Dcache fill */
		sprintf_dbm(interp,
			"Memory/EV6 bus Error:\n"
		 	"Address 0x%016lX, bit %d",
			L->C_ADDR,
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
		break;

	    default:
		break;
	    }
	}

	/* I-stream, other misc errors go on C_STAT alone */
	switch (L->C_STAT)
	{
	case EV6__C_STAT__ISTREAM_BC_ERR:
	    /* Bcache single-bit error on Icache fill (also MCHK) */
	    sprintf_dbm(interp,
			"Slot-B Error:\n"
			"Bcache/Icache - ECC bit %d",
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
	    break;

	case EV6__C_STAT__ISTREAM_MEM_ERR:
	    /* Memory single-bit error on Icache fill (also MCHK) */
	    sprintf_dbm(interp,
			"Memory/EV6 bus Error:\n"
			"Address 0x%016lX, bit %d",
			L->C_ADDR,
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
	    break;

	case EV6__C_STAT__PROBE_BC_ERR:
	    /* Bcache single-bit error on a probe hit */
	    sprintf_dbm(interp,
			"Slot-B Error:\n"
			"ECC on bcache probe, bit %d",
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
	    break;
	}
	break;

    case SCB_Q_PROCMCHK:

	/* Machine check errors described by DC_STAT */
	switch (L->DC_STAT) 
	{
	case EV6__DC_STAT__TPERR_P0:
	case EV6__DC_STAT__TPERR_P1:
	    /* Dcache tag parity error (on retry) */
	    sprintf_dbm(interp,
			"CPU error:\n"
			"Repeated Dcache tag parity errors" );
	    break;

	case EV6__DC_STAT__SEO:
	    /* Dcache second error on store */
	    sprintf_dbm(interp,
			"Processor internal error:\n"
			"Dcache error occurred during machine check handling" );
	    break;

	default:		/* some other kind of error */
	    break;
	}

	/* Machine check errors described by C_STAT */
	switch (L->C_STAT) 
	{
	case EV6__C_STAT__DC_PERR:
	    /* Dcache duplicate tag parity error */
	    sprintf_dbm(interp,
			"CPU error:\n"
			"Dcache duplicate tag error at address 0x%08lX",
			L->C_ADDR );
	    break;


	case EV6__C_STAT__BC_PERR:
	    /* Bcache tag parity error */
	    sprintf_dbm(interp,
			"Slot-B error:\n"
			"Bcache tag error at address 0x%08lX",
			L->C_ADDR);
	    break;


	case EV6__C_STAT__ISTREAM_BC_ERR:
	    /* Bcache single-bit error on Icache fill (also CRD) */
	    sprintf_dbm(interp,
			"Slot-B error:\n"
			"Bcache/Icache error at address 0x%08lX bit %d",
			L->C_ADDR,
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME));
	    break;


	case EV6__C_STAT__ISTREAM_MEM_ERR:
	    /* Memory single-bit error on Icache fill (also CRD) */
	    sprintf_dbm(interp,
			"Memory/EV6 bus error:\n",
			"ECC on Icache at address 0x%08lX, bit %d",
			L->C_ADDR,
			cpu_syn2bit(L->DC0_SYNDROME, L->DC1_SYNDROME) );
	    break;


	case EV6__C_STAT__ISTREAM_BC_DBL:
	    /* Bcache double-bit error on Icache fill */
	case EV6__C_STAT__DSTREAM_BC_DBL:
	    /* Bcache double-bit error on Dcache fill */

	    sprintf_dbm(interp,
			"Slot-B error:\n"
			"Multi-bit ECC error from Bcache at address 0x%08lX",
			L->C_ADDR );
	    break;


	case EV6__C_STAT__ISTREAM_MEM_DBL:
	    /* Memory double-bit error on Icache fill */
	case EV6__C_STAT__DSTREAM_MEM_DBL:
	    /* Memory double-bit error on Dcache fill */

	    sprintf_dbm(interp,
			"Memory/EV6 bus error:\n"
			"Multi-bit ECC error at address 0x%08lX",
			L->C_ADDR );
	    break;

	default:		/* something else */
	    break;
	}
	break;			/* machine check type loop (on vector) */

    default:
	break;
    }
}



/*----------------------------------------------------------------------*/
/* take the two syndromes from the CBOX error chain and convert them
 * into a bit number */

/* NOTE - since I currently don't know of any difference between C0 and C1
 * I just ignore C1, since in all cases I've seen so far they are identical */

static const unsigned char bit_to_syndrome[72] =
{
    0xce, 0xcb, 0xd3, 0xd5, 0xd6, 0xd9, 0xda, 0xdc,	/* 0 */
    0x23, 0x25, 0x26, 0x29, 0x2a, 0x2c, 0x31, 0x34,	/* 8 */
    0x0e, 0x0b, 0x13, 0x15, 0x16, 0x19, 0x1a, 0x1c,	/* 16 */
    0xe3, 0xe5, 0xe6, 0xe9, 0xea, 0xec, 0xf1, 0xf4,	/* 24 */
    0x4f, 0x4a, 0x52, 0x54, 0x57, 0x58, 0x5b, 0x5d,	/* 32 */
    0xa2, 0xa4, 0xa7, 0xa8, 0xab, 0xad, 0xb0, 0xb5,	/* 40 */
    0x8f, 0x8a, 0x92, 0x94, 0x97, 0x98, 0x9b, 0x9d,	/* 48 */
    0x62, 0x64, 0x67, 0x68, 0x6b, 0x6d, 0x70, 0x75,	/* 56 */
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80	/* 64 */
};


int cpu_syn2bit(unsigned long c0, unsigned long c1)
{
    int bit;

    for (bit = 0; bit < 72; bit++)
	if (bit_to_syndrome[bit] == c0)
	    return bit;

    return -1;			/* not found */
}


#ifdef TRACE_ENABLE

/*----------------------------------------------------------------------*/
/* Dump of PALcode impure region processor configuration details */


void cpu_impuredump( void )
{
    unsigned char myid = smp_myid();
    ImpureRgn_t *I = impure[myid];

    mobo_logf(
	LOG_INFO "Impure Region contents for CPU %d:\n"
	LOG_INFO "KSP:\t%lX\n"
	LOG_INFO "VA:\t%lX\n"
	LOG_INFO "VA_CTL:\t%lX\n"
	LOG_INFO "IER_CM:\t%lX\n"
	LOG_INFO "I_CTL:\t%lX\n"
	LOG_INFO "M_CTL:\t%lX\n"
	LOG_INFO "DC_CTL:\t%lX\n",
	myid,
	I->KSP, I->VA, I->VA_CTL, I->IER_CM, I->I_CTL, I->M_CTL, I->DC_CTL 
	);
}

#endif


/*----------------------------------------------------------------------*/
/* Register interesting information about this processor */

void cpu_info_submit( const unsigned myid )
{
    unsigned long cpu_freq;

    cpu_freq = 1000000000UL / impure[myid]->CYCLE_CNT;	/* in kHz */
    cpu_freq = (cpu_freq + 500) / 1000;			/* in MHz, rounded */

    info_submit( SRC_SROM, SUBJ_CPU, "Processor clock frequency",
		"CPU %d: %d MHz", myid, cpu_freq );

    info_submit( SRC_CPU, SUBJ_CPU, "Processor chip revision", "CPU %d: %s",
		myid, ev6_cpurev( impure[myid]->I_CTL ) );

    if ( impure[myid]->BEHAV_DATA != 0)
    {
         info_submit( SRC_SROM, SUBJ_CPU, "L2 cache size", "CPU %d: %d MBytes",
		 myid, ( impure[myid]->BEHAV_DATA >> 8 ) & 0xFFU );
    }
}

