/*----------------------------------------------------------------------*/
/* AlphaBIOS firmware loader */
/* Begun by Stig Telfer, Alpha Processor Inc, 28 May 1999 */
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* if you add a numerical argument to this function, it will attempt to 
 * jump to an image with matching fwid
 */

#undef TRACE_ENABLE		/* define this for verbose debug messages */

#include "lib.h"
#include "uilib.h"
#include "specifics.h"
#include "northbridge.h"
#include "cserve.h"
#include "ledcodes.h"

#include "rom.h"

#define MAX_ROM_SIZE (1024<<10)		/* a 1MB max limit on ROM size */

static unsigned *ld_addr;

static void usage( const String name )
{
    mobo_alertf( "Usage", 
		 "%s <Firmware ID>\r"
		 "%s -f <filename.rom>", name, name );
}


static DBM_STATUS load_from_rom( FWID I, rom_t **result )
{
    DBM_STATUS sval;
    rom_t *R;
    const String name = rom_I2name( I );

    TRACE( "Running...\n" );

    /* Search BootMem images for a match */
    TRACE( "Searching BootMem for an image of type %d\n", I );
    R = rom_search( I, rom_ram_map );
    if ( R != NULL )
    {
	TRACE( "Found bootmem match\n" );
	printf_dbm( "Found ROM-in-RAM match for '%s'\n", name );
    }
    else
    {
	TRACE( "Searching ROM for image of type %d\n", I );

	/* First attempt a quick check at all sector boundaries */
	/* If that fails then go for an extensive byte-by-byte search */

	printf_dbm("Scanning ROM for image of type %d (%s)...\n", I, name );

	R = rom_quickscan( I );
	if ( R == NULL )
	{
	    TRACE( "quickscan produced nothing, looking for full map\n" );
	    if ( !rom_mapped )
	    {
		TRACE( "Building full map\n" );
		sval = rom_index(); 
		if ( sval == STATUS_FAILURE )	return sval;
	    }

	    TRACE( "Searching physical map\n" );
	    R = rom_search( I, rom_phys_map );
	    
	    if ( R == NULL )
	    {
		printf_dbm( "Error in scanning for %s image\n", name );
		return STATUS_FAILURE;
	    }
	}
    }

    TRACE( "Returning match at 0x%lX\n", R );
    *result = R;
    return STATUS_SUCCESS;
}


/* FIXME: this could be done more neatly and using less heap by mallocing 
 * for a rom header and reading just that using fread, then loading the rest
 * directly to the correct destination
 */

static DBM_STATUS load_from_floppy( String fname, rom_t **result )
{
    unsigned isize;
    char *buf;
    DBM_STATUS sval;

    buf = malloc( MAX_ROM_SIZE );
    if ( buf == NULL )
    {
	mobo_logf( LOG_CRIT "Couldn't allocate buffer to load ROM file\n" );
	return STATUS_FAILURE;
    }

    printf_dbm( "Loading floppy file %s...\n", fname );
    isize = LoadAFile( fname, buf );			/* to download area */
    if (isize == -1)
    {
        mobo_alertf("Floppy load error", "Couldn't load file %s", cbuf );
        return STATUS_FAILURE;
    }

    /* now check that we have loaded a valid .ROM file */
    rom_free( rom_ram_map );
    rom_ram_map = NULL;
    sval = rom_ram_scan( buf, MAX_ROM_SIZE );
    if ( sval != STATUS_SUCCESS )
    {
	printf_dbm("File loaded is not a recognisable ROM-format file\n" );
	return STATUS_FAILURE;
    }

    *result = rom_ram_map;
  
    /* Might be a dangerous thing to do if the heap was corrupted by the 
     * above, but perhaps I'm a compulsively tidy person... */
    free( buf );

    return STATUS_SUCCESS;
}



DBM_STATUS go_bios( int argc, char *argv[] )
{
    FWID I;
    BOOLEAN do_jump = TRUE;
    rom_t *R;

    /* setup of interface */
    mobo_cls();
    swpipl( 7 );
    ld_addr = (unsigned *)NULL;


    /* look for command line arguments - first as string arguments */
    switch( argc ) {
	case 1:
	    if( load_from_rom( FW_SRM, &R ) == STATUS_SUCCESS )	/* no arg */
		break;
	    if( load_from_rom( FW_WNT, &R ) == STATUS_SUCCESS )	/* no arg */
		break;

	    printf_dbm( "Couldn't find any next-level firmware in the ROM\n" );
	    do_jump = FALSE;
	    break;

	case 2:
	    I = atoi( argv[ 1 ] );		/* argument as ROM FWID */
	    if( load_from_rom( I, &R ) == STATUS_SUCCESS )
		break;

	    printf_dbm( "Couldn't find the specified firmware in the ROM\n" );
	    usage( argv[0] );
	    do_jump = FALSE;
	    break;

	case 3:
	    if ( strcmp( argv[1], "-f" ) == 0 )
	    {
		if ( load_from_floppy( argv[2], &R ) == STATUS_SUCCESS )
		    break;

		printf_dbm( "Couldn't load the specified file on floppy\n" );
		do_jump = FALSE;
		break;
	    }
	    /* otherwise, we drop through here to the default (fail) case */

	default:
	    usage( argv[0] );
	    return STATUS_FAILURE;
    }


    if ( do_jump )
    {
	/* Just for kicks - if we're starting SRM, go blue & white */
	if( R->fwid == FW_SRM )
	{
	    printf_dbm( FG_WHITE BG_BLUE );
	    mobo_cls( );
	    printf_dbm( "Preparing console for SRM\n" );
	}

	/* We have the firmware image ready to go, so lets jump to it! */
	TRACE( "Jumping to loaded firmware image\n" );
	rom_exec( R );
    }

    printf_dbm( "Load failed: press any key to return\n" );
    mobo_key( 0 );
    return STATUS_FAILURE;
}

