/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Access to CMOS/Real-time clock */


#include "lib.h"
#include "uilib.h"
#include "northbridge.h"		/* pick up access for IO space */
#include "time.h"
#include "cmos_rtc.h"			/* interface for this library */
#include "nvram.h"

volatile unsigned long jiffies = 0;
unsigned short rtc_epoch=0;		/* Year signified by 0x00 in CMOS */


const unsigned TimerModes[15] = { 3906, 7813, 122, 244, 488, 977, 1953,
				  3906, 7813, 15625, 31250, 62500,
				  125000, 250000, 500000 };


enum CMOSport {
        CMOS_ADDR = 0x70,
        CMOS_DATA = 0x71
};


unsigned char cmosrb( enum CMOSaddr a )
{
    outportb( CMOS_ADDR, a );           /* address to CMOS Address Port */
    return inportb( CMOS_DATA );        /* Read back data from Data port */
}


void cmoswb( enum CMOSaddr a, unsigned char d )
{
    outportb( CMOS_ADDR, a );           /* address to CMOS Address Port */
    outportb( CMOS_DATA, d );           /* Write data out to the data port */
}


DBM_STATUS cmosready( void )
{
    /* check for whether the clock is ticking at all */
    unsigned long delta, start, target, ticks;

    /* 2 secs in clock ticks */
    delta = 2 * 1000000000000UL / primary_impure->CYCLE_CNT;
    start = rpcc();
    target = start + delta;

    /* Wait until an update is just finished */
    while( !(cmosrb( CM_STATUSA ) & 0x80) )
    {
	ticks = rpcc();
	if ( ticks < start ) 	ticks += 1UL<<32;	/* counter wrap */
	if ( ticks > target )
	{
	    mobo_logf( 	LOG_CRIT 
			"DATE/TIME: clock not ticking, update stuck low\n" );
	    return STATUS_FAILURE;			/* clock not ticking */
	}
    }
    while( cmosrb( CM_STATUSA ) & 0x80 )
    {
        ticks = rpcc(); 
        if ( ticks < start )    ticks += 1UL<<32;       /* counter wrap */
        if ( ticks > target )
	{
	    mobo_logf( 	LOG_CRIT 
			"DATE/TIME: clock not ticking, update stuck high\n" );
	    return STATUS_FAILURE;			/* clock not ticking */
	}
    }

    return STATUS_SUCCESS;
}


void RTC_Get( RTC_t *r )
{
    r->year   = cmosrb( CM_YEAR );
    r->month  = cmosrb( CM_MONTH );
    r->daymth = cmosrb( CM_DAYMTH );
    r->daywk  = cmosrb( CM_DAYWK );
    r->hralrm = cmosrb( CM_HRALRM );
    r->hours  = cmosrb( CM_HOURS );
    r->minalrm= cmosrb( CM_MINALRM );
    r->mins   = cmosrb( CM_MINS );
    r->secalrm= cmosrb( CM_SECALRM );
    r->secs   = cmosrb( CM_SECS );

}

void RTC_Set( RTC_t *r )
{
    cmoswb( CM_YEAR, r->year ); 
    cmoswb( CM_MONTH, r->month );
    cmoswb( CM_DAYMTH, r->daymth );
    cmoswb( CM_DAYWK, r->daywk );
    cmoswb( CM_HRALRM, r->hralrm );
    cmoswb( CM_HOURS, r->hours );
    cmoswb( CM_MINALRM, r->minalrm );
    cmoswb( CM_MINS, r->mins ); 
    cmoswb( CM_SECALRM, r->secalrm );
    cmoswb( CM_SECS, r->secs ); 
}


/* return seconds since Jan 1, 1970 */

time_t time(time_t * out)
{
    time_t curr_time;
    RTC_t t;

    /* wait for clock to be readable */
    if (cmosready() != STATUS_SUCCESS)		return 0;

    RTC_Get(&t);

    curr_time = (time_t) t.secs;
    curr_time += (time_t) (t.mins * 60);
    curr_time += (time_t) (t.hours * 60 * 60);
    curr_time += (time_t) ((t.daymth - 1) * 24 * 60 * 60);
    curr_time += (time_t) ((t.month - 1) * 31 * 24 * 60 * 60);
    curr_time += (time_t) ((t.year - 70) * 12 * 31 * 24 * 60 * 60);

    if (out != NULL)				*out = curr_time;
    return curr_time;
}


#define COM1BAUD_VALID_OK	69
#define COM1_MAXBAUD		115200

const String rtc_env_desc[ RTC_ENV_NVARS ] = {
    KEY_COM1_BAUD
};

static const unsigned com1_baud_list[] = {
    57600,
    38400,
    19200,
    9600,
    7200,
    4800,
    3600,
    2400,
    2000,
    1800,
    1200,
    600,
    300,
    150,
    134,
    110,
    75,
    50,

    /* Diags extension - this may not work with SRM console though...
     * Requires an SRM (eg SRM 5.8-65 or later) that knows this extension */
    115200
};

/* Lookup a string to see if it is a key in the CMOS environment.  If true,
 * return the CMOS NVRAM identifier which can then be used to access the
 * numerical value asssociated with this key
 */
 
int rtc_envmatch( const String key )
{
    int i;

    for ( i=0; i<RTC_ENV_NVARS; i++ )
    {
	if ( strcmp( key, rtc_env_desc[i] ) == 0 )
	    return i;
    }
    return RTC_ENV_NOKEY;
}


/* Read a CMOS byte used for holding an environment variable
 * The returned value may have variable-specific processing added to it
 */
int rtc_envrd( int var )
{
    int rval;
    uint8 rchar;

    if ( var < 0 || var >= RTC_ENV_NVARS )		/* input validation */
	return RTC_ENV_NOKEY;

    switch( var )
    {
	case RTC_COM1_BAUD:
	    if ( cmosrb( CM_COM1BAUD_VALID ) != COM1BAUD_VALID_OK )
		rval = -1;
	    else
	    {
		rchar = cmosrb( CM_COM1BAUD );
		if ( rchar >= ARRAYLEN( com1_baud_list ) )	/* paranoia */ 
		    rval = RTC_ENV_NOKEY;
		else 
		    rval = com1_baud_list[ rchar ];
	    }
	    break;

	default:
	    rval = RTC_ENV_NOKEY;		/* unknown CMOS variable */
	    break;
    }

    /* return the [interpreted] value of the CMOS byte (or -1 if error) */
    return rval;
}

void rtc_envwr( int var, int val )
{
    unsigned testval;
    int i;

    switch( var )
    {
	case RTC_COM1_BAUD:
	    if ( val == 0 )			/* paranoia */
		val = 9600;			/* a trusty default */

	    /* perform lookup to see if we have a recognised value */
	    for ( i=0; i< ARRAYLEN( com1_baud_list ); i++ )
	    {
		testval = com1_baud_list[ i ];

		/* If not this iteration, go on to next */
		if ( testval != val )
			continue;

		/* We have our match this time */
		cmoswb( CM_COM1BAUD, i );
		cmoswb( CM_COM1BAUD_VALID, COM1BAUD_VALID_OK );	/* marked OK */
		break;
	    }
	    break;

	default:			/* unknown CMOS variable */
	    break;
    }
}




/*--------------------------------------------------------------------*/
/* Convert the RTC into diags house format, broadly in keeping with 
 * all other Alpha firmware */

static unsigned char bcd2bin( const unsigned char bcdval )
{
    return (bcdval & 0xF) + ((bcdval >> 4) * 10);
}

void RTC_Canonicalize( void )
{
    unsigned char statusB;
    RTC_t t;
    unsigned char warn_dst=0, warn_24hr=0, warn_bcd=0, warn_hlt=0;
    String yearfmt;

    cmosready();		/* May fail if clock not ticking: that's OK */
    RTC_Get( &t );
    statusB = cmosrb( CM_STATUSB );

    if ( statusB & STATUSB_DST )
    {
 	warn_dst = 1;				/* do after writing back */
	statusB &= ~STATUSB_DST;		/* mask it out */
    }

    if ( (statusB & STATUSB_24HR) == 0 )
    {
	/* Fix the hours value (nb, may be BCD or binary here) */
 	warn_24hr = 1;				/* do after writing back */
	if ( statusB & STATUSB_BIN )		/* storage is binary */
	{
	    if ( t.hours & 0x80 )	t.hours = (t.hours & ~0x80) + 12;
	    if ( t.hralrm & 0x80 )	t.hralrm = (t.hralrm & ~0x80) + 12;
	} else {				/* storage is BCD */
	    if ( t.hours & 0x80 )	t.hours = (t.hours & ~0x80) + 0x12;
	    if ( t.hralrm & 0x80 )	t.hralrm = (t.hralrm & ~0x80) + 0x12;
	}
	statusB |= STATUSB_24HR;
    }

    if ( (statusB & STATUSB_BIN) == 0 ) 
    {
	/* note: any 24-hr hangups in the hour fields are catered for above */
 	warn_bcd = 1;				/* do after writing back */
	t.secs 		= bcd2bin( t.secs );
	t.secalrm 	= bcd2bin( t.secalrm );
	t.mins		= bcd2bin( t.mins );
	t.minalrm	= bcd2bin( t.minalrm );
	t.hours		= bcd2bin( t.hours );
	t.hralrm	= bcd2bin( t.hralrm );
	t.daymth	= bcd2bin( t.daymth );	/* no need to convert daywk */
	t.month		= bcd2bin( t.month );
	t.year		= bcd2bin( t.year );

	statusB |= STATUSB_BIN;
    }

    if ( statusB & STATUSB_HLT )
    {
 	warn_hlt = 1;				/* do after writing back */
	statusB &= ~STATUSB_HLT;
    }

    cmoswb( CM_STATUSB, statusB );
    RTC_Set( &t );			/* write home the changes */

    /* put out any warnings after writing back: 1 second read/write window */
    if( warn_dst )
	mobo_logf( LOG_WARN "CMOS: using defunct DST mode, correcting\n" );
    if( warn_24hr )
	mobo_logf( LOG_INFO "CMOS: converting clock to 24HR format\n" );
    if( warn_bcd )
	mobo_logf( LOG_WARN "CMOS: converting from BCD to binary storage\n" );
    if( warn_hlt )
	mobo_logf( LOG_WARN "CMOS: clock not ticking, starting it\n" );

    /* Guess whether the clock is in ARC or SRM year format */
    /* SRM console uses some kind of weird rolling epoch scheme */
    if (t.year >= 20 && t.year < 40)
    {
	rtc_epoch = EPOCH_ARC;
	yearfmt = "ARC";
    }
    else if (t.year >= 40 && t.year < 100)
    {
	rtc_epoch = EPOCH_SRM;
	yearfmt = "SRM";
    }
    else
    {
	rtc_epoch = 2000;
	yearfmt = "SRM (post 2000)";
    }
    
    mobo_logf( LOG_INFO "CMOS: %s console date format (epoch %d) detected\n",
		 yearfmt, rtc_epoch);

}


DBM_STATUS InitRTC(void)
{

    RTC_Canonicalize();			/* massage with any date format stuff */

/*
 * Initialize the real-time clock (RTC)
 *
 * Index into RTC Control Register A to set periodic interrupt rate
 * to 976.562 and set it running.
 */

    /* Enable and configure by setting this register */
    cmoswb(CM_STATUSA, CM_STATUSA__INIT);

/*
 * Index into RTC Control Register B and then enable periodic
 * interrupts, Daylight Savings (DSE), 24-Hour Mode, and Binary
 * mode.
 */

    cmoswb(CM_STATUSB, CM_STATUSB__INIT);

    /* Testing of the RTC is now moved into the ISA and CMOS diagnostics */
    /* See diagnostics/isadevs.c and diagnostics/cmos.c */

    cmosrb( CM_STATUSC );		/* ack and clear any stale interrupts */

    return STATUS_SUCCESS;
}

