/*
 *  linux/drivers/block/ide.c	Version 5.53  Jun  24, 1997
 *
 *  Copyright (C) 1994-1996  Linus Torvalds & authors (see below)
 */
#define _IDE_C		/* needed by <linux/blk.h> */

/*
 *  Maintained by Mark Lord  <mlord@pobox.com>
 *            and Gadi Oxman <gadio@netvision.net.il>
 *
 * This is the multiple IDE interface driver, as evolved from hd.c.
 * It supports up to four IDE interfaces, on one or more IRQs (usually 14 & 15).
 * There can be up to two drives per interface, as per the ATA-2 spec.
 *
 * Primary:    ide0, port 0x1f0; major=3;  hda is minor=0; hdb is minor=64
 * Secondary:  ide1, port 0x170; major=22; hdc is minor=0; hdd is minor=64
 * Tertiary:   ide2, port 0x???; major=33; hde is minor=0; hdf is minor=64
 * Quaternary: ide3, port 0x???; major=34; hdg is minor=0; hdh is minor=64
 *
 * It is easy to extend ide.c to handle more than four interfaces:
 *
 *	Change the MAX_HWIFS constant in ide.h.
 *
 *	Define some new major numbers (in major.h), and insert them into
 *	the ide_hwif_to_major table in ide.c.
 *
 *	Fill in the extra values for the new interfaces into the two tables
 *	inside ide.c:  default_io_base[]  and  default_irqs[].
 *
 *	Create the new request handlers by cloning "do_ide3_request()"
 *	for each new interface, and add them to the switch statement
 *	in the ide_init() function in ide.c.
 *
 *	Recompile, create the new /dev/ entries, and it will probably work.
 *
 *  From hd.c:
 *  |
 *  | It traverses the request-list, using interrupts to jump between functions.
 *  | As nearly all functions can be called within interrupts, we may not sleep.
 *  | Special care is recommended.  Have Fun!
 *  |
 *  | modified by Drew Eckhardt to check nr of hd's from the CMOS.
 *  |
 *  | Thanks to Branko Lankester, lankeste@fwi.uva.nl, who found a bug
 *  | in the early extended-partition checks and added DM partitions.
 *  |
 *  | Early work on error handling by Mika Liljeberg (liljeber@cs.Helsinki.FI).
 *  |
 *  | IRQ-unmask, drive-id, multiple-mode, support for ">16 heads",
 *  | and general streamlining by Mark Lord (mlord@pobox.com).
 *
 *  October, 1994 -- Complete line-by-line overhaul for linux 1.1.x, by:
 *
 *	Mark Lord	(mlord@pobox.com)		(IDE Perf.Pkg)
 *	Delman Lee	(delman@mipg.upenn.edu)		("Mr. atdisk2")
 *	Scott Snyder	(snyder@fnald0.fnal.gov)	(ATAPI IDE cd-rom)
 *
 *  This was a rewrite of just about everything from hd.c, though some original
 *  code is still sprinkled about.  Think of it as a major evolution, with
 *  inspiration from lots of linux users, esp.  hamish@zot.apana.org.au
 *
 *  Version 1.0 ALPHA	initial code, primary i/f working okay
 *  Version 1.3 BETA	dual i/f on shared irq tested & working!
 *  Version 1.4 BETA	added auto probing for irq(s)
 *  Version 1.5 BETA	added ALPHA (untested) support for IDE cd-roms,
 *  ...
 *  Version 3.5		correct the bios_cyl field if it's too small
 *  (linux 1.1.76)	 (to help fdisk with brain-dead BIOSs)
 *  Version 3.6		cosmetic corrections to comments and stuff
 *  (linux 1.1.77)	reorganise probing code to make it understandable
 *			added halfway retry to probing for drive identification
 *			added "hdx=noprobe" command line option
 *			allow setting multmode even when identification fails
 *  Version 3.7		move set_geometry=1 from do_identify() to ide_init()
 *			increase DRQ_WAIT to eliminate nuisance messages
 *			wait for DRQ_STAT instead of DATA_READY during probing
 *			  (courtesy of Gary Thomas gary@efland.UU.NET)
 *  Version 3.8		fixed byte-swapping for confused Mitsumi cdrom drives
 *			update of ide-cd.c from Scott, allows blocksize=1024
 *			cdrom probe fixes, inspired by jprang@uni-duisburg.de
 *  Version 3.9		don't use LBA if lba_capacity looks funny
 *			correct the drive capacity calculations
 *			fix probing for old Seagates without IDE_ALTSTATUS_REG
 *			fix byte-ordering for some NEC cdrom drives
 *  Version 3.10	disable multiple mode by default; was causing trouble
 *  Version 3.11	fix mis-identification of old WD disks as cdroms
 *  Version 3,12	simplify logic for selecting initial mult_count
 *			  (fixes problems with buggy WD drives)
 *  Version 3.13	remove excess "multiple mode disabled" messages
 *  Version 3.14	fix ide_error() handling of BUSY_STAT
 *			fix byte-swapped cdrom strings (again.. arghh!)
 *			ignore INDEX bit when checking the ALTSTATUS reg
 *  Version 3.15	add SINGLE_THREADED flag for use with dual-CMD i/f
 *			ignore WRERR_STAT for non-write operations
 *			added vlb_sync support for DC-2000A & others,
 *			 (incl. some Promise chips), courtesy of Frank Gockel
 *  Version 3.16	convert vlb_32bit and vlb_sync into runtime flags
 *			add ioctls to get/set VLB flags (HDIO_[SG]ET_CHIPSET)
 *			rename SINGLE_THREADED to SUPPORT_SERIALIZE,
 *			add boot flag to "serialize" operation for CMD i/f
 *			add optional support for DTC2278 interfaces,
 *			 courtesy of andy@cercle.cts.com (Dyan Wile).
 *			add boot flag to enable "dtc2278" probe
 *			add probe to avoid EATA (SCSI) interfaces,
 *			 courtesy of neuffer@goofy.zdv.uni-mainz.de.
 *  Version 4.00	tidy up verify_area() calls - heiko@colossus.escape.de
 *			add flag to ignore WRERR_STAT for some drives
 *			 courtesy of David.H.West@um.cc.umich.edu
 *			assembly syntax tweak to vlb_sync
 *			removable drive support from scuba@cs.tu-berlin.de
 *			add transparent support for DiskManager-6.0x "Dynamic
 *			 Disk Overlay" (DDO), most of this is in genhd.c
 *			eliminate "multiple mode turned off" message at boot
 *  Version 4.10	fix bug in ioctl for "hdparm -c3"
 *			fix DM6:DDO support -- now works with LILO, fdisk, ...
 *			don't treat some naughty WD drives as removable
 *  Version 4.11	updated DM6 support using info provided by OnTrack
 *  Version 5.00	major overhaul, multmode setting fixed, vlb_sync fixed
 *			added support for 3rd/4th/alternative IDE ports
 *			created ide.h; ide-cd.c now compiles separate from ide.c
 *			hopefully fixed infinite "unexpected_intr" from cdroms
 *			zillions of other changes and restructuring
 *			somehow reduced overall memory usage by several kB
 *			probably slowed things down slightly, but worth it
 *  Version 5.01	AT LAST!!  Finally understood why "unexpected_intr"
 *			 was happening at various times/places:  whenever the
 *			 ide-interface's ctl_port was used to "mask" the irq,
 *			 it also would trigger an edge in the process of masking
 *			 which would result in a self-inflicted interrupt!!
 *			 (such a stupid way to build a hardware interrupt mask).
 *			 This is now fixed (after a year of head-scratching).
 *  Version 5.02	got rid of need for {enable,disable}_irq_list()
 *  Version 5.03	tune-ups, comments, remove "busy wait" from drive resets
 *			removed PROBE_FOR_IRQS option -- no longer needed
 *			OOOPS!  fixed "bad access" bug for 2nd drive on an i/f
 *  Version 5.04	changed "ira %d" to "irq %d" in DEBUG message
 *			added more comments, cleaned up unexpected_intr()
 *			OOOPS!  fixed null pointer problem in ide reset code
 *			added autodetect for Triton chipset -- no effect yet
 *  Version 5.05	OOOPS!  fixed bug in revalidate_disk()
 *			OOOPS!  fixed bug in ide_do_request()
 *			added ATAPI reset sequence for cdroms
 *  Version 5.10	added Bus-Mastered DMA support for Triton Chipset
 *			some (mostly) cosmetic changes
 *  Version 5.11	added ht6560b support by malafoss@snakemail.hut.fi
 *			reworked PCI scanning code
 *			added automatic RZ1000 detection/support
 *			added automatic PCI CMD640 detection/support
 *			added option for VLB CMD640 support
 *			tweaked probe to find cdrom on hdb with disks on hda,hdc
 *  Version 5.12	some performance tuning
 *			added message to alert user to bad /dev/hd[cd] entries
 *			OOOPS!  fixed bug in atapi reset
 *			driver now forces "serialize" again for all cmd640 chips
 *			noticed REALLY_SLOW_IO had no effect, moved it to ide.c
 *			made do_drive_cmd() into public ide_do_drive_cmd()
 *  Version 5.13	fixed typo ('B'), thanks to houston@boyd.geog.mcgill.ca
 *			fixed ht6560b support
 *  Version 5.13b (sss)	fix problem in calling ide_cdrom_setup()
 *			don't bother invalidating nonexistent partitions
 *  Version 5.14	fixes to cmd640 support.. maybe it works now(?)
 *			added & tested full EZ-DRIVE support -- don't use LILO!
 *			don't enable 2nd CMD640 PCI port during init - conflict
 *  Version 5.15	bug fix in init_cmd640_vlb()
 *			bug fix in interrupt sharing code
 *  Version 5.16	ugh.. fix "serialize" support, broken in 5.15
 *			remove "Huh?" from cmd640 code
 *			added qd6580 interface speed select from Colten Edwards
 *  Version 5.17	kludge around bug in BIOS32 on Intel triton motherboards
 *  Version 5.18	new CMD640 code, moved to cmd640.c, #include'd for now
 *			new UMC8672 code, moved to umc8672.c, #include'd for now
 *			disallow turning on DMA when h/w not capable of DMA
 *  Version 5.19	fix potential infinite timeout on resets
 *			extend reset poll into a general purpose polling scheme
 *			add atapi tape drive support from Gadi Oxman
 *			simplify exit from _intr routines -- no IDE_DO_REQUEST
 *  Version 5.20	leave current rq on blkdev request list during I/O
 *			generalized ide_do_drive_cmd() for tape/cdrom driver use
 *  Version 5.21	fix nasty cdrom/tape bug (ide_preempt was messed up)
 *  Version 5.22	fix ide_xlate_1024() to work with/without drive->id
 *  Version 5.23	miscellaneous touch-ups
 *  Version 5.24	fix #if's for SUPPORT_CMD640
 *  Version 5.25	more touch-ups, fix cdrom resets, ...
 *			cmd640.c now configs/compiles separate from ide.c
 *  Version 5.26	keep_settings now maintains the using_dma flag
 *			fix [EZD] remap message to only output at boot time
 *			fix "bad /dev/ entry" message to say hdc, not hdc0
 *			fix ide_xlate_1024() to respect user specified CHS
 *			use CHS from partn table if it looks translated
 *			re-merged flags chipset,vlb_32bit,vlb_sync into io_32bit
 *			keep track of interface chipset type, when known
 *			add generic PIO mode "tuneproc" mechanism
 *			fix cmd640_vlb option
 *			fix ht6560b support (was completely broken)
 *			umc8672.c now configures/compiles separate from ide.c
 *			move dtc2278 support to dtc2278.c
 *			move ht6560b support to ht6560b.c
 *			move qd6580  support to qd6580.c
 *			add  ali14xx support in ali14xx.c
 * Version 5.27		add [no]autotune parameters to help cmd640
 *			move rz1000  support to rz1000.c
 * Version 5.28		#include "ide_modes.h"
 *			fix disallow_unmask: now per-interface "no_unmask" bit
 *			force io_32bit to be the same on drive pairs of dtc2278
 *			improved IDE tape error handling, and tape DMA support
 *			bugfix in ide_do_drive_cmd() for cdroms + serialize
 * Version 5.29		fixed non-IDE check for too many physical heads
 *			don't use LBA if capacity is smaller than CHS
 * Version 5.30		remove real_devices kludge, formerly used by genhd.c
 * Version 5.32		change "KB" to "kB"
 *			fix serialize (was broken in kernel 1.3.72)
 *			add support for "hdparm -I"
 *			use common code for disk/tape/cdrom IDE_DRIVE_CMDs
 *			add support for Promise DC4030VL caching card
 *			improved serialize support
 *			put partition check back into alphabetical order
 *			add config option for PCMCIA baggage
 *			try to make PCMCIA support safer to use
 *			improve security on ioctls(): all are suser() only
 * Version 5.33		improve handling of HDIO_DRIVE_CMDs that read data
 * Version 5.34		fix irq-sharing problem from 5.33
 *			fix cdrom ioctl problem from 5.33
 * Version 5.35		cosmetic changes
 *			fix cli() problem in try_to_identify()
 * Version 5.36		fixes to optional PCMCIA support
 * Version 5.37		don't use DMA when "noautotune" is specified
 * Version 5.37a (go)	fix shared irq probing (was broken in kernel 1.3.72)
 *			call unplug_device() from ide_do_drive_cmd()
 * Version 5.38		add "hdx=none" option, courtesy of Joel Maslak
 *			mask drive irq after use, if sharing with another hwif
 *			add code to help debug weird cmd640 problems
 * Version 5.39		fix horrible error in earlier irq sharing "fix"
 * Version 5.40		fix serialization -- was broken in 5.39
 *			help sharing by masking device irq after probing
 * Version 5.41		more fixes to irq sharing/serialize detection
 *			disable io_32bit by default on drive reset
 * Version 5.42		simplify irq-masking after probe
 *			fix NULL pointer deref in save_match()
 * Version 5.43		Ugh.. unexpected_intr is back: try to exterminate it
 * Version 5.44		Fix for "irq probe failed" on cmd640
 *			change path on message regarding MAKEDEV.ide
 *			add a throttle to the unexpected_intr() messages
 * Version 5.45		fix ugly parameter parsing bugs (thanks Derek)
 *			include Gadi's magic fix for cmd640 unexpected_intr
 *			include mc68000 patches from Geert Uytterhoeven
 *			add Gadi's fix for PCMCIA cdroms
 * Version 5.46		remove the mc68000 #ifdefs for 2.0.x
 * Version 5.47		fix set_tune race condition
 *			fix bug in earlier PCMCIA cdrom update
 * Version 5.48		if def'd, invoke CMD640_DUMP_REGS when irq probe fails
 *			lengthen the do_reset1() pulse, for laptops
 *			add idebus=xx parameter for cmd640 and ali chipsets
 *			no_unmask flag now per-drive instead of per-hwif
 *			fix tune_req so that it gets done immediately
 *			fix missing restore_flags() in ide_ioctl
 *			prevent use of io_32bit on cmd640 with no prefetch
 * Version 5.49		fix minor quirks in probing routines
 * Version 5.50		allow values as small as 20 for idebus=
 * Version 5.51		force non io_32bit in drive_cmd_intr()
 *			change delay_10ms() to delay_50ms() to fix problems
 * Version 5.52		fix incorrect invalidation of removable devices
 *			add "hdx=slow" command line option
 * Version 5.53		add ATAPI floppy drive support
 *			change default media for type 0 to floppy
 *			add support for Exabyte Nest
 *			add missing set_blocksize() in revalidate_disk()
 *			handle bad status bit sequencing in ide_wait_stat()
 *			support partition table translations with 255 heads
 *			probe all interfaces by default
 *			add probe for the i82371AB chipset
 *			acknowledge media change on removable drives
 *			add work-around for BMI drives
 *			remove "LBA" from boot messages
 * Version 5.53.1	add UDMA "CRC retry" support
 * Version 5.53.2	add Promise/33 auto-detection and DMA support
 *			fix MC_ERR handling
 *			fix mis-detection of NEC cdrom as floppy
 *			issue ATAPI reset and re-probe after "no response"
 *
 *  Some additional driver compile-time options are in ide.h
 *
 *  To do, in likely order of completion:
 *	- modify kernel to obtain BIOS geometry for drives on 2nd/3rd/4th i/f
*/

#undef REALLY_SLOW_IO		/* most systems can safely undef this */

#if 0
#include <linux/config.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/major.h>
#include <linux/blkdev.h>
#include <linux/errno.h>
#include <linux/hdreg.h>
#include <linux/genhd.h>
#include <linux/malloc.h>

#include <asm/byteorder.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/io.h>
#endif

/*SOOHOON*/
#define DEBUG
#if 0
#define DBG(X) 
#else
#define DBG(X) printk(X)
#endif
int printf_dbm(const char *f,...);
#define printk printf_dbm
#undef inb
#undef outb
#undef inw
#undef outw
#undef inl
#undef outl
#define inb inportb
#define inb_p inportb
#define outb(P,D) outportb((D), (P))
#define outb_p(P,D) outportb((D), (P))
#define inw inportw
#define outw(P,D) outportw((D), (P))
#define inl inportl
#define outl(P,D) outportl((D), (P))
unsigned int inportb(unsigned int);
unsigned int inportw(unsigned int);
unsigned int inportl(unsigned int);
void outportb(unsigned int,unsigned int);
void outportw(unsigned int,unsigned int);
void outportl(unsigned int,unsigned int);
void insb (unsigned long port, void *dst, unsigned long count);
void insw (unsigned long port, void *dst, unsigned long count);
void insl (unsigned long port, void *dst, unsigned long count);
void outsb (unsigned long port, const void *src, unsigned long count);
void outsw (unsigned long port, const void *src, unsigned long count);
void outsl (unsigned long port, const void *src, unsigned long count);

typedef unsigned char           u_char;
typedef unsigned short          u_short;
typedef unsigned int            u_int;
typedef unsigned long           u_long;

extern volatile unsigned long jiffies;  /* timer ticks a la linux kernel */

long int random(void);
void srandom(unsigned int seed);

void usleep(unsigned int usecs);
#define udelay usleep

#define IDE0_MAJOR      3
#define IDE1_MAJOR      22
#define IDE2_MAJOR      33
#define IDE3_MAJOR      34

#define HZ 1024

#define READ 0
#define WRITE 1
#define READA 2         /* read-ahead  - don't block if no resources */
#define WRITEA 3        /* write-ahead - don't block if no resources */

#ifndef NULL
#define NULL ((void *) 0)
#endif

#define MINORBITS       8
#define MINORMASK       ((1<<MINORBITS) - 1)

typedef unsigned short kdev_t;

#define MAJOR(dev)      ((dev) >> MINORBITS)
#define MINOR(dev)      ((dev) & MINORMASK)
#define HASHDEV(dev)    (dev)
#define NODEV           0
#define MKDEV(ma,mi)    (((ma) << MINORBITS) | (mi))

#define kmalloc(A,B) malloc(A)
#define kfree(A) free(A)

#ifdef CONFIG_PCI
#include <linux/bios32.h>
#include <linux/pci.h>
#endif /* CONFIG_PCI */

#include <linux/hdreg.h>
#include <linux/blk.h>
#include "linux/ide.h"
#include "linux/ide_modes.h"



static const byte       ide_hwif_to_major[MAX_HWIFS] = {IDE0_MAJOR, IDE1_MAJOR };
static const unsigned short default_io_base[MAX_HWIFS] = {0x1f0, 0x170};
static const byte	default_irqs[MAX_HWIFS]     = {14, 15};
static int	idebus_parameter; /* holds the "idebus=" parameter */
static int	system_bus_speed; /* holds what we think is VESA/PCI bus speed */

/*
 * This is declared extern in ide.h, for access by other IDE modules:
 */
ide_hwif_t	ide_hwifs[MAX_HWIFS];	/* master data repository */

#define SET_RECOVERY_TIMER(drive)

/*
 * Do not even *think* about calling this!
 */
static void init_hwif_data (unsigned int index)
{
	byte *p;
	unsigned int unit;
	ide_hwif_t *hwif = &ide_hwifs[index];

	/* bulk initialize hwif & drive info with zeros */
	p = ((byte *) hwif) + sizeof(ide_hwif_t);
	do {
		*--p = 0;
	} while (p > (byte *) hwif);

	/* fill in any non-zero initial values */
	hwif->index     = index;
	hwif->io_base	= default_io_base[index];
	hwif->ctl_port	= hwif->io_base ? hwif->io_base+0x206 : 0x000;
	hwif->major	= ide_hwif_to_major[index];
	hwif->name[0]	= 'i';
	hwif->name[1]	= 'd';
	hwif->name[2]	= 'e';
	hwif->name[3]	= '0' + index;
	for (unit = 0; unit < MAX_DRIVES; ++unit) {
		ide_drive_t *drive = &hwif->drives[unit];

		drive->select.all		= (unit<<4)|0xa0;
		drive->hwif			= hwif;
		drive->ctl			= 0x08;
		drive->ready_stat		= READY_STAT;
		drive->bad_wstat		= BAD_W_STAT;
		drive->special.b.recalibrate	= 1;
		drive->special.b.set_geometry	= 1;
		drive->name[0]			= 'h';
		drive->name[1]			= 'd';
		drive->name[2]			= 'a' + (index * MAX_DRIVES) + unit;
	}
}

/*
 * init_ide_data() sets reasonable default values into all fields
 * of all instances of the hwifs and drives, but only on the first call.
 * Subsequent calls have no effect (they don't wipe out anything).
 *
 * This routine is normally called at driver initialization time,
 * but may also be called MUCH earlier during kernel "command-line"
 * parameter processing.  As such, we cannot depend on any other parts
 * of the kernel (such as memory allocation) to be functioning yet.
 *
 * This is too bad, as otherwise we could dynamically allocate the
 * ide_drive_t structs as needed, rather than always consuming memory
 * for the max possible number (MAX_HWIFS * MAX_DRIVES) of them.
 */
#define MAGIC_COOKIE 0x12345678
static void init_ide_data (void)
{
	unsigned int index;
	static unsigned long magic_cookie = MAGIC_COOKIE;

	if (magic_cookie != MAGIC_COOKIE)
		return;		/* already initialized */
	magic_cookie = 0;

	for (index = 0; index < MAX_HWIFS; ++index)
		init_hwif_data(index);

	idebus_parameter = 0;
	system_bus_speed = 0;
}

/*
 * ide_system_bus_speed() returns what we think is the system VESA/PCI
 * bus speed (in Mhz).  This is used for calculating interface PIO timings.
 * The default is 40 for known PCI systems, 50 otherwise.
 * The "idebus=xx" parameter can be used to override this value.
 * The actual value to be used is computed/displayed the first time through.
 */
int ide_system_bus_speed (void)
{
	if (!system_bus_speed) {
		if (idebus_parameter)
			system_bus_speed = idebus_parameter;	/* user supplied value */
#ifdef CONFIG_PCI
		else if (pcibios_present())
			system_bus_speed = 40;	/* safe default value for PCI */
#endif /* CONFIG_PCI */
		else
			system_bus_speed = 50;	/* safe default value for VESA and PCI */
		printk("ide: Assuming %dMhz system bus speed for PIO modes; override with idebus=xx\n", system_bus_speed);
	}
	return system_bus_speed;
}

/*
 * This is used for most PIO data transfers *from* the IDE interface
 */
void ide_input_data (ide_drive_t *drive, void *buffer, unsigned int wcount)
{
	unsigned short io_base  = HWIF(drive)->io_base;
	unsigned short data_reg = io_base+IDE_DATA_OFFSET;
	byte io_32bit = drive->io_32bit;

	if (io_32bit) {
			insl(data_reg, buffer, wcount);
	} else {
			insw(data_reg, buffer, wcount<<1);
	}
}

/*
 * This is used for most PIO data transfers *to* the IDE interface
 */
void ide_output_data (ide_drive_t *drive, void *buffer, unsigned int wcount)
{
	unsigned short io_base  = HWIF(drive)->io_base;
	unsigned short data_reg = io_base+IDE_DATA_OFFSET;
	byte io_32bit = drive->io_32bit;

	if (io_32bit) {
			outsl(data_reg, buffer, wcount);
	} else {
			outsw(data_reg, buffer, wcount<<1);
	}
}

/*
 * This should get invoked any time we exit the driver to
 * wait for an interrupt response from a drive.  handler() points
 * at the appropriate code to handle the next interrupt, and a
 * timer is started to prevent us from waiting forever in case
 * something goes wrong (see the timer_expiry() handler later on).
 */
void ide_set_handler (ide_drive_t *drive, ide_handler_t *handler, unsigned int timeout)
{
printk("ide_set_handler\n");
}

/*
 * lba_capacity_is_ok() performs a sanity check on the claimed "lba_capacity"
 * value for this drive (from its reported identification information).
 *
 * Returns:	1 if lba_capacity looks sensible
 *		0 otherwise
 */
static int lba_capacity_is_ok (struct hd_driveid *id)
{
	unsigned long lba_sects   = id->lba_capacity;
	unsigned long chs_sects   = id->cyls * id->heads * id->sectors;
	unsigned long _10_percent = chs_sects / 10;

	/* very large drives (8GB+) may lie about the number of cylinders */
	if (id->cyls == 16383 && id->heads == 16 && id->sectors == 63 && lba_sects > chs_sects) {
		id->cyls = lba_sects / (16 * 63); /* correct cyls */
		return 1;	/* lba_capacity is our only option */
	}
	/* perform a rough sanity check on lba_sects:  within 10% is "okay" */
	if ((lba_sects - chs_sects) < _10_percent)
		return 1;	/* lba_capacity is good */

	/* some drives have the word order reversed */
	lba_sects = (lba_sects << 16) | (lba_sects >> 16);
	if ((lba_sects - chs_sects) < _10_percent) {
		id->lba_capacity = lba_sects;	/* fix it */
		return 1;	/* lba_capacity is (now) good */
	}
	return 0;	/* lba_capacity value is bad */
}

/*
 * current_capacity() returns the capacity (in sectors) of a drive
 * according to its current geometry/LBA settings.
 */
static unsigned long current_capacity (ide_drive_t  *drive)
{
	struct hd_driveid *id = drive->id;
	unsigned long capacity = drive->cyl * drive->head * drive->sect;

	if (!drive->present)
		return 0;
	if (drive->media != ide_disk)
		return 0x7fffffff;	/* cdrom or tape */
	drive->select.b.lba = 0;
	/* Determine capacity, and use LBA if the drive properly supports it */
	if (id != NULL && (id->capability & 2) && lba_capacity_is_ok(id)) {
		if (id->lba_capacity >= capacity) {
			drive->cyl = id->lba_capacity / (drive->head * drive->sect);
			capacity = id->lba_capacity;
			drive->select.b.lba = 1;
		}
	}
	return (capacity - drive->sect0);
}

#if 0 /*SOOHOON*/
static void do_reset1 (ide_drive_t *, int);		/* needed below */

#ifdef CONFIG_BLK_DEV_IDEATAPI
/*
 * atapi_reset_pollfunc() gets invoked to poll the interface for completion every 50ms
 * during an atapi drive reset operation. If the drive has not yet responded,
 * and we have not yet hit our maximum waiting time, then the timer is restarted
 * for another 50ms.
 */
static void atapi_reset_pollfunc (ide_drive_t *drive)
{
	ide_hwgroup_t *hwgroup = HWGROUP(drive);
	byte stat;

	OUT_BYTE (drive->select.all, IDE_SELECT_REG);
	udelay (10);

	if (OK_STAT(stat=GET_STAT(), 0, BUSY_STAT)) {
		printk("%s: ATAPI reset complete\n", drive->name);
	} else {
		if (jiffies < hwgroup->poll_timeout) {
			ide_set_handler (drive, &atapi_reset_pollfunc, HZ/20);
			return;	/* continue polling */
		}
		hwgroup->poll_timeout = 0;	/* end of polling */
		printk("%s: ATAPI reset timed-out, status=0x%02x\n", drive->name, stat);
		do_reset1 (drive, 1);	/* do it the old fashioned way */
		return;
	}
	hwgroup->poll_timeout = 0;	/* done polling */
}
#endif /* CONFIG_BLK_DEV_IDEATAPI */

/*
 * reset_pollfunc() gets invoked to poll the interface for completion every 50ms
 * during an ide reset operation. If the drives have not yet responded,
 * and we have not yet hit our maximum waiting time, then the timer is restarted
 * for another 50ms.
 */
static void reset_pollfunc (ide_drive_t *drive)
{
	ide_hwgroup_t *hwgroup = HWGROUP(drive);
	ide_hwif_t *hwif = HWIF(drive);
	byte tmp;

	if (!OK_STAT(tmp=GET_STAT(), 0, BUSY_STAT)) {
		if (jiffies < hwgroup->poll_timeout) {
			ide_set_handler (drive, &reset_pollfunc, HZ/20);
			return;	/* continue polling */
		}
		printk("%s: reset timed-out, status=0x%02x\n", hwif->name, tmp);
	} else  {
		printk("%s: reset: ", hwif->name);
		if ((tmp = GET_ERR()) == 1)
			printk("success\n");
		else {
#if FANCY_STATUS_DUMPS
			printk("master: ");
			switch (tmp & 0x7f) {
				case 1: printk("passed");
					break;
				case 2: printk("formatter device error");
					break;
				case 3: printk("sector buffer error");
					break;
				case 4: printk("ECC circuitry error");
					break;
				case 5: printk("controlling MPU error");
					break;
				default:printk("error (0x%02x?)", tmp);
			}
			if (tmp & 0x80)
				printk("; slave: failed");
			printk("\n");
#else
			printk("failed\n");
#endif /* FANCY_STATUS_DUMPS */
		}
	}
	hwgroup->poll_timeout = 0;	/* done polling */
}

/*
 * do_reset1() attempts to recover a confused drive by resetting it.
 * Unfortunately, resetting a disk drive actually resets all devices on
 * the same interface, so it can really be thought of as resetting the
 * interface rather than resetting the drive.
 *
 * ATAPI devices have their own reset mechanism which allows them to be
 * individually reset without clobbering other devices on the same interface.
 *
 * Unfortunately, the IDE interface does not generate an interrupt to let
 * us know when the reset operation has finished, so we must poll for this.
 * Equally poor, though, is the fact that this may a very long time to complete,
 * (up to 30 seconds worstcase).  So, instead of busy-waiting here for it,
 * we set a timer to poll at 50ms intervals.
 */
static void do_reset1 (ide_drive_t *drive, int  do_not_try_atapi)
{
	unsigned int unit;
	ide_hwif_t *hwif = HWIF(drive);
	ide_hwgroup_t *hwgroup = HWGROUP(drive);

#ifdef CONFIG_BLK_DEV_IDEATAPI
	/* For an ATAPI device, first try an ATAPI SRST. */
	if (drive->media != ide_disk) {
		if (!do_not_try_atapi) {
			if (!drive->keep_settings) {
				drive->unmask = 0;
				drive->io_32bit = 0;
			}
			OUT_BYTE (drive->select.all, IDE_SELECT_REG);
			udelay (20);
			OUT_BYTE (WIN_SRST, IDE_COMMAND_REG);
			hwgroup->poll_timeout = jiffies + WAIT_WORSTCASE;
			ide_set_handler (drive, &atapi_reset_pollfunc, HZ/20);
			return;
		}
	}
#endif /* CONFIG_BLK_DEV_IDEATAPI */

	/*
	 * First, reset any device state data we were maintaining
	 * for any of the drives on this interface.
	 */
	for (unit = 0; unit < MAX_DRIVES; ++unit) {
		ide_drive_t *rdrive = &hwif->drives[unit];
		rdrive->special.all = 0;
		rdrive->special.b.set_geometry = 1;
		rdrive->special.b.recalibrate  = 1;
		if (OK_TO_RESET_CONTROLLER)
			rdrive->mult_count = 0;
		if (!rdrive->keep_settings) {
			rdrive->mult_req = 0;
			rdrive->unmask = 0;
			rdrive->io_32bit = 0;
			if (rdrive->using_dma) {
				rdrive->using_dma = 0;
				printk("%s: disabled DMA\n", rdrive->name);
			}
		}
		if (rdrive->mult_req != rdrive->mult_count)
			rdrive->special.b.set_multmode = 1;
	}

#if OK_TO_RESET_CONTROLLER
	/*
	 * Note that we also set nIEN while resetting the device,
	 * to mask unwanted interrupts from the interface during the reset.
	 * However, due to the design of PC hardware, this will cause an
	 * immediate interrupt due to the edge transition it produces.
	 * This single interrupt gives us a "fast poll" for drives that
	 * recover from reset very quickly, saving us the first 50ms wait time.
	 */
	OUT_BYTE(drive->ctl|6,IDE_CONTROL_REG);	/* set SRST and nIEN */
	udelay(10);			/* more than enough time */
	OUT_BYTE(drive->ctl|2,IDE_CONTROL_REG);	/* clear SRST, leave nIEN */
	udelay(10);			/* more than enough time */
	hwgroup->poll_timeout = jiffies + WAIT_WORSTCASE;
	ide_set_handler (drive, &reset_pollfunc, HZ/20);
#endif	/* OK_TO_RESET_CONTROLLER */

}

/*
 * ide_do_reset() is the entry point to the drive/interface reset code.
 */
void ide_do_reset (ide_drive_t *drive)
{
	do_reset1 (drive, 0);
}
#endif /*0*/

/*
 * Error reporting, in human readable form (luxurious, but a memory hog).
 */
byte ide_dump_status (ide_drive_t *drive, const char *msg, byte stat)
{
	byte err = 0;

	printk("%s: %s: status=0x%02x", drive->name, msg, stat);
#if FANCY_STATUS_DUMPS
	if (drive->media == ide_disk) {
		printk(" { ");
		if (stat & BUSY_STAT)
			printk("Busy ");
		else {
			if (stat & READY_STAT)	printk("DriveReady ");
			if (stat & WRERR_STAT)	printk("DeviceFault ");
			if (stat & SEEK_STAT)	printk("SeekComplete ");
			if (stat & DRQ_STAT)	printk("DataRequest ");
			if (stat & ECC_STAT)	printk("CorrectedError ");
			if (stat & INDEX_STAT)	printk("Index ");
			if (stat & ERR_STAT)	printk("Error ");
		}
		printk("}");
	}
#endif	/* FANCY_STATUS_DUMPS */
	printk("\n");
	if ((stat & (BUSY_STAT|ERR_STAT)) == ERR_STAT) {
		err = GET_ERR();
		printk("%s: %s: error=0x%02x", drive->name, msg, err);
#if FANCY_STATUS_DUMPS
		if (drive->media == ide_disk) {
			printk(" { ");
			if (err & ICRC_ERR)	printk((err & ABRT_ERR) ? "BadCRC " : "BadSector ");
			if (err & ECC_ERR)	printk("UncorrectableError ");
			if (err & ID_ERR)	printk("SectorIdNotFound ");
			if (err & ABRT_ERR)	printk("DriveStatusError ");
			if (err & TRK0_ERR)	printk("TrackZeroNotFound ");
			if (err & MARK_ERR)	printk("AddrMarkNotFound ");
			printk("}");
			if (err & (BBD_ERR|ECC_ERR|ID_ERR|MARK_ERR)) {
				byte cur = IN_BYTE(IDE_SELECT_REG);
				if (cur & 0x40) {	/* using LBA? */
					printk(", LBAsect=%ld", (unsigned long)
					 ((cur&0xf)<<24)
					 |(IN_BYTE(IDE_HCYL_REG)<<16)
					 |(IN_BYTE(IDE_LCYL_REG)<<8)
					 | IN_BYTE(IDE_SECTOR_REG));
				} else {
					printk(", CHS=%d/%d/%d",
					 (IN_BYTE(IDE_HCYL_REG)<<8) +
					  IN_BYTE(IDE_LCYL_REG),
					  cur & 0xf,
					  IN_BYTE(IDE_SECTOR_REG));
				}
				if (drive->rq)
					printk(", sector=%ld", drive->rq->sector);
			}
		}
#endif	/* FANCY_STATUS_DUMPS */
		printk("\n");
	}
	return err;
}

/*
 * try_to_flush_leftover_data() is invoked in response to a drive
 * unexpectedly having its DRQ_STAT bit set.  As an alternative to
 * resetting the drive, this routine tries to clear the condition
 * by read a sector's worth of data from the drive.  Of course,
 * this may not help if the drive is *waiting* for data from *us*.
 */
static void try_to_flush_leftover_data (ide_drive_t *drive)
{
	int i = (drive->mult_count ? drive->mult_count : 1) * SECTOR_WORDS;

	while (i > 0) {
		unsigned long buffer[16];
		unsigned int wcount = (i > 16) ? 16 : i;
		i -= wcount;
		ide_input_data (drive, buffer, wcount);
	}
}

/*
 * ide_error() takes action based on the error returned by the controller.
 */
void ide_error (ide_drive_t *drive, const char *msg, byte stat)
{
	struct request *rq;
	byte err;

	err = ide_dump_status(drive, msg, stat);
	if ((rq = drive->rq) == NULL || drive == NULL)
		return;
	/* retry only "normal" I/O: */
	if (stat & BUSY_STAT) {		/* other bits are useless when BUSY */
		rq->errors |= ERROR_RESET;
	} else {
		if (drive->media == ide_disk && (stat & ERR_STAT)) {
			/* err has different meaning on cdrom and tape */
			if (err == ABRT_ERR) {
				if (drive->select.b.lba && IN_BYTE(IDE_COMMAND_REG) == WIN_SPECIFY)
					return;	/* some newer drives don't support WIN_SPECIFY */
			} else if ((err & (ABRT_ERR | ICRC_ERR)) == (ABRT_ERR | ICRC_ERR))
				; /* UDMA crc error -- just retry the operation */
			else if (err & (BBD_ERR | ECC_ERR))	/* retries won't help these */
				rq->errors = ERROR_MAX;
			else if (err & TRK0_ERR)	/* help it find track zero */
				rq->errors |= ERROR_RECAL;
			else if (err & MC_ERR)
				drive->special.b.mc = 1;
		}
		if ((stat & DRQ_STAT) && rq->cmd != WRITE)
			try_to_flush_leftover_data(drive);
	}
	if (GET_STAT() & (BUSY_STAT|DRQ_STAT))
		rq->errors |= ERROR_RESET;	/* Mmmm.. timing problem */

	if (rq->errors >= ERROR_MAX) {
 		drive->rq = drive->rq->next;
	}
	else {
		if ((rq->errors & ERROR_RESET) == ERROR_RESET) {
			++rq->errors;
printk("we need reset\n");
/*
			ide_do_reset(drive);
*/
			return;
		} else if ((rq->errors & ERROR_RECAL) == ERROR_RECAL)
			drive->special.b.recalibrate = 1;
		++rq->errors;
	}
}

/*
 * read_intr() is the handler for disk read/multread interrupts
 */
static void read_intr (ide_drive_t *drive)
{
	int i;
	unsigned int msect, nsect;
	struct request *rq;

do_more:
DBG("read_intr0\n");
	if (ide_wait_stat(drive,DATA_READY,BAD_R_STAT,WAIT_CMD)) {
DBG("read_intr1\n");
		return;
	}
	msect = drive->mult_count;
read_next:
	rq = drive->rq;
	if (msect) {
		if ((nsect = rq->current_nr_sectors) > msect)
			nsect = msect;
		msect -= nsect;
	} else
		nsect = 1;
	ide_input_data(drive, rq->buffer, nsect * SECTOR_WORDS);
#ifdef DEBUG
	printk("%s:  read: sectors(%ld-%ld), buffer=0x%08lx, remaining=%ld\n",
		drive->name, rq->sector, rq->sector+nsect-1,
		(unsigned long) rq->buffer+(nsect<<9), rq->nr_sectors-nsect);
#endif
	rq->sector += nsect;
	rq->buffer += nsect<<9;
	rq->errors = 0;
	i = (rq->nr_sectors -= nsect);
	if ((rq->current_nr_sectors -= nsect) <= 0) {
DBG("read_intr ide_end_req \n");
		drive->rq = drive->rq->next;
	}
	if (i > 0) {
		if (msect)
			goto read_next;
DBG("read_intr ide_set_hand again\n");
		goto do_more;
	}
DBG("end read_intr\n");
}

/*
 * write_intr() is the handler for disk write interrupts
 */
static void write_intr (ide_drive_t *drive)
{
	byte stat;
	int i;
	struct request *rq = drive->rq;

do_more:
	if (!ide_wait_stat(drive, DRIVE_READY,drive->bad_wstat,WAIT_CMD)) {
#ifdef DEBUG
		printk("%s: write: sector %ld, buffer=0x%08lx, remaining=%ld\n",
			drive->name, rq->sector, (unsigned long) rq->buffer,
			rq->nr_sectors-1);
#endif
		if ((rq->nr_sectors == 1) ^ ((stat & DRQ_STAT) != 0)) {
			rq->sector++;
			rq->buffer += 512;
			rq->errors = 0;
			i = --rq->nr_sectors;
			--rq->current_nr_sectors;
			if (rq->current_nr_sectors <= 0) 
				drive->rq = drive->rq->next;
			if (i > 0) {
				ide_output_data (drive, rq->buffer, SECTOR_WORDS);
				goto do_more;
			}
			return;
		}
	}
	ide_error(drive, "write_intr", stat);
}

/*
 * ide_multwrite() transfers a block of up to mcount sectors of data
 * to a drive as part of a disk multiple-sector write operation.
 */
void ide_multwrite (ide_drive_t *drive, unsigned int mcount)
{
	struct request *rq = &drive->wrq;

	do {
		unsigned int nsect = rq->current_nr_sectors;
		if (nsect > mcount)
			nsect = mcount;
		mcount -= nsect;

		ide_output_data(drive, rq->buffer, nsect<<7);
#ifdef DEBUG
		printk("%s: multwrite: sector %ld, buffer=0x%08lx, count=%d, remaining=%ld\n",
			drive->name, rq->sector, (unsigned long) rq->buffer,
			nsect, rq->nr_sectors - nsect);
#endif
		if ((rq->nr_sectors -= nsect) <= 0)
			break;
		if ((rq->current_nr_sectors -= nsect) == 0) {
/*
			if ((rq->bh = rq->bh->b_reqnext) != NULL) {
				rq->current_nr_sectors = rq->bh->b_size>>9;
				rq->buffer             = rq->bh->b_data;
			} else {
				panic("%s: buffer list corrupted\n", drive->name);
				break;
			}
*/
		} else {
			rq->buffer += nsect << 9;
		}
	} while (mcount);
}

/*
 * multwrite_intr() is the handler for disk multwrite interrupts
 */
static void multwrite_intr (ide_drive_t *drive)
{
	byte stat;
	int i;
	struct request *rq = &drive->wrq;

do_more:
	if (!ide_wait_stat(drive, DRIVE_READY,drive->bad_wstat,WAIT_CMD)) {
		if (stat & DRQ_STAT) {
			if (rq->nr_sectors) {
				ide_multwrite(drive, drive->mult_count);
				goto do_more;
				return;
			}
		} else {
			if (!rq->nr_sectors) {	/* all done? */
				rq = drive->rq;
				for (i = rq->nr_sectors; i > 0;){
					i -= rq->current_nr_sectors;
					drive->rq = drive->rq->next;
				}
				return;
			}
		}
	}
	ide_error(drive, "multwrite_intr", stat);
}

/*
 * Issue a simple drive command
 * The drive must be selected beforehand.
 */
static void ide_cmd(ide_drive_t *drive, byte cmd, byte nsect, ide_handler_t *handler)
{
	byte stat;

	OUT_BYTE(drive->ctl,IDE_CONTROL_REG);
	OUT_BYTE(nsect,IDE_NSECTOR_REG);
	OUT_BYTE(cmd,IDE_COMMAND_REG);

	if (ide_wait_stat(drive, READY_STAT,BAD_STAT,WAIT_CMD)) 
		return;

	switch(cmd) {
		case WIN_SETMULT: /*multimode*/
			if (OK_STAT(stat,READY_STAT,BAD_STAT)) {
				drive->mult_count = drive->mult_req;
			} else {
				drive->mult_req = drive->mult_count = 0;
				drive->special.b.recalibrate = 1;
				ide_dump_status(drive, "set_multmode", stat);
			}
			break;
		case WIN_SPECIFY: /*geometry*/
			if (!OK_STAT(stat,READY_STAT,BAD_STAT))
				ide_error(drive, "set_geometry_intr", stat);
			break;
		case WIN_RESTORE: /*recalibrate*/
			if (!OK_STAT(stat,READY_STAT,BAD_STAT))
				ide_error(drive, "recal_intr", stat);
			break;
		case WIN_ACKMC: /*mc*/
			if (!OK_STAT(stat,READY_STAT,BAD_STAT))
				ide_error(drive, "mc_intr", stat);
			drive->special.b.mc = 0;
			break;
		default: printk("ide_cmd : impossible cmd %d\n",cmd);
	}
		
}

/*
 * do_special() is used to issue WIN_SPECIFY, WIN_RESTORE, and WIN_SETMULT
 * commands to a drive.  It used to do much more, but has been scaled back.
 */
static void do_special (ide_drive_t *drive)
{
	special_t *s = &drive->special;

#ifdef DEBUG
	printk("%s: do_special: 0x%02x\n", drive->name, s->all);
#endif
	if (s->b.set_geometry) {
		s->b.set_geometry = 0;
		if (drive->media == ide_disk && !drive->no_geom) {
			OUT_BYTE(drive->sect,IDE_SECTOR_REG);
			OUT_BYTE(drive->cyl,IDE_LCYL_REG);
			OUT_BYTE(drive->cyl>>8,IDE_HCYL_REG);
			OUT_BYTE(((drive->head-1)|drive->select.all)&0xBF,IDE_SELECT_REG);
			ide_cmd(drive, WIN_SPECIFY, drive->sect, 0);
		}
	} else if (s->b.recalibrate) {
		s->b.recalibrate = 0;
		if (drive->media == ide_disk)
			ide_cmd(drive, WIN_RESTORE, drive->sect, 0);
	} else if (s->b.set_tune) {
		ide_tuneproc_t *tuneproc = HWIF(drive)->tuneproc;
		s->b.set_tune = 0;
		if (tuneproc != NULL)
			tuneproc(drive, drive->tune_req);
	} else if (s->b.set_multmode) {
		s->b.set_multmode = 0;
		if (drive->media == ide_disk) {
			if (drive->id && drive->mult_req > drive->id->max_multsect)
				drive->mult_req = drive->id->max_multsect;
			ide_cmd(drive, WIN_SETMULT, drive->mult_req, 0);
		} else
			drive->mult_req = 0;
	} else if (s->b.mc) {
		s->b.mc = 0;
		if (drive->media == ide_disk)
			ide_cmd(drive, WIN_ACKMC, drive->sect, 0);
	} else if (s->all) {
		int special = s->all;
		s->all = 0;
		printk("%s: bad special flag: 0x%02x\n", drive->name, special);
	}
}

/*
 * This routine busy-waits for the drive status to be not "busy".
 * It then checks the status for all of the "good" bits and none
 * of the "bad" bits, and if all is okay it returns 0.  All other
 * cases return 1 after invoking ide_error() -- caller should just return.
 *
 * This routine should get fixed to not hog the cpu during extra long waits..
 * That could be done by busy-waiting for the first jiffy or two, and then
 * setting a timer to wake up at half second intervals thereafter,
 * until timeout is achieved, before timing out.
 */
int ide_wait_stat (ide_drive_t *drive, byte good, byte bad, unsigned long timeout)
{
	byte stat;

	udelay(1);	/* spec allows drive 400ns to assert "BUSY" */
	if ((stat = GET_STAT()) & BUSY_STAT) {
		timeout += jiffies;
		while ((stat = GET_STAT()) & BUSY_STAT) {
			if (jiffies > timeout) {
				ide_error(drive, "status timeout", stat);
				return 1;
			}
		}
	}
	udelay(1);	/* allow status to settle, then read it again */
	if (OK_STAT((stat = GET_STAT()), good, bad))
		return 0;
	ide_error(drive, "status error", stat);
	return 1;
}

/*
 * do_rw_disk() issues READ and WRITE commands to a disk,
 * using LBA if supported, or CHS otherwise, to address sectors.
 * It also takes care of issuing special DRIVE_CMDs.
 */
void do_rw_disk (ide_drive_t *drive, struct request *rq, unsigned long block)
{
	ide_hwif_t *hwif = HWIF(drive);
	unsigned short io_base = hwif->io_base;

	OUT_BYTE(drive->ctl,IDE_CONTROL_REG);
	OUT_BYTE(rq->nr_sectors,io_base+IDE_NSECTOR_OFFSET);
	if (drive->select.b.lba) {
#ifdef DEBUG
		printk("%s: %sing: LBAsect=%ld, sectors=%ld, buffer=0x%08lx\n",
			drive->name, (rq->cmd==READ)?"read":"writ",
			block, rq->nr_sectors, (unsigned long) rq->buffer);
#endif
		OUT_BYTE(block,io_base+IDE_SECTOR_OFFSET);
		OUT_BYTE(block>>=8,io_base+IDE_LCYL_OFFSET);
		OUT_BYTE(block>>=8,io_base+IDE_HCYL_OFFSET);
		OUT_BYTE(((block>>8)&0x0f)|drive->select.all,io_base+IDE_SELECT_OFFSET);
	} else {
		unsigned int sect,head,cyl,track;
		track = block / drive->sect;
		sect  = block % drive->sect + 1;
		OUT_BYTE(sect,io_base+IDE_SECTOR_OFFSET);
		head  = track % drive->head;
		cyl   = track / drive->head;
		OUT_BYTE(cyl,io_base+IDE_LCYL_OFFSET);
		OUT_BYTE(cyl>>8,io_base+IDE_HCYL_OFFSET);
		OUT_BYTE(head|drive->select.all,io_base+IDE_SELECT_OFFSET);
#ifdef DEBUG
		printk("%s: %sing: CHS=%d/%d/%d, sectors=%ld, buffer=0x%08lx\n",
			drive->name, (rq->cmd==READ)?"read":"writ", cyl,
			head, sect, rq->nr_sectors, (unsigned long) rq->buffer);
#endif
	}
	if (rq->cmd == READ) {
#ifdef CONFIG_BLK_DEV_TRITON
                if (drive->using_dma && !(HWIF(drive)->dmaproc(ide_dma_read, drive)))
                        return;
#endif /* CONFIG_BLK_DEV_TRITON */

		OUT_BYTE(drive->mult_count ? WIN_MULTREAD : WIN_READ, io_base+IDE_COMMAND_OFFSET);
		read_intr(drive);
		return;
	}
	if (rq->cmd == WRITE) {
#ifdef CONFIG_BLK_DEV_TRITON
                if (drive->using_dma && !(HWIF(drive)->dmaproc(ide_dma_write, drive)))
                        return;
#endif /* CONFIG_BLK_DEV_TRITON */

		OUT_BYTE(drive->mult_count ? WIN_MULTWRITE : WIN_WRITE, io_base+IDE_COMMAND_OFFSET);
		if (ide_wait_stat(drive, DATA_READY, drive->bad_wstat, WAIT_DRQ)) {
			printk("%s: no DRQ after issuing %s\n", drive->name,
				drive->mult_count ? "MULTWRITE" : "WRITE");
			return;
		}
		if (drive->mult_count) {
			drive->wrq = *rq; /* scratchpad */
			ide_multwrite(drive, drive->mult_count);
			multwrite_intr(drive);
		} else {
			ide_output_data(drive, rq->buffer, SECTOR_WORDS);
			write_intr(drive);
		}
		return;
	}
	printk("%s: bad command: %d\n", drive->name, rq->cmd);
	drive->rq = drive->rq->next;
}

/*
 * do_request() initiates handling of a new I/O request
 */
static void do_request (ide_hwif_t *hwif, struct request *rq)
{
	unsigned int minor, unit;
	unsigned long block, blockend;
	ide_drive_t *drive;
	struct request *trq;

#ifdef DEBUG
	printk("%s: do_request: current=0x%08lx\n", hwif->name, (unsigned long) rq);
#endif
	minor = MINOR(rq->rq_dev);
	unit = minor >> PARTN_BITS;
	if (unit >= MAX_DRIVES) {
		printk("%s: bad device number: %d\n",
		       hwif->name, rq->rq_dev);
		goto kill_rq;
	}
	drive = &hwif->drives[unit];
	if (drive->rq == NULL) drive->rq = rq;
	else {
		trq = drive->rq;
		while (trq->next)
			trq = trq->next;
		trq->next = rq;
	}
do_more:
	block    = rq->sector;
	blockend = block + rq->nr_sectors;
	if ((blockend < block) || (blockend > drive->nr_sects)) {
		drive->nr_sects = current_capacity(drive);
		if ((blockend < block) || (blockend > drive->nr_sects)) {
			printk("%s%c: bad access: block=%ld, count=%ld %ld\n", drive->name,
			 (minor&PARTN_MASK)?'0'+(minor&PARTN_MASK):' ', block, rq->nr_sectors, drive->nr_sects);
			goto kill_rq;
		}
	}
	block += drive->sect0;

	SELECT_DRIVE(hwif,drive);
	if (ide_wait_stat(drive, drive->ready_stat, BUSY_STAT|DRQ_STAT, WAIT_READY)) {
		printk("%s: drive not ready for command\n", drive->name);
		return;
	}
	
	if (!drive->special.all) {
		switch (drive->media) {
			case ide_disk:
				do_rw_disk (drive, rq, block);
				if (drive->rq) {
					printk("do_req 0 is this sane?\n");
					goto do_more;
				}
				return;
#ifdef CONFIG_BLK_DEV_IDECD
			case ide_cdrom:
				ide_do_rw_cdrom (drive, block);
				return;
#endif /* CONFIG_BLK_DEV_IDECD */
			default:
				printk("%s: media type %d not supported\n",
					drive->name, drive->media);
				goto kill_rq;
		}
	}
	do_special(drive);
	if (drive->rq) {
		goto do_more;
	}
	return;
kill_rq:
	drive->rq = drive->rq->next;
	if (drive->rq)
		printk("do_req 2 is this sane?\n");
}

/*
 * This function is intended to be used prior to invoking ide_do_drive_cmd().
 */
void ide_init_drive_cmd (struct request *rq)
{
	rq->buffer = NULL;
	rq->cmd = IDE_DRIVE_CMD;
	rq->sector = 0;
	rq->nr_sectors = 0;
	rq->current_nr_sectors = 0;
	rq->next = NULL;

#if 0	/* these are done each time through ide_do_drive_cmd() */
	rq->errors = 0;
	rq->rq_status = RQ_ACTIVE;
	rq->rq_dev = ????;
#endif
}

static unsigned short * ntohs(unsigned short *pp)
{
        unsigned char t;

        t = *(char *)pp;
        *((char *)pp) = *(((char *)pp)+1) ;
        *(((char *)pp)+1) = t;
        return pp;
}

void ide_fixstring (byte *s, const int bytecount, const int byteswap)
{
	byte *p = s, *end = &s[bytecount & ~1]; /* bytecount must be even */

	if (byteswap) {
		/* convert from big-endian to host byte order */
		for (p = end ; p != s;) {
			unsigned short *pp = (unsigned short *) (p -= 2);
			*pp = ntohs(*pp);
		}
	}

	/* strip leading blanks */
	while (s != end && *s == ' ')
		++s;

	/* compress internal blanks and strip trailing blanks */
	while (s != end && *s) {
		if (*s++ != ' ' || (s != end && *s && *s != ' '))
			*p++ = *(s-1);
	}

	/* wipe out trailing garbage */
	while (p != end)
		*p++ = '\0';
}

static void do_identify (ide_drive_t *drive, byte cmd)
{
	int bswap;
	struct hd_driveid *id;
	unsigned long capacity, check;

	id = drive->id = kmalloc (SECTOR_WORDS*4, 0);
	ide_input_data(drive, id, SECTOR_WORDS);/* read 512 bytes of id info */

	/*
	 *  WIN_IDENTIFY returns little-endian info,
	 *  WIN_PIDENTIFY *usually* returns little-endian info.
	 */
	bswap = 1;
	if (cmd == WIN_PIDENTIFY) {
		if ((id->model[0] == 'N' && id->model[1] == 'E') /* NEC */
		 || (id->model[0] == 'F' && id->model[1] == 'X') /* Mitsumi */
		 || (id->model[0] == 'P' && id->model[1] == 'i'))/* Pioneer */
			bswap = 0;	/* Vertos drives may still be weird */
	}
	ide_fixstring (id->model,     sizeof(id->model),     bswap);
	ide_fixstring (id->fw_rev,    sizeof(id->fw_rev),    bswap);
	ide_fixstring (id->serial_no, sizeof(id->serial_no), bswap);

#ifdef CONFIG_BLK_DEV_IDEATAPI
	/*
	 * Check for an ATAPI device
	 */
	if (cmd == WIN_PIDENTIFY) {
		byte type = (id->config >> 8) & 0x1f;
		printk("%s: %s, ATAPI ", drive->name, id->model);
		if (!drive->ide_scsi) switch (type) {
			case 0:
				if (!strstr(id->model, "oppy") && !strstr(id->model, "poyp") && !strstr(id->model, "ZIP"))
					printk("cdrom or floppy?, assuming ");
				if (drive->media != ide_cdrom && !strstr(id->model, "CD-ROM")) {
					printk("FLOPPY ");
					break;
				}
				/* Early cdrom models used zero */
			case 5:
#ifdef CONFIG_BLK_DEV_IDECD
				printk ("CDROM drive\n");
				drive->media = ide_cdrom;
 				drive->present = 1;
				drive->removable = 1;
				return;
#else
				printk ("CDROM ");
				break;
#endif /* CONFIG_BLK_DEV_IDECD */
			case 1:
				printk ("TAPE ");
				break;
			default:
				drive->present = 0;
				printk("Type %d - Unknown device\n", type);
				return;
		}
		drive->present = 0;
		printk("- not supported by this kernel\n");
		return;
	}
#endif /* CONFIG_BLK_DEV_IDEATAPI */

	drive->media = ide_disk;
	/* Extract geometry if we did not already have one for the drive */
	if (!drive->present) {
		drive->present = 1;
		drive->cyl     = drive->bios_cyl  = id->cyls;
		drive->head    = drive->bios_head = id->heads;
		drive->sect    = drive->bios_sect = id->sectors;
	}
	/* Handle logical geometry translation by the drive */
	if ((id->field_valid & 1) && id->cur_cyls && id->cur_heads
	 && (id->cur_heads <= 16) && id->cur_sectors)
	{
		/*
		 * Extract the physical drive geometry for our use.
		 * Note that we purposely do *not* update the bios info.
		 * This way, programs that use it (like fdisk) will
		 * still have the same logical view as the BIOS does,
		 * which keeps the partition table from being screwed.
		 *
		 * An exception to this is the cylinder count,
		 * which we reexamine later on to correct for 1024 limitations.
		 */
		drive->cyl  = id->cur_cyls;
		drive->head = id->cur_heads;
		drive->sect = id->cur_sectors;

		/* check for word-swapped "capacity" field in id information */
		capacity = drive->cyl * drive->head * drive->sect;
		check = (id->cur_capacity0 << 16) | id->cur_capacity1;
		if (check == capacity) {	/* was it swapped? */
			/* yes, bring it into little-endian order: */
			id->cur_capacity0 = (capacity >>  0) & 0xffff;
			id->cur_capacity1 = (capacity >> 16) & 0xffff;
		}
	}
	/* Use physical geometry if what we have still makes no sense */
	if ((!drive->head || drive->head > 16) && id->heads && id->heads <= 16) {
		drive->cyl  = id->cyls;
		drive->head = id->heads;
		drive->sect = id->sectors;
	}

	/* calculate drive capacity, and select LBA if possible */
	(void) current_capacity (drive);

	/* Correct the number of cyls if the bios value is too small */
	if (drive->sect == drive->bios_sect && drive->head == drive->bios_head) {
		if (drive->cyl > drive->bios_cyl)
			drive->bios_cyl = drive->cyl;
	}

	if (!strncmp(id->model, "BMI ", 4) &&
	    strstr(id->model, " ENHANCED IDE ") &&
	    drive->select.b.lba)
		drive->no_geom = 1;

	printk ("%s: %.40s, %ldMB w/%dkB Cache, CHS=%d/%d/%d",
	 drive->name, id->model, current_capacity(drive)/2048L, id->buf_size/2,
	 drive->bios_cyl, drive->bios_head, drive->bios_sect);

	drive->mult_count = 0;
	if (id->max_multsect) {
		drive->mult_req = INITIAL_MULT_COUNT;
		if (drive->mult_req > id->max_multsect)
			drive->mult_req = id->max_multsect;
		if (drive->mult_req || ((id->multsect_valid & 1) && id->multsect))
			drive->special.b.set_multmode = 1;
	}

	if (drive->autotune != 2 && HWIF(drive)->dmaproc != NULL) {
		if (!(HWIF(drive)->dmaproc(ide_dma_check, drive))) {
			if ((id->field_valid & 4) && (id->dma_ultra & (id->dma_ultra >> 8) & 7))
				printk(", UDMA");
			else
				printk(", DMA");
		}
	}
	printk("\n");
}

/*
 * Delay for *at least* 50ms.  As we don't know how much time is left
 * until the next tick occurs, we wait an extra tick to be safe.
 * This is used only during the probing/polling for drives at boot time.
 */
static void delay_50ms (void)
{
	unsigned long timer = jiffies + ((HZ + 19)/20) + 1;
	while (timer > jiffies);
}
/*
 * try_to_identify() sends an ATA(PI) IDENTIFY request to a drive
 * and waits for a response.  It also monitors irqs while this is
 * happening, in hope of automatically determining which one is
 * being used by the interface.
 *
 * Returns:	0  device was identified
 *		1  device timed-out (no response to identify request)
 *		2  device aborted the command (refused to identify itself)
 */
static int try_to_identify (ide_drive_t *drive, byte cmd)
{
	int hd_status, rc;
	unsigned long timeout;

DBG("try_to_iden 0\n");
	OUT_BYTE(drive->ctl | 2,IDE_CONTROL_REG);	/* disable device irq */

	delay_50ms();				/* take a deep breath */
	if ((IN_BYTE(IDE_ALTSTATUS_REG) ^ IN_BYTE(IDE_STATUS_REG)) & ~INDEX_STAT) {
		printk("%s: probing with STATUS instead of ALTSTATUS\n", drive->name);
		hd_status = IDE_STATUS_REG;	/* ancient Seagate drives */
	} else
		hd_status = IDE_ALTSTATUS_REG;	/* use non-intrusive polling */

	OUT_BYTE(cmd,IDE_COMMAND_REG);		/* ask drive for ID */
	timeout = ((cmd == WIN_IDENTIFY) ? WAIT_WORSTCASE : WAIT_PIDENTIFY) / 2;
	timeout += jiffies;
	do {
		if (jiffies > timeout) {
			return 1;	/* drive timed-out */
		}
		delay_50ms();		/* give drive a breather */
	} while (IN_BYTE(hd_status) & BUSY_STAT);

	delay_50ms();		/* wait for IRQ and DRQ_STAT */
	if (OK_STAT(GET_STAT(),DRQ_STAT,BAD_R_STAT)) {
		do_identify(drive, cmd); /* drive returned ID */
		rc = 0;			/* drive responded with ID */
		(void) GET_STAT();	/* clear drive IRQ */
	} else
		rc = 2;			/* drive refused ID */
	return rc;
}

/*
 * do_probe() has the difficult job of finding a drive if it exists,
 * without getting hung up if it doesn't exist, without trampling on
 * ethernet cards, and without leaving any IRQs dangling to haunt us later.
 *
 * If a drive is "known" to exist (from CMOS or kernel parameters),
 * but does not respond right away, the probe will "hang in there"
 * for the maximum wait time (about 30 seconds), otherwise it will
 * exit much more quickly.
 *
 * Returns:	0  device was identified
 *		1  device timed-out (no response to identify request)
 *		2  device aborted the command (refused to identify itself)
 *		3  bad status from device (possible for ATAPI drives)
 *		4  probe was not attempted because failure was obvious
 */
static int do_probe (ide_drive_t *drive, byte cmd)
{
	int rc;
	ide_hwif_t *hwif = HWIF(drive);
	unsigned long timeout;
#ifdef CONFIG_BLK_DEV_IDEATAPI
	if (drive->present) {	/* avoid waiting for inappropriate probes */
		if ((drive->media != ide_disk) && (cmd == WIN_IDENTIFY))
			return 4;
	}
#endif	/* CONFIG_BLK_DEV_IDEATAPI */
#ifdef DEBUG
	printk("probing for %s: present=%d, media=%d, probetype=%s\n",
		drive->name, drive->present, drive->media,
		(cmd == WIN_IDENTIFY) ? "ATA" : "ATAPI");
#endif

	SELECT_DRIVE(hwif,drive);
DBG("do_probe 0\n");
	delay_50ms();
	if (IN_BYTE(IDE_SELECT_REG) != drive->select.all && !drive->present) {
		OUT_BYTE(0xa0,IDE_SELECT_REG);	/* exit with drive0 selected */
		delay_50ms();		/* allow BUSY_STAT to assert & clear */
		return 3;    /* no i/f present: avoid killing ethernet cards */
	}

	if (OK_STAT(GET_STAT(),READY_STAT,BUSY_STAT)
	 || drive->present || cmd == WIN_PIDENTIFY)
	{
		if ((rc = try_to_identify(drive,cmd)))   /* send cmd and wait */
			rc = try_to_identify(drive,cmd); /* failed: try again */
		if (rc == 1 && cmd == WIN_PIDENTIFY && drive->autotune != 2) {
			printk("%s: no response (status = 0x%02x), resetting drive\n", drive->name, GET_STAT());
			delay_50ms();
			OUT_BYTE (drive->select.all, IDE_SELECT_REG);
			delay_50ms();
			OUT_BYTE(WIN_SRST, IDE_COMMAND_REG);
			timeout = jiffies;
			while ((GET_STAT() & BUSY_STAT) && jiffies < timeout + WAIT_WORSTCASE)
				delay_50ms();
			rc = try_to_identify(drive, cmd);
		}
		if (rc == 1)
			printk("%s: no response (status = 0x%02x)\n", drive->name, GET_STAT());
		(void) GET_STAT();		/* ensure drive irq is clear */
	} else {
		rc = 3;				/* not present or maybe ATAPI */
	}
	if (drive->select.b.unit != 0) {
		OUT_BYTE(0xa0,IDE_SELECT_REG);	/* exit with drive0 selected */
		delay_50ms();
		(void) GET_STAT();		/* ensure drive irq is clear */
	}
	return rc;
}

/*
 * probe_for_drive() tests for existence of a given drive using do_probe().
 *
 * Returns:	0  no device was found
 *		1  device was found (note: drive->present might still be 0)
 */
static byte probe_for_drive (ide_drive_t *drive)
{
	if (drive->noprobe)			/* skip probing? */
		return drive->present;
	if (do_probe(drive, WIN_IDENTIFY) >= 2) { /* if !(success||timed-out) */
#ifdef CONFIG_BLK_DEV_IDEATAPI
		(void) do_probe(drive, WIN_PIDENTIFY); /* look for ATAPI device */
#endif	/* CONFIG_BLK_DEV_IDEATAPI */
	}
	if (!drive->present)
		return 0;			/* drive not found */
	if (drive->id == NULL) {		/* identification failed? */
		if (drive->media == ide_disk) {
			printk ("%s: non-IDE drive, CHS=%d/%d/%d\n",
			 drive->name, drive->cyl, drive->head, drive->sect);
		}
#ifdef CONFIG_BLK_DEV_IDECD
		else if (drive->media == ide_cdrom) {
			printk("%s: ATAPI cdrom (?)\n", drive->name);
		}
#endif	/* CONFIG_BLK_DEV_IDECD */
		else {
			drive->present = 0;	/* nuke it */
		}
	}
	return 1;	/* drive was found */
}

/*
 * This routine only knows how to look for drive units 0 and 1
 * on an interface, so any setting of MAX_DRIVES > 2 won't work here.
 */

#ifdef CONFIG_BLK_DEV_TRITON
#include "triton.c"
#endif

static void probe_hwif (ide_hwif_t *hwif)
{
	unsigned int unit;

	/*
	 * Second drive should only exist if first drive was found,
	 * but a lot of cdrom drives are configured as single slaves.
	 */
	for (unit = 0; unit < MAX_DRIVES; ++unit) {
		ide_drive_t *drive = &hwif->drives[unit];
#ifdef CONFIG_BLK_DEV_TRITON
		init_triton_dma (hwif);
#endif
		(void) probe_for_drive (drive);
		if (drive->present && drive->media == ide_disk) {
			if ((!drive->head || drive->head > 16) && !drive->select.b.lba) {
				printk("%s: INVALID GEOMETRY: %d PHYSICAL HEADS?\n",
				 drive->name, drive->head);
				drive->present = 0;
			}
		}
	}
}

/*
 * This is gets invoked once during initialization, to set *everything* up
 */
int ide_init (void)
{
	int index;

DBG("ide_init\n");
	init_ide_data ();

#ifdef CONFIG_BLK_DEV_TRITON
	ide_init_alim15x3 ();
#endif

DBG("probe_hwif\n");
	for (index = 0; index < MAX_HWIFS; ++index)
		probe_hwif (&ide_hwifs[index]);
DBG("ide_init end\n");

	return 0;
}



/*SUPPORT*/

unsigned long loops_per_sec = (1<<12);

int pcibios_present (void)
{
	return 0;
}

#if 0
/*
 * Read COUNT 8-bit bytes from port PORT into memory starting at
 * SRC.
 */
void insb (unsigned long port, void *dst, unsigned long count)
{
	while (((unsigned long)dst) & 0x3) {
		if (!count)
			return;
		count--;
		*(unsigned char *) dst = inb(port);
		((unsigned char *) dst)++;
	}

	while (count >= 4) {
		unsigned int w;
		count -= 4;
		w = inb(port);
		w |= inb(port) << 8;
		w |= inb(port) << 16;
		w |= inb(port) << 24;
		*(unsigned int *) dst = w;
		((unsigned int *) dst)++;
	}

	while (count) {
		--count;
		*(unsigned char *) dst = inb(port);
		((unsigned char *) dst)++;
	}
}
#endif


/*
 * Read COUNT 16-bit words from port PORT into memory starting at
 * SRC.  SRC must be at least short aligned.  This is used by the
 * IDE driver to read disk sectors.  Performance is important, but
 * the interfaces seems to be slow: just using the inlined version
 * of the inw() breaks things.
 */
void insw (unsigned long port, void *dst, unsigned long count)
{
	if (((unsigned long)dst) & 0x3) {
		if (((unsigned long)dst) & 0x1) {
			panic("insw: memory not short aligned");
		}
		if (!count)
			return;
		count--;
		*(unsigned short* ) dst = inw(port);
		((unsigned short *) dst)++;
	}

	while (count >= 2) {
		unsigned int w;
		count -= 2;
		w = inw(port);
		w |= inw(port) << 16;
		*(unsigned int *) dst = w;
		((unsigned int *) dst)++;
	}

	if (count) {
		*(unsigned short*) dst = inw(port);
	}
}


/*
 * Read COUNT 32-bit words from port PORT into memory starting at
 * SRC. Now works with any alignment in SRC. Performance is important,
 * but the interfaces seems to be slow: just using the inlined version
 * of the inl() breaks things.
 */
void insl (unsigned long port, void *dst, unsigned long count)
{
	unsigned int l = 0, l2;
	
	if (!count)
		return;
	
	switch (((unsigned long) dst) & 0x3)
	{
	 case 0x00:			/* Buffer 32-bit aligned */
		while (count--)
		{
			*(unsigned int *) dst = inl(port);
			((unsigned int *) dst)++;
		}
		break;
	
	/* Assuming little endian Alphas in cases 0x01 -- 0x03 ... */
	
	 case 0x02:			/* Buffer 16-bit aligned */
		--count;
		
		l = inl(port);
		*(unsigned short *) dst = l;
		((unsigned short *) dst)++;
		
		while (count--)
		{
			l2 = inl(port);
			*(unsigned int *) dst = l >> 16 | l2 << 16;
			((unsigned int *) dst)++;
			l = l2;
		}
		*(unsigned short *) dst = l >> 16;
		break;
	 case 0x01:			/* Buffer 8-bit aligned */
		--count;
		
		l = inl(port);
		*(unsigned char *) dst = l;
		((unsigned char *) dst)++;
		*(unsigned short *) dst = l >> 8;
		((unsigned short *) dst)++;
		while (count--)
		{
			l2 = inl(port);
			*(unsigned int *) dst = l >> 24 | l2 << 8;
			((unsigned int *) dst)++;
			l = l2;
		}
		*(unsigned char *) dst = l >> 24;
		break;
	 case 0x03:			/* Buffer 8-bit aligned */
		--count;
		
		l = inl(port);
		*(unsigned char *) dst = l;
		((unsigned char *) dst)++;
		while (count--)
		{
			l2 = inl(port);
			*(unsigned int *) dst = l << 24 | l2 >> 8;
			((unsigned int *) dst)++;
			l = l2;
		}
		*(unsigned short *) dst = l >> 8;
		((unsigned short *) dst)++;
		*(unsigned char *) dst = l >> 24;
		break;
	}
}


#if 0
/*
 * Like insb but in the opposite direction.
 * Don't worry as much about doing aligned memory transfers:
 * doing byte reads the "slow" way isn't nearly as slow as
 * doing byte writes the slow way (no r-m-w cycle).
 */
void outsb(unsigned long port, const void * src, unsigned long count)
{
	while (count) {
		count--;
		outb(*(char *)src, port);
		((char *) src)++;
	}
}
#endif

/*
 * Like insw but in the opposite direction.  This is used by the IDE
 * driver to write disk sectors.  Performance is important, but the
 * interfaces seems to be slow: just using the inlined version of the
 * outw() breaks things.
 */
void outsw (unsigned long port, const void *src, unsigned long count)
{
	if (((unsigned long)src) & 0x3) {
		if (((unsigned long)src) & 0x1) {
			panic("outsw: memory not short aligned");
		}
		outw(*(unsigned short*)src, port);
		((unsigned short *) src)++;
		--count;
	}

	while (count >= 2) {
		unsigned int w;
		count -= 2;
		w = *(unsigned int *) src;
		((unsigned int *) src)++;
		outw(w >>  0, port);
		outw(w >> 16, port);
	}

	if (count) {
		outw(*(unsigned short *) src, port);
	}
}


/*
 * Like insl but in the opposite direction.  This is used by the IDE
 * driver to write disk sectors.  Works with any alignment in SRC.
 *  Performance is important, but the interfaces seems to be slow:
 * just using the inlined version of the outl() breaks things.
 */
void outsl (unsigned long port, const void *src, unsigned long count)
{
	unsigned int l = 0, l2;
	
	if (!count)
		return;
	
	switch (((unsigned long) src) & 0x3)
	{
	 case 0x00:			/* Buffer 32-bit aligned */
		while (count--)
		{
			outl(*(unsigned int *) src, port);
			((unsigned int *) src)++;
		}
		break;
	
	/* Assuming little endian Alphas in cases 0x01 -- 0x03 ... */
	
	 case 0x02:			/* Buffer 16-bit aligned */
		--count;
		
		l = *(unsigned short *) src << 16;
		((unsigned short *) src)++;
		
		while (count--)
		{
			l2 = *(unsigned int *) src;
			((unsigned int *) src)++;
			outl (l >> 16 | l2 << 16, port);
			l = l2;
		}
		l2 = *(unsigned short *) src;
		outl (l >> 16 | l2 << 16, port);
		break;
	 case 0x01:			/* Buffer 8-bit aligned */
		--count;
		
		l  = *(unsigned char *) src << 8;
		((unsigned char *) src)++;
		l |= *(unsigned short *) src << 16;
		((unsigned short *) src)++;
		while (count--)
		{
			l2 = *(unsigned int *) src;
			((unsigned int *) src)++;
			outl (l >> 8 | l2 << 24, port);
			l = l2;
		}
		l2 = *(unsigned char *) src;
		outl (l >> 8 | l2 << 24, port);
		break;
	 case 0x03:			/* Buffer 8-bit aligned */
		--count;
		
		l  = *(unsigned char *) src << 24;
		((unsigned char *) src)++;
		while (count--)
		{
			l2 = *(unsigned int *) src;
			((unsigned int *) src)++;
			outl (l >> 24 | l2 << 8, port);
			l = l2;
		}
		l2  = *(unsigned short *) src;
		((unsigned short *) src)++;
		l2 |= *(unsigned char *) src << 16;
		outl (l >> 24 | l2 << 8, port);
		break;
	}
}

void panic(const char * fmt, ...)
{

    printk("panic called %s\n", fmt);

    halt();
}


void init_ide(void)
{
	int i;
	void InitRTC(void);
	static int inited=0;

	if (inited) return;

	InitRTC();
	{
		int ji = jiffies;
		int i;

		for (i=0; i< 0x100000; i++)
			continue;
		if (ji == jiffies) {
			printf_dbm("timer doesn't work\n");
			inited = 0;
			return;
		}
	}

	ide_init();

        for (i=0; i<MAX_HWIFS; i++)  {
                printk(" index=%d, present=%d, irq=%d, major=%d, io_base=%x\n",
                        ide_hwifs[i].index, ide_hwifs[i].present,
                        ide_hwifs[i].irq, ide_hwifs[i].major,
                        ide_hwifs[i].io_base);
#define D0 ide_hwifs[i].drives[0]
                if (ide_hwifs[i].drives[0].present)
                        printk("media %d id %s %d/%d/%d dma=%d mul=%d %d %d\n",
                                D0.media,D0.id->model, D0.head, D0.sect,
                                D0.cyl, D0.using_dma, D0.mult_count,
                                D0.id->multsect, D0.id->multsect_valid);
#define D1 ide_hwifs[i].drives[1]
                if (ide_hwifs[i].drives[1].present)
                        printk("media %d id %s %d/%d/%d dma=%d mul=%d %d %d\n",
                                D1.media,D1.id->model, D1.head, D1.sect,
                                D1.cyl, D1.using_dma, D1.mult_count,
                                D1.id->multsect, D1.id->multsect_valid);
        }

	inited = 1;
}


int read_ide(int dev, int sf, int ns, char *buf)
{
	struct request rq;
	int i;

	ide_init_drive_cmd(&rq);

	rq.rq_dev = MKDEV(ide_hwif_to_major[dev/2], (dev % 2) * 64);
	rq.cmd = READ;
	rq.nr_sectors = ns;
	rq.current_nr_sectors = ns;
	rq.sector = sf;
	rq.buffer = buf;

#if 0
	HWGROUP(&ide_hwifs[dev/2].drives[dev % 2])->rq = &rq;

	do_rw_disk(&ide_hwifs[dev/2].drives[dev % 2], &rq, sf);
	wait_intr(READ_INTR);
#else
	do_request(&ide_hwifs[dev/2], &rq);
#endif
DBG("wait\n");

}

int write_ide(int dev, int sf, int ns, char *buf)
{
	struct request rq;

	ide_init_drive_cmd(&rq);

	rq.rq_dev = MKDEV(ide_hwif_to_major[dev/2], (dev % 2) * 64);
	rq.cmd = WRITE;
	rq.nr_sectors = ns;
	rq.current_nr_sectors = ns;
	rq.sector = sf;
	rq.buffer = buf;

#if 0
	HWGROUP(&ide_hwifs[dev/2].drives[dev % 2])->rq = &rq;

	do_rw_disk(&ide_hwifs[dev/2].drives[dev % 2], &rq, sf);
	wait_intr(WRITE_INTR);
#else
	do_request(&ide_hwifs[dev/2], &rq);
#endif

}


/*
   drive = 0 : hda 
   drive = 1 : hdb
   drive = 2 : hdc 
   drive = 3 : hdd
	|
   drive = 7 : hdg
*/
void ide_test(int drive) 
{
	int buf[4096];
	int t, i, b, s;
	int seed;

	swpipl(0); /*for timer*/

	init_ide();

	read_ide(0, 100, 3, buf);
	read_ide(0, 103, 3, buf);
	read_ide(0, 106, 3, buf);
mobo_key();

	for (t=0; t<100; t++) {
#if 0
		seed = rpcc();
		srandom(seed);
		for (i=0; i<2048; i++) 
			buf[i] = random();
		write_ide(0, b=random()%2048, 2, buf);

		for (i=0; i<2048; i++) 
			buf[i] = 0;
#endif
		b = random()%2048; s=random()%8;
		if (b<0) b *= -1;
		if (s<0) s *= -1;
#ifdef DEBUG
printk("read b=%d, sz=%d ", b, s+1);
#endif
		read_ide(0, b, s+1, buf);
#if 0
		srandom(seed);
		for (i=0; i<1024; i++) 
			if (buf[i] != random())
				printk("IDE error %x\n", i);
#endif
	}

}


