/*****************************************************************************

       Copyright  1993, 1994 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

#include "lib.h"
#include "uilib.h"
#include "northbridge.h"

#include "kbd.h"
#include "kbdscan.h"


/* Internal function prototypes */
static void kbd_process(int scan_code);
static int kbd_84to101(int c);

/* Internal data */
static int prefix_flags;
static int ctrl_flags;
static int shift_flags;
static int scroll_lock;
static int num_lock;
static int caps_lock;
static int kbd_char;
static int kbd_type;


DBM_STATUS kbd_init(void)
{
    int d;
    int s;


    mobo_logf( LOG_INFO "KEYBOARD: Enable and test the keyboard\n" );
    kbd_ctl_cmd(KBD_CTL_ENABLE);

/*
   Flush pending input.
 */
    TRACE("Flush pending input.\n");
    d = 0;
    while (kbd_input() != -1) {
	/* Should be just a bit more than the Buffer size */
	if (d++ > 29) {	
	    mobo_logf( LOG_CRIT "KEYBOARD: buffer could not flush successfully\n");
	    return STATUS_FAILURE;	/* Give up eventually */
	}
    }
    TRACE("Flushed \"%d\" bytes out of Buffer.\n", d);

/*
   The keyboard is not initialized
   if the front panel lock switch is in the locked
   position. This lets you bring the console up on the
   serial port even if a keyboard is plugged in.
 */

#if 0
    if (((d = inportb(KBD_STAT)) & KBD_KBEN) == 0) {
        mobo_logf( LOG_CRIT "KEYBOARD: enable switch is disabled\n");
	return STATUS_FAILURE;
    }
    TRACE("kbd enable switch on\n");
#endif 

/*
   Global state.
 */
    prefix_flags = 0;
    ctrl_flags = 0;
    shift_flags = 0;
    caps_lock = FALSE;
    num_lock = FALSE;
    scroll_lock = FALSE;
    kbd_char = -1;
    kbd_type = 84;

/*
   Enable and test the keyboard
   interface in the VTI chip. This seems to
   be the only way to get it going.
 */
    TRACE("Enable and test the keyboard\n");
    kbd_ctl_cmd(KBD_CTL_ENABLE);
    usleep(300000);		/* Give the controller a chance to wake up */

#if 0
/*   Fails on the EB64.  Need to check this later. */
    TRACE("Load Mode register into buffer\n");
    kbd_ctl_cmd(KBD_CTL_RDMODE);
    TRACE("Read Mode register\n");
    s = kbd_input();
    TRACE("Mode Register = %x\n", s);
#endif

    d = 0;
    do {
	kbd_ctl_cmd(KBD_CTL_TEST1);
	s = kbd_input();
	TRACE("kbd_input() returns: %x\n", s);
	if (s == 0x55)
	    break;
	TRACE("keyboard test1 FAILED.\n");
	if (d++ > 10) {
	    mobo_logf( LOG_CRIT "KEYBOARD: self-test has failed!\n");
	    return STATUS_FAILURE;
	}
    } while (s != 0x55);
    TRACE("keyboard test1 Passed\n");

    d = 0;
    do {
	kbd_ctl_cmd(KBD_CTL_TEST2);
	s = kbd_input();
	TRACE("kbd_input() returns: %x\n", s);
	if (s == 0x00)
	    break;
	TRACE("keyboard test2 FAILED.\n");
	if (d++ > 10) {
	    mobo_logf( LOG_CRIT "KEYBOARD: interface test failed!\n");
	    return STATUS_FAILURE;
	}
    } while (s != 0x00);
    TRACE("keyboard test2 Passed\n");

/*
 *  we have to wake up the interface first before we can
 *  give the keyboard (p16) a jingle
 */

    TRACE("wake up the interface\n");
    while (kbd_input() != -1);
    if (kbd_locked()) {
	mobo_logf( LOG_WARN "KEYBOARD: is locked!\n");
	/* return STATUS_FAILURE; */		/* don't act on this yet */
    }
    TRACE("keyboard unlocked\n");

/*
   Flip off the HARD DISK light
   on the front panel. This is driven from
   an unused port on the 82C106.
 */
#if	1
    TRACE("Flip off the HARD DISK light\n");
    kbd_ctl_cmd(KBD_CTL_RDOUT);
    d = kbd_input();
    kbd_ctl_cmd(KBD_CTL_WROUT);
    kbd_ctl_output(d & (~HDLED));
#endif
/*
   Reset keyboard. If the read times out
   then the assumption is that no keyboard is
   plugged into the machine.
 */
    TRACE("Reset keyboard. If the read times out\n");
    if (kbd_output(KBD_RESET) != KBD_ACK) {
	mobo_logf( LOG_CRIT "KEYBOARD: didn't acknowledge to reset - plugged in OK?\n");
	return STATUS_FAILURE;
    }
    TRACE("Keyboard reset.\n");

    s = kbd_input();
    TRACE("kbd_input() returned: %x\n", s);
    if (s != 0xAA) {
	mobo_logf( LOG_CRIT "KEYBOARD: failed reset test\n");
	return STATUS_FAILURE;
    }
    TRACE("Keyboard reset test passed\n");

    kbd_ctl_cmd(KBD_CTL_RDMODE);
    s = kbd_input();
    kbd_ctl_cmd(KBD_CTL_WRMODE);
    s &= ~(KBD_EKI | KBD_EMI | KBD_SYS | KBD_DKB | KBD_DMS | KBD_KCC);
    kbd_ctl_output(s);

/*
   Assume that we have a 101 key keyboard which
   has powered up in mode 2 (the wrong mode). Switch it into
   mode 3. If we really have an 84 key keyboard then the
   first byte will be ACK'ed as a NOP, and the second byte will
   be RESEND'ed as garbage. On the 101 key keyboard we also
   need to reset the mode of the right ALT and CTRL keys.
 */
    if (kbd_output(KBD_SELECTCODE) != KBD_ACK) {
	mobo_logf( LOG_CRIT "KEYBOARD: didn't respond to mode-switching\n");
	return STATUS_FAILURE;
    }
    TRACE("Keyboard mode is correct\n");
    if (kbd_output_noverify(0x03) == KBD_ACK) {
	if (kbd_output(KBD_DEFAULTS) != KBD_ACK
	    || kbd_output(KBD_MAKEBREAK) != KBD_ACK
	    || kbd_output(0x39) != KBD_ACK
	    || kbd_output(KBD_MAKEBREAK) != KBD_ACK
	    || kbd_output(0x58) != KBD_ACK) {
	    mobo_logf( LOG_CRIT "KEYBOARD: not responding to configuration\n");
	    return STATUS_FAILURE;
	}
	kbd_type = 101;
    }
    if (kbd_output(KBD_ENABLE) != KBD_ACK)
    {
	mobo_logf( LOG_CRIT "KEYBOARD: not responding to enabling\n");
	return STATUS_FAILURE;
    }

    mobo_logf( LOG_INFO "KEYBOARD: Initialized successfully\n" );
    return STATUS_SUCCESS;
}


int kbd_charav(void)
{
    int s;
    int d;

    if (kbd_char >= 0)
	return (TRUE);
    s = inportb(KBD_STAT);
    if ((s & KBD_OBF) == 0)
	return (FALSE);
    d = inportb(KBD_DATA);
    if ((s & (KBD_PERR | KBD_GTO | KBD_MS_OBF)) != 0)
	return (FALSE);
    kbd_process(d);
    if (kbd_char >= 0)
	return (TRUE);
    return (FALSE);
}


/* changed by Stig - 23 August 1999 - returns -1 instantly if no key */

int kbd_getc(void)
{
    int c;
    int s, d;

    if ( kbd_char < 0 ) {			/* no key pre-fetched */

	s = inportb( KBD_STAT );
	if ((s & KBD_OBF) == 0)
				return -1;	/* no key buffered anywhere */

	d = inportb( KBD_DATA );		/* else fetch raw keycode */
	kbd_process( d );			/* sets kbd_char as output */
    }

    c = kbd_char;
    kbd_char = -1;				/* prepare state for next key */

    return (c);
}

static void kbd_process(int scan_code)
{
    int c;
    int prefix;

    if (scan_code == 0xE0) {
	prefix_flags |= E0SEEN;
	return;
    }
    if (scan_code == 0xF0) {
	prefix_flags |= F0SEEN;
	return;
    }
    prefix = prefix_flags;
    prefix_flags = 0;
    /* Macro key */
    if ((prefix & E0SEEN) != 0)
	return;
    if (kbd_type == 84)
	scan_code = kbd_84to101(scan_code);
    if (scan_code > 0x84)
	return;
    switch (xlate[scan_code].char_type) {
    case ASCII:
	if ((prefix & F0SEEN) != 0)
	    break;
	c = xlate[scan_code].unshifted;
	if (shift_flags != 0)
	    c = xlate[scan_code].shifted;
	if (ctrl_flags != 0) {
	    if (c == ' ')
		c = 0x00;
	    else if (c >= 'a' && c <= 'z')
		c -= 'a' - 1;
	    else if (c >= 0x40 && c <= 0x5F)
		c -= 0x40;
	    if (c == MCTRL('S')) {
		scroll_lock = TRUE;
		kbd_leds();
	    } else if (c == MCTRL('Q')) {
		scroll_lock = FALSE;
		kbd_leds();
	    }
	} else if (caps_lock != FALSE && c >= 'a' && c <= 'z')
	    c -= 0x20;
	kbd_char = c;
	break;

    case NUMPAD:
	if ((prefix & F0SEEN) != 0)
	    break;
	c = xlate[scan_code].unshifted;
	if (num_lock != FALSE)
	    c = xlate[scan_code].shifted;
	kbd_char = c;
	break;

    case FUNCTION:
	if ((prefix & F0SEEN) != 0)
	    break;
	/* ignore */
	break;

    case SCROLL:
	if ((prefix & F0SEEN) != 0)
	    break;
	scroll_lock = !scroll_lock;
	kbd_leds();
	if (scroll_lock != FALSE)
	    kbd_char = MCTRL('S');
	else
	    kbd_char = MCTRL('Q');
	break;

    case CAPS:
	if ((prefix & F0SEEN) != 0)
	    break;
	caps_lock = !caps_lock;
	kbd_leds();
	break;

    case NUMLOCK:
	if ((prefix & F0SEEN) != 0)
	    break;
	num_lock = !num_lock;
	kbd_leds();
	break;

    case RSHIFT:
	if ((prefix & F0SEEN) != 0)
	    shift_flags &= ~RFLAG;
	else
	    shift_flags |= RFLAG;
	break;

    case LSHIFT:
	if ((prefix & F0SEEN) != 0)
	    shift_flags &= ~LFLAG;
	else
	    shift_flags |= LFLAG;
	break;


    case RCTRL:
	if ((prefix & F0SEEN) != 0)
	    ctrl_flags &= ~RFLAG;
	else
	    ctrl_flags |= RFLAG;
	break;

    case LCTRL:
	if ((prefix & F0SEEN) != 0)
	    ctrl_flags &= ~LFLAG;
	else
	    ctrl_flags |= LFLAG;
	break;

    case ALT:
	/* ignore */
	break;
    }
}

static int kbd_84to101(int c)
{
    if (c == 0x01)
	c = 0x47;
    else if (c == 0x03)
	c = 0x27;
    else if (c == 0x04)
	c = 0x17;
    else if (c == 0x05)
	c = 0x07;
    else if (c == 0x06)
	c = 0x0F;
    else if (c == 0x09)
	c = 0x4F;
    else if (c == 0x0A)
	c = 0x3F;
    else if (c == 0x0B)
	c = 0x2F;
    else if (c == 0x0C)
	c = 0x1F;
    else if (c == 0x11)
	c = 0x19;
    else if (c == 0x14)
	c = 0x11;
    else if (c == 0x58)
	c = 0x14;
    else if (c == 0x5D)
	c = 0x5C;
    else if (c == 0x76)
	c = 0x08;
    else if (c == 0x77)
	c = 0x76;
    else if (c == 0x79)
	c = 0x7C;
    else if (c == 0x7B)
	c = 0x84;
    else if (c == 0x7C)
	c = 0x57;
    else if (c == 0x7E)
	c = 0x5F;
    else if (c == 0x83)
	c = 0x37;
    else if (c == 0x84)
	c = 0x00;
    return (c & 0x0FF);
}

void kbd_leds(void)
{
    int d;

    d = 0x00;
    if (scroll_lock != FALSE)
	d |= 0x01;
    if (num_lock != FALSE)
	d |= 0x02;
    if (caps_lock != FALSE)
	d |= 0x04;
    (void) kbd_output(KBD_SETLEDS);
    (void) kbd_output(d);
}

/*----------------------------------------------------------------------*/
/* is the keyboard ready for reading, writing, etc. ? */

#define KB_READY_TIMEOUT 1000

/* poll until there is space in the buffer for writing.
 * no writes may occur to either keyboard port until the input buffer 
 * is ready for it.  */

static int kb_ready_wr(void)
{
    int i, d;

    for (i = 0; i < KB_READY_TIMEOUT; i++) {
	usleep(1000);

	d = inportb( KBD_STAT ) & KBD_IBF;
	if ( d == 0 )		return 1;
    }

    /* timeout case */
    mobo_alertf( __FUNCTION__ , "Keyboard controller not accepting input" );
    return 0;
}

/* A read from port 0x60 is only valid when there is something there to be
 * read.
 */

static int kb_ready_rd(void)
{
    int i, d;

    for ( i=0; i < KB_READY_TIMEOUT; i++ ) {

	usleep(1000);

	d = inportb( KBD_STAT ) & KBD_OBF;
	if ( d == 1 )		return 1;	/* byte ready for collection */
    }

    /* timeout case - nothing available */
    return 0;
}


/* Empty the output buffer of any extraneous chaff */

static int kb_drain(void)
{
    int i, d;
    int dbyte;

    for ( i=0; i < KB_READY_TIMEOUT; i++ ) {

	usleep(1000);

	d = inportb( KBD_STAT ) & KBD_OBF;
	if ( d == 0 )		return i;		/* no more output */

	dbyte = inportb( KBD_DATA );			/* drain byte */
    }

    /* timeout case */
    mobo_alertf( __FUNCTION__, "Keyboard buffer overflowing with 0x%02X", d);
    return -1;
}

/*----------------------------------------------------------------------*/



#define MAX_RETRIES 10


void kbd_ctl_cmd(int c)
{
    if (!kb_ready_wr())		return;

    outportb(KBD_CMD, c);
}

void kbd_ctl_output(int c)
{
    if (!kb_ready_wr())		return;
    outportb(KBD_DATA, c);
}


int kbd_input(void)
{
    if ( kb_ready_rd() != 1)			return -1;
    return ( inportb( KBD_DATA ) & 0xFF );
}

int kbd_output(int c)
{
    int d;
    int n = MAX_RETRIES;	/* Resend this many times if necessary */
    do {
	if ( !kb_ready_wr() )	return -1;
	outportb(KBD_DATA, c);
	if ((d = kbd_input()) != KBD_RESEND)
	    return (d);
    } while (--n);
    return (d);
}

int kbd_output_noverify(int c)
{

    if ( !kb_ready_wr() )		return -1;
    outportb(KBD_DATA, c);
    return (kbd_input());
}


int kbd_locked(void)
{
    kbd_ctl_cmd(RDI);
    if ((kbd_input() & KKSW) == 0)
	return 1;
    return 0;
}



int kbd_get_status(void)
{
    ub status_reg;

    status_reg = inportb(KBD_STAT);

    return ((int) (status_reg & 0xff));
}

void kbd_enable_intrs(void)
{
    ub mode;

    kbd_ctl_cmd(KBD_CTL_RDMODE);
    mode = kbd_input();
    mode |= (KBD_EKI);		/* mouse too for now */
    kbd_ctl_cmd(KBD_CTL_WRMODE);
    kbd_output(mode);

}

void kbd_disable_intrs(void)
{
    ub mode;

    kbd_ctl_cmd(KBD_CTL_RDMODE);
    mode = kbd_input();
    mode &= ~(KBD_EKI);		/* mouse too for now */
    kbd_ctl_cmd(KBD_CTL_WRMODE);
    kbd_output(mode);
}



/*----------------------------------------------------------------------*/
/* Framework for Mouse */
/* This section is largely self-contained - the keyboard driver needs
 * to use similar access routines */


static void mouse_ctl( unsigned ctl_cmd )
{
    if (!kb_ready_wr())			return;
    outportb( KBD_CMD, ctl_cmd );
}


static void mouse_data( unsigned data_cmd )
{
    mouse_ctl( KBD_CTL_WRMOUSE );
    if ( !kb_ready_wr() )		return;
    outportb( KBD_DATA, data_cmd );
}


static const String msgs[] = { 	"No errors detected",
				"Clock line stuck low",
				"Clock line stuck high",
				"Data line stuck low",
				"Data line stuck high" };
#define OUTCOMES 4		/* max possible outcome */

static DBM_STATUS interface_test_analyse( int val, String *S )
{
    if ( (val >= 0) && (val <= OUTCOMES) )	*S = msgs[ val ];
    else					*S = "Unrecognised result!";

    if ( val == 0 )	return STATUS_SUCCESS;
    else		return STATUS_FAILURE;
}

static DBM_STATUS check_for_acknowledge( void )
{
    int rval = kbd_input();

    if ( rval == KBD_ACK )	return STATUS_SUCCESS;
    else {
	mobo_logf( LOG_WARN __FUNCTION__ ": got 0x%02X from kbd instead of 0x%02X\n",
			rval, KBD_ACK );
	return STATUS_FAILURE;
    }
}


#ifdef DEBUG_MOUSE
/* DEBUG */

static void showstate( void )					/* DEBUG */
{
    int rval;

    kb_drain();

    rval = inportb( 0x64 );
    mobo_logf( LOG_WARN __FUNCTION__ ": kbd controller says 0x%02X\n", rval );


    /* if we can, read the command byte - if IBF and OBF both clear */
    if ( ( rval & KBD_IBF ) == 0 )
    {
	kbd_ctl_cmd( KBD_CTL_RDMODE );
	rval = kbd_input( );
	mobo_logf( LOG_WARN __FUNCTION__ ": Command byte is 0x%02X\n", rval );
    }

    kbd_ctl_cmd(KBD_CTL_WRMODE);
    kbd_ctl_output( 0 );

    /* if we can, read the command byte - if IBF and OBF both clear */
    if ( ( rval & KBD_IBF ) == 0 )
    {
        kbd_ctl_cmd( KBD_CTL_RDMODE );
        rval = kbd_input( );
        mobo_logf( LOG_WARN __FUNCTION__ ": Command byte is 0x%02X\n", rval );
    }

}
#endif




DBM_STATUS mouse_init(void)
{
    int status;
    int retval;
    int try;
    String interp;
    DBM_STATUS sval;


    /*----------------------------------------------------------------------*/
    /* Reset */

    kb_drain();				/* clean the slate */

    mouse_data( MOUSE_RESET );		/* reset - leaves mouse disabled */
					/* returns no bytes */
    mouse_ctl( KBD_CTL_MOUSEEN );	/* enable mouse in controller */
					/* returns 1 byte? */
    kb_drain();			

    mouse_data( MOUSE_EN );		/* enable mouse */
					/* returns 1 byte? */
    kb_drain();			


    /*----------------------------------------------------------------------*/
    /* Configuration (I'm not sure if this is working...) */

    mouse_data( MOUSE_SAMPLE );
    if ( !kb_ready_wr() )
    {
	mobo_logf( LOG_CRIT "MOUSE: not responding to sample rate command\n" );
	return STATUS_FAILURE;
    }
    outportb( KBD_DATA, 100 );			/* 100 samples / sec */

    kb_drain();			

    mouse_data( MOUSE_RES );
    if ( !kb_ready_wr() )
    {
	mobo_logf( LOG_CRIT "MOUSE: not responding to resolution command\n" );
	return STATUS_FAILURE;
    }
    outportb( KBD_DATA, 3 );			/* 8 counts / mm */

    kb_drain();			

    mouse_data( MOUSE_SCALE21 );		/* 2:1 scaling */

					/* returns 1 byte */

    kb_drain();			


    /*----------------------------------------------------------------------*/
    /* Testing */

#define MOUSE_RETRIES 3
    for ( try = 0; try < MOUSE_RETRIES; try++ )
    { 
#ifdef DEBUG_MOUSE
	showstate();					/* DEBUG */
#endif
	mobo_logf( LOG_INFO "MOUSE: controller test...");
	mouse_ctl( KBD_CTL_TEST1 );
	retval = kbd_input( );

	if ( retval == KBD_CTL_TEST1_PASS ) {
	    mobo_logf( LOG_INFO "MOUSE: ...passed\n" );
	    break;
	} else {
	    mobo_logf( LOG_CRIT "MOUSE: ...failed! (returned 0x%02X, wanted 0x%02X)\n",
			retval, KBD_CTL_TEST1_PASS );
	    continue;
 	}
    }
    if ( retval != KBD_CTL_TEST1_PASS )			return STATUS_FAILURE;


#ifdef DEBUG_MOUSE
    showstate();					/* DEBUG */
#endif
    mobo_logf( LOG_INFO "MOUSE: interface test...");		/* mouse self-test */
    mouse_ctl( KBD_CTL_MTEST );	
    retval = kbd_input();
    sval = interface_test_analyse( retval, &interp );
    mobo_logf( "%s\n", interp );
    if ( sval != STATUS_SUCCESS )	return sval;	/* failed */


#if 0
/* Currently, these sections don't work - I get 0xFE from the mouse, 
 * signifying 'resend' due to error */

    for ( try = 0; try < MOUSE_RETRIES; try++ )
    {
	showstate();					/* DEBUG */
	mobo_logf( LOG_INFO "MOUSE: reading ID...\n");
	mouse_data( MOUSE_ID );
	msleep( 20 );				/* wait for mouse to respond */
	sval = check_for_acknowledge( );

	if ( sval == STATUS_SUCCESS ) {
	    mobo_logf( LOG_INFO "MOUSE: ...passed.\n" );
	    break;
	} else {
	    mobo_logf( LOG_CRIT "MOUSE: ...not responding - is it plugged in?\n");
	    continue;
	}
    }
    if ( sval != STATUS_SUCCESS )                 return sval;

    mobo_logf( LOG_INFO "   byte 1: 0x%02X", kbd_input() );
    mobo_logf( LOG_INFO "   byte 2: 0x%02X\n", kbd_input() );

    kb_drain();			


    for ( try = 0; try < MOUSE_RETRIES; try++ )
    {
	mobo_logf( LOG_INFO "MOUSE: reading info...\n");
	mouse_data( MOUSE_INFO );
	msleep( 20 );				/* wait for mouse to respond */
	sval = check_for_acknowledge( );

	if ( sval == STATUS_SUCCESS ) {
	    mobo_logf( LOG_INFO "MOUSE: ...passed.\n" );
	    break;
	} else { 
	    mobo_logf( LOG_CRIT "MOUSE: ...not responding - is it plugged in?\n");
	    continue;
	}
    }
    if ( sval != STATUS_SUCCESS )                 return sval;

    mobo_logf( LOG_INFO "   byte 1: 0x%02X", kbd_input() );
    mobo_logf( LOG_INFO "   byte 2: 0x%02X", kbd_input() );
    mobo_logf( LOG_INFO "   byte 3: 0x%02X\n", kbd_input() );

#endif				/* broken stuff */

    /*----------------------------------------------------------------------*/
    /* Cleanup */

    kb_drain();			
    return STATUS_SUCCESS;
}
